/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javax.sql;

import java.sql.NClob;
import java.sql.ResultSet;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.Ref;
import java.sql.SQLXML;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Map;
import java.io.InputStream;
import java.io.Reader;
import java.util.Calendar;
import java.math.BigDecimal;
import java.net.URL;

/**
 * A RowSet is an interface which provides access to data being sent from/to a
 * database and which extends the functionality of ResultSet into a form that
 * can be used as a JavaBeans component, perhaps being used in a visual
 * programming environment.
 * <p>
 * Facilities are provided for get/set of properties relating to the Database
 * and the SQL Command and for getting/setting data within the Rows represented
 * by the RowSet. The RowSet supports JavaBeans events so that other components
 * in an application can be informed when various changes happen to the RowSet,
 * such as changes in data values.
 * <p>
 * RowSet is implemented as a layer on top of the remainder of the JDBC API. A
 * RowSet may be <i>connected</i> where it maintains a connection to the
 * database throughout its lifecycle. A RowSet may be <i>disconnected</i> where
 * it establishes a connection to the database, gets data and then closes the
 * connection. Updates to a disconnected RowSet can be made and later send back
 * the changes to the database, but this requires the RowSet to first reconnect
 * to the database before the changes are sent back.
 * <p>
 * Disconnected RowSets may make use of RowSetReaders to populate the RowSet
 * with data, possibly from a non-relational database source. Disconnected
 * RowSets may also use RowSetWriters to send data back to the underlying data
 * store. There is considerable freedom in the way that RowSetReaders and
 * RowSetWriters are implemented to get and store data.
 */
public interface RowSet extends ResultSet {

    /**
     * Registers a supplied RowSetListener with this RowSet. Once registered,
     * the RowSetListener is notified of events generated by the RowSet.
     * 
     * @param theListener
     *            an object which implements the <code>rowSetListener</code>
     *            interface.
     */
    public void addRowSetListener(RowSetListener theListener);

    /**
     * Clears the parameters previously set for this RowSet.
     * <p>
     * Parameter values apply to repeated use of a RowSet object. Setting a new
     * value for a parameter clears its previous value.
     * <code>clearParameters</code> clears the values for all parameters with
     * one method call.
     * 
     * @throws SQLException
     *             if a problem occurs accessing the database
     */
    public void clearParameters() throws SQLException;

    /**
     * Fetches data for this RowSet. If successful, any existing data for the
     * RowSet is discarded and the metadata for the rowset is set.
     * <p>
     * Data is retrieved connects to the database and executes a Command. This
     * requires some or all of the following properties to be set: url, data
     * source name, user name, password, transaction isolation, type map ; plus
     * some or all of the properties: command, read only, maximum field size,
     * maximum rows, escape processing, and query timeout.
     * <p>
     * The RowSet may use a RowSetReader to access the database - in this case a
     * reader must be registered with the RowSet and the RowSet will then invoke
     * the <code>readData</code> method on the reader to fetch the data.
     * 
     * @throws SQLException
     *             if a problem occurs accessing the database or if the
     *             properties needed to access the database have not been set
     */
    public void execute() throws SQLException;

    /**
     * Gets the RowSet's Command property.
     * 
     * @return a string containing the RowSet's Command property - this is an
     *         SQL Query which can be executed to fetch data into the RowSet.
     */
    public String getCommand();

    /**
     * Gets the name of the datasource for this RowSet.
     * 
     * @return a String containing the name of the datasource.
     */
    public String getDataSourceName();

    /**
     * Reports if escape processing is enabled for this RowSet.
     * <p>
     * If <code>true</code> (the default) the driver will automatically
     * perform escape code processing on SQL statements prior to them being sent
     * to the database.
     * 
     * @return true if escape processing is enabled, false otherwise.
     * @throws SQLException
     *             if a problem occurs accessing the database
     */
    public boolean getEscapeProcessing() throws SQLException;

    /**
     * Gets the maximum number of bytes that can be returned for column values
     * which are of types BINARY, VARBINARY, LONGVARBINARYBINARY, CHAR, VARCHAR,
     * or LONGVARCHAR. Excess data is silently discarded if the number is
     * exceeded.
     * 
     * @return the current maximum size in bytes. 0 means no limit
     * @throws SQLException
     *             if a problem occurs accessing the database
     */
    public int getMaxFieldSize() throws SQLException;

    /**
     * Gets the maximum number of rows for this RowSet. Excess rows are
     * discarded silently if the limit is exceeded.
     * 
     * @return the previous maximum number of rows. 0 implies no limit.
     * @throws SQLException
     *             if a problem occurs accessing the database
     */
    public int getMaxRows() throws SQLException;

    /**
     * Gets the value of the password property for this RowSet. This property is
     * used when making a connection to the database and should be set before
     * invoking the <code>execute</code> method.
     * 
     * @return a String containing the value of the password property.
     */
    public String getPassword();

    /**
     * Gets the Timeout for the driver when executing a Query operation.
     * <p>
     * If a Query takes longer than the Timeout, an exception is thrown.
     * 
     * @return the Timeout value in seconds.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public int getQueryTimeout() throws SQLException;

    /**
     * Gets the transaction isolation property setting for this RowSet.
     * 
     * @return an integer holding the current transaction isolation setting. One
     *         of: one of Connection.TRANSACTION_READ_UNCOMMITTED,
     *         Connection.TRANSACTION_READ_COMMITTED,
     *         Connection.TRANSACTION_REPEATABLE_READ,
     *         Connection.TRANSACTION_SERIALIZABLE
     */
    public int getTransactionIsolation();

    /**
     * Gets the custom mapping of SQL types for this RowSet, if any.
     * 
     * @return a Map holding the custom mappings of SQL types to Java classes
     *         for this RowSet. By default, the Map is empty.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public Map<String, Class<?>> getTypeMap() throws SQLException;

    /**
     * Gets the URL property value for this RowSet. If there is no DataSource
     * object specified, the RowSet uses the URL to establish a connection to
     * the database. The default value for the URL is null.
     * 
     * @return a String holding the value of the URL property.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public String getUrl() throws SQLException;

    /**
     * Gets the value of the Username property for this RowSet. The Username is
     * used when establishing a connection to the database and should be set
     * before the <code>execute</code> method is invoked.
     * 
     * @return a String holing the value of the Username property.
     */
    public String getUsername();

    /**
     * Reports if this RowSet is read only.
     * 
     * @return true if this RowSet is read only, false if it is updateable.
     */
    public boolean isReadOnly();

    /**
     * Removes a specified RowSetListener object from the set of listeners which
     * will be notified of events by this RowSet.
     * 
     * @param theListener
     *            the RowSetListener to remove from the set of listeners for
     *            this RowSet.
     */
    public void removeRowSetListener(RowSetListener theListener);

    /**
     * Sets the specified ARRAY parameter in the RowSet command with the
     * supplied java.sql.Array value.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theArray
     *            the java.sql.Array value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setArray(int parameterIndex, Array theArray)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * ASCII data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInputStream
     *            an InputStream containing the ASCII data to set into the
     *            parameter value
     * @param length
     *            the length of the data in bytes
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setAsciiStream(int parameterIndex, InputStream theInputStream,
            int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * ASCII data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInputStream
     *            an InputStream containing the ASCII data to set into the
     *            parameter value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */

    public void setAsciiStream(int parameterIndex, InputStream theInputStream)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * ASCII data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theInputStream
     *            an InputStream containing the ASCII data to set into the
     *            parameter value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */

    public void setAsciiStream(String parameterName, InputStream theInputStream)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * ASCII data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theInputStream
     *            an InputStream containing the ASCII data to set into the
     *            parameter value
     * @param length
     *            the length of the data in bytes
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setAsciiStream(String parameterName,
            InputStream theInputStream, int length) throws SQLException;

    /**
     * Sets the value of the specified SQL NUMERIC parameter in the RowSet
     * command with the data in the supplied java.math.BigDecimal value.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theBigDecimal
     *            the BigDecimal containing the value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBigDecimal(int parameterIndex, BigDecimal theBigDecimal)
            throws SQLException;

    /**
     * Sets the value of the specified SQL NUMERIC parameter in the RowSet
     * command with the data in the supplied java.math.BigDecimal value.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theBigDecimal
     *            the BigDecimal containing the value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBigDecimal(String parameterName, BigDecimal theBigDecimal)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * binary data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @param length
     *            the length of the data in bytes
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBinaryStream(int parameterIndex, InputStream theInputStream,
            int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * binary data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBinaryStream(int parameterIndex, InputStream theInputStream)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * binary data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBinaryStream(String parameterName, InputStream theInputStream)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * binary data in the supplied java.io.InputStream value. Data is read from
     * the InputStream until end-of-file is reached.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @param length
     *            the length of the data in bytes
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBinaryStream(String parameterName,
            InputStream theInputStream, int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Blob.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theBlob
     *            the Blob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(int parameterIndex, Blob theBlob) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.InputStream. Data is read from the
     * InputStream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(int parameterIndex, InputStream theInputStream)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.InputStream. Data is read from the
     * InputStream until end-of-file is reached.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @param length
     *            the length of the data in bytes
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(int parameterIndex, InputStream theInputStream,
            long length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.InputStream. Data is read from the
     * InputStream until end-of-file is reached.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(String parameterName, InputStream theInputStream)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.InputStream. Data is read from the
     * InputStream until end-of-file is reached.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theInputStream
     *            an InputStream containing the binary data to set into the
     *            parameter value
     * @param length
     *            the length of the data in bytes
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(String parameterName, InputStream theInputStream,
            long length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Blob.
     * 
     * @param parameterName
     *            the name for parameter
     * @param theBlob
     *            the Blob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBlob(String parameterName, Blob theBlob) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied boolean.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theBoolean
     *            the boolean value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBoolean(int parameterIndex, boolean theBoolean)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied boolean.
     * 
     * @param parameterName
     *            name for parameter
     * @param theBoolean
     *            the boolean value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBoolean(String parameterName, boolean theBoolean)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied byte value.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theByte
     *            the byte value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setByte(int parameterIndex, byte theByte) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied byte value.
     * 
     * @param parameterName
     *            name for parameter
     * @param theByte
     *            the byte value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setByte(String parameterName, byte theByte) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied byte array value.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theByteArray
     *            the array of bytes to set into the parameter.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBytes(int parameterIndex, byte[] theByteArray)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied byte array value.
     * 
     * @param parameterName
     *            name for parameter
     * @param theByteArray
     *            the array of bytes to set into the parameter.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setBytes(String parametername, byte[] theByteArray)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * sequence of Unicode characters carried by the supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCharacterStream(int parameterIndex, Reader theReader,
            int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * sequence of Unicode characters carried by the supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCharacterStream(int parameterIndex, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * sequence of Unicode characters carried by the supplied java.io.Reader.
     * 
     * @param parameterName
     *            name for parameter
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCharacterStream(String parameterName, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * sequence of Unicode characters carried by the supplied java.io.Reader.
     * 
     * @param parameterName
     *            name for parameter
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCharacterStream(String parameterName, Reader theReader,
            int length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Clob.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theClob
     *            the Clob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(int parameterIndex, Clob theClob) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(int parameterIndex, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(int parameterIndex, Reader theReader, long length)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Clob.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param Clob
     *            the specific Clob object
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(String parameterName, Clob x) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterName
     *            name for parameter
     * @param theClob
     *            the Clob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(String parameterName, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setClob(String parameterName, Reader theReader, long length)
            throws SQLException;

    /**
     * Sets the Command property for this RowSet - the command is an SQL Query
     * which runs when the <code>execute</code> method is invoked. This
     * property is optional for datasources that do not support commands.
     * 
     * @param cmd
     *            a String containing the SQL Query. Can be null.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setCommand(String cmd) throws SQLException;

    /**
     * Sets the concurrency property of this RowSet. The default value is
     * ResultSet.CONCUR_READ_ONLY.
     * 
     * @param concurrency
     *            the new concurrency value - one of: ResultSet.CONCUR_READ_ONLY
     *            or ResultSet.CONCUR_UPDATABLE
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setConcurrency(int concurrency) throws SQLException;

    /**
     * Sets the Data Source Name property for the RowSet.
     * <p>
     * The Data Source Name can be used to find a <code>DataSource</code>
     * which has been registered with a naming service - the DataSource can then
     * be used to create a connection to the database.
     * 
     * @param name
     *            a String with the new Data Source Name.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDataSourceName(String name) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Date.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theDate
     *            the Date to use
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDate(int parameterIndex, Date theDate) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Date, where the conversion of the Date to an
     * SQL DATE value is calculated using a supplied Calendar.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theDate
     *            the Date to use
     * @param theCalendar
     *            the Calendar to use in converting the Date to an SQL DATE
     *            value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDate(int parameterIndex, Date theDate, Calendar theCalendar)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Date, where the conversion of the Date to an
     * SQL DATE value is calculated using a supplied Calendar.
     * 
     * @param parameterName
     *            name for parameter
     * @param theDate
     *            the Date to use
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDate(String parameterName, Date theDate) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Date, where the conversion of the Date to an
     * SQL DATE value is calculated using a supplied Calendar.
     * 
     * @param parameterName
     *            name for parameter
     * @param theDate
     *            the Date to use
     * @param theCalendar
     *            the Calendar to use in converting the Date to an SQL DATE
     *            value
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDate(String parameterName, Date theDate, Calendar theCalendar)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied double.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theDouble
     *            the double value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDouble(int parameterIndex, double theDouble)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied double.
     * 
     * @param parameterName
     *            name for parameter
     * @param theDouble
     *            the double value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setDouble(String parameterName, double theDouble)
            throws SQLException;

    /**
     * Sets the Escape Processing status for this RowSet. If escape processing
     * is on, the driver performs escape substitution before sending an SQL
     * command to the database. The default value for escape processing is on.
     * 
     * @param enable
     *            true to enable Escape Processing, false to turn it off.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setEscapeProcessing(boolean enable) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied float.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theFloat
     *            the float value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setFloat(int parameterIndex, float theFloat)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied float.
     * 
     * @param parameterName
     *            name for parameter
     * @param theFloat
     *            the float value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setFloat(String parameterName, float theFloat)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied integer.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theInteger
     *            the integer value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setInt(int parameterIndex, int theInteger) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied integer.
     * 
     * @param parameterName
     *            name for parameter
     * @param theInteger
     *            the integer value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setInt(String parameterName, int theInteger)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied long.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theLong
     *            the long value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setLong(int parameterIndex, long theLong) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * supplied long.
     * 
     * @param parameterName
     *            name for parameter
     * @param theLong
     *            the long value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setLong(String parameterName, long theLong) throws SQLException;

    /**
     * Sets the maximum number of bytes which can be returned for a column value
     * where the column type BINARY, VARBINARY, LONGVARBINARYBINARY, CHAR,
     * VARCHAR, or LONGVARCHAR. Data which exceeds this limit is silently
     * discarded. For portability, a value greater than 256 is recommended.
     * 
     * @param max
     *            the maximum size of the returned column value in bytes. 0
     *            means unlimited.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setMaxFieldSize(int max) throws SQLException;

    /**
     * Sets the maximum number of rows which can be held by the RowSet. Any
     * additional rows are silently discarded.
     * 
     * @param max
     *            the maximum number of rows which can be held in the RowSet. 0
     *            means no limit.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setMaxRows(int max) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNCharacterStream(int parameterIndex, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNCharacterStream(int parameterIndex, Reader theReader,
            long length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterName
     *            name for parameter
     * @param theClob
     *            the Clob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNCharacterStream(String parameterName, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNCharacterStream(String parameterName, Reader theReader,
            long length) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.Clob.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theClob
     *            the Clob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNClob(int parameterIndex, NClob theNClob)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNClob(int parameterIndex, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNClob(int parameterIndex, Reader theReader, long length)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.sql.NClob.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param value
     *            the given NClob object
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNClob(String parameterName, NClob value) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterName
     *            name for parameter
     * @param theNClob
     *            the NClob value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNClob(String parameterName, Reader theReader)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command with the
     * value of a supplied java.io.Reader.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theReader
     *            the Reader which contains the Unicode data to set into the
     *            parameter
     * @param length
     *            the length of the data in the Reader in characters
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNClob(String parameterName, Reader theReader, long length)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied NString
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theNString
     *            the NString value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNString(int parameterIndex, String theNString)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied NString.
     * 
     * @param parameterName
     *            name for parameter
     * @param theNString
     *            the NString value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNString(String parameterName, String theNString)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to SQL
     * NULL.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param sqlType
     *            the type of the parameter, as defined by java.sql.Types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNull(int parameterIndex, int sqlType) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to SQL
     * NULL. This form of the <code>setNull</code> method should be used for
     * User Defined Types and REF parameters.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param sqlType
     *            the type of the parameter, as defined by java.sql.Types.
     * @param typeName
     *            the fully qualified name of an SQL User Defined Type or the
     *            name of the SQL structured type referenced by a REF type.
     *            Ignored if the sqlType is not a UDT or REF type.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNull(int parameterIndex, int sqlType, String typeName)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to SQL
     * NULL. This form of the <code>setNull</code> method should be used for
     * User Defined Types and REF parameters.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param sqlType
     *            the type of the parameter, as defined by java.sql.Types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNull(String parameterName, int sqlType) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to SQL
     * NULL. This form of the <code>setNull</code> method should be used for
     * User Defined Types and REF parameters.
     * 
     * @param parameterName
     *            name for parameter
     * @param sqlType
     *            the type of the parameter, as defined by java.sql.Types.
     * @param typeName
     *            the fully qualified name of an SQL User Defined Type or the
     *            name of the SQL structured type referenced by a REF type.
     *            Ignored if the sqlType is not a UDT or REF type.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setNull(String parameterName, int sqlType, String typeName)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied Java object.
     * <p>
     * The JDBC specification provides a standard mapping for Java objects to
     * SQL data types. Database specific types can be mapped by JDBC driver
     * specific Java types.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theObject
     *            the Java object containing the data value.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(int parameterIndex, Object theObject)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied Java object.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theObject
     *            the Java object containing the data value.
     * @param targetSqlType
     *            the SQL type to send to the database, as defined in
     *            java.sql.Types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(int parameterIndex, Object theObject,
            int targetSqlType) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied Java object.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theObject
     *            the Java object containing the data value.
     * @param targetSqlType
     *            the SQL type to send to the database, as defined in
     *            java.sql.Types.
     * @param scale
     *            the number of digits after the decimal point, for
     *            java.sql.Types.DECIMAL and java.sql.Types.NUMERIC types.
     *            Ignored for all other types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(int parameterIndex, Object theObject,
            int targetSqlType, int scale) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied Java object.
     * 
     * @param parameterName
     *            name for parameter
     * @param theObject
     *            the Java object containing the data value.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(String parameterName, Object theObject)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied Java object.
     * 
     * @param parameterName
     *            name for parameter
     * @param theObject
     *            the Java object containing the data value.
     * @param targetSqlType
     *            the SQL type to send to the database, as defined in
     *            java.sql.Types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(String parameterName, Object theObject,
            int targetSqlType) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied Java object.
     * 
     * @param parameterName
     *            name for parameter
     * @param theObject
     *            the Java object containing the data value.
     * @param targetSqlType
     *            the SQL type to send to the database, as defined in
     *            java.sql.Types.
     * @param scale
     *            the number of digits after the decimal point, for
     *            java.sql.Types.DECIMAL and java.sql.Types.NUMERIC types.
     *            Ignored for all other types.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setObject(String parameterName, Object theObject,
            int targetSqlType, int scale) throws SQLException;

    /**
     * Sets the database Password for this RowSet.
     * 
     * @param password
     *            a string holding the new password
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setPassword(String password) throws SQLException;

    /**
     * Sets the Timeout value for this RowSet. The timeout is the maximum time
     * that the driver will wait while executing a command - after this time, an
     * SQLException is thrown.
     * 
     * @param seconds
     *            the number of seconds for the Timeout.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setQueryTimeout(int seconds) throws SQLException;

    /**
     * Sets whether the RowSet is read only or is updateable.
     * 
     * @param readOnly
     *            true to set the RowSet to readonly state, false to allow
     *            updates.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setReadOnly(boolean readOnly) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Ref. This is sent to the database as an SQL REF value.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theRef
     *            the Ref value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setRef(int parameterIndex, Ref theRef) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied short integer.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theShort
     *            the short value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setShort(int parameterIndex, short theShort)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied short integer.
     * 
     * @param parameterName
     *            name for parameter
     * @param theShort
     *            the short value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setShort(String parameterName, short theShort)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied String. The String is placed into the database as a VARCHAR or
     * LONGVARCHAR SQL value, depending on the database limits for the length of
     * VARCHAR values.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theString
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setString(int parameterIndex, String theString)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied String. The String is placed into the database as a VARCHAR or
     * LONGVARCHAR SQL value, depending on the database limits for the length of
     * VARCHAR values.
     * 
     * @param parameterName
     *            name for parameter
     * @param theString
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setString(String parameterName, String theString)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied RowId
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theRowId
     *            the RowId value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setRowId(int parameterIndex, RowId theRowId)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied RowId.
     * 
     * @param parameterName
     *            name for parameter
     * @param theRowId
     *            the RowId value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setRowId(String parameterName, RowId theRowId)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied SQLXML
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theSQLXML
     *            the SQLXML value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setSQLXML(int parameterIndex, SQLXML theSQLXML)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to the
     * supplied SQLXML.
     * 
     * @param parameterName
     *            name for parameter
     * @param theSQLXML
     *            the SQLXML value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setSQLXML(String parameterName, SQLXML theSQLXML)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Time, converting to an SQL TIME value using the system
     * default Calendar.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theTime
     *            the Time value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTime(int parameterIndex, Time theTime) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Time, converting to an SQL TIME value using a supplied
     * Calendar.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theTime
     *            the Time value to set
     * @param theCalendar
     *            the Calendar to use in the conversion operation
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTime(int parameterIndex, Time theTime, Calendar theCalendar)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Time, converting to an SQL TIME value using a supplied
     * Calendar.
     * 
     * @param parameterName
     *            name for parameter
     * @param theTime
     *            the Time value to set
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTime(String parameterName, Time theTime) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Time, converting to an SQL TIME value using a supplied
     * Calendar.
     * 
     * @param parameterName
     *            name for parameter
     * @param theTime
     *            the Time value to set
     * @param theCalendar
     *            the Calendar to use in the conversion operation
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTime(String parameterName, Time theTime, Calendar theCalendar)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Timestamp, converting to an SQL TIMESTAMP value using
     * the system default Calendar.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theTimestamp
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTimestamp(int parameterIndex, Timestamp theTimestamp)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Timestamp converting to an SQL TIMESTAMP value using a
     * supplied Calendar.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theTimestamp
     * @param theCalendar
     *            the Calendar to use in the conversion operation
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTimestamp(int parameterIndex, Timestamp theTimestamp,
            Calendar theCalendar) throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Timestamp converting to an SQL TIMESTAMP value using a
     * supplied Calendar.
     * 
     * @param parameterName
     *            name for parameter
     * @param theCalendar
     *            the Calendar to use in the conversion operation
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTimestamp(String parameterName, Timestamp theTimestamp)
            throws SQLException;

    /**
     * Sets the value of the specified parameter in the RowSet command to a
     * supplied java.sql.Timestamp converting to an SQL TIMESTAMP value using a
     * supplied Calendar.
     * 
     * @param parameterName
     *            name for parameter
     * @param theTimestamp
     * @param theCalendar
     *            the Calendar to use in the conversion operation
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTimestamp(String parameterName, Timestamp theTimestamp,
            Calendar theCalendar) throws SQLException;

    /**
     * Updates the target instance's transaction isolation level to one of a
     * discrete set of possible values.
     * 
     * @param level
     *            the new transaction isolation level. One of:
     *            Connection.TRANSACTION_READ_UNCOMMITTED,
     *            Connection.TRANSACTION_READ_COMMITTED,
     *            Connection.TRANSACTION_REPEATABLE_READ, or
     *            Connection.TRANSACTION_SERIALIZABLE
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTransactionIsolation(int level) throws SQLException;

    /**
     * Sets the type of this RowSet. By default, the type is non-scrollable.
     * 
     * @param type
     *            the new type for the RowSet. One of:
     *            ResultSet.TYPE_FORWARD_ONLY,
     *            ResultSet.TYPE_SCROLL_INSENSITIVE, or
     *            ResultSet.TYPE_SCROLL_SENSITIVE
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setType(int type) throws SQLException;

    /**
     * Sets the Map used to map SQL User Defined Types to Java classes.
     * 
     * @param theTypeMap
     *            a Map which defines the names of SQL UDTs and the Java classes
     *            to which they are mapped.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setTypeMap(Map<String, Class<?>> theTypeMap)
            throws SQLException;

    /**
     * Sets the URL used by this RowSet to access the database via a
     * <code>DriverManager</code>. The URL is optional - an alternative is to
     * use a Data Source Name to create a connection.
     * 
     * @param theURL
     *            a String containing the URL for the database. Can be null.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setUrl(String theURL) throws SQLException;

    /**
     * Sets the URL used by this RowSet to access the database via a
     * <code>DriverManager</code>. The URL is optional - an alternative is to
     * use a Data Source Name to create a connection.
     * 
     * @param parameterIndex
     *            index of the parameter to set, where the first parameter has
     *            index = 1.
     * @param theURL
     *            a java.net.URL containing the URL for the database.
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setURL(int parameterIndex, URL theURL) throws SQLException;

    /**
     * Sets the Username property for the RowSet, used to authenticate a
     * connection to the database.
     * 
     * @param theUsername
     *            a String containing the User Name
     * @throws SQLException
     *             if an error occurs accessing the database.
     */
    public void setUsername(String theUsername) throws SQLException;
}
