/*
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) Technology Edition, v6
 *  (C) Copyright IBM Corp. 2004, 2005. All Rights Reserved
 *
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 * ===========================================================================
 */

/*
 * @(#)UNIXProcess.java.linux	1.39 06/09/12
 *
 * Copyright 1995-2000 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 */

package java.lang;

import java.io.*;
/*Start 119100*/
import com.ibm.jvm.io.ConsoleInputStream;
import com.ibm.jvm.io.ConsolePrintStream;
import java.util.Vector;
import sun.security.action.GetPropertyAction;
import java.security.AccessController;
import java.util.StringTokenizer;
/*End 119100*/


/* java.lang.Process subclass in the UNIX environment.
 *
 * @author Mario Wolczko and Ross Knippel.
 * @author Konstantin Kladko (ported to Linux)
 */

final class UNIXProcess extends Process {
    private FileDescriptor stdin_fd;
    private FileDescriptor stdout_fd;
    private FileDescriptor stderr_fd;
    private int pid;
    private int exitcode;
    private boolean hasExited;

    private OutputStream stdin_stream;
    private InputStream  stdout_stream;
    private InputStream  stderr_stream;

    /* this is for the reaping thread */
    private native int waitForProcessExit(int pid);

    private native int forkAndExec(byte[] prog,
				   byte[] argBlock, int argc,
				   byte[] envBlock, int envc,
				   byte[] dir,
				   boolean redirectErrorStream,
				   FileDescriptor stdin_fd,
				   FileDescriptor stdout_fd,
				   FileDescriptor stderr_fd)
	throws IOException;

    /* In the process constructor we wait on this gate until the process    */
    /* has been created. Then we return from the constructor.               */
    /* fork() is called by the same thread which later waits for the process */
    /* to terminate */

    private static class Gate {

        private boolean exited = false;
        private IOException savedException;

        synchronized void exit() { /* Opens the gate */
           exited = true;
           this.notify();
        }

        synchronized void waitForExit() { /* wait until the gate is open */
            boolean interrupted = false;
            while (!exited) {
                try {
                    this.wait();
                } catch (InterruptedException e) {
                    interrupted = true;
                }
            }
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }

        void setException (IOException e) {
            savedException = e;
        }

        IOException getException() {
            return savedException;
        }
    }

    UNIXProcess(final byte[] prog,
		final byte[] argBlock, final int argc,
		final byte[] envBlock, final int envc,
		final byte[] dir,
		final boolean redirectErrorStream)
    throws IOException {
	stdin_fd  = new FileDescriptor();
	stdout_fd = new FileDescriptor();
	stderr_fd = new FileDescriptor();

        final UNIXProcess proc = this; /*119100*/
        final Gate gate = new Gate();
	/*
	 * For each subprocess forked a corresponding reaper thread
	 * is started.  That thread is the only thread which waits
	 * for the subprocess to terminate and it doesn't hold any
	 * locks while doing so.  This design allows waitFor() and
	 * exitStatus() to be safely executed in parallel (and they
	 * need no native code).
	 */

	java.security.AccessController.doPrivileged(
			    new java.security.PrivilegedAction() {
	    public Object run() {
		Thread t = new Thread("process reaper") {
		    public void run() {
                        try {
                            pid = forkAndExec(prog,
					      argBlock, argc,
					      envBlock, envc,
					      dir,
					      redirectErrorStream,
					      stdin_fd, stdout_fd, stderr_fd);
                        } catch (IOException e) {
                            gate.setException(e); /*remember to rethrow later*/
                            gate.exit();
                            return;
                        }
                        java.security.AccessController.doPrivileged(
                        new java.security.PrivilegedAction() {
                            public Object run() {
                            
                            try {
                            stdin_stream = new BufferedOutputStream(new
                                                    FileOutputStream(stdin_fd));
                            /*Start 119100*/
                            InputStream s = new FileInputStream(stdout_fd);
                            ProcessInputStream ps =
                                new ProcessInputStream(proc, s, "stdout reader "+proc); 
                            stdout_stream = ConsoleInputStream.localize(ps);
                            s = new FileInputStream(stderr_fd);
                            ps = new ProcessInputStream
                                        (proc, s, "stderr reader "+proc);
                            stderr_stream = ConsoleInputStream.localize(ps);
                            return null;
                            }
                            catch (IOException ie) {
                                try{
                                stdin_stream.close();
                                stdout_stream.close();
                                stderr_stream.close();
                                } catch (IOException ie1) {
                                   //ignore
                                }
                                gate.setException(ie);
                                return null;
                            }
                            /*End 119100*/
                        }
                        });
                        gate.exit(); /* exit from constructor */
			int res = waitForProcessExit(pid);
			synchronized (UNIXProcess.this) {
			    hasExited = true;
			    exitcode = res;
                            /*We can close stdin now that process has exited. */
                            try {
                               stdin_stream.close();
                            }
                            catch (IOException ie) { /*Ignore*/ }
			    UNIXProcess.this.notifyAll();
			}
		    }
		};
                t.setDaemon(true);
                t.start();
		return null;
	    }
	});
        gate.waitForExit();
        IOException e = gate.getException();
        if (e != null)
            throw new IOException(e.toString());
    }

    public OutputStream getOutputStream() {
	return stdin_stream;
    }

    public InputStream getInputStream() {
	return stdout_stream;
    }

    public InputStream getErrorStream() {
	return stderr_stream;
    }

    public synchronized int waitFor() throws InterruptedException {
        while (!hasExited) {
	    wait();
	}
	return exitcode;
    }

    public synchronized int exitValue() {
	if (!hasExited) {
	    throw new IllegalThreadStateException("process hasn't exited");
	}
	return exitcode;
    }

    private static native void destroyProcess(int pid);
    public void destroy() {
	// There is a risk that pid will be recycled, causing us to
	// kill the wrong process!  So we only terminate processes
	// that appear to still be running.  Even with this check,
	// there is an unavoidable race condition here, but the window
	// is very small, and OSes try hard to not recycle pids too
	// soon, so this is quite safe.
	synchronized (this) {
	    if (!hasExited)
		destroyProcess(pid);
	}
        try {
            stdin_stream.close();
            stdout_stream.close();
            stderr_stream.close();
        } catch (IOException e) {
            // ignore
        }
    }

    /* This routine initializes JNI field offsets for the class */
    private static native void initIDs();

    static {
	initIDs();
    }
}

/*Start 119100*/
/*********************************************************************/
/* A kind of PipedInputStream that won't block the associated        */
/* PipedOutputStream from writing when the internal buffer fills up. */
/* Here, we chain buffers for reading and tack a new one on the end  */
/* for writing.                                                      */
/*                                                                   */
/* It has a run() method b/c it's meant to be wrapped in a thread to */
/* read from a process's stdout/stderr and store in the buffer chain */
/* for the PipedInputStream (this)                                   */
/*********************************************************************/
class ProcessPipedInputStream extends PipedInputStream {
    byte[] writeBuf = null; // the current buffer where written bytes get put.
                            // reads come from the superclass's buffer[]. 
    boolean chaining;       // Whether our current write buffer is the
                            // superclass's write buffer - i.e., is the mbuf
                            // chain active? 
    int inPos;              // current write position - valid only if
                            // chaining==true
    Vector chain;           // Where we store the chain of mbuf's.  Operates as
                            // a queue - when the current read buffer gets
                            // depleted, we switch to the next one in the chain.
    
    public ProcessPipedInputStream(PipedOutputStream src) throws IOException {
        super(src);
        writeBuf = null;
        chaining = false;
        inPos = 0;
        chain = new Vector();
    }
    
    public ProcessPipedInputStream() {
        writeBuf = null;
        chaining = false;
        inPos = 0;
        chain = new Vector();
    }
    
    protected synchronized void receive(int b) throws IOException {
        if (chaining) {
            if (inPos == PIPE_SIZE) {
                /* current buffer full - chain it */
                writeBuf = new byte[PIPE_SIZE];
                chain.addElement(writeBuf);
                inPos = 0;
            }
            writeBuf[inPos++] = (byte)b;
        } else {
            super.receive(b);
            if (in == out) { /* now full - start chaining */
                inPos = 0;
                chaining = true;
                writeBuf = new byte[PIPE_SIZE];
                chain.addElement(writeBuf);
            }
        }
    }

    synchronized void receive(byte b[], int off, int len)  throws IOException { 
        for (int i=0; i<len; i++) {
            receive(b[off+i]);
        }
    }
    
    public synchronized int read() throws IOException {
        if (!chaining) {
            return super.read();
        }
        if (in == -1 && chain.size() != 0) {
            /* superclass buffer depleted - cycle to next in chain */
            buffer = (byte[]) chain.elementAt(0);
            chain.removeElementAt(0);
            in = out = 0;
            if (chain.size() == 0) {
                /* chain empty */
                if (inPos == 0) {
                    in = -1;
                } else if (inPos == PIPE_SIZE) {
                    in = 0;
                } else {
                    in = inPos;
                }
                chaining = false;
                /* set end of stream to last byte we wrote in last mbuf */
            }
        }
        return super.read();
    }

    public int available() throws IOException {
        int superAvailable = super.available();
        int chainSize = chain.size();
        if (chainSize == 0) {
            return superAvailable;
        }
        return superAvailable + PIPE_SIZE * chainSize - (PIPE_SIZE - inPos);
    }
}

/*********************************************************************/
/* The current problem with UNIXProcess is that the file descriptors */
/* associated with stdout and stderr must be closed when the process */
/* exits.  Ideally, we'd close these in the Process's finalize(),    */
/* but practice shows that the finalizer doesn't get run quickly     */
/* enough: under stress-testing we run out of file descriptors and   */
/* the whole * runtime dies (ugly).                                  */
/* Closing the fd's after the exec'd process exits creates the race  */
/* condition that the caller of exec() must read() the entire stream */
/* before exit, which doesn't work reliably.                         */
/* As a workaround, we create a thread each to read from             */
/* stdout/stderr and save the data in a buffer, and                  */
/* Process.getInputStream()/getErrorStream() read from these buffers.*/
/* It doesn't matter in this case that the fd's are closed.  The     */
/* process's output can be read long after it exits.  The code that  */
/* closes the streams is synchronized around the readers finishing.  */
/*********************************************************************/
class ProcessInputStream extends FilterInputStream implements Runnable {  

    ProcessPipedInputStream pipe;
    InputStream ins;             // the raw input stream of the process
    PipedOutputStream outs;      // where we put what the process is saying
    UNIXProcess p;               // the process we're associated with

    ProcessInputStream(UNIXProcess p, InputStream i, String name)
            throws IOException {
        super(i);
        outs = new PipedOutputStream();
        pipe = new ProcessPipedInputStream(outs);
        ins = i;
        this.p = p;

        Thread reader = new Thread(this, name); 
        reader.setDaemon(true); 
        reader.start();
    }

    public synchronized int read() throws IOException {
        return pipe.read();
    }

    public int read(byte b[]) throws IOException { 
        return pipe.read(b, 0, b.length); 
    } 

    public int read(byte b[], int off, int len) throws IOException { 
        return pipe.read(b, off, len); 
    } 

    public int available() throws IOException {
        return pipe.available();
    }

    public void close() throws IOException {                    /*ibm@84262*/
        insClose();                                             /*ibm@84262*/
        pipe.close();
    }

    private synchronized void insClose() throws IOException {   /*ibm@84262*/
        ins.close();                                            /*ibm@84262*/
    }                                                           /*ibm@84262*/

    public void run() {
        /* simply loop, reading from in and pushing to out, until in dies */
        byte[] buf = new byte[512];
        int    nread;
        while (true) {
            try {
                if ((nread = ins.read(buf)) < 0) break;
                pipe.receive(buf, 0, nread);
                synchronized (pipe) { 
                    pipe.notifyAll(); 
                } 
            } catch (IOException e) {
                break;
            }
        }
        try {
            outs.close();
        } catch (IOException e) {}
        try {
            insClose();   /*ibm@84262*/
        } catch (IOException e) {}
   }
}
/*End 119100*/

