

/*
 * ===========================================================================
 (C) Copyright Sun Microsystems Inc, 1992, 2004. All rights reserved.
 * ===========================================================================
 */


/*  
* =========================================================================== 
* Change activity: 
* 
* Reason  Date    Origin    Description 
* ------  ------  ------    -------------------------------------------------
* 079344  101104  cwhite    Merge to 5.0
* 084262  220305  stalleyj  Fix race condition is ProcessInputStream
* 87093   050505  cwhite    Don't call System.getProperty directly
*       
* =========================================================================== 
*/ 

package java.lang;

import java.io.*;
import com.ibm.jvm.io.ConsoleInputStream;
import com.ibm.jvm.io.ConsolePrintStream;
import java.util.Vector;
import sun.security.action.GetPropertyAction;
import java.security.AccessController;
import java.util.StringTokenizer;

/* java.lang.Process subclass in the UNIX environment.
 *
 * @author Mario Wolczko and Ross Knippel.
 * @author Konstantin Kladko (ported to Linux)
 */

final class ISERIESProcess extends Process {
    private int pid;
    private int exitcode;
    private boolean hasExited;

    private OutputStream stdin_stream;
    private InputStream  stdout_stream;
    private InputStream  stderr_stream;

    private SystemPipe stdin_pipe;
    private SystemPipe stdout_pipe;
    private SystemPipe stderr_pipe;
   
    private static native void destroyProcess(int pid);

    private native int forkAndExec(byte[] prog,
                                   byte[] argBlock, int argc,
                                   byte[] envBlock, int envc,
                                   byte[] dir,
                                   FileDescriptor c_stdin_fd, 
                                   FileDescriptor c_stdout_fd, 
                                   FileDescriptor c_stderr_fd)
        throws java.io.IOException; 
    
    private native int getPipeFDs(FileDescriptor p_stdin_fd, 
                                  FileDescriptor c_stdin_fd, 
                                  FileDescriptor p_stdout_fd, 
                                  FileDescriptor c_stdout_fd, 
                                  FileDescriptor p_stderr_fd, 
                                  FileDescriptor c_stderr_fd)
        throws java.io.IOException; 

    private native int statExecutable(byte []filename_bytes);
    
    private static native int setup();

    private static native int initi5mapping();
    
    /* this is for the reaping thread */
    private native int waitForProcessExit(int pid);

    static String platformEncoding;
    
    static boolean forkFromReaperThread;

    /* Perform initialisation */
    static {
        platformEncoding = (String)AccessController.doPrivileged                /*ibm@87093...*/
                               (new GetPropertyAction("ibm.system.encoding"));
        if (platformEncoding == null) {
            platformEncoding = (String)AccessController.doPrivileged
                                   (new GetPropertyAction("console.encoding"));
        }
        if (platformEncoding == null) {
            platformEncoding = (String)AccessController.doPrivileged
                                   (new GetPropertyAction("file.encoding"));    /*...ibm@87093*/
        }
	
        /* Note that due to the threading model for Linux the reaper thread */
        /* can only wait on the child process if it was the one that forked */
        /* the process. However for z/OS the parent thread must create the  */
        /* process (for security reasons).                                  */
        String osName = (String)AccessController.doPrivileged                   /*ibm@87093*/
                                    (new GetPropertyAction("os.name"));         /*ibm@87093*/
        if (osName.equals("Linux")) {                                           /*ibm@87093*/
            forkFromReaperThread = true;
        } else {
            forkFromReaperThread = false;
        }

        setup();
        initi5mapping();
    } 

    /* In the process constructor we wait on this gate until the process     */
    /* has been created. Then we return from the constructor.                */
    /* fork() is called by the same thread which later waits for the process */
    /* to terminate                                                          */
    private static class Gate {

        private boolean exited = false;
        private IOException savedException;

        synchronized void exit() { /* Opens the gate */
           exited = true;
           this.notify();
        }

        synchronized void waitForExit() { /* wait until the gate is open */
            boolean interrupted = false;
            while (!exited) {
                try {
                    this.wait();
                } catch (InterruptedException e) {
                    interrupted = true;
                }
            }
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }

        void setException (IOException e) {
            savedException = e;
        }

        IOException getException() {
            return savedException;
        }
    }

    /*
     * Constructor for a ISERIESProcess. Creates the child process, sets up the
     * I/O streams for child process communication, and creates a reaper
     * thread to monitor child termination.
     */
    ISERIESProcess(final byte[] prog,
       		final byte[] argBlock, final int argc,
		final byte[] envBlock, final int envc,
		final byte[] dir,
                final boolean redirectErrorStream) throws IOException {	

        /* Convert passed program name to its fullpath name */
        final byte fullprog[] = fullPath(prog);

        /* Open pipes for process stdin, stdout, and stderr */
        stdin_pipe = new SystemPipe();
        stdout_pipe = new SystemPipe();
        if (redirectErrorStream) {
            stderr_pipe = stdout_pipe;
        } else {
            stderr_pipe = new SystemPipe();
        }

        final ISERIESProcess proc = this;
        final Gate gate = new Gate();

        /*
         * Setup the Input and Output streams for child process I/O
         */
        java.security.AccessController.doPrivileged(
                                    new java.security.PrivilegedAction() {
            public Object run() {
                stdin_stream = ConsolePrintStream.localize(
                    new BufferedOutputStream(
                        new FileOutputStream(stdin_pipe.getInfd())));
                try {
                    /* Create and InputStream for the child process's stdout */
                    /* note that this localized in the same way as System.in */
                    /* and hence can be treated similarly                    */
                    InputStream s = new FileInputStream(stdout_pipe.getOutfd());
                    ISProcessInputStream ps = 
                        new ISProcessInputStream(proc, s, "stdout reader "+proc); 
                    stdout_stream = ConsoleInputStream.localize(ps);

                    /* Create and InputStream for the child process's stderr */
                    if (redirectErrorStream) {
                        stderr_stream = stdout_stream;
                    } else {
                        s = new FileInputStream(stderr_pipe.getOutfd());
                        ps = new ISProcessInputStream
                                    (proc, s, "stderr reader "+proc);
                        stderr_stream = ConsoleInputStream.localize(ps);
                    }

                    if (!forkFromReaperThread) {
                        /* fork and exec the child process */
                        pid = forkAndExec(fullprog, argBlock, argc,
                                          envBlock, envc, dir,
                                          stdin_pipe.getOutfd(),
                                          stdout_pipe.getInfd(),
                                          stderr_pipe.getInfd());
                    }
                } catch (IOException e) {
                    stdin_pipe.close();
                    stdout_pipe.close();
                    stderr_pipe.close();
                    gate.setException(e);
                }
                return null;
            }
        });
        
        /* If streams were not setup correctly don't continue */
        IOException e = gate.getException();
        if (e != null) throw new IOException(e.toString());

        /*
         * For each subprocess forked a corresponding reaper thread
         * is started.  That thread is the only thread which waits
         * for the subprocess to terminate and it doesn't hold any
         * locks while doing so.  This design allows waitFor() and
         * exitStatus() to be safely executed in parallel (and they
         * need no native code).
         *
         * Note that for Linux we must call forkAndExec from the
         * reaper thread so that waitForProcessExit can see the
         * termination of the child (via waitpid)
         */
        java.security.AccessController.doPrivileged(
                                    new java.security.PrivilegedAction() {
            public Object run() {
		Thread t = new Thread("process reaper "+proc) {
		    public void run() {
                        if (forkFromReaperThread) {
                          try {
                            /* fork and exec the child process */
                            pid = forkAndExec(fullprog, argBlock, argc,
                                              envBlock, envc, dir,
                                              stdin_pipe.getOutfd(),
                                              stdout_pipe.getInfd(),
                                              stderr_pipe.getInfd());

                          } catch (IOException e) {
                            stdin_pipe.close();
                            stdout_pipe.close();
                            stderr_pipe.close();
                            gate.setException(e);
                          }
                        }

                        gate.exit(); /* exit from constructor */
                        int res = waitForProcessExit(pid);
                        synchronized (ISERIESProcess.this) {
                            hasExited = true;
                            exitcode = res;
                            try {
                                stdin_stream.close();
                            } catch (IOException e) {} 
                            ISERIESProcess.this.notifyAll();
                        }
                    }
                };
                t.setDaemon(true);
                t.start();                    
                return null;                
            }
        });

        /* Wait for the reaper thread to start and return any exception */
        /* that may have occurred                                       */
        gate.waitForExit();
        e = gate.getException();
        if (e != null) throw new IOException(e.toString());
    }

    public OutputStream getOutputStream() {
	return stdin_stream;
    }

    public InputStream getInputStream() {
        return stdout_stream;
    }

    public InputStream getErrorStream() {
        return stderr_stream;
    }

    public synchronized int waitFor() throws InterruptedException {
        while (!hasExited) {
	    wait();
	}
	return exitcode;
    }

    public synchronized int exitValue() {
	if (!hasExited) {
	    throw new IllegalThreadStateException("process hasn't exited");
	}
	return exitcode;
    }

    public void destroy() {
	destroyProcess(pid);
    }

    /**
     * Returns the full path name for the specified filename. 
     */
    public byte[] fullPath(byte []filename_bytes) throws IOException {
        String sep = (String)AccessController.doPrivileged                      /*ibm@87093*/
                         (new GetPropertyAction("file.separator"));             /*ibm@87093*/
        String PATH = ProcessEnvironment.getenv("PATH");
        String ps = (String)AccessController.doPrivileged                       /*ibm@87093*/
                         (new GetPropertyAction("path.separator"));             /*ibm@87093*/
        String fullfilename = null;
        byte fullfilename_bytes[] = null;
        
        String filename;
        try {
            filename = new String(filename_bytes,platformEncoding);
        } catch (UnsupportedEncodingException e) {
            filename = new String(filename_bytes);
        }

        /* If the filename we want to exec has any slashes in it then  */
        /* we shouldn't do a path search, as in /foo ./foo or foo/bar. */
        if ((filename.indexOf(sep) < 0) && PATH != null) { 
            StringTokenizer st = new StringTokenizer(PATH,ps); 
            boolean foundit = false; 
            while (st.hasMoreTokens() && !foundit) {
                fullfilename = st.nextToken() + sep + filename;
                try {
                    fullfilename_bytes =
                        fullfilename.getBytes(platformEncoding);
                } catch (UnsupportedEncodingException e) {
                    fullfilename_bytes = fullfilename.getBytes();
                }
                if ( statExecutable(fullfilename_bytes) == 0) {
                    foundit = true;
                }
            }
            if (!foundit) throw new IOException(filename+ ": not found");
        } else {
            int status = statExecutable(filename_bytes);
            switch (status) {
                case -1: throw new IOException(filename+ ": not found");
                case -2: throw new IOException(filename+ ": cannot execute");
            }
            fullfilename_bytes = filename_bytes;
        }

        return fullfilename_bytes;
    }
}

/*********************************************************************/
/* A kind of PipedInputStream that won't block the associated        */
/* PipedOutputStream from writing when the internal buffer fills up. */
/* Here, we chain buffers for reading and tack a new one on the end  */
/* for writing.                                                      */
/*                                                                   */
/* It has a run() method b/c it's meant to be wrapped in a thread to */
/* read from a process's stdout/stderr and store in the buffer chain */
/* for the PipedInputStream (this)                                   */
/*********************************************************************/
class ISProcessPipedInputStream extends PipedInputStream {
    byte[] writeBuf = null; // the current buffer where written bytes get put.
                            // reads come from the superclass's buffer[]. 
    boolean chaining;       // Whether our current write buffer is the
                            // superclass's write buffer - i.e., is the mbuf
                            // chain active? 
    int inPos;              // current write position - valid only if
                            // chaining==true
    Vector chain;           // Where we store the chain of mbuf's.  Operates as
                            // a queue - when the current read buffer gets
                            // depleted, we switch to the next one in the chain.
    
    public ISProcessPipedInputStream(PipedOutputStream src) throws IOException {
        super(src);
        writeBuf = null;
        chaining = false;
        inPos = 0;
        chain = new Vector();
    }
    
    public ISProcessPipedInputStream() {
        writeBuf = null;
        chaining = false;
        inPos = 0;
        chain = new Vector();
    }
    
    protected synchronized void receive(int b) throws IOException {
        if (chaining) {
            if (inPos == PIPE_SIZE) {
                /* current buffer full - chain it */
                writeBuf = new byte[PIPE_SIZE];
                chain.addElement(writeBuf);
                inPos = 0;
            }
            writeBuf[inPos++] = (byte)b;
        } else {
            super.receive(b);
            if (in == out) { /* now full - start chaining */
                inPos = 0;
                chaining = true;
                writeBuf = new byte[PIPE_SIZE];
                chain.addElement(writeBuf);
            }
        }
    }

    synchronized void receive(byte b[], int off, int len)  throws IOException { 
        for (int i=0; i<len; i++) {
            receive(b[off+i]);
        }
    }
    
    public synchronized int read() throws IOException {
        if (!chaining) {
            return super.read();
        }
        if (in == -1 && chain.size() != 0) {
            /* superclass buffer depleted - cycle to next in chain */
            buffer = (byte[]) chain.elementAt(0);
            chain.removeElementAt(0);
            in = out = 0;
            if (chain.size() == 0) {
                /* chain empty */
                if (inPos == 0) {
                    in = -1;
                } else if (inPos == PIPE_SIZE) {
                    in = 0;
                } else {
                    in = inPos;
                }
                chaining = false;
                /* set end of stream to last byte we wrote in last mbuf */
            }
        }
        return super.read();
    }

    public int available() throws IOException {
        int superAvailable = super.available();
        int chainSize = chain.size();
        if (chainSize == 0) {
            return superAvailable;
        }
        return superAvailable + PIPE_SIZE * chainSize - (PIPE_SIZE - inPos);
    }
}

/*********************************************************************/
/* The current problem with ISERIESProcess is that the file descriptors */
/* associated with stdout and stderr must be closed when the process */
/* exits.  Ideally, we'd close these in the Process's finalize(),    */
/* but practice shows that the finalizer doesn't get run quickly     */
/* enough: under stress-testing we run out of file descriptors and   */
/* the whole * runtime dies (ugly).                                  */
/* Closing the fd's after the exec'd process exits creates the race  */
/* condition that the caller of exec() must read() the entire stream */
/* before exit, which doesn't work reliably.                         */
/* As a workaround, we create a thread each to read from             */
/* stdout/stderr and save the data in a buffer, and                  */
/* Process.getInputStream()/getErrorStream() read from these buffers.*/
/* It doesn't matter in this case that the fd's are closed.  The     */
/* process's output can be read long after it exits.  The code that  */
/* closes the streams is synchronized around the readers finishing.  */
/*********************************************************************/
class ISProcessInputStream extends FilterInputStream implements Runnable {  

    ISProcessPipedInputStream pipe;
    InputStream ins;             // the raw input stream of the process
    PipedOutputStream outs;      // where we put what the process is saying
    ISERIESProcess p;               // the process we're associated with

    ISProcessInputStream(ISERIESProcess p, InputStream i, String name)
            throws IOException {
        super(i);
        outs = new PipedOutputStream();
        pipe = new ISProcessPipedInputStream(outs);
        ins = i;
        this.p = p;

        Thread reader = new Thread(this, name); 
        reader.setDaemon(true); 
        reader.start();
    }

    public synchronized int read() throws IOException {
        return pipe.read();
    }

    public int read(byte b[]) throws IOException { 
        return pipe.read(b, 0, b.length); 
    } 

    public int read(byte b[], int off, int len) throws IOException { 
        return pipe.read(b, off, len); 
    } 

    public int available() throws IOException {
        return pipe.available();
    }

    public void close() throws IOException {                    /*ibm@84262*/
        insClose();                                             /*ibm@84262*/
        pipe.close();
    }

    private synchronized void insClose() throws IOException {   /*ibm@84262*/
        ins.close();                                            /*ibm@84262*/
    }                                                           /*ibm@84262*/

    public void run() {
        /* simply loop, reading from in and pushing to out, until in dies */
        byte[] buf = new byte[512];
        int    nread;
        while (true) {
            try {
                if ((nread = ins.read(buf)) < 0) break;
                pipe.receive(buf, 0, nread);
                synchronized (pipe) { 
                    pipe.notifyAll(); 
                } 
            } catch (IOException e) {
                break;
            }
        }
        try {
            outs.close();
        } catch (IOException e) {}
        try {
            insClose();   /*ibm@84262*/
        } catch (IOException e) {}
   }
}

/*********************************************/
/* A class to create a system pipe and allow */
/* access to its the file descriptors        */
/*********************************************/
class SystemPipe {
    private FileDescriptor in;
    private FileDescriptor out;

    private native void open(FileDescriptor in, FileDescriptor out)
                            throws java.io.IOException;
    private native void close(FileDescriptor fd);
    
    public SystemPipe() throws java.io.IOException {
        in  = new FileDescriptor();
        out = new FileDescriptor();
        open(in,out);
    }

    public void close() {
        closeInfd();
        closeOutfd();
    }

    public void closeInfd() {
        if (in != null) {
            close(in);
            in = null;
        }
    }
    
    public void closeOutfd() {
        if (out != null) {
            close(out);
            out = null;
        }
    }

    public FileDescriptor getInfd() {
        return in;
    }

    public FileDescriptor getOutfd() {
        return out;
    }
}
