/*
 * @(#)src/classes/sov/javax/swing/Popup.java, swing, asdev, 20051029 1.25
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v5.0
 * (C) Copyright IBM Corp. 1998, 2005. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 (C) Copyright Sun Microsystems Inc, 1992, 2004. All rights reserved.
 * ===========================================================================
 */

/* 
 *
 * Change activity:
 *
 * Reason  Date   Origin    Description
 * ------  ----   ------    -------------------------------------------------- 
 * 56756  031202  mchapman  Scrolling menus feature for 1.4.1
 * 71872  140404  chenerys  Handle submenus properly in fix 56756
 * 71873  230404  chenerys  Handle not going off bottom of screen in fix 56756
 * 71875  150404  odamg     Add default color settings for PopupScrollButton
 * 72122  260404  chenerys  Extended fix 56756 to some large non-JMenu popups
 * 73375  020604  odamg     Move incorrectly placed closing bracket
 * 74556  090604  jenningm  Back out 73375 & 72122
 * 74904  270904  odamg		Upgrade to 5.0 Swing 
 * 80708  061204 chenerys  Fix popup's scroll buttons
 * 94966  200905  jenningm  Fix size of JToolTip
 * 97486  111105  rlee      Undo 94966
 * 96155.1 021105 corbin     Add an option to display menus without scrollbars.
 *                          This is for a workaround due to a problem with the
 *                          JAWS screenreader software not working correctly
 *                          with scrollbars and menus
 *
 *
 * ===========================================================================
 * Module Information:
 * 
 * DESCRIPTION: IBM.WRITEME
 * ===========================================================================
 */

/*
 * @(#)Popup.java	1.17 03/12/19
 *
 */

package javax.swing;

import java.awt.*;
import sun.awt.ModalExclude;

import javax.swing.event.ChangeListener;                       /* ibm@56756 */
import javax.swing.event.ChangeEvent;                          /* ibm@56756 */
import java.awt.event.*;                                       /* ibm@56756 */

import java.security.AccessController;          //ibm.96155.1
import java.security.PrivilegedAction;          //ibm.96155.1
/**
 * Popups are used to display a <code>Component</code> to the user, typically
 * on top of all the other <code>Component</code>s in a particular containment
 * hierarchy. <code>Popup</code>s have a very small life cycle. Once you
 * have obtained a <code>Popup</code>, and hidden it (invoked the
 * <code>hide</code> method), you should no longer
 * invoke any methods on it. This allows the <code>PopupFactory</code> to cache
 * <code>Popup</code>s for later use.
 * <p>
 * The general contract is that if you need to change the size of the
 * <code>Component</code>, or location of the <code>Popup</code>, you should
 * obtain a new <code>Popup</code>.
 * <p>
 * <code>Popup</code> does not descend from <code>Component</code>, rather
 * implementations of <code>Popup</code> are responsible for creating
 * and maintaining their own <code>Component</code>s to render the
 * requested <code>Component</code> to the user.
 * <p>
 * You typically do not explicitly create an instance of <code>Popup</code>,
 * instead obtain one from a <code>PopupFactory</code>.
 *
 * @see PopupFactory
 *
 * @version 1.17 12/19/03
 * @since 1.4
 */
public class Popup {
    /**
     * The Component representing the Popup.
     */
    private Component component;

    /**
     * Creates a <code>Popup</code> for the Component <code>owner</code>
     * containing the Component <code>contents</code>. <code>owner</code>
     * is used to determine which <code>Window</code> the new
     * <code>Popup</code> will parent the <code>Component</code> the
     * <code>Popup</code> creates to.
     * A null <code>owner</code> implies there is no valid parent.
     * <code>x</code> and
     * <code>y</code> specify the preferred initial location to place
     * the <code>Popup</code> at. Based on screen size, or other paramaters,
     * the <code>Popup</code> may not display at <code>x</code> and
     * <code>y</code>.
     *
     * @param owner    Component mouse coordinates are relative to, may be null
     * @param contents Contents of the Popup
     * @param x        Initial x screen coordinate
     * @param y        Initial y screen coordinate
     * @exception IllegalArgumentException if contents is null
     */
    protected Popup(Component owner, Component contents, int x, int y) {
        this();
        if (contents == null) {
            throw new IllegalArgumentException("Contents must be non-null");
        }
        reset(owner, contents, x, y);
    }

    //ibm.96155.1 starts
    static final boolean longMenus;
    static {
        PrivilegedAction pa = new sun.security.action.GetPropertyAction("ibm.swing.longMenu");
        String useLongMenu = (String)AccessController.doPrivileged(pa);
        longMenus = useLongMenu != null;
    }
    //ibm.96155.1 ends
    /**
     * Creates a <code>Popup</code>. This is provided for subclasses.
     */
    protected Popup() {
    }

    /**
     * Makes the <code>Popup</code> visible. If the <code>Popup</code> is
     * currently visible, this has no effect.
     */
    public void show() {
        Component component = getComponent();

        if (component != null) {
            component.show();
        }
    }

    /**
     * Hides and disposes of the <code>Popup</code>. Once a <code>Popup</code>
     * has been disposed you should no longer invoke methods on it. A
     * <code>dispose</code>d <code>Popup</code> may be reclaimed and later used
     * based on the <code>PopupFactory</code>. As such, if you invoke methods
     * on a <code>disposed</code> <code>Popup</code>, indeterminate
     * behavior will result.
     */
    public void hide() {
        Component component = getComponent();

        if (component instanceof JWindow) {
            component.hide();
            ((JWindow)component).getContentPane().removeAll();
        }
        dispose();
    }

    /**
     * Frees any resources the <code>Popup</code> may be holding onto.
     */
    void dispose() {
        Component component = getComponent();

        if (component instanceof JWindow) {
            ((Window)component).dispose();
            component = null;
        }
    }

    /**
     * Resets the <code>Popup</code> to an initial state.
     */
    void reset(Component owner, Component contents, int ownerX, int ownerY) {
        if (getComponent() == null) {
            component = createComponent(owner);
        }

        Component c = getComponent();

        if (c instanceof JWindow) {
            JWindow component = (JWindow)getComponent();

            component.setLocation(ownerX, ownerY);

            //ibm.96155.1 starts

            /*
             * If longMenus is true then display the menu on the screen without scrollbars
             * and if needed, display the menu across the whole height of the screen if
             * the menu has a lot of items.  This is how Sun currently works
             */
            if(longMenus) {
                component.getContentPane().add(contents, BorderLayout.CENTER);
                contents.invalidate();
                pack();
                return;
            }
            //ibm.96155.1 ends
	    /* ibm@56756 start */
	    /* Instead of adding the contents directly to the component
	     * we add the contents to a panel which in turn is added to
	     * a scrollpane (we can't add the contents directly to the
	     * scrollpane because the contents component is a JPopupMenu
	     * which overrides setLocation which prevents setViewPosition
	     * on the scrollpane's viewport from working).
	     */
            JComponent contentPane = (JComponent)component.getContentPane();
                contentPane.setPreferredSize(contents.getPreferredSize());
//	    if (contents instanceof JPopupMenu){ /*ibm@72122 ibm@74556 */
	    if (contents instanceof JPopupMenu
                && ((JPopupMenu)contents).getInvoker() instanceof JMenu) {
		JPopupMenu jcontents = (JPopupMenu)contents;   
                /* ibm@74556 start*/
		/*ibm@72122 start*/
//		boolean canAddScroller=false;
//                if(jcontents.getInvoker() instanceof JMenu) {
//		    canAddScroller = true;
//		}else if (!(jcontents.getInvoker() instanceof JComboBox)){
//		    MenuElement[] me=jcontents.getSubElements();
//		    if(me.length>0) canAddScroller = true;
//		    int n = jcontents.getComponentCount();  
//		    if(n>me.length){
//			for(int i=0;i<n;i++){
//			    Component child=jcontents.getComponent(i);
//			    if(!(child instanceof MenuElement) &&!(child instanceof JSeparator)){
//				canAddScroller=false;
//				break;
//			    }
//			}
//		    }
//		}
//		if(canAddScroller){
                /* ibm@74556 end*/
		/*ibm@72122 end*/                
		    Point position = new Point(ownerX,ownerY);
		    Dimension screenSize;
		    Insets screenIns = new Insets(0,0,0,0);
		    GraphicsConfiguration gc = component.getGraphicsConfiguration();
		    if (gc == null) {
			screenSize = component.getToolkit().getScreenSize();
		    }
		    else {
			Rectangle sBounds = gc.getBounds();
			screenSize = sBounds.getSize();
			position.x -= sBounds.x;
			position.y -= sBounds.y;
			screenIns = component.getToolkit().getScreenInsets(gc);
		    }
		    
		    Point invPos = jcontents.getInvoker().getLocationOnScreen();
		    // determine whether the menu is appearing upwards or downwards
		    // and calculate the available space in that direction
		    int maxHeight;
//		    if(jcontents.getInvoker() instanceof JMenu){/*ibm@72122 ibm@74556 */
			if (position.y
			    + (Math.max(Math.abs(UIManager.getInt("Menu.menuPopupOffsetY")),
					Math.abs(UIManager.getInt("Menu.submenuPopupOffsetY"))))
			    < invPos.y) { // upwards
			    maxHeight = invPos.y - position.y;
			    if(jcontents.getInvoker().getParent() instanceof JPopupMenu ){/*ibm@71872*/
				maxHeight += jcontents.getInvoker().getHeight() -         /*ibm@71872*/
				             UIManager.getInt("Menu.submenuPopupOffsetY");/*ibm@71872*/
			    }else{                                                        /*ibm@71873*/
				maxHeight -= UIManager.getInt("Menu.menuPopupOffsetY");   /*ibm@71873*/
			    }                                                             /*ibm@71872*/
			    maxHeight = Math.min(maxHeight,screenSize.height - position.y - screenIns.bottom);/*ibm@71873*/
			} else { // downwards
			    maxHeight = screenSize.height - position.y - screenIns.bottom;
			}
//		    }else{                                                            /*ibm@72122 ibm@74556 */
//			maxHeight = screenSize.height - position.y - screenIns.bottom;/*ibm@72122 ibm@74556*/
//		    }                                                                 /*ibm@72122 ibm@74556*/
		    
		    Dimension size = contents.getPreferredSize();
		    if (size.height > maxHeight) {
			size.height = maxHeight; // constrain height
			
			contentPane.setPreferredSize(size);
			
			JPanel panel = new JPanel(new BorderLayout());
			panel.add(contents,BorderLayout.CENTER);
			JScrollPane scroller = new JScrollPane(panel,
							       ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER,
							       ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
			
			// set the border on the popup's content page
			// instead of the popup menu so that the arrow
			// buttons appear inside the border
			contentPane.setBorder(UIManager.getBorder("PopupMenu.border"));
			jcontents.setBorder(null);
			scroller.setBorder(null);
			
			final PopupScrollButton b_up =
			    new PopupScrollButton(SwingConstants.NORTH,scroller,jcontents);
			final PopupScrollButton b_down =
			    new PopupScrollButton(SwingConstants.SOUTH,scroller,jcontents);
			
			// listener to enable/disable arrow buttons when top
			// and bottom of menu is reached
			scroller.getViewport().addChangeListener(new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
				    JViewport view = (JViewport)e.getSource();
				    Point position = view.getViewPosition();
                                // disable up button if view is at top of menu, otherwise enable
				    b_up.setEnabled(position.y!=0);
                                // disable down button if view is at bottom of menu, otherwise enable
				    b_down.setEnabled(position.y + view.getExtentSize().height
						      < view.getViewSize().height);
				}
			    });
			
			contentPane.add(b_up, BorderLayout.NORTH);
			contentPane.add(scroller, BorderLayout.CENTER);
			contentPane.add(b_down, BorderLayout.SOUTH);
		    } else {
			if (jcontents.getBorder()==null) { // make sure border is set
			    jcontents.setBorder(UIManager.getBorder("PopupMenu.border"));
			}                    
			contentPane.setBorder(null);
			contentPane.add(contents, BorderLayout.CENTER);
		    }
//	      } /*ibm@72122 ibm@73375 ibm@74556 */
		} else {
		    component.getContentPane().add(contents, BorderLayout.CENTER);
		}
//	    } /*ibm@72122 ibm@73375 ibm@74556 */
	    /* ibm@56756 end */

            contents.invalidate();
            if(component.isVisible()) {
                // Do not call pack() if window is not visible to
                // avoid early native peer creation
                pack();
            }
        }
    }


    /* ibm@56756 start */
    private class PopupScrollButton extends JButton {
	private int direction;
	private JScrollPane scroller;
	private JComponent contents;
        private Color shadow = null;      /* ibm@71875 */ 
        private Color darkShadow = null;  /* ibm@71875 */
        private Color highlight = null;   /* ibm@71875 */
        private Color background = null;  /* ibm@71875 */ 

	public PopupScrollButton(int direction, JScrollPane scroller,
				 JComponent contents) {
	    super();
	    this.direction = direction;
	    this.scroller = scroller;
	    this.contents = contents;
	    this.shadow = UIManager.getColor("controlShadow");
	    this.darkShadow = UIManager.getColor("menuText");
	    this.highlight = UIManager.getColor("controlLtHighlight");

	    /* ibm@71875 start */ 
	    if (this.shadow == null) this.shadow = Color.GRAY;          
	    if (this.darkShadow == null) this.darkShadow = Color.BLACK;
	    if (this.highlight == null) this.highlight = Color.WHITE;
	    this.background = UIManager.getColor("menu");
	    if (this.background == null) this.background = Color.LIGHT_GRAY;
		setBackground(this.background);
	    /* ibm@71875 end */

	    addMouseListener(new PopupScrollAction());
	    setName("###PopupScrollButton###");/*ibm@80708*/
	}

	private class PopupScrollAction extends MouseAdapter {
            private final int intialDelay = 500; //milliseconds
            private final int delay = 100; //milliseconds
            private Timer timer;
            private ActionListener task;
            private Component button;

	    public void mousePressed(MouseEvent e) {
                button = (Component)e.getSource();
                if (!button.isEnabled()) {
                    return;
                }
                scrollAction();
                if (task==null) {
                    task = new ActionListener() {
                            public void actionPerformed(ActionEvent evt) {
                                if (button.isEnabled()) {
                                    scrollAction();
                                } else {
                                    timer.stop();
                                }
                            }
                        };
                }
                if (timer==null) {
                    timer=new Timer(delay, task);
                    timer.setInitialDelay(intialDelay);
                    timer.start();
                } else {
                    timer.restart();
                }
            }
	    public void mouseReleased(MouseEvent e) {
                if (timer!=null) {
                    timer.stop();
                }
            }

            private void scrollAction() {
		JViewport view = scroller.getViewport();
		Point position = view.getViewPosition();
		Component first = contents.findComponentAt(position);
		if (first==null) {
                    return;
                }
		// adjust position to find the component above or below
		if (direction==NORTH) { // scroll up
		    position.y -= 1;
		} else { // scroll down
		    position.y += first.getSize().height;
		}
		Component new_comp = contents.findComponentAt(position);
		if (new_comp != null) {
		    view.setViewPosition(new_comp.getLocation());
		}
	    }
	}	

	public Dimension getPreferredSize() {
            return new Dimension(16, 16);
        }

        public Dimension getMinimumSize() {
            return new Dimension(5, 5);
        }

        public Dimension getMaximumSize() {
            return new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
        }
    
    	public boolean isFocusTraversable() {
	  return false;
	}

	// rendering code modified from BasicArrowButton
	public void paint(Graphics g) {
	    Color origColor;
	    boolean isPressed, isEnabled;
	    int w, h, size;

            w = getSize().width;
            h = getSize().height;
	    origColor = g.getColor();
	    isPressed = getModel().isPressed();
	    isEnabled = isEnabled();

            g.setColor(getBackground());
            g.fillRect(0, 0, w, h);

            // Draw the proper Border
             if (isPressed) {
                 g.setColor(shadow);
                 g.drawRect(0, 0, w-1, h-1);
             } else {
                 g.setColor(highlight);    // inner 3D border
                 g.drawLine(0, 0, 0, h);
             }

            // If there's no room to draw arrow, bail
            if(h < 5 || w < 5)      {
                g.setColor(origColor);
                return;
            }

            if (isPressed) {
                g.translate(1, 1);
            }

            // Draw the arrow
            size = Math.min((h - 4) / 3, (w - 4) / 3);
            size = Math.max(size, 2);
	    paintTriangle(g, (w - size) / 2, (h - size) / 2,
				size, direction, isEnabled);

            // Reset the Graphics back to it's original settings
            if (isPressed) {
                g.translate(-1, -1);
	    }
	    g.setColor(origColor);

        }

	public void paintTriangle(Graphics g, int x, int y, int size, 
					int direction, boolean isEnabled) {
	    Color oldColor = g.getColor();
	    int mid, i, j;

	    j = 0;
            size = Math.max(size, 2);
	    mid = (size / 2) - 1;
	
	    g.translate(x, y);
	    if(isEnabled)
		g.setColor(darkShadow);
	    else
		g.setColor(shadow);

            switch(direction)       {
            case NORTH:
                for(i = 0; i < size; i++)      {
                    g.drawLine(mid-i, i, mid+i, i);
                }
                if(!isEnabled)  {
                    g.setColor(highlight);
                    g.drawLine(mid-i+2, i, mid+i, i);
                }
                break;
            case SOUTH:
                if(!isEnabled)  {
                    g.translate(1, 1);
                    g.setColor(highlight);
                    for(i = size-1; i >= 0; i--)   {
                        g.drawLine(mid-i, j, mid+i, j);
                        j++;
                    }
		    g.translate(-1, -1);
		    g.setColor(shadow);
		}
		
		j = 0;
                for(i = size-1; i >= 0; i--)   {
                    g.drawLine(mid-i, j, mid+i, j);
                    j++;
                }
                break;
            case WEST:
                for(i = 0; i < size; i++)      {
                    g.drawLine(i, mid-i, i, mid+i);
                }
                if(!isEnabled)  {
                    g.setColor(highlight);
                    g.drawLine(i, mid-i+2, i, mid+i);
                }
                break;
            case EAST:
                if(!isEnabled)  {
                    g.translate(1, 1);
                    g.setColor(highlight);
                    for(i = size-1; i >= 0; i--)   {
                        g.drawLine(j, mid-i, j, mid+i);
                        j++;
                    }
		    g.translate(-1, -1);
		    g.setColor(shadow);
                }

		j = 0;
                for(i = size-1; i >= 0; i--)   {
                    g.drawLine(j, mid-i, j, mid+i);
                    j++;
                }
		break;
            }
	    g.translate(-x, -y);	
	    g.setColor(oldColor);
	}
    }
    /* ibm@56756 end */

    /**
     * Causes the <code>Popup</code> to be sized to fit the preferred size
     * of the <code>Component</code> it contains.
     */
    void pack() {
        Component component = getComponent();

        if (component instanceof Window) {
            ((Window)component).pack();
        }
    }

    /**
     * Returns the <code>Window</code> to use as the parent of the
     * <code>Window</code> created for the <code>Popup</code>. This creates
     * a new <code>Frame</code> each time it is invoked. Subclasses that wish
     * to support a different <code>Window</code> parent should override
     * this.
     */
    private Window getParentWindow(Component owner) {
        Window window = null;

        if (owner instanceof Window) {
            window = (Window)owner;
        }
        else if (owner != null) {
            window = SwingUtilities.getWindowAncestor(owner);
        }
        if (window == null) {
            window = new DefaultFrame();
        }
        return window;
    }

    /**
     * Creates the Component to use as the parent of the <code>Popup</code>.
     * The default implementation creates a <code>Window</code>, subclasses
     * should override.
     */
    Component createComponent(Component owner) {
        if (GraphicsEnvironment.isHeadless()) {
            // Generally not useful, bail.
            return null;
        }
        return new HeavyWeightWindow(getParentWindow(owner));
    }

    /**
     * Returns the <code>Component</code> returned from
     * <code>createComponent</code> that will hold the <code>Popup</code>.
     */
    Component getComponent() {
        return component;
    }


    /**
     * Component used to house window.
     */
    static class HeavyWeightWindow extends JWindow implements ModalExclude {
        HeavyWeightWindow(Window parent) {
            super(parent);
            setFocusableWindowState(false);
            setName("###overrideRedirect###");
        }

        public void update(Graphics g) {
            paint(g);
        }

	public void show() {
	    this.pack();
	    super.show();
	}
    }


    /**
     * Used if no valid Window ancestor of the supplied owner is found.
     * <p>
     * PopupFactory uses this as a way to know when the Popup shouldn't
     * be cached based on the Window.
     */
    static class DefaultFrame extends Frame {
    }
}
