package java.lang;

import java.io.InputStream;
import java.security.ProtectionDomain;
import java.security.AllPermission;
import java.security.Permissions;
import java.lang.reflect.*;
import java.net.URL;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * An instance of class Class is the in-image representation
 * of a Java class. There are three basic types of Classes
 * <dl>
 * <dt><em>Classes representing object types (classes or interfaces)</em></dt>
 * <dd>These are Classes which represent the class of a
 *     simple instance as found in the class hierarchy.
 *     The name of one of these Classes is simply the
 *     fully qualified class name of the class or interface
 *     that it represents. Its <em>signature</em> is
 *     the letter "L", followed by its name, followed
 *     by a semi-colon (";").</dd>
 * <dt><em>Classes representing base types</em></dt>
 * <dd>These Classes represent the standard Java base types.
 *     Although it is not possible to create new instances
 *     of these Classes, they are still useful for providing
 *     reflection information, and as the component type
 *     of array classes. There is one of these Classes for
 *     each base type, and their signatures are:
 *     <ul>
 *     <li><code>B</code> representing the <code>byte</code> base type</li>
 *     <li><code>S</code> representing the <code>short</code> base type</li>
 *     <li><code>I</code> representing the <code>int</code> base type</li>
 *     <li><code>J</code> representing the <code>long</code> base type</li>
 *     <li><code>F</code> representing the <code>float</code> base type</li>
 *     <li><code>D</code> representing the <code>double</code> base type</li>
 *     <li><code>C</code> representing the <code>char</code> base type</li>
 *     <li><code>Z</code> representing the <code>boolean</code> base type</li>
 *     <li><code>V</code> representing void function return values</li>
 *     </ul>
 *     The name of a Class representing a base type
 *     is the keyword which is used to represent the
 *     type in Java source code (i.e. "int" for the
 *     <code>int</code> base type.</dd>
 * <dt><em>Classes representing array classes</em></dt>
 * <dd>These are Classes which represent the classes of
 *     Java arrays. There is one such Class for all array
 *     instances of a given arity (number of dimensions)
 *     and leaf component type. In this case, the name of the
 *     class is one or more left square brackets (one per
 *     dimension in the array) followed by the signature ofP
 *     the class representing the leaf component type, which
 *     can be either an object type or a base type. The
 *     signature of a Class representing an array type
 *     is the same as its name.</dd>
 * </dl>
 *
 * @author		OTI
 * @version		initial
 */
public final class Class implements java.io.Serializable {
	static final long serialVersionUID = 3206093459760846163L;
	private static ProtectionDomain AllPermissionsPD;

	private static final int UNINITIALIZED = 0;
	private static final int INITIALIZED = 1;
	private static final int FAILED = 2;
	private static final int UNVERIFIED = 3;

	private static final int j9Version = 0x0101022e;

	private static final long j9Config = 0x7363617200000000L;	// 'scar\0'

	private static Class[] EmptyParameters = new Class[0];

/**
 * Prevents this class from being instantiated. Instances
 * created by the virtual machine only.
 */
private Class() {}

private void checkMemberAccess(int type) {
	SecurityManager security = System.getSecurityManager();
	if (security == null ||
		ClassLoader.getStackClassLoader(2) == ClassLoader.systemClassLoader)
			return;
	security.checkMemberAccess(this, type);
	String packageName = getPackageName();
	if(packageName != "")
		security.checkPackageAccess(packageName);
}

/**
 * Answers a Class object which represents the class
 * named by the argument. The name should be the name
 * of a class as described in the class definition of
 * java.lang.Class, however Classes representing base
 * types can not be found using this method.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		className	The name of the non-base type class to find
 * @return		the named Class
 * @throws		ClassNotFoundException If the class could not be found
 *
 * @see			java.lang.Class
 */
public static Class forName(String className) throws ClassNotFoundException {
	ClassLoader defaultClassLoader = ClassLoader.callerClassLoader();
	return forNameImpl(className, true, defaultClassLoader);
}

/**
 * Answers a Class object which represents the class
 * named by the argument. The name should be the name
 * of a class as described in the class definition of
 * java.lang.Class, however Classes representing base
 * types can not be found using this method.
 * Security rules will be obeyed.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		className			The name of the non-base type class to find
 * @param		initializeBoolean	A boolean indicating whether the class should be
 *									initialized
 * @param		classLoader			The classloader to use to load the class
 * @return		the named class.
 * @throws		ClassNotFoundException If the class could not be found
 *
 * @see			java.lang.Class
 */
public static Class forName(String className, boolean initializeBoolean, ClassLoader classLoader)
	throws ClassNotFoundException
{
	/* perform security checks */
	if (null == classLoader) {
		ClassLoader callerClassLoader = ClassLoader.callerClassLoader();
		if (callerClassLoader != null) {
			/* only allowed if caller has RuntimePermission("getClassLoader") permission */
			SecurityManager security = System.getSecurityManager();
			if (null != security)
				security.checkPermission(RuntimePermission.permissionToGetClassLoader);
		}
	}
	return forNameImpl(className, initializeBoolean, classLoader);
}

/**
 * Answers a Class object which represents the class
 * named by the argument. The name should be the name
 * of a class as described in the class definition of
 * java.lang.Class, however Classes representing base
 * types can not be found using this method.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		className			The name of the non-base type class to find
 * @param		initializeBoolean	A boolean indicating whether the class should be
 *									initialized
 * @param		classLoader			The classloader to use to load the class
 * @return		the named class.
 * @throws		ClassNotFoundException If the class could not be found
 *
 * @see			java.lang.Class
 */
private static native Class forNameImpl(String className,
                            boolean initializeBoolean,
                            ClassLoader classLoader)
	throws ClassNotFoundException;

/**
 * Answers an array containing all public class members
 * of the class which the receiver represents and its
 * superclasses and interfaces
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the class' public class members
 * @throws		SecurityException If member access is not allowed
 *
 * @see			java.lang.Class
 */
public Class[] getClasses() {
	SecurityManager security = System.getSecurityManager();
	if (security != null && ClassLoader.callerClassLoader() != null) {
		Class current = this;
		while (current != null) {
			security.checkMemberAccess(current, Member.PUBLIC);
			String packageName = current.getPackageName();
			if (packageName != "")
				security.checkPackageAccess(packageName);
			current = current.getSuperclass();
		}
	}

	java.util.Vector publicClasses = new java.util.Vector();
	Class current = this;
	Class[] classes;
	while(current != null) {
		classes = current.getDeclaredClassesImpl();
		for (int i = 0; i < classes.length; i++)
			if (Modifier.isPublic(classes[i].getModifiers()))
				publicClasses.addElement(classes[i]);
		current = current.getSuperclass();
	}
	classes = new Class[publicClasses.size()];
	publicClasses.copyInto(classes);
	return classes;
}

void verify() {
	Class superclass;

	while (true) {
		switch (getInitStatus()) {
			case INITIALIZED:
			case UNINITIALIZED:
			case FAILED:
				return;

			case UNVERIFIED:
				synchronized (this) {
					if (getInitStatus() != UNVERIFIED) break;
					setInitThread();
				}
				superclass = getSuperclass();
				try {
					// verify the superclass
					if (superclass != null)
						superclass.verify();
					// verify this class
					verifyImpl();
				} catch (Error e) {
					setInitStatus(UNVERIFIED);
					throw e;
				} catch (Throwable e) {
					setInitStatus(UNVERIFIED);
					throw new VerifyError(e.toString());
				}
				synchronized (this) {
					if (getInitThread() && ((getInitStatus() & 0x3) == UNVERIFIED))
						setInitStatus(UNINITIALIZED);
				}
				return;

			default: // INPROGRESS
				synchronized (this) {
					int status = getInitStatus();

					if (((status & ~0x3) == 0)) break;
					if ((status & 0x3) != UNVERIFIED) return;
					if (!getInitThread()) {
						try { wait(); } catch (InterruptedException e) {}
						break;
					}
				}

				try {
					verifyImpl();
				} catch (Error e) {
					setInitStatus(UNVERIFIED);
					throw e;
				} catch (Throwable e) {
					setInitStatus(UNVERIFIED);
					throw new VerifyError(e.toString());
				}
				synchronized (this) {
					if (getInitThread() && ((getInitStatus() & 0x3) == UNVERIFIED))
						setInitStatus(UNINITIALIZED);
				}
		}
	}
}

private native void verifyImpl() throws Throwable;

/**
 * Sent internally by the VM to initiatiate
 * initialization of the receiver.  See chapter
 * 2.17.5 of the JVM Specification (2nd ed)
 *
 * @author		OTI
 * @version		initial
 *
 * @throws		Throwable
 */
private void initialize() throws Throwable {
	Class superclass;

	while (true) {
		switch (getInitStatus()) {
			case INITIALIZED:
				return;
			case UNVERIFIED:
				verify();
				break;
			case FAILED:
				throw new NoClassDefFoundError(getName());
			case UNINITIALIZED:
				synchronized (this) {
					if (getInitStatus() != UNINITIALIZED) break;
					setInitThread();
				}

				// initialize the superclass
				superclass = getSuperclass();
				if (superclass != null) {
					try {
						superclass.initialize();
					} catch (Error err) {
						setInitStatus(FAILED);
						throw err;
					}
				}

				// initialize this class
				try {
					/* When we are initializing the statics of the class
					 * we want to be in the correct memory space for the class loader.
					 * If the class loader does not have a memory space, then we want
					 * to initialize it in the base memory space.  In the situation where
					 * we are not in mutli-memory space mode, we will get null back when
					 * we request the memory space for the class loader, and we won't need
					 * to switch memory spaces.  If we have the system class loader then we
					 * want to initialize in the base memory space.  Their are problems treating
					 * the system class loader the same as others because not everything required
					 * by MemorySpace.getMemorySpaceForClassLoader has been initialized the first
					 * time the method is called.
					 */
					initializeImpl();
				} catch (Error err) {
					setInitStatus(FAILED);
					throw err;
				} catch (Throwable t) {
					setInitStatus(FAILED);
					throw new ExceptionInInitializerError(t);
				}

				setInitStatus(INITIALIZED);
				return;
			default: // INPROGRESS
				synchronized (this) {
					int status = getInitStatus();

					if ((status & ~0x3) == 0) break;
					if ((status & 0x3) == UNINITIALIZED) {
						if (getInitThread()) return;
						try { wait(); } catch (InterruptedException e) {}
						break;
					}
				}
				verify();
		}
	}
}

/**
 * Used to indicate the end of class initialization.
 * Sets the initialization status and notifies any
 * threads which may have been waiting for
 * initialization to complete
 *
 * @author		OTI
 * @version		initial
 *
 * @param		status
 *					INITIALIZED (1)
 *					FAILED (2)
 *
 */
private synchronized void setInitStatus(int status) {
	setInitStatusImpl(status);
	notifyAll();
}

/**
 * Answers the receiver's initialization status
 *
 * @author		OTI
 * @version		initial
 *
 * @return		status
 *					UNINITIALIZED (0)
 *					INITIALIZED (1)
 *					FAILED (2)
 *					INPROGRESS (0xnnnnnnn[048C])
 *
 */
private native int getInitStatus();

/**
 * Set the receiver's initialization status
 *
 * @author		OTI
 * @version		initial
 *
 * @param		status
 *					INITIALIZED (1)
 *					FAILED (2)
 *
 */
private native void setInitStatusImpl(int status);

/**
 * Run the receiver's <clinit> method and initialize
 * any static variables
 *
 * @author		OTI
 * @version		initial
 *
 * @throws		Throwable Any exception may be thrown
 */
private native void initializeImpl() throws Throwable;

/**
 * Answers true if the current thread is currently
 * initializing this class
 *
 * @author		OTI
 * @version		initial
 *
 * @return		true if the current thread is initializing the receiver
 *
 */
private native boolean getInitThread();

/**
 * Set the receiver's initialize status to be 'in
 * progress' and save the current thread as the
 * initializing thread.
 *
 * @author		OTI
 * @version		initial
 *
 */
private native void setInitThread();

/**
 * Answers the classloader which was used to load the
 * class represented by the receiver. Answer null if the
 * class was loaded by the system class loader
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's class loader or nil
 *
 * @see			java.lang.ClassLoader
 */
public ClassLoader getClassLoader() {
	ClassLoader loader = getClassLoaderImpl();
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
		// No check for systemClassLoader, contrary to
		// spec but testing reveals this is correct behavior.
		if (loader == ClassLoader.systemClassLoader) return null;
		ClassLoader callersClassLoader = ClassLoader.callerClassLoader();
		if (callersClassLoader != null && callersClassLoader != loader
			&& !callersClassLoader.isAncestorOf(loader))
				security.checkPermission(
					RuntimePermission.permissionToGetClassLoader);
	}
	if (loader == ClassLoader.systemClassLoader) return null;
	return loader;
}

/**
 * Return the ClassLoader for this Class without doing any security
 * checks. The bootstrap ClassLoader is returned, unlike getClassLoader()
 * which returns null in place of the bootstrap ClassLoader.
 *
 * @return the ClassLoader
 *
 * @see ClassLoader#isSystemClassLoader()
 */
native ClassLoader getClassLoaderImpl();

/**
 * Answers a Class object which represents the receiver's
 * component type if the receiver represents an array type.
 * Otherwise answers nil. The component type of an array
 * type is the type of the elements of the array.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the component type of the receiver.
 *
 * @see			java.lang.Class
 */
public native Class getComponentType();

private void throwNoSuchMethodException(String name, Class[] types) throws NoSuchMethodException {
	StringBuffer error = new StringBuffer();
	error.append(getName()).append('.').append(name).append('(');
	if (types.length > 0) {
		error.append(types[0] == null ? null : types[0].getName());
		for (int i=1; i<types.length; i++) {
			error.append(", ").append(types[i] == null ? null : types[i].getName());
		}
	}
	error.append(')');
	throw new NoSuchMethodException(error.toString());
}

/**
 * Answers a public Constructor object which represents the
 * constructor described by the arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		parameterTypes	the types of the arguments.
 * @return		the constructor described by the arguments.
 * @throws		NoSuchMethodException if the constructor could not be found.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getConstructors
 */
public Constructor getConstructor(Class parameterTypes[]) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.PUBLIC);

	// Handle the default constructor case upfront
	if(parameterTypes == null || parameterTypes.length == 0) {
		Constructor rc = getConstructorImpl(EmptyParameters, "()V");
		if (rc == null) throwNoSuchMethodException("<init>", EmptyParameters);
		return rc;
	}

	// Build a signature for the requested method.
	int total = 3;
	String[] sigs = new String[parameterTypes.length];
	for(int i = 0; i < parameterTypes.length; i++) {
		if (parameterTypes[i] != null) {
			sigs[i] = parameterTypes[i].getSignature();
			total += sigs[i].length();
		} else throwNoSuchMethodException("<init>", parameterTypes);
	}
	StringBuffer signature = new StringBuffer(total);
	signature.append('(');
	for(int i = 0; i < parameterTypes.length; i++)
		signature.append(sigs[i]);
	signature.append(")V");

	Constructor rc = getConstructorImpl((Class[])parameterTypes.clone(), signature.toString());
	if (rc == null) throwNoSuchMethodException("<init>", parameterTypes);
	return rc;
}

/**
 * Answers a public Constructor object which represents the
 * constructor described by the arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		parameterTypes	the types of the arguments.
 * @param		signature		the signature of the method.
 * @return		the constructor described by the arguments.
 *
 * @see			#getConstructors
 */
private native Constructor getConstructorImpl(Class parameterTypes[], String signature);

/**
 * Answers an array containing Constructor objects describing
 * all constructors which are visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		all visible constructors starting from the receiver.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getMethods
 */
public Constructor[] getConstructors() throws SecurityException {
	checkMemberAccess(Member.PUBLIC);
	return getConstructorsImpl();
}

/**
 * Answers an array containing Constructor objects describing
 * all constructors which are visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		all visible constructors starting from the receiver.
 *
 * @see			#getMethods
 */
private native Constructor[] getConstructorsImpl();

/**
 * Answers an array containing all class members of the class
 * which the receiver represents. Note that some of the fields
 * which are returned may not be visible in the current
 * execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the class' class members
 * @throws		SecurityException if member access is not allowed
 *
 * @see			java.lang.Class
 */
public Class[] getDeclaredClasses() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredClassesImpl();
}

/**
 * Answers an array containing all class members of the class
 * which the receiver represents. Note that some of the fields
 * which are returned may not be visible in the current
 * execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the class' class members
 *
 * @see			java.lang.Class
 */
private native Class[] getDeclaredClassesImpl();

/**
 * Answers a Constructor object which represents the
 * constructor described by the arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		parameterTypes	the types of the arguments.
 * @return		the constructor described by the arguments.
 * @throws		NoSuchMethodException if the constructor could not be found.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getConstructors
 */
public Constructor getDeclaredConstructor(Class parameterTypes[]) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.DECLARED);

	// Handle the default constructor case upfront
	if(parameterTypes == null || parameterTypes.length == 0) {
		Constructor rc = getDeclaredConstructorImpl(EmptyParameters, "()V");
		if (rc == null) throwNoSuchMethodException("<init>", EmptyParameters);
		return rc;
	}

	// Build a signature for the requested method.
	int total = 3;
	String[] sigs = new String[parameterTypes.length];
	for(int i = 0; i < parameterTypes.length; i++) {
		if (parameterTypes[i] != null) {
			sigs[i] = parameterTypes[i].getSignature();
			total += sigs[i].length();
		} else throwNoSuchMethodException("<init>", parameterTypes);
	}
	StringBuffer signature = new StringBuffer(total);
	signature.append('(');
	for(int i = 0; i < parameterTypes.length; i++)
		signature.append(sigs[i]);
	signature.append(")V");

	Constructor rc = getDeclaredConstructorImpl((Class[])parameterTypes.clone(), signature.toString());
	if (rc == null) throwNoSuchMethodException("<init>", parameterTypes);
	return rc;
}

/**
 * Answers a Constructor object which represents the
 * constructor described by the arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		parameterTypes	the types of the arguments.
 * @param		signature		the signature of the method.
 * @return		the constructor described by the arguments.
 *
 * @see			#getConstructors
 */
private native Constructor getDeclaredConstructorImpl(Class parameterTypes[], String signature);

/**
 * Answers an array containing Constructor objects describing
 * all constructor which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's constructors.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getMethods
 */
public Constructor[] getDeclaredConstructors() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredConstructorsImpl();
}

/**
 * Answers an array containing Constructor objects describing
 * all constructor which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's constructors.
 *
 * @see			#getMethods
 */
private native Constructor[] getDeclaredConstructorsImpl();

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument. Note that the Constructor may not be
 * visible from the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException if the requested field could not be found
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getDeclaredFields
 */
public Field getDeclaredField(String name) throws NoSuchFieldException, SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredFieldImpl(name);
}

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument. Note that the Constructor may not be
 * visible from the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException If the given field does not exist
 *
 * @see			#getDeclaredFields
 */
private native Field getDeclaredFieldImpl(String name) throws NoSuchFieldException;

/**
 * Answers an array containing Field objects describing
 * all fields which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's fields.
 * @throws		SecurityException If member access is not allowed
 *
 * @see			#getFields
 */
public Field[] getDeclaredFields() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredFieldsImpl();
}

/**
 * Answers an array containing Field objects describing
 * all fields which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's fields.
 *
 * @see			#getFields
 */
private native Field[] getDeclaredFieldsImpl();

/**
 * Answers a Method object which represents the method
 * described by the arguments. Note that the associated
 * method may not be visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name			the name of the method
 * @param		parameterTypes	the types of the arguments.
 * @return		the method described by the arguments.
 * @throws		NoSuchMethodException if the method could not be found.
 * @throws		SecurityException If member access is not allowed
 *
 * @see			#getMethods
 */
public Method getDeclaredMethod(String name, Class parameterTypes[]) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.DECLARED);

	// Handle the no parameter case upfront
	if(name == null || parameterTypes == null || parameterTypes.length == 0) {
		Method rc = getDeclaredMethodImpl(name, EmptyParameters, "()");
		if (rc == null) throwNoSuchMethodException(name, EmptyParameters);
		return rc;
	}

	// Build a signature for the requested method.
	int total = 2;
	String[] sigs = new String[parameterTypes.length];
	for(int i = 0; i < parameterTypes.length; i++) {
		if (parameterTypes[i] != null) {
			sigs[i] = parameterTypes[i].getSignature();
			total += sigs[i].length();
		} else throwNoSuchMethodException(name, parameterTypes);
	}
	StringBuffer signature = new StringBuffer(total);
	signature.append('(');
	for(int i = 0; i < parameterTypes.length; i++)
		signature.append(sigs[i]);
	signature.append(')');

	Method rc = getDeclaredMethodImpl(name, (Class[])parameterTypes.clone(), signature.toString());
	if (rc == null) throwNoSuchMethodException(name, parameterTypes);
	return rc;
}

/**
 * Answers a Method object which represents the method
 * described by the arguments. Note that the associated
 * method may not be visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 *
 * @param		name				the name of the method
 * @param		parameterTypes		the types of the arguments.
 * @param		partialSignature	the signature of the method, without return type.
 * @return		the method described by the arguments.
 *
 * @see			#getMethods
 */
private native Method getDeclaredMethodImpl(String name, Class parameterTypes[], String partialSignature);

/**
 * Answers an array containing Method objects describing
 * all methods which are defined by the receiver. Note that
 * some of the methods which are returned may not be visible
 * in the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @throws		SecurityException	if member access is not allowed
 * @return		the receiver's methods.
 *
 * @see			#getMethods
 */
public Method[] getDeclaredMethods() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredMethodsImpl();
}

/**
 * Answers an array containing Method objects describing
 * all methods which are defined by the receiver. Note that
 * some of the methods which are returned may not be visible
 * in the current execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's methods.
 *
 * @see			#getMethods
 */
private native Method[] getDeclaredMethodsImpl();

/**
 * Answers the class which declared the class represented
 * by the receiver. This will return null if the receiver
 * is a member of another class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declaring class of the receiver.
 */
public Class getDeclaringClass() {
	return getDeclaringClassImpl();
}

/**
 * Answers the class which declared the class represented
 * by the receiver. This will return null if the receiver
 * is a member of another class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the declaring class of the receiver.
 */
private native Class getDeclaringClassImpl();

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument which must be visible from the current
 * execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException If the given field does not exist
 * @throws		SecurityException If access is denied
 *
 * @see			#getDeclaredFields
 */
public Field getField(String name) throws NoSuchFieldException, SecurityException {
	checkMemberAccess(Member.PUBLIC);
	return getFieldImpl(name);
}

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument which must be visible from the current
 * execution context.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException If the given field does not exist
 *
 * @see			#getDeclaredFields
 */
private native Field getFieldImpl(String name) throws NoSuchFieldException;

/**
 * Answers an array containing Field objects describing
 * all fields which are visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		all visible fields starting from the receiver.
 * @throws		SecurityException If member access is not allowed
 *
 * @see			#getDeclaredFields
 */
public Field[] getFields() throws SecurityException {
	checkMemberAccess(Member.PUBLIC);
	return getFieldsImpl();
}

/**
 * Answers an array containing Field objects describing
 * all fields which are visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		all visible fields starting from the receiver.
 *
 * @see			#getDeclaredFields
 */
private native Field[] getFieldsImpl();

/**
 * Answers an array of Class objects which match the interfaces
 * specified in the receiver classes <code>implements</code>
 * declaration
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Class[]
 *					the interfaces the receiver claims to implement.
 */
public native Class[] getInterfaces();

/**
 * Answers a Method object which represents the method
 * described by the arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name String
 *					the name of the method
 * @param		parameterTypes Class[]
 *					the types of the arguments.
 * @return		Method
 *					the method described by the arguments.
 * @throws	NoSuchMethodException
 *					if the method could not be found.
 * @throws	SecurityException
 *					if member access is not allowed
 *
 * @see			#getMethods
 */
public Method getMethod(String name, Class parameterTypes[]) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.PUBLIC);

	// Handle the no parameter case upfront
	if(name == null || parameterTypes == null || parameterTypes.length == 0) {
		Method rc = getMethodImpl(name, EmptyParameters, "()");
		if (rc == null) throwNoSuchMethodException(name, EmptyParameters);
		return rc;
	}

	// Build a signature for the requested method.
	int total = 2;
	String[] sigs = new String[parameterTypes.length];
	for(int i = 0; i < parameterTypes.length; i++) {
		if (parameterTypes[i] != null) {
			sigs[i] = parameterTypes[i].getSignature();
			total += sigs[i].length();
		} else throwNoSuchMethodException(name, parameterTypes);
	}
	StringBuffer signature = new StringBuffer(total);
	signature.append('(');
	for(int i = 0; i < parameterTypes.length; i++)
		signature.append(sigs[i]);
	signature.append(')');

	Method rc = getMethodImpl(name, (Class[])parameterTypes.clone(), signature.toString());
	if (rc == null) throwNoSuchMethodException(name, parameterTypes);
	return rc;
}

/**
 * Answers a Method object which represents the method
 * described by the arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name String
 *					the name of the method
 * @param		parameterTypes Class[]
 *					the types of the arguments.
 * @param		partialSignature String
 *					the signature of the method, without return type.
 * @return		Method
 *					the method described by the arguments.
 *
 * @see			#getMethods
 */
private native Method getMethodImpl(String name, Class parameterTypes[], String partialSignature);

/**
 * Answers an array containing Method objects describing
 * all methods which are visible from the current execution
 * context.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Method[]
 *					all visible methods starting from the receiver.
 * @throws	SecurityException
 *					if member access is not allowed
 *
 * @see			#getDeclaredMethods
 */
public Method[] getMethods() throws SecurityException {
	checkMemberAccess(Member.PUBLIC);
	Method[] methods;

	if(isPrimitive()) return new Method[0];
	if(isInterface())
	{
		return getInterfaceMethodsImpl();
	}
	else
	{
		int vCount = getVirtualMethodCountImpl();

		methods = (Method[])Method.class.allocateAndFillArray(vCount + getStaticMethodCountImpl());
		getVirtualMethodsImpl(methods, 0);
		getStaticMethodsImpl(methods, vCount);
		return methods;
	}
}

private boolean methodsEqual(Method m1, Method m2) {
	Class[] m1Parms, m2Parms;

	if(!m1.getName().equals(m2.getName())) {
		return false;
	}
	if(!m1.getReturnType().equals(m2.getReturnType())) {
		return false;
	}
	m1Parms = m1.getParameterTypes();
	m2Parms = m2.getParameterTypes();
	if(m1Parms.length != m2Parms.length) {
		return false;
	}
	for(int i = 0; i < m1Parms.length; i++) {
		if(m1Parms[i] != m2Parms[i]) {
			return false;
		}
	}
	return true;
}

private int getInterfaceMethodCountImpl()
{
	Class[] parents;
	int count;

	count = getDeclaredMethods().length;
	parents = getInterfaces();
	for(int i = 0; i < parents.length; i++) {
		count += parents[i].getInterfaceMethodCountImpl();
	}
	return count;
}

private Method[] getInterfaceMethodsImpl()
{
	Method[] scratch, unique, local;
	Class[] parents;
	int index, count;

	/* Get a pessimal amount of scratch space */
	scratch = new Method[getInterfaceMethodCountImpl()];

	/* Obtain the local methods. These are guaranteed to be returned */
	local = getDeclaredMethods();
	index = 0;
	for(int i = 0; i < local.length; i++) {
		/* <clinit> is never abstract */
		if(Modifier.isAbstract(local[i].getModifiers())) {
			scratch[index++] = local[i];
		} else {
			local[i] = null;
		}
	}

	/* Walk each superinterface */
	parents = getInterfaces();
	for(int i = 0; i < parents.length; i++) {
		/* Get the superinterface's (swept) methods */
		Method[] parentMethods = parents[i].getInterfaceMethodsImpl();
		for(int j = 0; j < local.length; j++) {
			if(local[j] != null) {
				/* Sweep out any local overrides */
				for(int k = 0; k < parentMethods.length; k++) {
					if((parentMethods[k] != null)&& methodsEqual(local[j], parentMethods[k])) {
						parentMethods[k] = null;
					}
				}
			}
		}
		/* Store any remaining methods */
		for(int j = 0; j < parentMethods.length; j++) {
			if(parentMethods[j] != null) {
				scratch[index++] = parentMethods[j];
			}
		}
	}

	/* Sweep out any duplicates */
	for(int i = 0; i < scratch.length; i++) {
		if(scratch[i] != null) {
			for(int j = i+1; j < scratch.length; j++) {
				if((scratch[j] != null) && scratch[i].equals(scratch[j])) {
					scratch[j] = null;
				}
			}
		}
	}

	/* Copy into a correctly sized array and return */
	count = 0;
	for(int i = 0; i < scratch.length; i++) {
		if(scratch[i] != null) count++;
	}
	unique = new Method[count];
	index = 0;
	for(int i = 0; i < scratch.length; i++) {
		if(scratch[i] != null) {
			unique[index++] = scratch[i];
		}
	}

	return unique;
}

private native int getVirtualMethodCountImpl();
private native void getVirtualMethodsImpl(Method[] array, int start);
private native int getStaticMethodCountImpl();
private native void getStaticMethodsImpl(Method[] array, int start);
private native Object[] allocateAndFillArray(int size);

/**
 * Answers an integer which which is the receiver's modifiers.
 * Note that the constants which describe the bits which are
 * returned are implemented in class java.lang.reflect.Modifier
 * which may not be available on the target.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's modifiers
 */
public native int getModifiers();

/**
 * Answers the name of the class which the receiver represents.
 * For a description of the format which is used, see the class
 * definition of java.lang.Class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's name.
 *
 * @see			java.lang.Class
 */
public native String getName();

/**
 * Answers the ProtectionDomain of the receiver.
 * <p>
 * Note: In order to conserve space in embedded targets, we allow this
 * method to answer null for classes in the system protection domain
 * (i.e. for system classes). System classes are always given full
 * permissions (i.e. AllPermission). This is not changeable via the
 * java.security.Policy.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		ProtectionDomain
 *					the receiver's ProtectionDomain.
 *
 * @see			java.lang.Class
 */
public ProtectionDomain getProtectionDomain() {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToGetProtectionDomain);

	ProtectionDomain result = getPDImpl();
	if (result != null) return result;

	if (AllPermissionsPD == null) {
		Permissions collection = new Permissions();
		collection.add(new AllPermission());
		AllPermissionsPD = new ProtectionDomain(null, collection);
	}
	return AllPermissionsPD;
}

/**
 * Answers the ProtectionDomain of the receiver.
 * <p>
 * This method is for internal use only.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		ProtectionDomain
 *					the receiver's ProtectionDomain.
 *
 * @see			java.lang.Class
 */
native ProtectionDomain getPDImpl();

/**
 * Sets the ProtectionDomain of the receiver.
 * <p>
 * This method is for internal use only.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		pd ProtectionDomain
 *					the receiver's new ProtectionDomain.
 *
 * @see			java.lang.Class
 */
native void setPDImpl(ProtectionDomain pd);

/**
 * Answers the name of the package to which the receiver belongs.
 * For example, Object.class.getPackageName() returns "java.lang".
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's package name.
 *
 * @see			#getPackage
 */
private String getPackageName() {
	String name = getName();
	int index = name.lastIndexOf('.');
	if (index >= 0) return name.substring(0, index);
	return "";
}

/**
 * Answers a read-only stream on the contents of the
 * resource specified by resName. The mapping between
 * the resource name and the stream is managed by the
 * class' class loader.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		resName 	the name of the resource.
 * @return		a stream on the resource.
 *
 * @see			java.lang.ClassLoader
 */
public URL getResource(String resName) {
	ClassLoader loader = this.getClassLoaderImpl();
	if (loader == ClassLoader.systemClassLoader)
		return ClassLoader.getSystemResource(this.toResourceName(resName));
	else
		return loader.getResource(this.toResourceName(resName));
}

/**
 * Answers a read-only stream on the contents of the
 * resource specified by resName. The mapping between
 * the resource name and the stream is managed by the
 * class' class loader.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		resName		the name of the resource.
 * @return		a stream on the resource.
 *
 * @see			java.lang.ClassLoader
 */
public InputStream getResourceAsStream(String resName) {
	ClassLoader loader = this.getClassLoaderImpl();
	if (loader == ClassLoader.systemClassLoader)
		return ClassLoader.getSystemResourceAsStream(this.toResourceName(resName));
	else
		return loader.getResourceAsStream(this.toResourceName(resName));
}

/**
 * Answers a String object which represents the class's
 * signature, as described in the class definition of
 * java.lang.Class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the signature of the class.
 *
 * @see			java.lang.Class
 */
private String getSignature() {
	if(isArray()) return getName(); // Array classes are named with their signature
	if(isPrimitive()) {
		// Special cases for each base type.
		if(this == void.class) return "V";
		if(this == boolean.class) return "Z";
		if(this == byte.class) return "B";
		if(this == char.class) return "C";
		if(this == short.class) return "S";
		if(this == int.class) return "I";
		if(this == long.class) return "J";
		if(this == float.class) return "F";
		if(this == double.class) return "D";
	}

	// General case.
	// Create a StringBuffer of the correct size
	String name = getName();
	return new StringBuffer(name.length() + 2).
		append('L').append(name).append(';').toString();
}

/**
 * Answers the signers for the class represented by the
 * receiver, or null if there are no signers.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the signers of the receiver.
 *
 * @see			#getMethods
 */
public Object[] getSigners() {
	ClassLoader loader = getClassLoader();
	if (loader == null) return null;
	 return loader.getSigners(this);
}

/**
 * Answers the Class which represents the receiver's
 * superclass. For Classes which represent base types,
 * interfaces, and for java.lang.Object the method
 * answers null.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's superclass.
 */
public native Class getSuperclass();

/**
 * Answers true if the receiver represents an array class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					if the receiver represents an array class
 *              <code>false</code>
 *                  if it does not represent an array class
 */
public native boolean isArray();

/**
 * Answers true if the type represented by the argument
 * can be converted via an identity conversion or a widening
 * reference conversion (i.e. if either the receiver or the
 * argument represent primitive types, only the identity
 * conversion applies).
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					the argument can be assigned into the receiver
 *              <code>false</code>
 *					the argument cannot be assigned into the receiver
 * @param		cls	Class
 *					the class to test
 * @throws	NullPointerException
 *					if the parameter is null
 *
 */
public native boolean isAssignableFrom(Class cls);

/**
 * Answers true if the argument is non-null and can be
 * cast to the type of the receiver. This is the runtime
 * version of the <code>instanceof</code> operator.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					the argument can be cast to the type of the receiver
 *              <code>false</code>
 *					the argument is null or cannot be cast to the
 *					type of the receiver
 *
 * @param		object Object
 *					the object to test
 */
public native boolean isInstance(Object object);

/**
 * Answers true if the receiver represents an interface.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					if the receiver represents an interface
 *              <code>false</code>
 *                  if it does not represent an interface
 */
public boolean isInterface() {
	return (getModifiers() & 512 /* AccInterface */) != 0;
}

/**
 * Answers true if the receiver represents a base type.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					if the receiver represents a base type
 *              <code>false</code>
 *                  if it does not represent a base type
 */
public native boolean isPrimitive();

/**
 * Answers a new instance of the class represented by the
 * receiver, created by invoking the default (i.e. zero-argument)
 * constructor. If there is no such constructor, or if the
 * creation fails (either because of a lack of available memory or
 * because an exception is thrown by the constructor), an
 * InstantiationException is thrown. If the default constructor
 * exists, but is not accessible from the context where this
 * message is sent, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a new instance of the class represented by the receiver.
 * @throws		IllegalAccessException if the constructor is not visible to the sender.
 * @throws		InstantiationException if the instance could not be created.
 */
public Object newInstance() throws IllegalAccessException, InstantiationException {
	checkMemberAccess(Member.PUBLIC);
	return newInstanceImpl();
}

/**
 * Used as a prototype for the jit.
 *
 * @param 		cl
 * @return		the object
 * @throws 		InstantiationException
 */
private Object newInstancePrototype(Class cl) throws InstantiationException {
	throw new InstantiationException();
}

/**
 * Answers a new instance of the class represented by the
 * receiver, created by invoking the default (i.e. zero-argument)
 * constructor. If there is no such constructor, or if the
 * creation fails (either because of a lack of available memory or
 * because an exception is thrown by the constructor), an
 * InstantiationException is thrown. If the default constructor
 * exists, but is not accessible from the context where this
 * message is sent, an IllegalAccessException is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a new instance of the class represented by the receiver.
 * @throws		IllegalAccessException if the constructor is not visible to the sender.
 * @throws		InstantiationException if the instance could not be created.
 */
private native Object newInstanceImpl()
	throws IllegalAccessException, InstantiationException;

/**
 * Answers a string describing a path to the receiver's appropriate
 * package specific subdirectory, with the argument appended if the
 * argument did not begin with a slash. If it did, answer just the
 * argument with the leading slash removed.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *					the path to the resource.
 * @param		resName	String
 *					the name of the resource.
 *
 * @see			#getResource
 * @see			#getResourceAsStream
 */
private String toResourceName(String resName) {
	// Turn package name into a directory path
	if (resName.length() > 0 && resName.charAt(0) == '/')
		return resName.substring(1);

	String qualifiedClassName = getName();
	int classIndex = qualifiedClassName.lastIndexOf('.');
	if (classIndex == -1) return resName; // from a default package
	return qualifiedClassName.substring(0, classIndex + 1).replace('.', '/') + resName;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	// Note change from 1.1.7 to 1.2: For primitive types,
	// return just the type name.
	if (isPrimitive()) return getName();
	return (isInterface() ? "interface " : "class ") + getName();
}

/**
 * Returns the Package of which this class is a member.
 * A class has a Package iff it was loaded from a SecureClassLoader
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Package the Package of which this class is a member or null
 *
 */
public Package getPackage() {
	return getClassLoaderImpl().getPackage(getPackageName());
}

static Class getPrimitiveClass(String name) {
	if (name.equals("float"))
		return new float[0].getClass().getComponentType();
	if (name.equals("double"))
		return new double[0].getClass().getComponentType();
	if (name.equals("int"))
		return new int[0].getClass().getComponentType();
	if (name.equals("long"))
		return new long[0].getClass().getComponentType();
	if (name.equals("char"))
		return new char[0].getClass().getComponentType();
	if (name.equals("byte"))
		return new byte[0].getClass().getComponentType();
	if (name.equals("boolean"))
		return new boolean[0].getClass().getComponentType();
	if (name.equals("short"))
		return new short[0].getClass().getComponentType();
	if (name.equals("void")) {
		try {
			java.lang.reflect.Method method = Runnable.class.getMethod("run", new Class[0]);
			return method.getReturnType();
		} catch (Exception e) {
			com.ibm.oti.vm.VM.dumpString("Cannot initialize Void.TYPE\n");
		}
	}
	throw new Error("Unknown primitive type: " + name);
}

/**
 * Returns the assertion status for this class.
 * Assertion is enabled/disabled based on
 * classloader default, package or class default at runtime
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the assertion status for this class
 */
public boolean desiredAssertionStatus() {
	ClassLoader cldr = getClassLoaderImpl();
	if (cldr != null) {
		return cldr.getClassAssertionStatus(getName());
	}
	return false;
}

/**
 * Answer the class at depth.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 *	 2) The item at index zero describes the caller of this method.
 *
 * @param 		depth
 * @return		the class at the given depth
 */
static final native Class getStackClass(int depth);

/**
 * Walk the stack and answer an array containing the maxDepth
 * most recent classes on the stack of the calling thread.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 *	 2) The item at index zero in the result array describes the caller of
 *		the caller of this method.
 *
 * @param 		maxDepth			maximum depth to walk the stack, -1 for the entire stack
 * @param 		stopAtPrivileged	stop at priviledged classes
 * @return		the array of the most recent classes on the stack
 */
static final native Class[] getStackClasses(int maxDepth, boolean stopAtPrivileged);

/**
 * Called from JVM_ClassDepth.
 * Answers the index in the stack of the first method which
 * is contained in a class called <code>name</code>. If no
 * methods from this class are in the stack, return -1.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name String
 *					the name of the class to look for.
 * @return		int
 *					the depth in the stack of a the first
 *					method found.
 */
static int classDepth (String name) {
	Class[] classes = getStackClasses(-1, false);
	for (int i=1; i<classes.length; i++)
		if (classes[i].getName().equals(name))
			return i - 1;
	return -1;
}

/**
 * Called from JVM_ClassLoaderDepth.
 * Answers the index in the stack of thee first class
 * whose class loader is not a system class loader.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the frame index of the first method whose class was loaded by a non-system class loader.
 */
static int classLoaderDepth() {
	// Now, check if there are any non-system class loaders in
	// the stack up to the first privileged method (or the end
	// of the stack.
	Class[] classes = getStackClasses(-1, true);
	for (int i=1; i<classes.length; i++) {
		ClassLoader cl = classes[i].getClassLoaderImpl();
		if (!cl.isSystemClassLoader()) return i - 1;
	}
	return -1;
}

/**
 * Called from JVM_CurrentClassLoader.
 * Answers the class loader of the first class in the stack
 * whose class loader is not a system class loader.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the most recent non-system class loader.
 */
static ClassLoader currentClassLoader() {
	// Now, check if there are any non-system class loaders in
	// the stack up to the first privileged method (or the end
	// of the stack.
	Class[] classes = getStackClasses(-1, true);
	for (int i=1; i<classes.length; i++) {
		ClassLoader cl = classes[i].getClassLoaderImpl();
		if (!cl.isSystemClassLoader()) return cl;
	}
	return null;
}

/**
 * Called from JVM_CurrentLoadedClass.
 * Answers the first class in the stack which was loaded
 * by a class loader which is not a system class loader.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the most recent class loaded by a non-system class loader.
 */
static Class currentLoadedClass() {
	// Now, check if there are any non-system class loaders in
	// the stack up to the first privileged method (or the end
	// of the stack.
	Class[] classes = getStackClasses(-1, true);
	for (int i=1; i<classes.length; i++) {
		ClassLoader cl = classes[i].getClassLoaderImpl();
		if (!cl.isSystemClassLoader()) return classes[i];
	}
	return null;
}

}
