
package com.ibm.oti.vm;

import java.util.Vector;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * Represents the running virtual machine. All VM specific API
 * are implemented on this class.
 * <p>
 * Note that all methods in VM are static. There is no singleton
 * instance which represents the actively running VM.
 */
public final class VM {

/**
 * Prevents this class from being instantiated.
 */
private VM() {
}

/**
 * Answer the class at depth.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 *	 2) The item at index zero describes the caller of this method.
 *
 */
static final native Class getStackClass(int depth);

/**
 * Walk the stack and answer the ClassLoader at the specified depth
 * on the stack of the calling thread.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 *	 2) The item at depth one is the caller of the caller of this method
 *
 */
static final native ClassLoader getStackClassLoader(int depth);

/**
 * Walk the stack and answer the most recent non-null and non-bootstrap
 * ClassLoaders on the stack of the calling thread. If no such ClassLoader
 * is found, null is returned.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 */
static public final native ClassLoader getNonBootstrapClassLoader();

/**
 * Initialize the classloader.
 *
 * @param 		loader ClassLoader
 *					the ClassLoader instance
 * @param 		bootLoader boolean
 *					true for the bootstrap class loader
 */
public final static native void initializeClassLoader(ClassLoader loader, boolean bootLoader);

/**
 * Return whether or not a ClassLoader is the bootstrap class loader.
 *
 * @param 		loader ClassLoader,
 *					the ClassLoader instance to test
 */
static private final native boolean isBootstrapClassLoader(ClassLoader loader);

/**
 * Native used to find and load a class using the VM
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		java.lang.Class
 *					the class or null.
 * @param 		className String
 *					the name of the class to search for.
 *				classLoader
 *					the classloader to do the work
 */
static native Class findClassOrNull(String className, ClassLoader classLoader);

private static ClassLoader caller2ClassLoader() {
	ClassLoader loader = getStackClassLoader(3);
	if (isBootstrapClassLoader(loader)) return null;
	return loader;
}

/**
 * Native used to dump a string to the system console for debugging.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		str String
 *					the String to display
 */
public static native void dumpString(String str);

/**
 * Native used to set the classpath for an OTI implemented classloader
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		classLoader
 *					the classloader to give the path to.
 *				classPath
 *					the string used as the class path.
 */
public static void setClassPathImpl(ClassLoader classLoader, String classPath) {
}

/**
 * Tags a class as hot-swap enabled, this flag is intended for
 * use by OTI implemented class loaders.  This flag implies that
 * when a class is changed in the IDE it's replacement can be found
 * by searching the bootstrap classpath and classpath entries.
 *
 * User implemented classLoaders can set this flag if they respect
 * these limitations.
 *
 * @param 		hotSwapClass
 *					the Class to tag as hot-swap enabled.
 *
 * @author		OTI
 * @version		initial
 *
 * @deprecated	No longer used for hot-swap
 */
public static void enableClassHotSwap(Class hotSwapClass) {
}

/**
 * Sets the ProtectionDomain of the first arg.
 * <p>
 * This method is for internal use only.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		targetClass Class
 *					the class to set the protection domain of.
 *				pd ProtectionDomain
 *					targetClass' new ProtectionDomain.
 *
 * @see			java.lang.Class
 */
static native void setPDImpl(Class targetClass, Object pd);

/**
 * Get the classpath entry that was used to load the class that is the arg.
 * <p>
 * This method is for internal use only.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		targetClass Class
 *					the class to set the classpath of.
 *
 * @see			java.lang.Class
 */
static native int getCPIndexImpl(Class targetClass);

/**
 * Does internal initializaion required by VM.
 *
 * @author		OTI
 * @version		initial
 */
static void initializeVM() {
}

/**
 * Internal use only. Shutdown the JCL.
 * Do not use Threads, as they are already shutdown.
 *
 * @author		OTI
 * @version		initial
 */
private static void shutdown() {
}

static final int CPE_TYPE_UNKNOWN = 0;
static final int CPE_TYPE_DIRECTORY = 1;
static final int CPE_TYPE_JAR = 2;
static final int CPE_TYPE_TCP = 3;
static final int CPE_TYPE_JXE = 4;
static final int CPE_TYPE_UNUSABLE = 5;
static final int CPE_TYPE_PALMDB = 6;
static final native int getClassPathEntryType(Object classLoader, int cpIndex);

/**
 * Returns command line arguments passed to the VM. Internally these are broken into
 * optionString and extraInfo. This only returns the optionString part.
 * <p>
 * @return		a String array containing the optionString part of command line arguments
 *
 * @author		OTI
 * @version		initial
 */
public static native String [] getVMArgs();

static native int getClassPathCount();
static native byte[] getPathFromClassPath(int index);

/**
 * Answer if native implementations should be used.
 */
public final static boolean useNatives() {
	return useNativesImpl();
}

private final static native boolean useNativesImpl();

}
