package com.ibm.oti.vm;

import java.io.*;

/**
 * <p>This class provides utilities to load/unload/relocate jxes.
 *
 * <pre>
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corp. 1999, 2004  All Rights Reserved
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 * </pre>
 */
public final class JxeUtil {

	private static Jxe[] baseJxes;

	static {
		try {
			System.loadLibrary("iverel" + System.getProperty("com.ibm.oti.vm.library.version", "22"));
		} catch (UnsatisfiedLinkError e) {
		}
	}

/**
 * Answer the rom image pointer of a jxe.
 */
private static native long nativeGetRomImagePointerFromJxePointer(long jxePointer);

/**
 * Answer the rom segment pointer of a rom image.
 */
private static native long nativeGetRomSegmentPointerFromRomImagePointer(long romImagePointer);

/**
 * Answer the string form of a relocation message.
 */
private static native String nativeGetRelocationMessage(int relocationMessageNumber);

/**
 * Answer the string form of a relocation message.
 */
static String getRelocationMessage(int relocationMessageNumber) {
	return nativeGetRelocationMessage(relocationMessageNumber);
}

/**
 * Return a byte[] to pass to defineClass() to load className, or
 * null if the class does not exist.
 */
public static byte[] getRomClass(Jxe jxe, String className) {
	byte romClassCookie[] = new byte[36];
	if (jxe.romSegmentPointer == 0L) return null;
	if (!nativeGetRomClassCookie(jxe.romSegmentPointer, jxe.getJxePointer(), className, romClassCookie))
		return null;
	return romClassCookie;
}

private static native boolean nativeGetRomClassCookie(long romSegmentPointer, long jxePointer, String className, byte []romClassCookie);

/**
 * Load a rom image.
 */
private static native long nativeRomImageLoad(long pointer, ClassLoader classLoader, long jxePointer, long jxeAlloc);

/**
 * Load a rom image.
 */
public static void romImageLoad(Jxe jxe, ClassLoader classLoader) throws JxeException {
	long romImagePointer;
	long romSegmentPointer;

	if (jxe.romSegmentPointer != 0L)
		throw new JxeException(com.ibm.oti.util.Msg.getString("K0228"));

	romImagePointer = nativeGetRomImagePointerFromJxePointer(jxe.getJxePointer());
	if (0 == romImagePointer)
		throw new JxeException(com.ibm.oti.util.Msg.getString("K01a1"));

	romSegmentPointer = nativeRomImageLoad(romImagePointer, classLoader, jxe.getJxePointer(), jxe.getJxeAlloc());
	if (0L == romSegmentPointer)
		throw new JxeException(com.ibm.oti.util.Msg.getString("K01a2"));

	jxe.romSegmentPointer = romSegmentPointer;
}

/**
 * Relocate a jxe in place.
 */
private static native int nativeRelocateJxeInPlace(long pointer);

/**
 * Relocate a jxe in place.  The system memory represented by the jxePointer argument
 * will be updated so that it is located to the that address.  The system memory must
 * allow direct read and write access.
 */
static void relocateJxeInPlace(long jxePointer) throws JxeException {
	int rc;

	rc = nativeRelocateJxeInPlace(jxePointer);
	if (0 != rc) {
		throw JxeException.jxeExceptionFromRelocationReturnCode(rc);
	}
}

/**
 * Answer the list of base jxes as pointers.
 */
private static native long[] nativeGetBaseJxePointers();

/**
 * Answer the list of base jxes as pointers.
 */
public static Jxe[] getBaseJxes() {
	long[]   baseJxePointers = null;
	long romImagePointer;

	if (null != baseJxes) return (Jxe[])baseJxes.clone();

	try {
		baseJxePointers = nativeGetBaseJxePointers();
	} catch (UnsatisfiedLinkError e) {}
	if (null == baseJxePointers) {
		baseJxes = new Jxe[0];
		return baseJxes;
	}

	baseJxes = new Jxe[baseJxePointers.length];

	for (int i=0; i<baseJxes.length; i++) {
		baseJxes[i] = new Jxe(baseJxePointers[i], false);
		romImagePointer = nativeGetRomImagePointerFromJxePointer(baseJxes[i].getJxePointer());
		baseJxes[i].romSegmentPointer = nativeGetRomSegmentPointerFromRomImagePointer(romImagePointer);
	}

	return (Jxe[])baseJxes.clone();
}

/**
 * Answer the value of an environment variable.
 */
private static native String nativeGetEnvironmentVariable(String key);

/**
 * Answer the value of an environment variable.
 */
public static String getEnvironmentVariable(String key) {
	return nativeGetEnvironmentVariable(key);
}

/**
 * Native to perform structural verification of a jxe.
 */
private static native int nativeVerifyJxe(long pointer);

/**
 * Perform structural verification of a jxe.
 */
public static void verifyJxe(long romImageAddress) throws JxeException {
	int rc;

	rc = nativeVerifyJxe(romImageAddress);
	if (0 != rc) {
		throw JxeException.jxeExceptionFromRelocationReturnCode(rc);
	}
}

/**
 * Answer the rom image version.
 */
private static native int nativeGetCurrentRomImageVersion();

/**
 * Answer the rom image version.
 */
public static long getCurrentRomImageVersion() {
	return nativeGetCurrentRomImageVersion();
}

/**
 * Answer the lowest compatible rom image version.
 */
private static native int nativeGetLowestRomImageVersion();

/**
 * Answer the lowest compatible rom image version.
 */
public static long getLowestRomImageVersion() {
	return nativeGetLowestRomImageVersion();
}

/**
 * Register the jxe so its memory will not be free'd
 */
static boolean registerJxe(Jxe jxe) {
	long romImagePointer = nativeGetRomImagePointerFromJxePointer(jxe.getJxePointer());
	if (0 == romImagePointer) return false;
	nativeRegisterJxe(romImagePointer, jxe.getJxePointer(), jxe.getJxeAlloc());
	return true;
}

static native void nativeRegisterJxe(long romImagePointer, long jxePointer, long jxeAlloc);

/**
 * Unregister the jxe so its memory will be free'd
 */
static boolean unregisterJxe(Jxe jxe) {
	long romImagePointer = nativeGetRomImagePointerFromJxePointer(jxe.getJxePointer());
	if (0 == romImagePointer) return false;
	nativeUnregisterJxe(romImagePointer);
	return true;
}

static native void nativeUnregisterJxe(long romImagePointer);
}
