package com.ibm.oti.vm;

import java.io.*;
import java.util.*;

/**
 * This class is used to access a ZIP file which is stored in OS memory.
 * Note that compressed zip files are not supported.
 *
 * <pre>
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corp. 1999, 2004 All Rights Reserved
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 * </pre>
 */
class JxeResourceTable {
	private static final boolean useNative = com.ibm.oti.vm.VM.useNatives();
	private Hashtable pointerTable;

	private InputStream	iStream;
	private JxeResource	osmEntry;
	private int			offset;
	private long			pointer;
	private byte[] header = new byte[LOCHDR];
	private byte[] nameBytes = new byte[16];
	private char[] nameChars = new char[0];

	private static final int LOCHDR = 30;
	private static final int SIG_LOCAL  = 0x04034B50;
	private static final int SIG_CEN    = 0x02014B50;
	private static final int SIG_ENDCEN = 0x06054B50;

/**
 * Answer a new JxeResourceTable
 */
JxeResourceTable(InputStream stream) throws IOException {
	iStream = stream;
}

/**
 * Answer a new JxeResourceTable initialized from an InputStream
 */
JxeResourceTable(MemInputStream stream) throws IOException {
	iStream = stream;
	pointer = stream.getPointer();
	readEntries();
}

/**
 * Read the entries from the zip file.
 */
private void readEntries() throws IOException {
	offset = 0;
	pointerTable = new Hashtable(15);

	String fileName;
	while ((fileName = getNext()) != null) {
		int size = osmEntry.getSize();
		long lrc = iStream.skip(size);
		offset += size;
		if (lrc != size) throw new IOException(com.ibm.oti.util.Msg.getString("K01a0"));

		if (!fileName.equals("rom.classes"))
			pointerTable.put(fileName, osmEntry);
	}
}

JxeResource getResource() {
	return osmEntry;
}

Hashtable getTable() {
	return pointerTable;
}

/**
 * Gets the next entry, returns false when done.
 */
String getNext() throws IOException {
	long   sig;
	int    gpBits;
	int    method;
	long   compressedSize;
	int    fileNameLength;
	int    extraLength;
	String fileName;
	int    rc;

	osmEntry = null;

	rc = iStream.read(header, 0, header.length);
	sig = getInt(header, 0); // signature 0
	if (SIG_CEN == sig || SIG_ENDCEN == sig)
		return null;
	if (SIG_LOCAL != sig) throw new IOException(com.ibm.oti.util.Msg.getString("K019c"));
	if (rc != LOCHDR)
		throw new IOException(com.ibm.oti.util.Msg.getString("K01a0"));
	offset += rc;

	// version 4
	gpBits = getShort(header, 6); // flags 6
	method = getShort(header, 8); // method 8
	// time 10
	// date 12
	// crc 14
	compressedSize = getInt(header, 18); // compressedSize 18
	// size 22
	fileNameLength = getShort(header, 26); // fileNameLength 26
	extraLength    = getShort(header, 28); // extraLength 28

//	if (0x10 != (version & 0x10)) throw new IOException("Invalid Zip version");
	if (Jxe.STORED != method && Jxe.DEFLATED != method)
		throw new IOException(com.ibm.oti.util.Msg.getString("K019d"));
	if (0 != (0x01 & gpBits))     throw new IOException(com.ibm.oti.util.Msg.getString("K019e"));
	if (0 != (0x80 & gpBits))     throw new IOException(com.ibm.oti.util.Msg.getString("K019f"));

	if (fileNameLength > nameBytes.length)
		nameBytes = new byte[fileNameLength + 32];
	rc = iStream.read(nameBytes, 0, fileNameLength);
	if (rc != fileNameLength) throw new IOException(com.ibm.oti.util.Msg.getString("K01a0"));
	offset += fileNameLength;
	if (useNative)
		fileName = com.ibm.oti.util.Util.convertFromUTF8(nameBytes, 0, fileNameLength);
	else {
		if (nameChars.length < nameBytes.length)
			nameChars = new char[nameBytes.length];
		fileName = com.ibm.oti.util.Util.convertUTF8WithBuf(nameBytes, nameChars, 0, fileNameLength);
	}

	rc = (int)iStream.skip(extraLength);
	if (rc != extraLength) throw new IOException(com.ibm.oti.util.Msg.getString("K01a0"));
	offset += extraLength;

	osmEntry = new JxeResource(pointer + offset,(int)compressedSize, method);

	return fileName;
}

/**
 * Gets the 4 byte int.
 */
private long getInt(byte[] bytes, int offset) {
	return ((bytes[offset+2] & 0xff) << 16) +
		((bytes[offset+1] & 0xff) << 8) +
		(bytes[offset] & 0xff) +
		((long)(bytes[offset+3] & 0xff) << 24);
}

/**
 * Gets the 2 byte int.
 */
private int getShort(byte[] bytes, int offset) {
	return ((bytes[offset+1] & 0xff) << 8) +
		(bytes[offset] & 0xff);
}
}
