package com.ibm.oti.vm;

import java.io.*;
import java.util.*;

/**
 * <p>This class is a model of a loaded jxe.
 *
 * <pre>
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corp. 1999, 2004  All Rights Reserved
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 * </pre>
 */
public final class Jxe {
	long                    romSegmentPointer;
	private boolean		allocated;
	private String		uuid;
	private long		jxePointer;
	private Hashtable       resTable;

	static final int DEFLATED = 8;
	static final int STORED = 0;

/**
 * Answers a new Jxe from a stream of bytes; these bytes
 * will be copied to system memory and relocated.
 */
public static Jxe fromInputStream(InputStream iStream, int length) throws JxeException, IOException {
	long malloc = nativeMalloc(length + 7);
	if (0 == malloc) throw new OutOfMemoryError(com.ibm.oti.util.Msg.getString("K019a", length + 7));
	// Make sure the pointer is 8 byte aligned
	int offset = (int)(malloc & 7);
	long pointer = offset == 0 ? malloc : malloc + (8 - offset);

	try {
		copyStreamToMemory(iStream, length, pointer);
	} catch (IOException e) {
		nativeFree(malloc);
		throw e;
	}

	Jxe jxe;
	try {
		jxe = new Jxe(malloc, true);
		JxeMetaData metaData = jxe.getJxeMetaData();
		JxeUtil.relocateJxeInPlace(pointer);
		jxe.uuid = metaData.getUuid();
	}
	catch (JxeException e) {
		nativeFree(malloc);
		throw e;
	}
	JxeUtil.registerJxe(jxe);
	return jxe;
}

static void copyStreamToMemory(InputStream iStream, int size, long pointer) throws IOException {
	byte[] buffer = new byte[size < 4096 ? size : 4096];
	int read = 0;
	while (read < size) {
		int left = size - read;
		int chunk = iStream.read(buffer, 0, left <= buffer.length ? left : buffer.length);
		if (-1 == chunk) throw new IOException(com.ibm.oti.util.Msg.getString("K019b", read));
		nativeMemcpy(pointer + read, buffer, 0, chunk);
		read += chunk;
	}
}

/**
 * Answers a new Jxe from the bytes in a File; these bytes
 * will be copied to system memory and relocated.
 */
public static Jxe fromFile(File file) throws JxeException, IOException {
	int             length;
	FileInputStream fiStream;
	Jxe             jxe;

	if (!file.exists()) {
		throw new FileNotFoundException(file.getPath());
	}

	length = (int) file.length();

	jxe      = null;
	fiStream = new FileInputStream(file);
	try {
		jxe = fromInputStream(fiStream,length);
	}
	finally {
		fiStream.close();
	}

	return jxe;
}

/**
 * Construct a new Jxe with a specified pointer.
 */
Jxe(long pointer, boolean allocated) {
	this.jxePointer = pointer;
	this.allocated = allocated;
}

/**
 * Construct a new Jxe with a specified pointer.
 */
public static Jxe fromPointer(long pointer) throws JxeException {
	JxeUtil.verifyJxe(pointer);
	return new Jxe(pointer, false);
}

/**
 * Answer the pointer to this jxe.
 */
long getJxePointer() {
	if (allocated) {
		int offset = (int)(jxePointer & 7);
		if (offset != 0) return jxePointer + (8 - offset);
	}
	return jxePointer;
}

/**
 * Answer the pointer to the allocated memory.
 */
long getJxeAlloc() {
	return allocated ? jxePointer : 0L;
}

/**
 * Answer the meta data associated with this jxe.
 */
public JxeMetaData getJxeMetaData() {
	return new JxeMetaData(this);
}

/**
 * Return a resource as an InputStream.
 */
public InputStream getResourceAsStream(String name) {
	SecurityManager sm = System.getSecurityManager();
	if (sm != null)
		sm.checkPermission(new JxePermission(getUuid()));

	return internalGetResourceAsStream(name);
}

/**
 * Return a resource as an InputStream.
 */
InputStream internalGetResourceAsStream(String name) {
	if (jxePointer == 0L) return null;
	initializeResTable();

	JxeResource entry;
	while (true) {
		entry = (JxeResource)resTable.get(name);
		if (null != entry) {
			InputStream stream = new MemInputStream(entry.getPointer(), entry.getSize(), this);
			if (entry.getMethod() != STORED)
				return new java.util.zip.InflaterInputStream(stream, new java.util.zip.Inflater(true));
			return stream;
		}

		if (!name.startsWith("/", 0)) return null;
		name = name.substring(1);
	}
}

/**
 * Answer the uuid of the jxe.
 */
public String getUuid() {
	if (uuid == null)
		uuid = getJxeMetaData().getUuid();
	return uuid;
}

/**
 * Free the memory allocated for this jxe.
 */
private void free() {
	if (allocated) {
		if (jxePointer != 0L && romSegmentPointer == 0L) {
			nativeFree(jxePointer);
			jxePointer = 0;
		}
	}
}

public void finalize() {
	if (!JxeUtil.unregisterJxe(this)) free();
}

/**
 * Initialize the resource table.
 */
private void initializeResTable() {
	JxeResourceTable resources;

	if (null != resTable) return;

	long pointer;
	try {
		resources = new JxeResourceTable(new MemInputStream(getJxePointer(), Integer.MAX_VALUE, null));
	} catch (IOException e) {
		resTable = new Hashtable(0);
		return;
	}

	resTable = resources.getTable();
}

private static native void nativeMemcpy(long pointer, byte[] bytes, int offset, int length);
private static native long nativeMalloc(int length);
private static native void nativeFree(long pointer);

}
