/*
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * $Id: StylesheetRoot.java,v 1.54 2004/02/16 20:32:32 minchau Exp $
 */
package org.apache.xalan.templates;

import java.util.Hashtable;
import java.util.Properties;

import javax.xml.transform.ErrorListener;
import javax.xml.transform.Templates;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.sax.TemplatesHandler;

import org.apache.xalan.processor.XSLTSchema;

import org.apache.xml.utils.TreeWalker;
import org.apache.xml.utils.DOM2Helper;
import org.apache.xml.utils.PrefixResolver;
import org.apache.xpath.XPathContext;
import org.apache.xpath.WhitespaceStrippingElementMatcher;

import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;
import org.xml.sax.SAXException;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import java.io.IOException;

/**
 * A wrapper class for StylesheetRoot
 */
public final class StylesheetRootProxy implements Templates, XSLTVisitable, 
                WhitespaceStrippingElementMatcher, PrefixResolver
{
        private StylesheetRoot m_root = null;
  /**
   * Uses an XSL stylesheet document.
   * @throws TransformerConfigurationException if the baseIdentifier can not be resolved to a URL.
   */
  public StylesheetRootProxy(ErrorListener errorListener) throws TransformerConfigurationException
  {
    m_root = new StylesheetRoot(errorListener);
  }

  public StylesheetRootProxy(XMLReader reader, InputSource source) throws SAXException, IOException{
          reader.parse(source);
          m_root = (StylesheetRoot)((TemplatesHandler) reader.getContentHandler()).getTemplates();          
  }
  
  public StylesheetRootProxy(TemplatesHandler builder, Node node) throws SAXException, IOException{
          TreeWalker walker = new TreeWalker(builder,
                                             new DOM2Helper(),
                                             builder.getSystemId());

          walker.traverse(node);
          m_root = (StylesheetRoot) builder.getTemplates();          
  }  
  /**
   * Creates a wrapper for StylesheetRoot and retains a pointer to the schema used to create this
   * StylesheetRoot.  The schema may be needed later for an element-available() function call.
   * 
   * @param schema The schema used to create this stylesheet
   * @throws TransformerConfigurationException if the baseIdentifier can not be resolved to a URL.
   */
  public StylesheetRootProxy(XSLTSchema schema, ErrorListener listener) throws TransformerConfigurationException
  {
        m_root = new StylesheetRoot(schema, listener); 
  }

  //============== Templates Interface ================

  /**
   * Create a new transformation context for this Templates object.
   *
   * @return A Transformer instance, never null.
   */
  public Transformer newTransformer()
  {
    return m_root.newTransformer();
  }
  

  public Properties getDefaultOutputProps()
  {
    return m_root.getDefaultOutputProps();
  }

  /**
   * Get the static properties for xsl:output.  The object returned will
   * be a clone of the internal values, and thus it can be mutated
   * without mutating the Templates object, and then handed in to
   * the process method.
   *
   * <p>For XSLT, Attribute Value Templates attribute values will
   * be returned unexpanded (since there is no context at this point).</p>
   *
   * @return A Properties object, not null.
   */
  public Properties getOutputProperties()
  {    
    return (Properties)getDefaultOutputProps().clone();
  }

  //============== End Templates Interface ================  

  //============== XSLTVisitable Interface ================
               
  /**
   * Call the children visitors.
   * @param visitor The visitor whose appropriate method will be called.
   */
  public void callVisitors(XSLTVisitor visitor)
  {
        m_root.callVisitors(visitor);
  }
  
  //============== End XSLTVisitable Interface ================
  
  //============== WhitespaceStrippingElementMatcher Interface ================
  
  /**
   * Get information about whether or not an element should strip whitespace.
   * @see <a href="http://www.w3.org/TR/xslt#strip">strip in XSLT Specification</a>
   *
   * @param support The XPath runtime state.
   * @param targetElement Element to check
   *
   * @return true if the whitespace should be stripped.
   *
   * @throws TransformerException
   */
  public boolean shouldStripWhiteSpace(
          XPathContext support, Element targetElement) throws TransformerException
  {
    return m_root.shouldStripWhiteSpace(support, targetElement);
  }
  
  /**
   * Get information about whether or not whitespace can be stripped.
   * @see <a href="http://www.w3.org/TR/xslt#strip">strip in XSLT Specification</a>
   *
   * @return true if the whitespace can be stripped.
   */
  public boolean canStripWhiteSpace()
  {
    return m_root.canStripWhiteSpace();
  }

  //============= End WhitespaceStrippingElementMatcher Interface =============
  //============== PrefixResolver Interface ================
    
  /**
   * Return the base identifier.
   *
   * @return The base identifier 
   */
  public String getBaseIdentifier()
  {
    return m_root.getBaseIdentifier();
  }
  /**
   * Fullfill the PrefixResolver interface.  Calling this for this class 
   * will throw an error.
   *
   * @param prefix The prefix to look up, which may be an empty string ("") 
   *               for the default Namespace.
   * @param context The node context from which to look up the URI.
   *
   * @return null if the error listener does not choose to throw an exception.
   */
  public String getNamespaceForPrefix(String prefix, org.w3c.dom.Node context)
  {
    return m_root.getNamespaceForPrefix(prefix, context);
  }

  /**
   * Given a namespace, get the corrisponding prefix.
   * 9/15/00: This had been iteratively examining the m_declaredPrefixes
   * field for this node and its parents. That makes life difficult for
   * the compilation experiment, which doesn't have a static vector of
   * local declarations. Replaced a recursive solution, which permits
   * easier subclassing/overriding.
   *
   * @param prefix non-null reference to prefix string, which should map 
   *               to a namespace URL.
   *
   * @return The namespace URL that the prefix maps to, or null if no 
   *         mapping can be found.
   */
  public String getNamespaceForPrefix(String prefix)
  {
        return m_root.getNamespaceForPrefix(prefix); 
  }
  
  /**
   * @see PrefixResolver#handlesNullPrefixes()
   */
  public boolean handlesNullPrefixes() {
          return m_root.handlesNullPrefixes();
  } 
  
  /**
   * Required by the debugger tool
   * @return The underline StylesheetRoot object
   */
  public StylesheetRoot getStylesheetRoot()
  {
    return m_root;
  }   
      
}
