/*
 * @(#)src/classes/sov/java/security/Security.java, security, as142, 20050517 1.35.2.3
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v1.4.2
 * (C) Copyright IBM Corp. 1998, 2004. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * ===========================================================================
 */



/*
 * ===========================================================================
 * Change activity:
 *
 * Reason  Date   Origin   Description
 * ------  ----   ------   ----------------------------------------------------
 * 009058  280300 hdclw    ScJVM: Dirty checks
 * 025627  241000 hdejs    ScJVM: Add parameter to SetJVMUnresettable...
 * 042762  190402 cheesemp Updated initializeStatic() to use correct providers
 * 043014  230402 pabbott  Correct name of JCE provider class
 * 043191  100502 pabbott  Add boostrap provider
 * 052302  270602 pabbott  Make deleteBootstrap create an instance of MD
 * 055583  270902 pabbott  Move bootstrap provider to end of list
 * 055691  031002 pabbott  Allow retry of deleteBootstrap on failure
 * 055864  201002 pabbott  Fix out of order recursive load of providers
 * 056401  111102 pabbott  Add configurable digestProvider name
 * 056542  131102 pabbott  Remove digest reload from deleteBootstrap
 * 056238  130103 pabbott  Create bootstrap provider.
 * 058577  040303 pabbott  Synchronize message digest check
 * 059503  300403 ansriniv SVT: JCE hangs for heavy stress 
 * 063626  130803 pabbott  Simplify provider load debug messages
 * 070846  050304 eldergil Ensure algorithmCache correct (prevent recursion)
 * 083188  180205 hardillb Correct a deadlock with classloader 
 * 084478  180405 hardillb back out 83188
 * 
 */

package java.security;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import com.ibm.jvm.Constants;
import com.ibm.jvm.ExtendedSystem;
import com.ibm.security.bootstrap.BootstrapProvider;
import com.ibm.security.util.PropertyExpander;
import sun.security.util.Debug;

/**
 * <p>This class centralizes all security properties and common security
 * methods. One of its primary uses is to manage providers.
 *
 */

public final class Security {

    private static final Debug sdebug =
    Debug.getInstance("properties");

    private static final Debug pdebug =
    Debug.getInstance("provider");

    private static final Debug adebug =
    Debug.getInstance("algorithm");

    /* Flag used in provider vector to denote a provider 
     * which is in the process of loading.
     */
    private static final String providerLoading = "Provider Loading...";

    /* Bootstrap provider used when recursion detected 
     */
    private static Provider bootstrapProvider;

    /* The java.security properties 
     */
    private static Properties props;

    /* A vector of providers, in order of priority.
     * This structure is initialised to contain n null
     * elements, where n is the number of statically registered
     * providers listed in the java.security file.
     * The providers vector may contain 4 entry types;
     *   a Provider object,
     *   a reference to the Security.providerLoading flag,
     *   a LoadFailed object or
     *   null
     */
    private static Vector providers;

    /* Where we cache algorithm class names
     */
    private static Hashtable algorithmCache;

    /* Where we record the message digests we are loading
     */
    private static Hashtable digestBeingLoaded;

    /* Where we cache search results
     */
    private static Hashtable searchResultsCache;

    /* Do we need to clear the cache before using it?
     */
    private static boolean searchResultsCacheInvalid = false;

    /* An element in the algorithm cache
     */
    private static class ProviderProperty {
        String className;
        Provider provider;
    }

    /* A provider vector entry which records a failed load, why it failed
     * and the name of the provider class.
     */
    private static class LoadFailed {
        Exception loadException;
        String providerClassName;

        public String toString() {
            return "Load Failed, class="+providerClassName+", reason="+loadException;
        }
    }

    static {
        // doPrivileged here because there are multiple
        // things in initialize that might require privs.
        // (the FileInputStream call and the File.exists call, etc)
        AccessController.doPrivileged(new PrivilegedAction() {
                                          public Object run() {
                                              initialize();
                                              return null;
                                          }
                                      });
    }

    private static void initialize() {
        props                   = new Properties();
        providers               = new Vector(10);
        algorithmCache          = new Hashtable(20);
        searchResultsCache      = new Hashtable(5);
        digestBeingLoaded       = new Hashtable(11);
        bootstrapProvider       = new BootstrapProvider();

        boolean loadedProps = false;
        boolean overrideAll = false;
        String sep = File.separator;

        // first load the system properties file
        // to determine the value of security.overridePropertiesFile
        File propFile =  new File(System.getProperty("java.home") + sep + "lib" + sep +
                                  "security" + sep + Constants.java_security);

        if (propFile.exists()) {
            try {
                FileInputStream is = new FileInputStream(propFile);
                // Inputstream has been buffered in Properties class
                props.load(is);
                is.close();
                loadedProps = true;

                if (sdebug != null) {
                    sdebug.println("reading file " + propFile);
                }
            } catch (IOException e) {
                if (sdebug != null) {
                    sdebug.println("unable to load security properties from " +
                                   propFile);
                    sdebug.printStackTrace(e);
                }
            }
        } else if (sdebug != null) {
            sdebug.println("unable to read file " + propFile);
        }

        if (props.getProperty("security.overridePropertiesFile", "false")
            .equalsIgnoreCase("true")) {

            String extraPropFile = System.getProperty
                                   ("java.security.properties");
            if (extraPropFile != null && extraPropFile.startsWith("=")) {
                overrideAll = true;
                extraPropFile = extraPropFile.substring(1);
            }

            if (overrideAll) {
                props = new Properties();
                if (sdebug != null) {
                    sdebug.println("overriding other security properties files!");
                }
            }

            // now load the user-specified file so its values
            // will win if they conflict with the earlier values
            if (extraPropFile != null) {
                try {
                    URL propURL;

                    extraPropFile = PropertyExpander.expand(extraPropFile);
                    propFile = new File(extraPropFile);
                    if (propFile.exists()) {
                        propURL = new URL("file:" + propFile.getCanonicalPath());
                    } else {
                        propURL = new URL(extraPropFile);
                    }
                    BufferedInputStream bis = new BufferedInputStream
                                              (propURL.openStream());
                    props.load(bis);
                    bis.close();
                    loadedProps = true;

                    if (sdebug != null) {
                        sdebug.println("reading security properties file: " +
                                       propURL);
                        if (overrideAll) {
                            sdebug.println("overriding other security properties files!");
                        }
                    }
                } catch (Exception e) {
                    if (sdebug != null) {
                        sdebug.println("unable to load security properties from " +
                                       extraPropFile);
                        sdebug.printStackTrace(e);
                    }
                }
            }
        }

        /*
         * If we succeed in loading a properties file
         * extract the static provider names and copy them
         * to the providers vector.
         */
        if (loadedProps) {
            if (pdebug != null) {
                pdebug.println("statically registered providers");
            }
            copyStaticProviders();
        } else {
            if (pdebug != null) {
                pdebug.println("no statically registered providers, using defaults");
            }
            defaultStaticProviders();
        }

        if (pdebug != null) {
            for (int i=0; i<providers.size(); i++) {
                pdebug.println("["+i+"] "+providers.elementAt(i));
            }

            pdebug.println();
        }
    }

    /*
     * Initialize to default values, if <java.home>/lib/java.security
     * is not found.
     */
    private static void defaultStaticProviders() {
        providers.add("com.ibm.jsse.IBMJSSEProvider");
        providers.add("com.ibm.crypto.provider.IBMJCE");
        providers.add("com.ibm.security.jgss.IBMJGSSProvider");
        providers.add("com.ibm.security.cert.IBMCertPath");
    }

    /**
     * Don't let anyone instantiate this.
     */
    private Security() {
    }

    /**
     * Loops through provider declarations, which are expected to be
     * of the form:
     *
     * security.provider.1=com.ibm.jsse.IBMJSSEProvider
     * security.provider.2=com.ibm.crypto.provider.IBMJCE
     * etc.
     *
     * and copy the class names into the providers vector.
     * The order determines the default search order when looking for
     * an algorithm.
     */
    private static void copyStaticProviders() {

        int i = 1;

        while (true) {
            String name = props.getProperty("security.provider." + i);
            if (name == null) {
                break;
            } else {
                String fullClassName = name.trim();
                if (fullClassName.length() == 0) {
                    if (pdebug != null) {
                        pdebug.println("invalid entry for security.provider." + i);
                    }
                    break;
                } else {
                    // Get rid of duplicate providers.
                    if (!providers.contains(fullClassName)) {
                        providers.add(fullClassName);
                    }
                    i++;
                }
            }          
        }
    }

    /**
     * Retrieve provider from given position, if the provider
     * class has not been loaded try to load it.The given slot
     * position may contain either;
     *
     * String object      Representing the provider class name
     * Provider object    The loaded provider
     * LoadFailed object  Representing a failed provider load
     * Loading flag       Representing a provider which is in the
     *                    process of loading.
     *
     * @param pos position of provider to return, it is assumed
     *            that this is a legal value, no range checking is 
     *            done.
     *
     * @return Provider the provider or bootstrap provider if
     *                  recursion detected. May be null if load
     *                  failed.
     */
    private static synchronized Provider getProviderAt(int pos) {

        String className = null;

        /*
         * First check to see what is currently in the
         * provider vector at slot position pos.
         */
        Object slotEntry = providers.elementAt(pos);

        if (slotEntry instanceof Provider) {
            return(Provider)slotEntry;
        } else if (slotEntry == providerLoading) {
            return bootstrapProvider;
        } else if (slotEntry instanceof String) {
            className = (String)slotEntry;
        } else if (slotEntry instanceof LoadFailed) {
            className = ((LoadFailed)slotEntry).providerClassName;
        } else {
            // This should never happen, but for safety lets stop here

            if (pdebug != null) {
                pdebug.println("slot "+pos+", contains unrecognised entry, aborting. Type ="+slotEntry.getClass());                
            }

            return null;
        }

        /*
         * If the slot entry is a String or LoadFailed object
         * then try to load the provider.
         */

        providers.setElementAt(providerLoading, pos);

        final String name = className;

        if (pdebug != null) {
            pdebug.println("loading provider ["+className+"] into slot "+pos+" ... "); /*ibm@63626*/
        }

        try {
            Provider loadedProvider = (Provider)AccessController.doPrivileged(
                new PrivilegedExceptionAction() {
                    public Object run() throws Exception {
                        return Provider.loadProvider(name);
                    }
                });

            providers.setElementAt(loadedProvider, pos);

            searchResultsCacheInvalid = true;
            algorithmCache.clear();

            if (pdebug != null) {
                pdebug.println("slot "+pos+" loaded with ["+loadedProvider+"]");
            }

            return loadedProvider;

        } catch (Exception e) {
            LoadFailed lf = new LoadFailed();
            lf.loadException = e;
            lf.providerClassName = name;
            providers.setElementAt(lf, pos);

            if (pdebug != null) {
                pdebug.println("slot "+pos+", load failed, "+e);
                pdebug.printStackTrace(e);
            }

            return null;
        }

    }

    /**
     * Returns the property (if any) mapping the key for the given provider.
     *
     * @param key name of property to get from the named provider. Normally 
     *            an algorithm name of the form "MessageDigest.SHA"
     * @param provider named provider object from which to get the property.
     *
     * @return String propery value, normally a class name or alias name
     *                may be null if key not found.
     */
    private static String getProviderProperty(String key, Provider provider) {

        String prop = provider.getProperty(key);

        if (prop == null) {
            Enumeration enum = provider.keys();
            String matchKey;
            while (enum.hasMoreElements()) {
                matchKey = (String)enum.nextElement();
                if (key.equalsIgnoreCase(matchKey)) {
                    return provider.getProperty(matchKey);
                }
            }
        }
        return prop;
    }

    /**
     * Gets a specified property for an algorithm. The algorithm name
     * should be a standard name.
     *
     * @param algName the algorithm name.
     *
     * @param propName the name of the property to get.
     *
     * @return the value of the specified property.
     *
     * @deprecated This method used to return the value of a proprietary
     * property in the master file of the "SUN" Cryptographic Service
     * Provider in order to determine how to parse algorithm-specific
     * parameters. Use the new provider-based and algorithm-independent
     * <code>AlgorithmParameters</code> and <code>KeyFactory</code> engine
     * classes (introduced in the Java 2 platform) instead.
     */
    public static String getAlgorithmProperty(String algName,
                                              String propName) {

        String matchKey;
        Provider prov;   
        String prop;
        Enumeration enum;

        String key = "Alg." + propName + "." + algName;

        for (int i = 0; i < providers.size(); i++) {

            prov = getProviderAt(i);

            if (prov != null) {

                prop = prov.getProperty(key);

                if (prop != null) {
                    return prop;
                } else {
                    enum = prov.keys();
                    while (enum.hasMoreElements()) {
                        matchKey = (String)enum.nextElement();
                        if (key.equalsIgnoreCase(matchKey)) {
                            return prov.getProperty(matchKey);
                        }
                    }
                }

            }
        }

        return null;
    }

    /**
     * Lookup the algorithm in our list of providers. Process
     * each provider in priority order one at a time looking for
     * either the direct engine property or a matching alias.
     *
     * @param algName Name of the algorithm to find
     * @param engineType Name of engine, i.e. MessageDigest, Signature ...
     *
     * @return ProviderProperty structure containing the algorithm class
     *                          name and the suppling provider.
     */
    private static ProviderProperty getAlgClassName(String algName, String engineType)
    throws NoSuchAlgorithmException
    {
        ProviderProperty pp;
        String key;

        /* First check the cache to see if the required algorithm
         * has already been found.
         */
        if (algName == null) {
            key = engineType;
        } else {
            key = engineType + "." + algName;
        }

        pp = (ProviderProperty)algorithmCache.get(key);

        if (pp != null) {
            return pp;
        }

        /* If the algorithm was not found check each provider 
         * to see if it can supply the algorithm.
         */
        synchronized (Security.class) {
            Provider prov = null;
            for (int i = 0; i < providers.size(); i++) {
                prov = getProviderAt(i);

                if (prov != null) {

                    // Try to get the algorithm class name from the provider
                    String className = getProviderProperty(key, prov);
                    if (className == null) {

                        // Now check the provider's aliases to see if we can find
                        // a match
                        String stdName = 
                        getProviderProperty("Alg.Alias." + engineType + "." + algName,
                                            prov);
                        if (stdName != null) {
                            /*key = engineType + "." + stdName; ibm@70846*/
                            className = getProviderProperty(engineType + "." + stdName, prov); /*ibm@70846*/
                        }
                    }

                    // If we were able to find a class name for the required algorithm
                    // package it up and return it.
                    if (className != null) {
                        pp = new ProviderProperty();
                        pp.className = className;
                        pp.provider = prov;
                        algorithmCache.put(key, pp);
                        return pp;
                    }
                }
            }

        }
        throw new NoSuchAlgorithmException(algName.toUpperCase() + " " +
                                           engineType + " not available");
    }


    /**
     * Lookup the algorithm in the spcified provider.
     *
     * @param algName Name of the algorithm to find
     * @param provider Name of provider which will supply the required
     *                 algorithm.
     * @param engineType Name of engine, i.e. MessageDigest, Signature ...
     *
     * @return ProviderProperty structure containing the algorithm class
     *                          name and the suppling provider.
     */
    private static ProviderProperty getAlgClassName(String algName,
                                                    String provider,
                                                    String engineType)
    throws NoSuchAlgorithmException, NoSuchProviderException
    {
        if (provider == null) {
            return getAlgClassName(algName, engineType);
        }

        // check if the provider is installed
        Provider prov = getProvider(provider);
        if (prov == null) {
            throw new NoSuchProviderException("no such provider: " +
                                              provider);
        }

        return getAlgClassName(algName, prov, engineType); 
    }

    /**
     * Lookup the algorithm in the spcified provider.The parameter 
     * provider cannot be null.
     *
     * @param algName Name of the algorithm to find
     * @param provider Provider object which will supply the required
     *                 algorithm.
     * @param engineType Name of engine, i.e. MessageDigest, Signature ...
     *
     * @return ProviderProperty structure containing the algorithm class
     *                          name and the suppling provider.
     */
    private static ProviderProperty getAlgClassName(String algName,
                                                    Provider provider, 
                                                    String engineType) 
    throws NoSuchAlgorithmException
    {
        String key;
        if (engineType.equalsIgnoreCase("SecureRandom") && algName == null)
            key = engineType;
        else
            key = engineType + "." + algName;

        String className = getProviderProperty(key, provider);
        if (className == null) {
            if (engineType.equalsIgnoreCase("SecureRandom") && algName == null) {
                throw new NoSuchAlgorithmException
                ("SecureRandom not available for provider " + provider.getName());
            } else {
                // try algName as alias name
                String stdName = 
                getProviderProperty("Alg.Alias." + engineType + "." + algName,
                                    provider);

                if (stdName != null) {
                    key = engineType + "." + stdName;
                }
                if ((stdName == null)
                    || (className = getProviderProperty(key, provider)) == null) {
                    throw new NoSuchAlgorithmException("no such algorithm: " +
                                                       algName
                                                       + " for provider " +
                                                       provider.getName());
                }
            }
        }

        ProviderProperty entry = new ProviderProperty();
        entry.className = className;
        entry.provider = provider;

        return entry;
    }

    /**
     * Adds a new provider, at a specified position. The position is
     * the preference order in which providers are searched for
     * requested algorithms. Note that it is not guaranteed that this
     * preference will be respected. The position is 1-based, that is,
     * 1 is most preferred, followed by 2, and so on.
     *
     * <p>If the given provider is installed at the requested position,
     * the provider that used to be at that position, and all providers
     * with a position greater than <code>position</code>, are shifted up
     * one position (towards the end of the list of installed providers).
     *
     * <p>A provider cannot be added if it is already installed.
     *
     * <p>First, if there is a security manager, its
     * <code>checkSecurityAccess</code>
     * method is called with the string
     * <code>"insertProvider."+provider.getName()</code>
     * to see if it's ok to add a new provider.
     * If the default implementation of <code>checkSecurityAccess</code>
     * is used (i.e., that method is not overriden), then this will result in
     * a call to the security manager's <code>checkPermission</code> method
     * with a
     * <code>SecurityPermission("insertProvider."+provider.getName())</code>
     * permission.
     *
     * @param provider the provider to be added.
     *
     * @param position the preference position that the caller would
     * like for this provider.
     *
     * @return the actual preference position in which the provider was
     * added, or -1 if the provider was not added because it is
     * already installed.
     *
     * @throws  SecurityException
     *          if a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkSecurityAccess}</code> method
     *          denies access to add a new provider
     *
     * @see #getProvider
     * @see #removeProvider
     * @see java.security.SecurityPermission
     */
    public static synchronized int insertProviderAt(Provider provider,
                                                    int position) {
        if (pdebug != null) {
            pdebug.println("insert provider "+provider+" at "+position);
        }

        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkSecurityAccess("insertProvider."+provider.getName());
        }

        /* First check if the provider is already installed */
        Provider already = getProvider(provider.getName());
        if (already != null) {
            return -1;
        }

        int size = providers.size();
        if (position > size || position <= 0) {
            position = size+1;
        }

        providers.insertElementAt(provider, position-1);

        searchResultsCacheInvalid = true;
        algorithmCache.clear();

        ExtendedSystem.setJVMUnresettableConditionally(               /*SHIRAZ*/
            ExtendedSystem.SCJVM_MODIFYING_SECURITY,
            new String("Modified security in Security.insertProvidersAt()"));/*ibm@25627*//*SHIRAZ*/

        return position;
    }

    /**
     * Adds a provider to the next position available.
     *
     * <p>First, if there is a security manager, its
     * <code>checkSecurityAccess</code>
     * method is called with the string
     * <code>"insertProvider."+provider.getName()</code>
     * to see if it's ok to add a new provider.
     * If the default implementation of <code>checkSecurityAccess</code>
     * is used (i.e., that method is not overriden), then this will result in
     * a call to the security manager's <code>checkPermission</code> method
     * with a
     * <code>SecurityPermission("insertProvider."+provider.getName())</code>
     * permission.
     *
     * @param provider the provider to be added.
     *
     * @return the preference position in which the provider was
     * added, or -1 if the provider was not added because it is
     * already installed.
     *
     * @throws  SecurityException
     *          if a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkSecurityAccess}</code> method
     *          denies access to add a new provider
     *
     * @see #getProvider
     * @see #removeProvider
     * @see java.security.SecurityPermission
     */
    public static int addProvider(Provider provider) {
        return insertProviderAt(provider, 0);
    }

    /**
     * Removes the provider with the specified name.
     *
     * <p>When the specified provider is removed, all providers located
     * at a position greater than where the specified provider was are shifted
     * down one position (towards the head of the list of installed
     * providers).
     *
     * <p>This method returns silently if the provider is not installed.
     *
     * <p>First, if there is a security manager, its
     * <code>checkSecurityAccess</code>
     * method is called with the string <code>"removeProvider."+name</code>
     * to see if it's ok to remove the provider.
     * If the default implementation of <code>checkSecurityAccess</code>
     * is used (i.e., that method is not overriden), then this will result in
     * a call to the security manager's <code>checkPermission</code> method
     * with a <code>SecurityPermission("removeProvider."+name)</code>
     * permission.
     *
     * @param name the name of the provider to remove.
     *
     * @throws  SecurityException
     *          if a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkSecurityAccess}</code> method
     *          denies
     *          access to remove the provider
     *
     * @see #getProvider
     * @see #addProvider
     */
    public static synchronized void removeProvider(String name) {
        if (pdebug != null) {
            pdebug.println("remove provider "+name);
        }

        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkSecurityAccess("removeProvider."+name);
        }

        Provider provider = getProvider(name);
        if (provider != null) {
            providers.remove(provider);

            searchResultsCacheInvalid = true;
            algorithmCache.clear();

            ExtendedSystem.setJVMUnresettableConditionally(           /*SHIRAZ*/
                ExtendedSystem.SCJVM_MODIFYING_SECURITY,
                new String("Modified security in Security.removeProvider()"));/*ibm@25627*//*SHIRAZ*/
        }
    }

    /**
     * Returns an array containing all the installed providers. The order of
     * the providers in the array is their preference order.
     *
     * @return an array of all the installed providers.
     */
    public static synchronized Provider[] getProviders() {
        Vector filtered = new Vector(providers.size());
        Provider slot;

        /* Remove any slot entries in the providers vector 
         * which are not Provider objects, failed load flags etc
         */
        for (int i=0; i<providers.size(); i++) {
            slot = getProviderAt(i);
            if (slot != null) {
                filtered.addElement(slot);
            }
        }
        Provider[] result = new Provider[filtered.size()];
        filtered.copyInto(result);

        return result;
    }

    /**
     * Returns the provider installed with the specified name, if
     * any. Returns null if no provider with the specified name is
     * installed.
     *
     * @param name the name of the provider to get.
     *
     * @return the provider of the specified name.
     *
     * @see #removeProvider
     * @see #addProvider
     */
    public static synchronized Provider getProvider(String name) {

        Provider prov;
        for (int i = 0; i < providers.size(); i++) {
            prov = getProviderAt(i);

            if (prov != null && prov.getName().equals(name)) {
                return prov;
            }
        }
        return null;
    }

    /**
    * Returns an array containing all installed providers that satisfy the
    * specified selection criterion, or null if no such providers have been
    * installed. The returned providers are ordered
    * according to their <a href=
    * "#insertProviderAt(java.security.Provider, int)">preference order</a>.
    *
    * <p> A cryptographic service is always associated with a particular
    * algorithm or type. For example, a digital signature service is
    * always associated with a particular algorithm (e.g., DSA),
    * and a CertificateFactory service is always associated with
    * a particular certificate type (e.g., X.509).
    *
    * <p>The selection criterion must be specified in one of the following two formats:
    * <ul>
    * <li> <i>&lt;crypto_service>.&lt;algorithm_or_type></i> <p> The
    * cryptographic service name must not contain any dots.
    * <p> A
    * provider satisfies the specified selection criterion iff the provider implements the
    * specified algorithm or type for the specified cryptographic service.
    * <p> For example, "CertificateFactory.X.509"
    * would be satisfied by any provider that supplied
    * a CertificateFactory implementation for X.509 certificates.
    * <li> <i>&lt;crypto_service>.&lt;algorithm_or_type> &lt;attribute_name>:&lt attribute_value></i>
    * <p> The cryptographic service name must not contain any dots. There
    * must be one or more space characters between the <i>&lt;algorithm_or_type></i>
    * and the <i>&lt;attribute_name></i>.
    * <p> A provider satisfies this selection criterion iff the
    * provider implements the specified algorithm or type for the specified
    * cryptographic service and its implementation meets the
    * constraint expressed by the specified attribute name/value pair.
    * <p> For example, "Signature.SHA1withDSA KeySize:1024" would be
    * satisfied by any provider that implemented
    * the SHA1withDSA signature algorithm with a keysize of 1024 (or larger).
    *
    * </ul>
    *
    * <p> See Appendix A in the <a href=
    * "../../../guide/security/CryptoSpec.html#AppA">
    * Java Cryptogaphy Architecture API Specification &amp; Reference </a>
    * for information about standard cryptographic service names, standard
    * algorithm names and standard attribute names.
    *
    * @param filter the criterion for selecting
    * providers. The filter is case-insensitive.
    *
    * @return all the installed providers that satisfy the selection
    * criterion, or null if no such providers have been installed.
    *
    * @ see #getProviders(java.util.Map)
    */
    public static Provider[] getProviders(String filter) {
        String key = null;
        String value = null;
        int index = filter.indexOf(':');

        if (index == -1) {
            key = new String(filter);
            value = "";
        } else {
            key = filter.substring(0, index);
            value = filter.substring(index + 1);
        }

        Hashtable hashtableFilter = new Hashtable(1);
        hashtableFilter.put(key, value);

        return getProviders(hashtableFilter);
    }

    /**
     * Returns an array containing all installed providers that satisfy the specified
     * selection criteria, or null if no such providers have been installed.
     * The returned providers are ordered
     * according to their <a href=
     * "#insertProviderAt(java.security.Provider, int)">preference order</a>.
     *
     * <p>The selection criteria are represented by a map.
     * Each map entry represents a selection criterion.
     * A provider is selected iff it satisfies all selection
     * criteria. The key for any entry in such a map must be in one of the
     * following two formats:
     * <ul>
     * <li> <i>&lt;crypto_service>.&lt;algorithm_or_type></i>
     * <p> The cryptographic service name must not contain any dots.
     * <p> The value associated with the key must be an empty string.
     * <p> A provider
     * satisfies this selection criterion iff the provider implements the
     * specified algorithm or type for the specified cryptographic service.
     * <li>  <i>&lt;crypto_service>.&lt;algorithm_or_type> &lt;attribute_name></i>
     * <p> The cryptographic service name must not contain any dots. There
     * must be one or more space characters between the <i>&lt;algorithm_or_type></i>
     * and the <i>&lt;attribute_name></i>.
     * <p> The value associated with the key must be a non-empty string.
     * A provider satisfies this selection criterion iff the
     * provider implements the specified algorithm or type for the specified
     * cryptographic service and its implementation meets the
     * constraint expressed by the specified attribute name/value pair.
     * </ul>
     *
     * <p> See Appendix A in the <a href=
     * "../../../guide/security/CryptoSpec.html#AppA">
     * Java Cryptogaphy Architecture API Specification &amp; Reference </a>
     * for information about standard cryptographic service names, standard
     * algorithm names and standard attribute names.
     *
     * @param filter the criteria for selecting
     * providers. The filter is case-insensitive.
     *
     * @return all the installed providers that satisfy the selection
     * criteria, or null if no such providers have been installed.
     *
     * @see #getProviders(java.lang.String)
     */
    public static Provider[] getProviders(Map filter) {
        // Get all installed providers first.
        // Then only return those providers who satisfy the selection criteria.
        Provider[] allProviders = getProviders();
        Set keySet = filter.keySet();
        HashSet candidates = new HashSet(5);

        // Returns all installed providers
        // if the selection criteria is null.
        if ((keySet == null) || (allProviders == null)) {
            return allProviders;
        }

        boolean firstSearch = true;

        // For each selection criterion, remove providers
        // which don't satisfy the criterion from the candidate set.
        for (Iterator ite = keySet.iterator(); ite.hasNext();) {
            String key = (String)ite.next();
            String value = (String)filter.get(key);

            HashSet newCandidates = getAllQualifyingCandidates(key, value,
                                                               allProviders);
            if (firstSearch) {
                candidates = newCandidates;
                firstSearch = false;
            }

            if ((newCandidates != null) && !newCandidates.isEmpty()) {
                // For each provider in the candidates set, if it
                // isn't in the newCandidate set, we should remove
                // it from the candidate set.
                for (Iterator cansIte = candidates.iterator();
                    cansIte.hasNext();) {
                    Provider prov = (Provider)cansIte.next();
                    if (!newCandidates.contains(prov)) {
                        cansIte.remove();
                    }
                }
            } else {
                candidates = null;
                break;
            }
        }

        if ((candidates == null) || (candidates.isEmpty()))
            return null;

        Object[] candidatesArray = candidates.toArray();
        Provider[] result = new Provider[candidatesArray.length];

        for (int i = 0; i < result.length; i++) {
            result[i] = (Provider)candidatesArray[i];
        }

        return result;
    }

    private static boolean checkSuperclass(Class subclass, Class superclass) {
        while (!subclass.equals(superclass)) {
            subclass = subclass.getSuperclass();
            if (subclass == null) {
                return false;
            }
        }
        return true;
    }

    /*
     * Returns an array of objects: the first object in the array is
     * an instance of an implementation of the requested algorithm
     * and type, and the second object in the array identifies the provider
     * of that implementation.
     * The <code>provider</code> argument can be null, in which case all
     * configured providers will be searched in order of preference.
     */
    static Object[] getImpl(String algorithm, String type, String provider)
    throws NoSuchAlgorithmException, NoSuchProviderException
    {
        if (adebug != null) {
            adebug.println("request for "+type+" "+algorithm+" from "+provider);
            adebug.printStackTrace(null);
        }

        ProviderProperty pp = getAlgClassName(algorithm, provider, type);
        if (adebug != null) {
            adebug.println("request for "+algorithm+" can be met by "+pp.provider);
        }

        Object[] r = createAlgInstance(type, pp);
        if (adebug != null) {
            adebug.println(algorithm+" created, "+r[0].getClass());
        }

        return r;
    }

    static Object[] getImpl(String algorithm, String type, String provider, Object params)
    throws NoSuchAlgorithmException, NoSuchProviderException,InvalidAlgorithmParameterException
    {
        if (adebug != null) {
            adebug.println("request for "+type+" "+algorithm+" from "+provider+" Params=["+params+"]");
            adebug.printStackTrace(null);
        }

        ProviderProperty pp = getAlgClassName(algorithm, provider, type);
        if (adebug != null) {
            adebug.println("request for "+algorithm+" can be met by "+pp.provider);
        }

        Object[] r = createAlgInstance(type, pp, params);
        if (adebug != null) {
            adebug.println(algorithm+" created, "+r[0].getClass());
        }

        return r;
    }

    /*
     * Returns an array of objects: the first object in the array is
     * an instance of an implementation of the requested algorithm
     * and type, and the second object in the array identifies the provider
     * of that implementation.
     * The <code>provider</code> argument cannot be null.
     */
    static Object[] getImpl(String algorithm, String type, Provider provider)
    throws NoSuchAlgorithmException
    {
        if (adebug != null) {
            adebug.println("request for "+type+" "+algorithm+" from ("+provider+")");
            adebug.printStackTrace(null);
        }

        ProviderProperty pp = getAlgClassName(algorithm, provider, type);
        if (adebug != null) {
            adebug.println("request for "+algorithm+" can be met by "+pp.provider);
        }

        Object[] r = createAlgInstance(type, pp);
        if (adebug != null) {
            adebug.println(algorithm+" created, "+r[0].getClass());
        }

        return r;
    }

    static Object[] getImpl(String algorithm, String type, Provider provider, Object params)
    throws NoSuchAlgorithmException, InvalidAlgorithmParameterException
    {
        if (adebug != null) {
            adebug.println("request for "+type+" "+algorithm+" from ("+provider+") Params=["+params+"]");
            adebug.printStackTrace(null);
        }

        ProviderProperty pp = getAlgClassName(algorithm, provider, type);
        if (adebug != null) {
            adebug.println("request for "+algorithm+" can be met by "+pp.provider);
        }

        Object[] r = createAlgInstance(type, pp, params);
        if (adebug != null) {
            adebug.println(algorithm+" created, "+r[0].getClass());
        }

        return r;
    }

    private static Object[] createAlgInstance(String type, ProviderProperty pp)
    throws NoSuchAlgorithmException
    {
        try {
            Object[] result;

            /* If the type is message digest then there is a danger
             * of recursive class instantiation so look in the cache
             * for a key equal to the class name.
             * If we detect recursion build a new class name out of 
             * the bootstrap package name + the algorithm name.
             *
             * ibm@58577 Put code into a synchronize block to prevent multiple
             *           threads thinking they are in a recursive loop.
             */
            if (type.equals("MessageDigest")) {
                String className = pp.className;
                /* 
                 * ibm@59503
                 * Replace synchronized block with thread name added to class 
                 * name in cache. The cache can now have 2 entries both with 
                 * the same class name but for different threads.
                 */
                if (digestBeingLoaded.containsKey(className+Thread.currentThread().getName())) {
                    ProviderProperty npp = new ProviderProperty();
                    npp.className = "com.ibm.security.bootstrap."+
                                    className.substring(className.lastIndexOf(".")+1);
                    npp.provider = bootstrapProvider;
                    result =  createAlgInstance(type, npp, null);
                } else {
                    digestBeingLoaded.put(className+Thread.currentThread().getName(), "value");
                    result = createAlgInstance(type, pp, null);
                    digestBeingLoaded.remove(className+Thread.currentThread().getName());
                }
            } else {
                result = createAlgInstance(type, pp, null);
            }
            return result;

        } catch (InvalidAlgorithmParameterException e) {
            // should not occur
            throw new NoSuchAlgorithmException(e.getMessage());
        }
    }

    private static Object[] createAlgInstance(String type, ProviderProperty pp, Object params)
    throws NoSuchAlgorithmException, InvalidAlgorithmParameterException
    { 
        String className = pp.className;
        Provider provider = pp.provider;

        try {
            // java.security.<type>.Spi is a system class, therefore
            // Class.forName() always works
            Class typeClass;
            if (type.equals("CertificateFactory") ||
                type.equals("CertPathBuilder") ||
                type.equals("CertPathValidator") ||
                type.equals("CertStore")) {
                typeClass = Class.forName("java.security.cert." + type + "Spi");
            } else {
                typeClass = Class.forName("java.security." + type + "Spi");
            }

            /* Load the implementation class using the same class loader that
             * was used to load the associated provider.
             * In order to get the class loader of a class, the caller's class
             * loader must be the same as or an ancestor of the class loader
             * being returned.
             * Since java.security.Security is a system class, it can get the
             * class loader of any class (the system class loader is an
             * ancestor of all class loaders).
             */
            ClassLoader cl = provider.getClass().getClassLoader();
            Class implClass;
            if (cl != null) {
                implClass = cl.loadClass(className);
            } else {
                implClass = Class.forName(className);
            }

            if (checkSuperclass(implClass, typeClass)) {
                Object obj;
                if (type.equals("CertStore")) {
                    Constructor cons = 
                    implClass.getConstructor(new Class[] 
                                             { Class.forName
                                                 ("java.security.cert.CertStoreParameters")});
                    obj = cons.newInstance(new Object[] {params});
                } else
                    obj = implClass.newInstance();

                return new Object[] { obj, provider};
            } else {
                if (adebug != null) {
                    adebug.println("class configured for " + type + ": " + className + " not a " + type);
                    adebug.printStackTrace(null);
                }
                throw new NoSuchAlgorithmException("class configured for " +
                                                   type + ": " + className +
                                                   " not a " + type);
            }
        } catch (ClassNotFoundException e) {
            if (adebug != null) {
                adebug.println("class configured for " + type + " cannot be found, "+e.getMessage());
                adebug.printStackTrace(e);
            }
            throw new NoSuchAlgorithmException("class configured for " +
                                               type + "(provider: " +
                                               provider + ")" + 
                                               "cannot be found.\n" +
                                               e.getMessage());
        } catch (InstantiationException e) {
            if (adebug != null) {
                adebug.println("class " + className + " cannot be instantiated, "+e.getMessage());
                adebug.printStackTrace(e);
            }
            throw (NoSuchAlgorithmException) new NoSuchAlgorithmException("class " + className + 
                                                                          " configured for " + type +
                                                                          "(provider: " + provider + 
                                                                          ") cannot be " +
                                                                          "instantiated.\n").initCause(e);
        } catch (IllegalAccessException e) {
            if (adebug != null) {
                adebug.println("class " + className + " cannot be accessed, "+e.getMessage());
                adebug.printStackTrace(e);
            }
            throw new NoSuchAlgorithmException("class " + className +
                                               " configured for " + type +
                                               "(provider: " + provider +
                                               ") cannot be accessed.\n" +
                                               e.getMessage());
        } catch (SecurityException e) {
            if (adebug != null) {
                adebug.println("class " + className + " cannot be accessed, "+e.getMessage());
                adebug.printStackTrace(e);
            }
            throw new NoSuchAlgorithmException("class " + className +
                                               " configured for " + type +
                                               "(provider: " + provider +
                                               ") cannot be accessed.\n" +
                                               e.getMessage());
        } catch (NoSuchMethodException e) {
            if (adebug != null) {
                adebug.println("constructor for " + className + " cannot be found, "+e.getMessage());
                adebug.printStackTrace(e);
            }
            throw new NoSuchAlgorithmException("constructor for " +
                                               "class " + className + 
                                               " configured for " + type +
                                               "(provider: " + provider +
                                               ") cannot be instantiated.\n" + 
                                               e.getMessage());
        } catch (InvocationTargetException e) {
            if (adebug != null) {
                adebug.println("invalid argument for constructor of " + className + ", "+e.getMessage());
                adebug.printStackTrace(e);
            }
            Throwable t = e.getCause();
            if (t != null && t instanceof InvalidAlgorithmParameterException)
                throw (InvalidAlgorithmParameterException) t;
            else
                throw new InvalidAlgorithmParameterException("constructor " +
                                                             "for class " + className + 
                                                             " configured for " + type +
                                                             "(provider: " + provider +
                                                             ") cannot be instantiated.\n" + 
                                                             e.getMessage());
        }
    }

    /**
     * Gets a security property value.
     *
     * <p>First, if there is a security manager, its
     * <code>checkPermission</code>  method is called with a
     * <code>java.security.SecurityPermission("getProperty."+key)</code>
     * permission to see if it's ok to retrieve the specified
     * security property value..
     *
     * @param key the key of the property being retrieved.
     *
     * @return the value of the security property corresponding to key.
     *
     * @throws  SecurityException
     *          if a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkPermission}</code> method
     *          denies
     *          access to retrieve the specified security property value
     *
     * @see #setProperty
     * @see java.security.SecurityPermission
     */
    public static String getProperty(String key) {
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkPermission(new SecurityPermission("getProperty."+key));
        }

        String name = props.getProperty(key);
        if (name != null)
            name = name.trim(); // could be a class name with trailing ws
        return name;
    }

    /**
     * Sets a security property value.
     *
     * <p>First, if there is a security manager, its
     * <code>checkPermission</code> method is called with a
     * <code>java.security.SecurityPermission("setProperty."+key)</code>
     * permission to see if it's ok to set the specified
     * security property value.
     *
     * @param key the name of the property to be set.
     *
     * @param datum the value of the property to be set.
     *
     * @throws  SecurityException
     *          if a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkPermission}</code> method
     *          denies access to set the specified security property value
     *
     * @see #getProperty
     * @see java.security.SecurityPermission
     */
    public static void setProperty(String key, String datum) {
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkPermission(
                new SecurityPermission("setProperty."+key));
        }

        props.put(key, datum);
        invalidateSMCache(key);
        ExtendedSystem.setJVMUnresettableConditionally(               /*SHIRAZ*/
            ExtendedSystem.SCJVM_MODIFYING_SECURITY,
            new String("Modified security in Security.setProperty()"));/*ibm@25627*//*SHIRAZ*/
    }

    /*
     * Implementation detail:  If the property we just set in
     * setProperty() was either "package.access" or
     * "package.definition", we need to signal to the SecurityManager
     * class that the value has just changed, and that it should
     * invalidate it's local cache values.
     *
     * Rather than create a new API entry for this function,
     * we use reflection to set a private variable.
     */
    private static void invalidateSMCache(String key) {

        final boolean pa = key.equals("package.access");
        final boolean pd = key.equals("package.definition");

        if (pa || pd) {
            AccessController.doPrivileged(new PrivilegedAction() {
                public Object run() {
                    try {
                        /* Get the class via the bootstrap class loader. */
                        Class cl = Class.forName(
                            "java.lang.SecurityManager", false, null);
                        Field f = null;
                        boolean accessible = false;

                        if (pa) {
                            f = cl.getDeclaredField("packageAccessValid");
                            accessible = f.isAccessible();
                            f.setAccessible(true);
                        } else {
                            f = cl.getDeclaredField("packageDefinitionValid");
                            accessible = f.isAccessible();
                            f.setAccessible(true);
                        }
                        f.setBoolean(f, false);
                        f.setAccessible(accessible);
                    } catch (Exception e1) {
                        /* If we couldn't get the class, it hasn't
                         * been loaded yet.  If there is no such
                         * field, we shouldn't try to set it.  There
                         * shouldn't be a security exception, as we
                         * are loaded by boot class loader, and we
                         * are inside a doPrivileged() here.
                         *
                         * NOOP: don't do anything...
                         */
                    }
                    return null;
                }
                });
        }
    }

    /*
    * Returns all providers who satisfy the specified
    * criterion.
    */
    private static HashSet getAllQualifyingCandidates(String filterKey,
                                                      String filterValue,
                                                      Provider[] allProviders) {
        if (searchResultsCacheInvalid == true) {
            searchResultsCache.clear();
            searchResultsCacheInvalid = false;
        }

        String[] filterComponents = getFilterComponents(filterKey,
                                                        filterValue);

        // The first component is the service name.
        // The second is the algorithm name.
        // If the third isn't null, that is the attribute name.
        String serviceName = filterComponents[0];
        String algName = filterComponents[1];
        String attrName = filterComponents[2];

        // Check whether we can find anything in the cache
        String cacheKey = serviceName + '.' + algName;
        HashSet candidates = (HashSet)searchResultsCache.get(cacheKey);

        // If there is no entry for the cacheKey in the cache,
        // let's build an entry for it first.
        HashSet forCache = getProvidersNotUsingCache(serviceName,
                                                     algName,
                                                     null,
                                                     null,
                                                     null,
                                                     allProviders);

        if ((forCache == null) || (forCache.isEmpty())) {
            return null;
        } else {
            searchResultsCache.put(cacheKey, forCache);
            if (attrName == null) {
                return forCache;
            }
            return getProvidersNotUsingCache(serviceName, algName, attrName,
                                             filterValue, candidates,
                                             allProviders);
        }
    }

    private static HashSet getProvidersNotUsingCache(String serviceName,
                                                     String algName,
                                                     String attrName,
                                                     String filterValue,
                                                     HashSet candidates,
                                                     Provider[] allProviders) {
        if ((attrName != null) && (candidates != null) &&
            (!candidates.isEmpty())) {
            for (Iterator cansIte = candidates.iterator();
                cansIte.hasNext();) {
                Provider prov = (Provider)cansIte.next();
                if (!isCriterionSatisfied(prov, serviceName, algName,
                                          attrName, filterValue)) {
                    cansIte.remove();
                }
            }
        }

        if ((candidates == null) || (candidates.isEmpty())) {
            if (candidates == null)
                candidates = new HashSet(5);
            for (int i = 0; i < allProviders.length; i++) {
                if (isCriterionSatisfied(allProviders[i], serviceName,
                                         algName,
                                         attrName, filterValue)) {
                    candidates.add(allProviders[i]);
                }
            }
        }

        return candidates;
    }

    /*
     * Returns true if the given provider satisfies
     * the selection criterion key:value.
     */
    private static boolean isCriterionSatisfied(Provider prov,
                                                String serviceName,
                                                String algName,
                                                String attrName,
                                                String filterValue) {
        String key = serviceName + '.' + algName;

        if (attrName != null) {
            key += ' ' + attrName;
        }
        // Check whether the provider has a property
        // whose key is the same as the given key.
        String propValue = getProviderProperty(key, prov);

        if (propValue == null) {
            // Check whether we have an alias instead
            // of a standard name in the key.
            String standardName = getProviderProperty("Alg.Alias." +
                                                      serviceName + "." +
                                                      algName,
                                                      prov);
            if (standardName != null) {
                key = serviceName + "." + standardName;

                if (attrName != null) {
                    key += ' ' + attrName;
                }

                propValue = getProviderProperty(key, prov);
            }

            if (propValue == null) {
                // The provider doesn't have the given
                // key in its property list.
                return false;
            }
        }

        // If the key is in the format of:
        // <crypto_service>.<algorithm_or_type>,
        // there is no need to check the value.

        if (attrName == null) {
            return true;
        }

        /* If we get here, the key must be in the
         * format of <crypto_service>.<algorithm_or_provider> <attribute_name>.
         */
        if (attrName.equalsIgnoreCase("KeySize")) {
            return Integer.parseInt(filterValue,10) <= Integer.parseInt(propValue,10);

        } else if (attrName.equalsIgnoreCase("ImplementedIn")) {
            return filterValue.equalsIgnoreCase(propValue);

        } else {
            return filterValue.equalsIgnoreCase(propValue);
        }
    }

    static String[] getFilterComponents(String filterKey, String filterValue) {
        int algIndex = filterKey.indexOf('.');

        if (algIndex < 0) {
            // There must be a dot in the filter, and the dot
            // shouldn't be at the beginning of this string.
            throw new InvalidParameterException("Invalid filter");
        }

        String serviceName = filterKey.substring(0, algIndex);
        String algName = null;
        String attrName = null;

        if (filterValue.length() == 0) {
            // The filterValue is an empty string. So the filterKey
            // should be in the format of <crypto_service>.<algorithm_or_type>.
            algName = filterKey.substring(algIndex + 1).trim();
            if (algName.length() == 0) {
                // There must be a algorithm or type name.
                throw new InvalidParameterException("Invalid filter");
            }
        } else {
            // The filterValue is a non-empty string. So the filterKey must be
            // in the format of
            // <crypto_service>.<algorithm_or_type> <attribute_name>
            int attrIndex = filterKey.indexOf(' ');

            if (attrIndex == -1) {
                // There is no attribute name in the filter.
                throw new InvalidParameterException("Invalid filter");
            } else {
                attrName = filterKey.substring(attrIndex + 1).trim();
                if (attrName.length() == 0) {
                    // There is no attribute name in the filter.
                    throw new InvalidParameterException("Invalid filter");
                }
            }

            // There must be an algorithm name in the filter.
            if ((attrIndex < algIndex) ||
                (algIndex == attrIndex - 1)) {
                throw new InvalidParameterException("Invalid filter");
            } else {
                algName = filterKey.substring(algIndex + 1, attrIndex);
            }
        }

        String[] result = new String[3];
        result[0] = serviceName;
        result[1] = algName;
        result[2] = attrName;

        return result;
    }

    /**
     * Returns a Set of Strings containing the names of all available
     * algorithms or types for the specified Java cryptographic service
     * (e.g., Signature, MessageDigest, Cipher, Mac, KeyStore). Returns
     * an empty Set if there is no provider that supports the  
     * specified service. For a complete list of Java cryptographic
     * services, please see the 
     * <a href="../../../guide/security/CryptoSpec.html">Java 
     * Cryptography Architecture API Specification &amp; Reference</a>.
     * Note: the returned set is immutable.
     *
     * @param serviceName the name of the Java cryptographic 
     * service (e.g., Signature, MessageDigest, Cipher, Mac, KeyStore).
     * Note: this parameter is case-insensitive.
     *
     * @return a Set of Strings containing the names of all available 
     * algorithms or types for the specified Java cryptographic service
     * or an empty set if no provider supports the specified service.
     *
     * @since 1.4
     **/
    public static Set getAlgorithms(String serviceName) {
        HashSet result = new HashSet();

        if ((serviceName == null) || (serviceName.length() == 0) ||
            (serviceName.endsWith("."))) {
            return result;
        }

        for (int i = 0; i < providers.size(); i++) {
            // Check the keys for each provider.
            for (Enumeration e = getProviderAt(i).keys(); e.hasMoreElements();) {
                String currentKey = ((String)e.nextElement()).toUpperCase();
                if (currentKey.startsWith(serviceName.toUpperCase())) {
                    // We should skip the currentKey if it contains a 
                    // whitespace. The reason is: such an entry in the
                    // provider property contains attributes for the
                    // implementation of an algorithm. We are only interested
                    // in entries which lead to the implementation
                    // classes.
                    if (currentKey.indexOf(" ") < 0) {
                        result.add(currentKey.substring(serviceName.length() + 1));
                    }
                }
            }       
        }
        return Collections.unmodifiableSet(result);
    }
}

