/*
 * Copyright 1999-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
package org.apache.xpath.functions;

import org.apache.xpath.ExtensionsProvider;
import org.apache.xpath.XPathContext;
import org.apache.xpath.objects.XNull;
import org.apache.xpath.objects.XObject;
import org.apache.xpath.res.XPATHErrorResources;

/**
 * An object of this class represents an extension call expression.  The
 * function is supported internally but is considered an extension function
 * because it is not defined in the XPath 1.0 spec.  
 * When the expression executes, it calls ExtensionsTable#extFunction, and
 * then converts the result to the appropriate XObject.
 * @xsl.usage advanced
 */
public class FuncCoreExtFunction extends FuncExtFunction {

    static final long serialVersionUID = 8776968851259518578L;

    /**
     * Create a new FuncCoreExtFunction based on the qualified name of the
     * extension, and a unique method key.
     * 
     * @param namespace The namespace for the extension function
     * @param extensionName The local name of the extension.
     * @param methodKey Unique method key, which is passed to
     * ExtensionsTable#extFunction in order to allow caching
     * of the method.
     */
    public FuncCoreExtFunction(String namespace, String extensionName,
                               Object methodKey) {
        super(namespace, extensionName, methodKey);
    }
	
    /**
     * Execute the function.  The function must return a valid object.
     * @param xctxt The current execution context.
     * @return A valid XObject.
     *
     * @throws javax.xml.transform.TransformerException
     */
    public XObject execute(XPathContext xctxt)
        throws javax.xml.transform.TransformerException
    {
    
        ExtensionsProvider provider =
            (ExtensionsProvider)xctxt.getOwnerObject();

        if (provider == null
                || !provider.functionAvailable(getNamespace(),
                                               getFunctionName())) {
            error(xctxt, XPATHErrorResources.ER_COULDNOT_FIND_FUNCTION,
                  new Object[]{ getFunctionName() }); 
            return new XNull();
        }

        return super.execute(xctxt);
    }

}
