/*
 * @(#)src/classes/sov/java/util/prefs/XmlSupport.java, xml, as142, 20050517 1.9.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v1.4.2
 * (C) Copyright IBM Corp. 1998, 2004. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * ===========================================================================
 */




package java.util.prefs;

import java.util.*;
import java.io.*;
import javax.xml.parsers.*;
import org.xml.sax.*;
import org.w3c.dom.*;
//ibm@41560 use xerces not crimson /* import org.apache.crimson.tree.*; */
import org.apache.xerces.dom.*;

/**
 * XML Support for java.util.prefs. Methods to import and export preference
 * nodes and subtrees.
 *
 * @author  Josh Bloch and Mark Reinhold
 * @version 1.10, 12/03/01
 * @see     Preferences
 * @since   1.4
 */
class XmlSupport {
    // The required DTD URI for exported preferences
    private static final String PREFS_DTD_URI =
        "http://java.sun.com/dtd/preferences.dtd";

    // The actual DTD corresponding to the URI
    private static final String PREFS_DTD =
        "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" +

        "<!-- DTD for preferences -->"               +

        "<!ELEMENT preferences (root) >"             +
        "<!ATTLIST preferences"                      +
        " EXTERNAL_XML_VERSION CDATA \"0.0\"  >"     +

        "<!ELEMENT root (map, node*) >"              +
        "<!ATTLIST root"                             +
        "          type (system|user) #REQUIRED >"   +

        "<!ELEMENT node (map, node*) >"              +
        "<!ATTLIST node"                             +
        "          name CDATA #REQUIRED >"           +

        "<!ELEMENT map (entry*) >"                   +
        "<!ATTLIST map"                              +
        "  MAP_XML_VERSION CDATA \"0.0\"  >"         +
        "<!ELEMENT entry EMPTY >"                    +
        "<!ATTLIST entry"                            +
        "          key CDATA #REQUIRED"              +
        "          value CDATA #REQUIRED >"          ;
    /**
     * Version number for the format exported preferences files.
     */
    private static final String EXTERNAL_XML_VERSION = "1.0";

    /*
     * Version number for the internal map files.
     */
    private static final String MAP_XML_VERSION = "1.0";

    /**
     * Export the specified preferences node and, if subTree is true, all
     * subnodes, to the specified output stream.  Preferences are exported as
     * an XML document conforming to the definition in the Preferences spec.
     *
     * @throws IOException if writing to the specified output stream
     *         results in an <tt>IOException</tt>.
     * @throws BackingStoreException if preference data cannot be read from
     *         backing store.
     * @throws IllegalStateException if this node (or an ancestor) has been
     *         removed with the {@link #removeNode()} method.
     */
    static void export(OutputStream os, final Preferences p, boolean subTree)
        throws IOException, BackingStoreException {
        if (((AbstractPreferences)p).isRemoved())
            throw new IllegalStateException("Node has been removed");
        //ibm@41560
        CoreDocumentImpl doc = new CoreDocumentImpl();
        //ibm@41560 doc.setDoctype(null, PREFS_DTD_URI), null);
        doc.appendChild(doc.createDocumentType("preferences", null, PREFS_DTD_URI));  //ibm@42235
        Element preferences =  (Element)
            doc.appendChild(doc.createElement("preferences"));
        preferences.setAttribute("EXTERNAL_XML_VERSION", EXTERNAL_XML_VERSION);
        Element xmlRoot =  (Element)
        preferences.appendChild(doc.createElement("root"));
        xmlRoot.setAttribute("type", (p.isUserNode() ? "user" : "system"));

        // Get bottom-up list of nodes from p to root, excluding root
        List ancestors = new ArrayList();

        for (Preferences kid = p, dad = kid.parent(); dad != null;
                                   kid = dad, dad = kid.parent()) {
            ancestors.add(kid);
        }
        Element e = xmlRoot;
        for (int i=ancestors.size()-1; i >= 0; i--) {
            e.appendChild(doc.createElement("map"));
            e = (Element) e.appendChild(doc.createElement("node"));
            e.setAttribute("name", ((Preferences)ancestors.get(i)).name());
        }
        putPreferencesInXml(e, doc, p, subTree);
        //ibm@41560 doc.write(os);
        writeXMLPreferences(doc, os);
    }

    /**
     * Put the preferences in the specified Preferences node into the
     * specified XML element which is assumed to represent a node
     * in the specified XML document which is assumed to conform to
     * PREFS_DTD.  If subTree is true, create children of the specified
     * XML node conforming to all of the children of the specified
     * Preferences node and recurse.
     *
     * @throws BackingStoreException if it is not possible to read
     *         the preferences or children out of the specified
     *         preferences node.
     */
    private static void putPreferencesInXml(Element elt, Document doc,
               Preferences prefs, boolean subTree) throws BackingStoreException
    {
        Preferences[] kidsCopy = null;
        String[] kidNames = null;

        // Node is locked to export its contents and get a
        // copy of children, then lock is released,
        // and, if subTree = true, recursive calls are made on children
        synchronized (((AbstractPreferences)prefs).lock) {
            // Check if this node was concurrently removed. If yes
            // remove it from XML Document and return.
            if (((AbstractPreferences)prefs).isRemoved()) {
                elt.getParentNode().removeChild(elt);
                return;
            }
            // Put map in xml element
            String[] keys = prefs.keys();
            Element map = (Element) elt.appendChild(doc.createElement("map"));
            for (int i=0; i<keys.length; i++) {
                Element entry = (Element)
                    map.appendChild(doc.createElement("entry"));
                entry.setAttribute("key", keys[i]);
                // NEXT STATEMENT THROWS NULL PTR EXC INSTEAD OF ASSERT FAIL
                entry.setAttribute("value", prefs.get(keys[i], null));
            }
            // Recurse if appropriate
            if (subTree) {
                /* Get a copy of kids while lock is held */
                kidNames = prefs.childrenNames();
                kidsCopy = new Preferences[kidNames.length];
                for (int i = 0; i <  kidNames.length; i++)
                    kidsCopy[i] = prefs.node(kidNames[i]);
            }
        // release lock
        }

        if (subTree) {
            for (int i=0; i < kidNames.length; i++) {
                Element xmlKid = (Element)
                    elt.appendChild(doc.createElement("node"));
                xmlKid.setAttribute("name", kidNames[i]);
                putPreferencesInXml(xmlKid, doc, kidsCopy[i], subTree);
            }
        }
    }

    /**
     * Import preferences from the specified input stream, which is assumed
     * to contain an XML document in the format described in the Preferences
     * spec.
     *
     * @throws IOException if reading from the specified output stream
     *         results in an <tt>IOException</tt>.
     * @throws InvalidPreferencesFormatException Data on input stream does not
     *         constitute a valid XML document with the mandated document type.
     */
    static void importPreferences(InputStream is)
        throws IOException, InvalidPreferencesFormatException
    {

        try {

            Element em = null;
            String xmlVersion = null;
            Document doc = load(is);

            /* ibm@42235 use doc version */

            if (doc instanceof DeferredDocumentImpl) {
                DeferredDocumentImpl ddoc = (DeferredDocumentImpl) doc;
                xmlVersion = ddoc.getXmlVersion();              /*ibm@68277*/
            }

            if (xmlVersion.compareTo(EXTERNAL_XML_VERSION) > 0)
                throw new InvalidPreferencesFormatException(
                "Exported preferences file format version " + xmlVersion +
                " is not supported. This java installation can read" +
                " versions " + EXTERNAL_XML_VERSION + " or older. You may need" +
                " to install a newer version of JDK.");

            /* ibm@42235 use CHildNode(2) from Xerces parser -
               Sun use ChildNode(1) from Crimson parser 
               ibm@58331 revert back to ChildNode(1) now that the parser 
               is set to ignore comments. ( see also ibm@57976 )
               */
            Element xmlRoot = (Element) doc.getChildNodes().item(1).
                getChildNodes().item(0);

            Preferences prefsRoot =
                (xmlRoot.getAttribute("type").equals("user") ?
                            Preferences.userRoot() : Preferences.systemRoot());

            ImportSubtree(prefsRoot, xmlRoot);

         } catch(SAXException e) {
            throw new InvalidPreferencesFormatException(e);
        }
    }

    /**
     * Load an XML document from specified input stream, which must
     * have the requisite DTD URI.
     */
    private static Document load(InputStream in)
        throws SAXException, IOException
    {
        Resolver r = new Resolver();
	DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
	dbf.setIgnoringElementContentWhitespace(true);
	dbf.setValidating(true);
	dbf.setCoalescing(true);
        dbf.setIgnoringComments(true);
	try {
	    DocumentBuilder db = dbf.newDocumentBuilder();
	    db.setEntityResolver(new Resolver());
	    db.setErrorHandler(new EH());
            return db.parse(new InputSource(in));
	} catch (ParserConfigurationException x) {
            System.err.println(x);
	    throw new Error(x);
	}
    }

    /**
     * Recursively traverse the specified preferences node and store
     * the described preferences into the system or current user
     * preferences tree, as appropriate.
     */
    private static void ImportSubtree(Preferences prefsNode, Element xmlNode) {
        NodeList xmlKids = xmlNode.getChildNodes();
        int numXmlKids = xmlKids.getLength();
        /*
         * We first lock the node, import its contents and get
         * child nodes. Then we unlock the node and go to children
         * Since some of the children might have been concurrently
         * deleted we check for this.
         */
        Preferences[] prefsKids;
        /* Lock the node */
        synchronized (((AbstractPreferences)prefsNode).lock) {
            //If removed, return silently
            if (((AbstractPreferences)prefsNode).isRemoved())
                return;

            // Import any preferences at this node
            Element firstXmlKid = (Element) xmlKids.item(0);
            ImportPrefs(prefsNode, firstXmlKid);
            prefsKids = new Preferences[numXmlKids - 1];

            // Get involved children
            for (int i=1; i < numXmlKids; i++) {
                Element xmlKid = (Element) xmlKids.item(i);
                prefsKids[i-1] = prefsNode.node(xmlKid.getAttribute("name"));
            }
        } // unlocked the node
        // import children
        for (int i=1; i < numXmlKids; i++)
            ImportSubtree(prefsKids[i-1], (Element)xmlKids.item(i));
    }

    /**
     * Import the preferences described by the specified XML element
     * (a map from a preferences document) into the specified
     * preferences node.
     */
    private static void ImportPrefs(Preferences prefsNode, Element map) {
        NodeList entries = map.getChildNodes();
        for (int i=0, numEntries = entries.getLength(); i < numEntries; i++) {
            Element entry = (Element) entries.item(i);
            prefsNode.put(entry.getAttribute("key"),
                          entry.getAttribute("value"));
        }
    }

    /**
     * Export the specified Map<String,String> to a map document on
     * the specified OutputStream as per the prefs DTD.  This is used
     * as the internal (undocumented) format for FileSystemPrefs.
     *
     * @throws IOException if writing to the specified output stream
     *         results in an <tt>IOException</tt>.
     */
    static void exportMap(OutputStream os, Map map) throws IOException {
        CoreDocumentImpl doc = new CoreDocumentImpl();
        //ibm@41560 doc.setDoctype(null, PREFS_DTD_URI, null);
        doc.appendChild(doc.createDocumentType("map", null, PREFS_DTD_URI));  //ibm@42235

        Element xmlMap = (Element) doc.appendChild(doc.createElement("map"));
        xmlMap.setAttribute("MAP_XML_VERSION", MAP_XML_VERSION);

        for (Iterator i = map.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e = (Map.Entry) i.next();
            Element xe = (Element)
                xmlMap.appendChild(doc.createElement("entry"));
            xe.setAttribute("key",   (String) e.getKey());
            xe.setAttribute("value", (String) e.getValue());
        }

        //ibm@41560 doc.write(os);
        writeXMLPreferences(doc, os);
    }

    /**
     * Import Map from the specified input stream, which is assumed
     * to contain a map document as per the prefs DTD.  This is used
     * as the internal (undocumented) format for FileSystemPrefs.  The
     * key-value pairs specified in the XML document will be put into
     * the specified Map.  (If this Map is empty, it will contain exactly
     * the key-value pairs int the XML-document when this method returns.)
     *
     * @throws IOException if reading from the specified output stream
     *         results in an <tt>IOException</tt>.
     * @throws InvalidPreferencesFormatException Data on input stream does not
     *         constitute a valid XML document with the mandated document type.
     */
    static void importMap(InputStream is, Map m)
        throws IOException, InvalidPreferencesFormatException
    {
        try {
            String mapVersion = null;
            Document doc = load(is);

            /* ibm@42235 use ChildNOde(2) from Xerces parser -
               Sun use ChildNode(1) from Crimson parser 
               ibm@57976 revert back to ChildNode(1) since comments 
               [ dbf.setIgnoringComments(true) ] are now ignored
            */
            Element xmlMap = (Element) doc.getChildNodes().item(1);

            // check version
            /* ibm@42235 use Document version instead of Map attribute
            String mapVersion = xmlMap.getAttribute("MAP_XML_VERSION");
            */
            if (doc instanceof DeferredDocumentImpl) {
                DeferredDocumentImpl ddoc = (DeferredDocumentImpl) doc;
                mapVersion = ddoc.getXmlVersion();          /*ibm@68277*/
            }

            if (mapVersion.compareTo(MAP_XML_VERSION) > 0)
                throw new InvalidPreferencesFormatException(
                "Preferences map file format version " + mapVersion +
                " is not supported. This java installation can read" +
                " versions " + MAP_XML_VERSION + " or older. You may need" +
                " to install a newer version of JDK.");

            NodeList entries = xmlMap.getChildNodes();
            for (int i=0, numEntries=entries.getLength(); i<numEntries; i++) {
                Element entry = (Element) entries.item(i);
                m.put(entry.getAttribute("key"), entry.getAttribute("value"));
            }
        } catch(SAXException e) {
            throw new InvalidPreferencesFormatException(e);
        }
    }

    /**
     * ibm@41560 - since Xerces doesn't supply a DOM writer, added one
     * here to write the XML document formed from the preferences for
     * this node (and sub-nodes if included).
     *
     * @throws IOException if      */
    private static void writeXMLPreferences(CoreDocumentImpl doc, OutputStream os)
        throws IOException
    {
        int length;

        //set up the writer for the OutputStream
        java.io.Writer writer = new OutputStreamWriter(os, "UTF8");
        PrintWriter fOut = new PrintWriter(writer);

        //print XMl version and encoding info
        fOut.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        fOut.println();

        /* sort out the DOCTYPE tag at the start of the output -
         * Crimson provides a setDoctype method on XmlDocument, but
         * for Xerces we have to hack this a bit.
         */
        /* ibm@42235 should be covered by doc.createDocumentType call now
        fOut.println("<!DOCTYPE preferences SYSTEM '" + PREFS_DTD_URI + "'>");
        fOut.println();
        fOut.flush();   */

        //loop through the nodes and write them
        length = doc.getLength();
        if (length == 0)
            return;
        for (int i = 0; i < length; i++) {
            writeXMLNode(doc.item(i), fOut, 0);
            fOut.println();
            fOut.flush();
        }
    }
    /** Writes the specified node, recursively. */
    private static void writeXMLNode(Node node, PrintWriter fOut, int level) {
        int indentLevel = level;

        // is there anything to do?
        if (node == null) {
            return;
        }
        // add some pretty-print handling for testing
        //fOut.println();
        for (int i = 0 ; i < indentLevel ; i++) {
            fOut.print(' ');
        }

        short type = node.getNodeType();
        switch (type) {

            case Node.DOCUMENT_TYPE_NODE: {
                DocumentType doctype = (DocumentType)node;
                fOut.print("<!DOCTYPE ");
                fOut.print(doctype.getName());
                String publicId = doctype.getPublicId();
                String systemId = doctype.getSystemId();
                if (publicId != null) {
                    fOut.print(" PUBLIC '");
                    fOut.print(publicId);
                    fOut.print("' '");
                    fOut.print(systemId);
                    fOut.print('\'');
                }
                else {
                    fOut.print(" SYSTEM '");
                    fOut.print(systemId);
                    fOut.print('\'');
                }
                String internalSubset = doctype.getInternalSubset();
                if (internalSubset != null) {
                    fOut.println(" [");
                    fOut.print(internalSubset);
                    fOut.print(']');
                }
                fOut.println('>');
                //fOut.println();
                break;
            }

            case Node.ELEMENT_NODE: {
                fOut.print('<');
                fOut.print(node.getNodeName());
                Attr attrs[] = sortAttributes(node.getAttributes());
                for (int i = 0; i < attrs.length; i++) {
                    Attr attr = attrs[i];
                    fOut.print(' ');
                    fOut.print(attr.getNodeName());
                    fOut.print("=\"");
                    normalizeAndPrint(attr.getNodeValue(), fOut);
                    fOut.print('"');
                }

                Node child = node.getFirstChild();
                /* handle a chain of nodes */
                if (child != null) {
                    fOut.print('>');
                    fOut.println();
                    while (child != null) {
                        writeXMLNode(child, fOut, indentLevel+2);
                        child = child.getNextSibling();
                    }
                    /* that's taken us to the end of the chain,
                     * now we come back with the /node list
                     */
                    for (int i = 0 ; i < indentLevel ; i++) {  // prettify the output
                        fOut.print(' ');
                    }
                    fOut.print("</");
                    fOut.print(node.getNodeName());
                    fOut.println('>');  //println() instead of print()
                }
                else {
                    /* handle a single element */
                    fOut.println(" />");
                }
                fOut.flush();
                break;
            }

            case Node.ENTITY_REFERENCE_NODE: {
                fOut.print('&');
                fOut.print(node.getNodeName());
                fOut.print(';');
                fOut.flush();
                break;
            }

            case Node.CDATA_SECTION_NODE: {
                fOut.print("<![CDATA[");
                fOut.print(node.getNodeValue());
                fOut.print("]]>");
                fOut.flush();
                break;
            }

            case Node.TEXT_NODE: {
                normalizeAndPrint(node.getNodeValue(), fOut);
                fOut.flush();
                break;
            }

            case Node.PROCESSING_INSTRUCTION_NODE: {
                fOut.print("<?");
                fOut.print(node.getNodeName());
                String data = node.getNodeValue();
                if (data != null && data.length() > 0) {
                    fOut.print(' ');
                    fOut.print(data);
                }
                fOut.println("?>");
                fOut.flush();
                break;
            }
        }

    } // write(Node)

    /** Normalizes and prints the given string. */
    protected static void normalizeAndPrint(String s, PrintWriter fOut) {

        int len = (s != null) ? s.length() : 0;
        for (int i = 0; i < len; i++) {
            char c = s.charAt(i);
            normalizeAndPrint(c, fOut);
        }

    } // normalizeAndPrint(String)

    /** Normalizes and print the given character. */
    protected static void normalizeAndPrint(char c, PrintWriter fOut) {

        switch (c) {
            case '<': {
                fOut.print("&lt;");
                break;
            }
            case '>': {
                fOut.print("&gt;");
                break;
            }
            case '&': {
                fOut.print("&amp;");
                break;
            }
            case '"': {
                fOut.print("&quot;");
                break;
            }
            case '\r':
            case '\n':
            default: {
                fOut.print(c);
            }
        }

    } // normalizeAndPrint(char)

        /** Returns a sorted list of attributes. */
    private static Attr[] sortAttributes(NamedNodeMap attrs) {

        int len = (attrs != null) ? attrs.getLength() : 0;
        Attr array[] = new Attr[len];
        for (int i = 0; i < len; i++) {
            array[i] = (Attr)attrs.item(i);
        }
        for (int i = 0; i < len - 1; i++) {
            String name = array[i].getNodeName();
            int index = i;
            for (int j = i + 1; j < len; j++) {
                String curName = array[j].getNodeName();
                if (curName.compareTo(name) < 0) {
                    name = curName;
                    index = j;
                }
            }
            if (index != i) {
                Attr temp = array[i];
                array[i] = array[index];
                array[index] = temp;
            }
        }

        return array;

    } // sortAttributes(NamedNodeMap):Attr[]
    /* ibm@41560 ends */

    private static class Resolver implements EntityResolver {
        public InputSource resolveEntity(String pid, String sid)
            throws SAXException
        {
            if (sid.equals(PREFS_DTD_URI)) {
                InputSource is;
                is = new InputSource(new StringReader(PREFS_DTD));
                is.setSystemId(PREFS_DTD_URI);
                return is;
            }
            throw new SAXException("Invalid system identifier: " + sid);
        }
    }

    private static class EH implements ErrorHandler {
        public void error(SAXParseException x) throws SAXException {
            throw x;
        }
        public void fatalError(SAXParseException x) throws SAXException {
            throw x;
        }
        public void warning(SAXParseException x) throws SAXException {
            throw x;
        }
    }

}
