/*
 * @(#)src/classes/sov/java/net/URLClassLoader.java, net, as142, 20050517 1.30.2.2
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v1.4.2
 * (C) Copyright IBM Corp. 1998, 2004. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * ===========================================================================
 */


/*
 *
 * Change activity:
 *
 * Reason  Date   Origin  Description
 * ------  ----   ------  ----------------------------------------------------
 * 003683  051199 hdrl    Sun 1.3 rollup
 * 006740  280100 hdmaw   Sun 1.3R
 * 005654  160600 hdpgr   ScJVM: Loading checked extensions
 * 005654  160600 hdpgr   ScJVM: Loading checked extensions
 * 022900  160800 hdmaw   Replace SCJVM_NO_EXTENSION_LOADER with InternalError
 * 025066  131000 hdejs   ScJVM,PERF: Add ibmJVMGetExcludedFields
 * 025627  241000 hdejs:  ScJVM: Add parameter to SetJVMUnresettable...
 * 026352  221100 hdmaw   ScJVM: Use manifest entry to mark compatible extensions
 * 026494  271100 hdejs   ScJVM: Cross heap reference checking
 * 055138  190902 kennard Improve diagnostics for class loading
 * 055302  250902 kennard Change showloading property to verbose
 * 056111  301002 stalleyj Merge 1.4.1 changes 
 * 055792.1240203 cwhite   Allow EXTENSION_COMPATIBLE in any part of the MANIFEST.MF
 * 060643  130603 nichanir ScJVM: Cache URLClassPath & create lazily.
 * 062522  020703 ansriniv JCK:api/java_net/URLClassLoader/NewInst1Tests.java failed
 * 064539  220903 shankar  Fix in loadClass() for Sun Security Bulletin #00115
 * 064147  211003 riclau   1.4.2 merge (without SecurityConstants dependency)
 * 064147  212003 riclau   1.4.2 merge (SecurityConstants part)
 * 080916  161204 adevegow NoClassDefFoundError while starting the applet
 *
 * ===========================================================================
 * Module Information:
 *
 * DESCRIPTION: URLClassLoader           
 * ===========================================================================
 */
 
package java.net;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.io.File;
import java.io.FilePermission;
import java.io.InputStream;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandlerFactory;
import java.util.Enumeration;
import java.util.NoSuchElementException;			/*ibm@10954*/
import java.util.StringTokenizer;
import java.util.jar.Manifest;
import java.util.jar.Attributes;
import java.util.jar.Attributes.Name;
import java.util.Vector;                                            /*ibm@5654*/
import java.util.regex.Pattern;                                     /*ibm@55138*/
import java.util.regex.Matcher;                                     /*ibm@55138*/
import java.util.regex.PatternSyntaxException;                      /*ibm@55138*/
import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;
import java.security.AccessController;
import java.security.AccessControlContext;
import java.security.SecureClassLoader;
import java.security.CodeSource;
import java.security.Permission;
import java.security.PermissionCollection;
import sun.misc.Resource;
import sun.misc.URLClassPath;
import sun.net.www.ParseUtil;
import sun.security.util.SecurityConstants;
import java.lang.reflect.Field;                                     /*ibm@5654*/
import com.ibm.jvm.ExtendedSystem;                                  /*ibm@5654*/
import com.ibm.jvm.ClassLoaderDiagnosticsHelper;                    /*ibm@55138*/
import java.util.Set;                                               /*ibm@55792.1*/
import java.util.Map;                                               /*ibm@55792.1*/
import java.util.Iterator;                                          /*ibm@55792.1*/

/**
 * This class loader is used to load classes and resources from a search
 * path of URLs referring to both JAR files and directories. Any URL that
 * ends with a '/' is assumed to refer to a directory. Otherwise, the URL
 * is assumed to refer to a JAR file which will be opened as needed.
 * <p>
 * The AccessControlContext of the thread that created the instance of
 * URLClassLoader will be used when subsequently loading classes and
 * resources.
 * <p>
 * The classes that are loaded are by default granted permission only to
 * access the URLs specified when the URLClassLoader was created.
 *
 * @author  David Connelly
 * @version 1.80, 06/07/03
 * @since   1.2
 */
public class URLClassLoader extends SecureClassLoader {
    /* The search path for classes and resources */
    private URLClassPath ucp;
    private URL[] cachedURLs;                           /*ibm@60643*/
    private URLStreamHandlerFactory cachedFactory;      /*ibm@60643*/
    

    /* The context to be used when loading classes and resources */
    private AccessControlContext acc;
    
    private ClassFinder loader = new ClassFinder();                  /*ibm@802*/

    /* Removed checkedExtURLs ibm@26352*/


    /**
     * Constructs a new URLClassLoader for the given URLs. The URLs will be
     * searched in the order specified for classes and resources after first
     * searching in the specified parent class loader. Any URL that ends with
     * a '/' is assumed to refer to a directory. Otherwise, the URL is assumed
     * to refer to a JAR file which will be downloaded and opened as needed.
     *
     * <p>If there is a security manager, this method first
     * calls the security manager's <code>checkCreateClassLoader</code> method
     * to ensure creation of a class loader is allowed.
     * 
     * @param urls the URLs from which to load classes and resources
     * @param parent the parent class loader for delegation
     * @exception  SecurityException  if a security manager exists and its  
     *             <code>checkCreateClassLoader</code> method doesn't allow 
     *             creation of a class loader.
     * @see SecurityManager#checkCreateClassLoader
     */
    public URLClassLoader(URL[] urls, ClassLoader parent) {
        super(parent);
        // this is to make the stack depth consistent with 1.1
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkCreateClassLoader();
        }
        /*
         * "if statement" is used to throw null pointer exception
         */
        if (urls.length >= 0) {                                    /*ibm@62522*/                 
            cachedURLs = urls;                                     /*ibm@60643*/
        }                                                          /*ibm@62522*/
        acc = AccessController.getContext();
        loader.classloader = this;                                  /*ibm@55138*/
    }

    /**
     * Constructs a new URLClassLoader for the specified URLs using the
     * default delegation parent <code>ClassLoader</code>. The URLs will
     * be searched in the order specified for classes and resources after
     * first searching in the parent class loader. Any URL that ends with
     * a '/' is assumed to refer to a directory. Otherwise, the URL is
     * assumed to refer to a JAR file which will be downloaded and opened
     * as needed.
     *
     * <p>If there is a security manager, this method first
     * calls the security manager's <code>checkCreateClassLoader</code> method
     * to ensure creation of a class loader is allowed.
     * 
     * @param urls the URLs from which to load classes and resources
     *
     * @exception  SecurityException  if a security manager exists and its  
     *             <code>checkCreateClassLoader</code> method doesn't allow 
     *             creation of a class loader.
     * @see SecurityManager#checkCreateClassLoader
     */
    public URLClassLoader(URL[] urls) {
        super();
        // this is to make the stack depth consistent with 1.1
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkCreateClassLoader();
        }
        if (urls.length >= 0) {                                    /*ibm@62522*/
            cachedURLs = urls;                                     /*ibm@60643*/
        }                                                          /*ibm@62522*/
        acc = AccessController.getContext();
        loader.classloader = this;                                  /*ibm@55138*/
    }

    /**
     * Constructs a new URLClassLoader for the specified URLs, parent
     * class loader, and URLStreamHandlerFactory. The parent argument
     * will be used as the parent class loader for delegation. The
     * factory argument will be used as the stream handler factory to
     * obtain protocol handlers when creating new URLs.
     *
     * <p>If there is a security manager, this method first
     * calls the security manager's <code>checkCreateClassLoader</code> method
     * to ensure creation of a class loader is allowed.
     *
     * @param urls the URLs from which to load classes and resources
     * @param parent the parent class loader for delegation
     * @param factory the URLStreamHandlerFactory to use when creating URLs
     *
     * @exception  SecurityException  if a security manager exists and its  
     *             <code>checkCreateClassLoader</code> method doesn't allow 
     *             creation of a class loader.
     * @see SecurityManager#checkCreateClassLoader
     */
    public URLClassLoader(URL[] urls, ClassLoader parent,
            URLStreamHandlerFactory factory) {
        super(parent);
        // this is to make the stack depth consistent with 1.1
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkCreateClassLoader();
        }
        if (urls.length >= 0) {                                    /*ibm@62522*/
            cachedURLs = urls;                                     /*ibm@60643*/
        }                                                          /*ibm@62522*/
        cachedFactory = factory;                                   /*ibm@60643*/
        acc = AccessController.getContext();
        loader.classloader = this;                                  /*ibm@55138*/
    }

    /**
     * Appends the specified URL to the list of URLs to search for
     * classes and resources.
     *
     * @param url the URL to be added to the search path of URLs
     */
    protected void addURL(URL url) {
	(getUCP()).addURL(url);                            /*ibm@60643*/
    }

    /**
     * Returns the search path of URLs for loading classes and resources.
     * This includes the original list of URLs specified to the constructor,
     * along with any URLs subsequently appended by the addURL() method.
     * @return the search path of URLs for loading classes and resources.
     */
    public URL[] getURLs() {
	return (getUCP()).getURLs();                      /*ibm@60643*/
    }

    /**
     * Returns a cached URLClassPath if it exists or else creates a new
     * one based on a URL[] and/or URLStreamHandlerFactory cached in the
     * URLClassLoader constructor.
     */    /*ibm@60643*/
    private URLClassPath getUCP() {                       
        if (ucp == null) {
            if (cachedFactory != null) {
                ucp = new URLClassPath(cachedURLs,cachedFactory);
            }
            else {
                ucp = new URLClassPath(cachedURLs);
            }
        }
        return ucp;
    }

    /**
     * Returns a list of fields which can safely be excluded from promotion
     * on reset of a JVM. This method only applies to primordial classes
     * and can only be used on static fields. ibm@25066
     */
    private static final String ibmJVMGetExcludedFields() {
        return new String("extloader");                            /*ibm@26352*/
    }

    /*ibm@55138 start*/
    /* Stores a list of classes for which is show detailed class loading */
    private static Vector showClassLoadingFor = null;
    /* Caches whether the vector showClassLoadingFor is empty */
    private static boolean showLoadingMessages = false;
    /* Property to use to setup the detailed classloading output */
    private final static String showClassLoadingProperty = "ibm.cl.verbose"; /*ibm@55302*/

    /*
     * Initializes the showClassLoadingFor vector. All expressions are precompiled
     */
    static
    {
        Vector showClassLoadingFor = new Vector();
        String classes = System.getProperty(showClassLoadingProperty);
        /* If the property exists then process the supplied file expressions */
        if (classes != null) {
            StringTokenizer classMatchers = new StringTokenizer(classes, ",");
            while (classMatchers.hasMoreTokens()) {
                String classMatcher = classMatchers.nextToken();
                /* Do the replacements to allow more readable expressions to be supplied */
                String classMatcherExp = classMatcher.replaceAll("\\.", "\\.");
                classMatcherExp = classMatcherExp.replaceAll("\\*", ".*");
                Pattern pattern;
                /* Add the compiled pattern to the vector */
                try {
                    pattern = Pattern.compile(classMatcherExp);
                    showClassLoadingFor.addElement(pattern);
                } catch (PatternSyntaxException e) {
                    /*
                     * The user has supplied something which has is not
                     * a legal regular expression (or isn't now that it has been
                     * transformed!) Warn the user and ignore this expression
                     */
                    System.err.println("Illegal class matching expression \"" + classMatcher +
                        "\" supplied by property " + showClassLoadingProperty);
                }
            }
        }
        /*
         * Cache whether a check should be made to see whether to show loading messages for
         * a particular class
         */
        if (!showClassLoadingFor.isEmpty()) {
            showLoadingMessages = true;
        }
        URLClassLoader.showClassLoadingFor = showClassLoadingFor;
    }


    /*
     * Returns whether the class loading should be explicitly shown for a
     * particular class. This is determined by the system property ibm.cl.verbose
     * which contains a comma separated list of file expressions.
     * A file expression being a regular expression but with .* substituted for *
     * and \. substituted for . to allow a more readable form to be used
     * If no property exists or contains no expressions then showClassLoadingFor
     * will be an empty vector and this will be the only test each time this function
     * is called. Otherwise name will be matched against each expression in the vector
     * and if a match is found true is returned, otherwise false
     */
    private boolean showClassLoading(String name)
    {
        /* If there are supplied expressions try and match this class name against them */
        if (URLClassLoader.showLoadingMessages) {
            Enumeration patterns = URLClassLoader.showClassLoadingFor.elements();
            while (patterns.hasMoreElements()) {
                Pattern pattern = (Pattern)patterns.nextElement();
                Matcher matcher = pattern.matcher(name);
                if (matcher.matches()) {
                    return true;
                }
            }
        }
        /* Either no expressions or no matches */
        return false;
    }
    /*ibm@55138 end*/

    /**
     * Finds and loads the class with the specified name from the URL search
     * path. Any URLs referring to JAR files are loaded and opened as needed
     * until the class is found.
     *
     * @param name the name of the class
     * @return the resulting class
     * @exception ClassNotFoundException if the class could not be found
     */
    protected Class findClass(final String name) throws ClassNotFoundException
    {
        try {
            loader.name = name;                                      /*ibm@802*/
            boolean scl = showClassLoading(name);
            if (scl) {                                                          /*ibm@55138*/
                ClassLoaderDiagnosticsHelper.attemptingToLoadClass(this, name);/*ibm@55138*/
            }                                                                   /*ibm@55138*/
            Class clazz = (Class)AccessController.doPrivileged(loader, acc);
            if (clazz == null) {                                     /*ibm@802*/
                if (scl) {                                                      /*ibm@55138*/
                    ClassLoaderDiagnosticsHelper.failedToLoadClass(this, name); /*ibm@55138*/
                }                                                               /*ibm@55138*/
                throw new ClassNotFoundException(name);              /*ibm@802*/
            }
            if (scl) {                                                          /*ibm@55138*/
                ClassLoaderDiagnosticsHelper.loadedClass(this, name);           /*ibm@55138*/
            }
            return clazz;                                            /*ibm@802*/
        } catch (java.security.PrivilegedActionException pae) {      /*ibm@802*/
            throw (ClassNotFoundException) pae.getException();       /*ibm@802*/
        }    
    }

    /*
     * Defines a Class using the class bytes obtained from the specified
     * Resource. The resulting Class must be resolved before it can be
     * used.
     */
    private Class defineClass(String name, Resource res) throws IOException {
	int i = name.lastIndexOf('.');
	URL url = res.getCodeSourceURL();
	if (i != -1) {
	    String pkgname = name.substring(0, i);
	    // Check if package already loaded.
	    Package pkg = getPackage(pkgname);
	    Manifest man = res.getManifest();
	    if (pkg != null) {
		// Package found, so check package sealing.
		if (pkg.isSealed()) {
		    // Verify that code source URL is the same.
                    if (!pkg.isSealed(url)) {
                        throw new SecurityException(
                            "sealing violation: package " + pkgname + " is sealed");
                    }
		} else {
		    // Make sure we are not attempting to seal the package
		    // at this code source URL.
		    if ((man != null) && isSealed(pkgname,man)) {
                        throw new SecurityException(
                            "sealing violation: can't seal package " + pkgname +
                            ": already loaded");
                    }
		}
	    } else {

                /* 
                 * If this is the standard Extension ClassLoader we check to see
                 * if the manifest shows the JAR is compatible with Shiraz      
                 */
                /*ibm@5654 ibm@26352*/
	        if (this == getStandardExtensionClassLoader()) {                          
                    String value = null;

                    /* If we have a manifest find the entry */
                    if (null != man) {
                        Attributes attr = man.getMainAttributes();
                        value = attr.getValue(ExtendedSystem.EXTENSION_COMPATIBLE);
                        /*
                         * If the entry is not in the main section then check in the
                         * body. ibm@55792.1
                         */
                        if (null == value) {
                           Map map = man.getEntries();
                           Set set = map.keySet();
                           Iterator it = set.iterator();
                           while (it.hasNext() && null == value) {
                              attr = man.getAttributes((String) it.next());
                              value = attr.getValue(ExtendedSystem.EXTENSION_COMPATIBLE);
                           }
                        }
                    }

                    /*
                     * If we DON'T have the appropriate manifest entry mark the 
                     * JVM dirty
                     */
                    if (null == value) {               
                        File urlfile = new File(url.getFile());                   
                        String urlpath = urlfile.getCanonicalPath();           
		        ExtendedSystem.setJVMUnresettableConditionally(      
			    ExtendedSystem.SCJVM_LOADING_UNCHECKED_EXTENSION,
                            new String("Unchecked extension loading class: " + 
                                       name + 
                                       " from path " + 
                                       urlpath));                 /*ibm@25627*/
                    }
		}                                                       
		/*ibm@5654 ibm@26352 ends*/       


		if (man != null) {
		    definePackage(pkgname, man, url);
		} else {
		    definePackage(pkgname, null, null, null, null, null, null, null);
		}
	    }
	}
	// Now read the class bytes and define the class
	byte[] b = res.getBytes();
	java.security.cert.Certificate[] certs = res.getCertificates();
	CodeSource cs = new CodeSource(url, certs);
	return defineClass(name, b, 0, b.length, cs);
    }

    /*ibm@5654 starts*/
    private static URLClassLoader extLoader = null;
    private static boolean attemptedExtLoader = false;

    private static ClassLoader getStandardExtensionClassLoader(){
        if (!attemptedExtLoader) {
	    attemptedExtLoader=true;
	    AccessController.doPrivileged(new PrivilegedAction() {
	        public Object run() {
		    try {
		        sun.misc.Launcher l =
			    sun.misc.Launcher.getLauncher();
			Class launcher = l.getClass();
			Field field = launcher.getDeclaredField("extLoader");
			field.setAccessible(true);
			extLoader = (URLClassLoader)field.get(l);
			field.setAccessible(false);
			return Boolean.TRUE;
		    } catch (Exception e) {
                        throw new InternalError(e.toString());     /*ibm@22900*/
		    }
		}
	    });
	}
	return extLoader;
    } /* end getStandardExtensionClassLoader */
	
    /* Return an array of the directories specified in the */
    /* "java.checked.ext.dirs" system property             */
    private static File[] getCheckedExtDirs() {
        String s = System.getProperty("java.checked.ext.dirs");
        File[] dirs;
	if (s != null) {
	    StringTokenizer st = 
	        new StringTokenizer(s, File.pathSeparator);
	    int count = st.countTokens();
	    dirs = new File[count];
	    for (int i = 0; i < count; i++) {
	        dirs[i] = new File(st.nextToken());
	    }
	} else {
	    dirs = new File[0];
	}
	return dirs;
    } /* end getCheckedExtDirs */

    /* Return a Vetor of canonical paths for the checked URLs */
    private static Vector getCheckedExtURLs(File[] dirs) throws IOException {
        Vector urls = new Vector();
        for (int i = 0; i < dirs.length; i++) {
            String[] files = dirs[i].list();
	    if (files != null) {
		for (int j = 0; j < files.length; j++) {
		    File f = new File(dirs[i], files[j]);
		    urls.add(f.getCanonicalPath());
		}
	    }
        }
	return urls;
    } /* end getCheckedExtURLs */
     /*ibm@5654 ends*/


    /**
     * Defines a new package by name in this ClassLoader. The attributes
     * contained in the specified Manifest will be used to obtain package
     * version and sealing information. For sealed packages, the additional
     * URL specifies the code source URL from which the package was loaded.
     *
     * @param name  the package name
     * @param man   the Manifest containing package version and sealing
     *              information
     * @param url   the code source url for the package, or null if none
     * @exception   IllegalArgumentException if the package name duplicates
     *              an existing package either in this class loader or one
     *              of its ancestors
     * @return the newly defined Package object
     */
    protected Package definePackage(String name, Manifest man, URL url)
	throws IllegalArgumentException
    {
	String path = name.replace('.', '/').concat("/");
	String specTitle = null, specVersion = null, specVendor = null;
	String implTitle = null, implVersion = null, implVendor = null;
	String sealed = null;
	URL sealBase = null;

	Attributes attr = man.getAttributes(path);
	if (attr != null) {
	    specTitle   = attr.getValue(Name.SPECIFICATION_TITLE);
	    specVersion = attr.getValue(Name.SPECIFICATION_VERSION);
	    specVendor  = attr.getValue(Name.SPECIFICATION_VENDOR);
	    implTitle   = attr.getValue(Name.IMPLEMENTATION_TITLE);
	    implVersion = attr.getValue(Name.IMPLEMENTATION_VERSION);
	    implVendor  = attr.getValue(Name.IMPLEMENTATION_VENDOR);
	    sealed      = attr.getValue(Name.SEALED);
	}
	attr = man.getMainAttributes();
	if (attr != null) {
	    if (specTitle == null) {
		specTitle = attr.getValue(Name.SPECIFICATION_TITLE);
	    }
	    if (specVersion == null) {
		specVersion = attr.getValue(Name.SPECIFICATION_VERSION);
	    }
	    if (specVendor == null) {
		specVendor = attr.getValue(Name.SPECIFICATION_VENDOR);
	    }
	    if (implTitle == null) {
		implTitle = attr.getValue(Name.IMPLEMENTATION_TITLE);
	    }
	    if (implVersion == null) {
		implVersion = attr.getValue(Name.IMPLEMENTATION_VERSION);
	    }
	    if (implVendor == null) {
		implVendor = attr.getValue(Name.IMPLEMENTATION_VENDOR);
	    }
	    if (sealed == null) {
		sealed = attr.getValue(Name.SEALED);
	    }
	}
	if ("true".equalsIgnoreCase(sealed)) {
	    sealBase = url;
	}
	return definePackage(name, specTitle, specVersion, specVendor,
			     implTitle, implVersion, implVendor, sealBase);
    }

    /*
     * Returns true if the specified package name is sealed according to the
     * given manifest.
     */
    private boolean isSealed(String name, Manifest man) {
	String path = name.replace('.', '/').concat("/");
	Attributes attr = man.getAttributes(path);
	String sealed = null;
	if (attr != null) {
	    sealed = attr.getValue(Name.SEALED);
	}
	if (sealed == null) {
	    if ((attr = man.getMainAttributes()) != null) {
		sealed = attr.getValue(Name.SEALED);
	    }
	}
	return "true".equalsIgnoreCase(sealed);
    }

    /**
     * Finds the resource with the specified name on the URL search path.
     *
     * @param name the name of the resource
     * @return a <code>URL</code> for the resource, or <code>null</code>
     * if the resource could not be found.
     */
    /*ibm@6740*/
    public URL findResource(final String name) {
	/*
	 * The same restriction to finding classes applies to resources
	 */
    if (null == name) return null;                          /*ibm@26494*/
    final String s = new String(name);                      /*ibm@26494*/
	URL url =
	    (URL) AccessController.doPrivileged(new PrivilegedAction() {
		    public Object run() {
			return (getUCP()).findResource(s, true); /*ibm@60643*//*ibm@26494*/
		    }
		}, acc);
	
        return url != null ? (getUCP()).checkURL(url) : null;  /*ibm@60643*//*ibm@10954*/
    }

    /**
     * Returns an Enumeration of URLs representing all of the resources
     * on the URL search path having the specified name.
     *
     * @param name the resource name
     * @exception IOException if an I/O exception occurs
     * @return an <code>Enumeration</code> of <code>URL</code>s
     */
    /*ibm@6740*/ /*ibm@10954*/
    public Enumeration findResources(final String name) throws IOException {
	final Enumeration e = (getUCP()).findResources(name, true); /*ibm@60643*/

	return new Enumeration() {
	    private URL url = null;

            private boolean next() {
                if (url != null) {
                    return true;
                }

		do {
		    URL u = (URL)
			AccessController.doPrivileged(new PrivilegedAction() {
			    public Object run() {
				if (!e.hasMoreElements())
                               	    return null;
                            	return e.nextElement();
			    }
			}, acc);
		    if (u == null) 
			break;
		    url = (getUCP()).checkURL(u);     /*ibm@60643*/
		} while (url == null);
		return url != null;
	    }

	    public Object nextElement() {
		if (!next()) {
		    throw new NoSuchElementException();
		}
		URL u = url;
		url = null;
		return u;
	    }

	    public boolean hasMoreElements() {
		return next();
	    }
	};
    }

    /**
     * Returns the permissions for the given codesource object.
     * The implementation of this method first calls super.getPermissions
     * and then adds permissions based on the URL of the codesource.
     * <p>
     * If the protocol is "file"
     * and the path specifies a file, then permission to read that
     * file is granted. If protocol is "file" and the path is
     * a directory, permission is granted to read all files
     * and (recursively) all files and subdirectories contained in
     * that directory.
     * <p>
     * If the protocol is not "file", then
     * to connect to and accept connections from the URL's host is granted.
     * @param codesource the codesource
     * @return the permissions granted to the codesource
     */
    protected PermissionCollection getPermissions(CodeSource codesource)
    {
	PermissionCollection perms = super.getPermissions(codesource);

	URL url = codesource.getLocation();

	Permission p;
        URLConnection urlConnection;

	try {
            urlConnection = url.openConnection();
	    p = urlConnection.getPermission();
	} catch (java.io.IOException ioe) {
	    p = null;
            urlConnection = null;
	}

	if (p instanceof FilePermission) {
	    // if the permission has a separator char on the end,
	    // it means the codebase is a directory, and we need
	    // to add an additional permission to read recursively
	    String path = p.getName();
	    if (path.endsWith(File.separator)) {
		path += "-";
		p = new FilePermission(path, SecurityConstants.FILE_READ_ACTION);
	    }
	} else if ((p == null) && (url.getProtocol().equals("file"))) {
	    String path = url.getFile().replace('/', File.separatorChar);
            path = ParseUtil.decode(path);
	    if (path.endsWith(File.separator))
		path += "-";
	    p =  new FilePermission(path, SecurityConstants.FILE_READ_ACTION);
	} else {
            URL locUrl = url;
            if (urlConnection instanceof JarURLConnection) {
                locUrl = ((JarURLConnection)urlConnection).getJarFileURL();
            }
	    String host = locUrl.getHost();
	    if (host == null)
		host = "localhost";
	    p = new SocketPermission(host,
		SecurityConstants.SOCKET_CONNECT_ACCEPT_ACTION);
	}

	// make sure the person that created this class loader
	// would have this permission

	if (p != null) {
	    final SecurityManager sm = System.getSecurityManager();
	    if (sm != null) {
		final Permission fp = p;
		AccessController.doPrivileged(new PrivilegedAction() {
		    public Object run() throws SecurityException {
			sm.checkPermission(fp);
			return null;
		    }
		}, acc);
	    }
	    perms.add(p);
	}
	return perms;
    }

    /**
     * Creates a new instance of URLClassLoader for the specified
     * URLs and parent class loader. If a security manager is
     * installed, the <code>loadClass</code> method of the URLClassLoader
     * returned by this method will invoke the
     * <code>SecurityManager.checkPackageAccess</code> method before
     * loading the class.
     *
     * @param urls the URLs to search for classes and resources
     * @param parent the parent class loader for delegation
     * @return the resulting class loader
     */
    public static URLClassLoader newInstance(final URL[] urls,
					     final ClassLoader parent) {
	// Save the caller's context
	AccessControlContext acc = AccessController.getContext();
	// Need a privileged block to create the class loader
	URLClassLoader ucl =
	    (URLClassLoader) AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
		    return new FactoryURLClassLoader(urls, parent);
		}
	    });
	// Now set the context on the loader using the one we saved,
	// not the one inside the privileged block...
	ucl.acc = acc;
	return ucl;
    }

    /**
     * Creates a new instance of URLClassLoader for the specified
     * URLs and default parent class loader. If a security manager is
     * installed, the <code>loadClass</code> method of the URLClassLoader
     * returned by this method will invoke the
     * <code>SecurityManager.checkPackageAccess</code> before
     * loading the class.
     *
     * @param urls the URLs to search for classes and resources
     * @return the resulting class loader
     */
    public static URLClassLoader newInstance(final URL[] urls) {
	// Save the caller's context
	AccessControlContext acc = AccessController.getContext();
	// Need a privileged block to create the class loader
	URLClassLoader ucl = (URLClassLoader)
	    AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
		    return new FactoryURLClassLoader(urls);
		}
	    });

	// Now set the context on the loader using the one we saved,
	// not the one inside the privileged block...
	ucl.acc = acc;
	return ucl;
    }
    
  final class ClassFinder implements PrivilegedExceptionAction           /*ibm@802*/
  {
     String name;                                                        /*ibm@802*/
     ClassLoader classloader;                                            /*ibm@55138*/
    
     public Object run() throws ClassNotFoundException {                 /*ibm@802*/
	final String name1 = name;                                       //ibm@80916
        String path = name1.replace('.', '/').concat(".class");           /*ibm@802*/  //ibm@80916
        try {                                                            /*ibm@802*/
            Resource res = (getUCP()).getResource(path, false, classloader, showClassLoading(name1)); /*ibm@802*//*ibm@55138*//*ibm@60643*/ //ibm@80916
            if (res != null)                                             /*ibm@802*/
                return defineClass(name1, res);                           /*ibm@802*/  //ibm@80916
        } catch (IOException e) {                                        /*ibm@802*/
                throw new ClassNotFoundException(name1, e);               /*ibm@802*/  //ibm@80916
        }                                                                /*ibm@802*/
        return null;                                                     /*ibm@802*/
     }
  }
}

final class FactoryURLClassLoader extends URLClassLoader {

    FactoryURLClassLoader(URL[] urls, ClassLoader parent) {
	super(urls, parent);
    }

    FactoryURLClassLoader(URL[] urls) {
	super(urls);
    }

    public final synchronized Class loadClass(String name, boolean resolve)
	throws ClassNotFoundException
    {
	// First check if we have permission to access the package. This
	// should go away once we've added support for exported packages.
	SecurityManager sm = System.getSecurityManager();
	if (sm != null) {
            /*ibm@64539 start*/
            String cname = name.replace('/', '.');
            if (cname.startsWith("[")) {
                int b = cname.lastIndexOf('[') + 2;
                if (b > 1 && b < cname.length()) {
                    cname = cname.substring(b);
                }
            }
            int i = cname.lastIndexOf('.');
            if (i != -1) {
                sm.checkPackageAccess(cname.substring(0, i));
            }
            /*ibm@64539 end*/
	}
	return super.loadClass(name, resolve);
    }
}
