/*
 * @(#)src/classes/sov/java/math/BigDecimal.java, math, as142, 20050517 1.43.1.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v1.4.2
 * (C) Copyright IBM Corp. 1998, 2004. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * ===========================================================================
 */


 /* defect  Date   Author  Why
  * ------  ----   ------  ---
  * 40825.1 080202 kstares Correction on intLong support
  * 40919.1 200202 kstares Synchronisation of intVal with intLong for correct
  *                        serialisation behaviour
  * 41155.1 200202 kstares Add missing intLong = NO_LONG;
  * 041970  180302 kstares Fix NumberFormat exception 
  * 051550  110602 banfield Fix for signs when rounding in divide(BigDecimal,int,int)
  *                        with rounding modes ROUND_HALF_DOWN and ROUND_HALF_EVEN
  * 51972.1 130602 banfield Fix hashCode so that intVal and intLong versions produce
  *                        equivalent results
  * 052082  170602 banfield Fx new rounding error introduced by ibm@51550
  * 054826  110902 prakashr Fix the bugs in BigDecimal(String val) constructor
  * 057632  150103 psman   Fix typo
  * 57687.1 060203 andyt   allocating intVal on wrong heap makes jvm unresettable
  * 64701   061003 andyt   movePointRight is incorrect
  * 069926  020304 preece  Performance improvements from Robin Fernades
  * 069983  040304 preece  Fix to performance update above also from Robin
  * 074798  070704 andyt   Performance improvement for toString plus fix for subtract
  */


package java.math;

/**
 * Immutable, arbitrary-precision signed decimal numbers.  A BigDecimal
 * consists of an arbitrary precision integer <i>unscaled value</i> and a
 * non-negative 32-bit integer <i>scale</i>, which represents the number of
 * digits to the right of the decimal point.  The number represented by the
 * BigDecimal is <tt>(unscaledValue/10<sup>scale</sup>)</tt>.  BigDecimal
 * provides operations for basic arithmetic, scale manipulation, comparison,
 * hashing, and format conversion.
 * <p>

 * The BigDecimal class gives its user complete control over rounding
 * behavior, forcing the user to explicitly specify a rounding
 * behavior for operations capable of discarding precision ({@link
 * #divide(BigDecimal, int)}, {@link #divide(BigDecimal, int, int)},
 * and {@link #setScale}).  Eight <em>rounding modes</em> are provided
 * for this purpose.

 * <p>
 * Two types of operations are provided for manipulating the scale of a
 * BigDecimal: scaling/rounding operations and decimal point motion
 * operations.  Scaling/rounding operations (<tt>setScale</tt>) return a
 * BigDecimal whose value is approximately (or exactly) equal to that of the
 * operand, but whose scale is the specified value; that is, they increase or
 * decrease the precision of the number with minimal effect on its value.
 * Decimal point motion operations ({@link #movePointLeft} and
 * {@link #movePointRight}) return a BigDecimal created from the operand by
 * moving the decimal point a specified distance in the specified direction;
 * that is, they change a number's value without affecting its precision.
 * <p>
 * For the sake of brevity and clarity, pseudo-code is used throughout the
 * descriptions of BigDecimal methods.  The pseudo-code expression
 * <tt>(i + j)</tt> is shorthand for "a BigDecimal whose value is
 * that of the BigDecimal <tt>i</tt> plus that of the BigDecimal <tt>j</tt>."
 * The pseudo-code expression <tt>(i == j)</tt> is shorthand for
 * "<tt>true</tt> if and only if the BigDecimal <tt>i</tt> represents the same
 * value as the the BigDecimal <tt>j</tt>."  Other pseudo-code expressions are
 * interpreted similarly. 
 * <p>
 * Note: care should be exercised if BigDecimals are to be used as
 * keys in a {@link java.util.SortedMap} or elements in a {@link
 * java.util.SortedSet}, as BigDecimal's <i>natural ordering</i> is
 * <i>inconsistent with equals</i>.  See {@link Comparable}, {@link
 * java.util.SortedMap} or {@link java.util.SortedSet} for more
 * information.
 * <p>
 * All methods and constructors for this class
 * throw <CODE>NullPointerException</CODE> when passed
 * a null object reference for any input parameter.
 *
 * @see     BigInteger
 * @see	    java.util.SortedMap
 * @see	    java.util.SortedSet
 * @version 1.36, 09/21/01
 * @author Josh Bloch
 */
public class BigDecimal extends Number implements Comparable
{
   /**
    * The unscaled value of this BigDecimal, as returned by unscaledValue().
    *
    * @serial
    * @see #unscaledValue
    */
   private BigInteger intVal;
   private long intLong;
   private final static long MAX_DIGITS = 18;
   private final static long MAX_LONG = Long.MAX_VALUE-1;
   private final static long MIN_LONG = Long.MIN_VALUE+1;
   private final static long NO_LONG = Long.MIN_VALUE;

   /* ibm@69926 additional info for building a BD from a BI */
   private final static long MAX_BITS = 62;

   /**
    * The scale of this BigDecimal, as returned by scale().
    *
    * @serial
    * @see #scale
    */
   private int          scale = 0;

   /**
    * Lets us know when we've lost the ability to fit
    * the unscaled portion of the BigDecimal in a long
    */

   /* Appease the serialization gods */
   private static final long serialVersionUID = 6108874887143696463L;

   /* For fast LongtoString: 
      MinCharBufferSize =  3 + max(MaxScale, MaxUnscaledDigits) where 3 is for "-0."
      We have MaxUnscaledDigits=19 due to max long, so we choose MaxScale=19 and MinCharBufferSize=22 */
   final static int CHARBUFFER_SIZE = 22;
   final static int FASTTOSTRING_MAXSCALE = 19;

   /* Character arrays required for fast LongtoString conversion */
   final static char [] DigitTens = {
        '0', '0', '0', '0', '0', '0', '0', '0', '0', '0',
        '1', '1', '1', '1', '1', '1', '1', '1', '1', '1',
        '2', '2', '2', '2', '2', '2', '2', '2', '2', '2',
        '3', '3', '3', '3', '3', '3', '3', '3', '3', '3',
        '4', '4', '4', '4', '4', '4', '4', '4', '4', '4',
        '5', '5', '5', '5', '5', '5', '5', '5', '5', '5',
        '6', '6', '6', '6', '6', '6', '6', '6', '6', '6',
        '7', '7', '7', '7', '7', '7', '7', '7', '7', '7',
        '8', '8', '8', '8', '8', '8', '8', '8', '8', '8',
        '9', '9', '9', '9', '9', '9', '9', '9', '9', '9',
   };

   final static char [] DigitOnes = {
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
   };


   // Constructors

    /**
     * Translates the String representation of a BigDecimal into a
     * BigDecimal.  The String representation consists of an optional
     * sign, <tt>'+'</tt> (<tt>'&#92;u002B'</tt>) or <tt>'-'</tt>
     * (<tt>'&#92;u002D'</tt>), followed by a sequence of zero or more
     * decimal digits ("the integer"), optionally followed by a
     * fraction, optionally followed by an exponent.
     *
     * <p>The fraction consists of of a decimal point followed by zero or more
     * decimal digits.  The string must contain at least one digit in either
     * the integer or the fraction.  The number formed by the sign, the
     * integer and the fraction is referred to as the <i>significand</i>.
     *
     * <p>The exponent consists of the character <tt>'e'</tt>
     * (<tt>'&#92;u0075'</tt>) or <tt>'E'</tt> (<tt>'&#92;u0045'</tt>)
     * followed by one or more decimal digits.  The value of the
     * exponent must lie between -{@link Integer#MAX_VALUE} ({@link
     * Integer#MIN_VALUE}+1) and {@link Integer#MAX_VALUE}, inclusive.
     *
     * <p>More formally, the strings this constructor accepts are
     * described by the following grammar:
     * <blockquote>
     * <dl>
     * <dt><i>BigDecimalString:</i>
     * <dd><i>Sign<sub>opt</sub> Significand Exponent<sub>opt</sub></i>
     * <p>
     * <dt><i>Sign:</i>
     * <dd><code>+</code>
     * <dd><code>-</code>
     * <p>
     * <dt><i>Significand:</i>
     * <dd><i>IntegerPart</i> <code>.</code> <i>FractionPart<sub>opt</sub></i>
     * <dd><code>.</code> <i>FractionPart</i>
     * <dd><i>IntegerPart</i>
     * <p>
     * <dt><i>IntegerPart:
     * <dd>Digits</i>
     * <p>
     * <dt><i>FractionPart:
     * <dd>Digits</i>
     * <p>
     * <dt><i>Exponent:
     * <dd>ExponentIndicator SignedInteger</i>
     * <p>
     * <dt><i>ExponentIndicator:</i>
     * <dd><code>e</code>
     * <dd><code>E</code>
     * <p>
     * <dt><i>SignedInteger:
     * <dd>Sign<sub>opt</sub> Digits</i>
     * <p>
     * <dt><i>Digits:
     * <dd>Digit
     * <dd>Digits Digit</i>
     * <p>
     * <dt><i>Digit:</i>
     * <dd>any character for which {@link Character#isDigit}
     * returns <code>true</code>, including 0, 1, 2 ...
     * </dl>
     * </blockquote>
     *
     * <p>The scale of the returned BigDecimal will be the number of digits in
     * the fraction, or zero if the string contains no decimal point, subject
     * to adjustment for any exponent:  If the string contains an exponent, the
     * exponent is subtracted from the scale.  If the resulting scale is
     * negative, the scale of the returned BigDecimal is zero and the unscaled
     * value is multiplied by the appropriate power of ten so that, in every
     * case, the resulting BigDecimal is equal to <i>significand</i> &times;
     * 10<i><sup>exponent</sup></i>. (If in the future this specification is 
     * amended to permit negative scales, the final step of zeroing the scale
     * and adjusting the unscaled value will be eliminated.)
     *
     * <p>The character-to-digit mapping is provided by {@link
     * java.lang.Character#digit} set to convert to radix 10.  The
     * String may not contain any extraneous characters (whitespace,
     * for example).
     *
     * <p>Note: For values other <tt>float</tt> and <tt>double</tt>
     * NaN and &plusmn;Infinity, this constructor is compatible with
     * the values returned by {@link Float#toString} and {@link
     * Double#toString}.  This is generally the preferred way to
     * convert a <tt>float</tt> or <tt>double</tt> into a BigDecimal,
     * as it doesn't suffer from the unpredictability of the {@link
     * #BigDecimal(double)} constructor.
     *
     * <p>Note: the optional leading plus sign and trailing exponent were
     * added in release 1.3.
     *
     * @param val String representation of BigDecimal.
     * @throws NumberFormatException <tt>val</tt> is not a valid representation
     *	       of a BigDecimal.
     */
   public BigDecimal(String val)
   {
      // Empty string not accepted
      if (val.length() == 0)
          throw new NumberFormatException("Zero length BigDecimal");   /*ibm@38094*/

      //  Change this method completely
      /* Start of ibm@54826 */
      if (val.charAt(0) == '+'){
         val = val.substring(1);
	     if (val.length() == 0 ||          /* "+" illegal! */
             val.charAt(0) == '-')       /* "+-7" illegal! */
             throw new NumberFormatException(val);
      }
      /* End of ibm@54826 */

      String intString = val;
      String fractString = "";
      int exponent = 0;
      int strLength = val.length();

      scale = 0;

      int pointPos = val.indexOf('.');
      int ePos = val.indexOf('e');
      if (ePos == -1)
         ePos = val.indexOf('E');
      if (ePos != -1)
      {
         /* Start of ibm@41970*/
         String exp = val.substring(ePos+1);
         if (exp.length() == 0)              /* "1.2e" illegal! */
             throw new NumberFormatException();
         if (exp.charAt(0) == '+') {
             exp = exp.substring(1);         /* Discard leading '+' */
			 if (exp.length() == 0 ||        /* "1.2e+" illegal! */ /*ibm@54826*/
                 exp.charAt(0) == '-')       /* "123.456e+-7" illegal! */
                 throw new NumberFormatException();
         }
         exponent = Integer.parseInt(exp);
         /* End of ibm@41970*/ 

         strLength = ePos;
         intString = val.substring(0, ePos);

      }
      if (pointPos != -1)
      {
         if (pointPos == 0)
         {
            if (val.charAt(pointPos+1) == '-')   /* ".-123" illegal! */
               throw new NumberFormatException();
         }
         fractString = val.substring(pointPos+1,strLength--);
         scale = fractString.length();
         intString = intString.substring(0,pointPos);
      }

      /* Start of ibm@54826 */
      long longScale = (long)scale - (long)exponent;
	  if ( longScale > Integer.MAX_VALUE )
         throw new NumberFormatException("Final scale out of range");    /*ibm@57632*/   
      scale = (int)longScale;
      /* End of ibm@54826 */

      if (scale < 0)
      {
         strLength -= scale;
      }
      if (strLength <= MAX_DIGITS)
      {
         intLong = Long.parseLong(intString + fractString);
         for (;scale<0; scale++)
         {
            intLong *= 10L;
         }
      }
      else
      {   // must handle as BigInteger
         intLong = NO_LONG;
         intVal = new BigInteger(intString + fractString);
         if (scale < 0)
         {
            intVal = timesTenToThe(intVal, -scale);
            scale = 0;
         }
      }
      // End updates to this method
   }

   /**
    * Translates a <code>double</code> into a BigDecimal.  The scale
    *  of the BigDecimal is the smallest value such that
    *  <tt>(10<sup>scale</sup> * val)</tt> is an integer.
    * <p>
    * Note: the results of this constructor can be somewhat unpredictable.
    * One might assume that <tt>new BigDecimal(.1)</tt> is exactly equal
    * to .1, but it is actually equal
    * to .1000000000000000055511151231257827021181583404541015625.
    * This is so because .1 cannot be represented exactly as a double
    * (or, for that matter, as a binary fraction of any finite length).
    * Thus, the long value that is being passed <i>in</i> to the constructor is
    * not exactly equal to .1, appearances notwithstanding.
    * <p>
    * The (String) constructor, on the other hand, is perfectly predictable:
    * <tt>new BigDecimal(".1")</tt> is <i>exactly</i> equal to .1, as one
    * would expect.  Therefore, it is generally recommended that the (String)
    * constructor be used in preference to this one.
    *
    * @param val <code>double</code> value to be converted to BigDecimal.
    * @throws NumberFormatException <tt>val</tt> if <tt>val</tt> is
    *         infinite or NaN.
    */
   public BigDecimal(double val)
   {
      if (Double.isInfinite(val) || Double.isNaN(val))
         throw new NumberFormatException("Infinite or NaN");

      /*
       * Translate the double into sign, exponent and mantissa, according
       * to the formulae in JLS, Section 20.10.22.
       */
      long valBits = Double.doubleToLongBits(val);
      int sign = ((valBits >> 63)==0 ? 1 : -1);
      int exponent = (int) ((valBits >> 52) & 0x7ffL);
      long mantissa = (exponent==0 ? (valBits & ((1L<<52) - 1)) << 1
                       : (valBits & ((1L<<52) - 1)) | (1L<<52));
      exponent -= 1075;
      /* At this point, val == sign * mantissa * 2**exponent */

      /*
       * Special case zero to to supress nonterminating normalization
       * and bogus scale calculation.
       */
      if (mantissa == 0)
      {
         intLong = 0L;
         return;
      }

      /* ibm@69926 start */
	  if (exponent==0) // val == sign*mantissa*2**0 == sign*mantissa
	  {	
		 // mantissa is already a long and sign is 1 or -1. 
		 // There is still a very small chance we will be 
		 // out of bounds for intLong:
		 long testIntLong = sign*mantissa;
		 
		 if ((testIntLong <= MAX_LONG) && (testIntLong >= MIN_LONG))  
		 
		 intLong = testIntLong;
	     return; 		//intLong is already correct, don't need intVal. 
	  }		  
      /* ibm@69926 end */


      /* Normalize */
      while ((mantissa & 1) == 0)
      {    /*  i.e., Mantissa is even */
         mantissa >>= 1;
         exponent++;
      }

      /* Calculate intVal and scale */
      //  START Updates for intLong
      intLong = sign*mantissa;
      long multval = 1L;

      if (exponent < 0 && exponent > -27)       // ibm@69983 was -57
      {                                         // floor(ln(abs(Long.MIN_VALUE))/ln(5)) = 27
         for (int i=exponent; i<0; i++)
            multval *= 5L;
         multIntLong(multval);
         scale = -exponent;
      }
      else if (exponent > 0 && exponent < 63)	//ibm@69926 added else -  //ibm@69983 was 57
      {                                         // floor(ln(abs(Long.MIN_VALUE))/ln(2)) = 63
         for (int i=exponent; i>0; i--)
            multval *= 2L;
         multIntLong(multval);
      }
      //  END   Updates for intLong
      else
      {
         intLong = NO_LONG;
         intVal = BigInteger.valueOf(sign*mantissa);
         if (exponent < 0)
         {
            intVal = intVal.multiply(BigInteger.valueOf(5).pow(-exponent));
            scale = -exponent;
         }
         else if (exponent > 0)
            intVal = intVal.multiply(BigInteger.valueOf(2).pow(exponent));
      }
   }

   /**
    * Translates a BigInteger into a BigDecimal.  The scale of the BigDecimal
    * is zero.
    *
    * @param val BigInteger value to be converted to BigDecimal.
    */
   public BigDecimal(BigInteger val)
   {
	/* ibm@69926 Perhaps the BI could fit in long? */
		if (val.bitLength()>MAX_BITS)
		{	
			intVal = val;		
			intLong = NO_LONG;
	
		}
		else
			intLong=val.longValue();
	/* end ibm@69926 */

   }

   /**
    * Translates a BigInteger unscaled value and an <code>int</code>
    * scale into a BigDecimal.  The value of the BigDecimal is
    * <tt>(unscaledVal/10<sup>scale</sup>)</tt>.
    *
    * @param unscaledVal unscaled value of the BigDecimal.
    * @param scale scale of the BigDecimal.
    * @throws NumberFormatException scale is negative
    */
   public BigDecimal(BigInteger unscaledVal, int scale)
   {
      if (scale < 0)
         throw new NumberFormatException("Negative scale");

	/* ibm@69926 Perhaps the BI could fit in long? */
		if (unscaledVal.bitLength()>MAX_BITS)
		{	
			intVal = unscaledVal;		
			intLong = NO_LONG;
	
		}
		else
			intLong=unscaledVal.longValue();
	/* end ibm@69926 */
	
		this.scale = scale;
   }

   // START New constructor based on long value
   private BigDecimal(long unscaledVal, int scale)
   {

      if (scale < 0)
         throw new NumberFormatException("Negative scale");

      if (unscaledVal > MAX_LONG || unscaledVal < MIN_LONG)
      {
         intLong = NO_LONG;
         intVal = BigInteger.valueOf(unscaledVal);
      }
      else
         intLong = unscaledVal;
      this.scale = scale;

   }
   // END   New constructor based on long value


   // Static Factory Methods

   /**
    * Translates a <code>long</code> unscaled value and an
    * <code>int</code> scale into a BigDecimal.  This &quot;static factory
    * method&quot; is provided in preference to a (<code>long</code>,
    * <code>int</code>) constructor because it allows for reuse of
    * frequently used BigDecimals.
    *
    * @param unscaledVal unscaled value of the BigDecimal.
    * @param scale scale of the BigDecimal.
    * @return a BigDecimal whose value is
    *	       <tt>(unscaledVal/10<sup>scale</sup>)</tt>.
    */
   public static BigDecimal valueOf(long unscaledVal, int scale)
   {
      return new BigDecimal(unscaledVal, scale);
   }

   /**
    * Translates a <code>long</code> value into a BigDecimal with a
    * scale of zero.  This &quot;static factory method&quot; is provided in
    * preference to a (<code>long</code>) constructor because it
    * allows for reuse of frequently used BigDecimals.
    *
    * @param val value of the BigDecimal.
    * @return a BigDecimal whose value is <tt>val</tt>.
    */
   public static BigDecimal valueOf(long val)
   {
      return new BigDecimal(val, 0);
   }


   // Arithmetic Operations

   /**
    * Returns a BigDecimal whose value is <tt>(this + val)</tt>, and whose
    * scale is <tt>max(this.scale(), val.scale())</tt>.
    *
    * @param  val value to be added to this BigDecimal.
    * @return <tt>this + val</tt>
    */
   public BigDecimal add(BigDecimal val)
   {
      BigDecimal arg[] = new BigDecimal[2];
      arg[0] = this; arg[1] = val;
      matchScale(arg);
      // START intLong support
      long iLong0 = arg[0].intLong;
      long iLong1 = arg[1].intLong;
      if ((iLong0 != NO_LONG) & (iLong1 != NO_LONG))
      {
         boolean overflow = false;
         if (iLong0<0)
         {
            if (iLong1<0)
               if (iLong1 < MIN_LONG-iLong0)
                  overflow=true;
         }
         else
         {
            if (iLong1>0)
               if (iLong1 > MAX_LONG-iLong0)
                  overflow=true;
         }
         if (!overflow)
            return new BigDecimal(iLong0+iLong1, arg[0].scale);
      }
      if (arg[0].intVal == null)
         arg[0].intVal = BigInteger.valueOf(iLong0);
      if (arg[1].intVal == null) {
         try {
            final long lTemp = iLong1;
            BigInteger biTemp = (BigInteger)
               (java.security.AccessController.doPrivileged(
                  new java.security.PrivilegedExceptionAction() {
                     public Object run() {
                        return BigInteger.valueOf(lTemp);
                     }
                  }
               ));
            arg[1].intVal = biTemp;
         } catch (java.security.PrivilegedActionException pae) {
            arg[1].intVal = BigInteger.valueOf(iLong1);
         }
      }
      // END   intLong support
      return new BigDecimal(arg[0].intVal.add(arg[1].intVal), arg[0].scale);
   }

   /**
    * Returns a BigDecimal whose value is <tt>(this - val)</tt>, and whose
    * scale is <tt>max(this.scale(), val.scale())</tt>.
    *
    * @param  val value to be subtracted from this BigDecimal.
    * @return <tt>this - val</tt>
    */
   public BigDecimal subtract(BigDecimal val)
   {
      BigDecimal arg[] = new BigDecimal[2];
      arg[0] = this; arg[1] = val;
      matchScale(arg);
      // BEGIN intLong support
      long iLong0 = arg[0].intLong;
      long iLong1 = arg[1].intLong;

      if ((iLong0 != NO_LONG) & (iLong1 != NO_LONG))
      {

         boolean overflow = false;
         if (iLong0<0)
         {
            if (iLong1>0)
               if (iLong0 < MIN_LONG + iLong1)
                  overflow=true;
         }
         else
         {
            if (iLong1<0)
               if (iLong0 > MAX_LONG + iLong1)
                  overflow=true;
         }
         if (!overflow)
            return new BigDecimal(iLong0-iLong1, arg[0].scale);
      }
      if (arg[0].intVal == null)
         arg[0].intVal = BigInteger.valueOf(iLong0);
      if (arg[1].intVal == null) {
         try {
            final long lTemp = iLong1;
            BigInteger biTemp = (BigInteger)
               (java.security.AccessController.doPrivileged(
                  new java.security.PrivilegedExceptionAction() {
                     public Object run() {
                        return BigInteger.valueOf(lTemp);
                     }
                  }
               ));
            arg[1].intVal = biTemp;
         } catch (java.security.PrivilegedActionException pae) {
            arg[1].intVal = BigInteger.valueOf(iLong1);
         }
      }
      //End intLong support
      return new BigDecimal(arg[0].intVal.subtract(arg[1].intVal),
                            arg[0].scale);
   }

   /**
    * Returns a BigDecimal whose value is <tt>(this * val)</tt>, and whose
    * scale is <tt>(this.scale() + val.scale())</tt>.
    *
    * @param  val value to be multiplied by this BigDecimal.
    * @return <tt>this * val</tt>
    */
   public BigDecimal multiply(BigDecimal val)
   {
      //Begin intLong support


      long iLong0 = intLong;
      long iLong1 = val.intLong;

      if ((iLong0 != NO_LONG) & (iLong1 != NO_LONG))
      {
         boolean overflow = false;
         if (iLong1 != 0)
         {

            if (iLong0 < 0)
            {
               if (iLong1 < 0)
               {
                  if (iLong0 < MAX_LONG/iLong1)
                     overflow = true;
               }
               else
               {
                  if (iLong0 < MIN_LONG/iLong1)
                     overflow = true;
               }
            }
            else
            {
               if (iLong1 < 0)
               {
                  if (iLong0 > MIN_LONG/iLong1)
                     overflow = true;
               }
               else
               {
                  if (iLong0 > MAX_LONG/iLong1)
                     overflow = true;
               }
            }
         }
         if (!overflow)
            return new BigDecimal(iLong0*iLong1, scale+val.scale);
      }
      if (intVal == null)
         this.intVal = BigInteger.valueOf(iLong0);
      if (val.intVal == null) {
         try {
            final long lTemp = iLong1;
            BigInteger biTemp = (BigInteger)
               (java.security.AccessController.doPrivileged(
                  new java.security.PrivilegedExceptionAction() {
                     public Object run() {
                        return BigInteger.valueOf(lTemp);
                     }
                  } 
               ));
            val.intVal = biTemp;
         } catch (java.security.PrivilegedActionException pae) {
            val.intVal = BigInteger.valueOf(iLong1);
         }
      }
      //End intLong support


      return new BigDecimal(intVal.multiply(val.intVal), scale+val.scale);
   }

   /**
    * Returns a BigDecimal whose value is <tt>(this / val)</tt>, and whose
    * scale is as specified.  If rounding must be performed to generate a
    * result with the specified scale, the specified rounding mode is
    * applied.
    *
    * @param  val value by which this BigDecimal is to be divided.
    * @param  scale scale of the BigDecimal quotient to be returned.
    * @param  roundingMode rounding mode to apply.
    * @return <tt>this / val</tt>
    * @throws ArithmeticException <tt>val</tt> is zero, <tt>scale</tt> is
    *	       negative, or <tt>roundingMode==ROUND_UNNECESSARY</tt> and
    *	       the specified scale is insufficient to represent the result
    *	       of the division exactly.
    * @throws IllegalArgumentException <tt>roundingMode</tt> does not
    *	       represent a valid rounding mode.
    * @see    #ROUND_UP
    * @see    #ROUND_DOWN
    * @see    #ROUND_CEILING
    * @see    #ROUND_FLOOR
    * @see    #ROUND_HALF_UP
    * @see    #ROUND_HALF_DOWN
    * @see    #ROUND_HALF_EVEN
    * @see    #ROUND_UNNECESSARY
    */

   public BigDecimal divide(BigDecimal val, int scale, int roundingMode)
   {
      if (scale < 0)
         throw new ArithmeticException("Negative scale");
      if (roundingMode < ROUND_UP || roundingMode > ROUND_UNNECESSARY)
         throw new IllegalArgumentException("Invalid rounding mode");

      //Begin intLong support
      int scaleIn = scale;

      BigDecimal dividend = this; BigDecimal divisor = val;

      long intOut, rem;
      int scaleFactor = this.scale-val.scale;
      if(scaleIn < MAX_DIGITS)
      {
      if (scaleIn != scaleFactor)
      {
         if (scaleFactor<scaleIn)
            dividend = scaleUp(this,scaleIn-scaleFactor);
         else
            divisor = scaleUp(val,scaleFactor-scaleIn);
      }

      if (dividend.intLong != NO_LONG && divisor.intLong != NO_LONG)
      {
         intOut     = dividend.intLong/divisor.intLong;
         rem        = dividend.intLong%divisor.intLong;


         if (rem != 0)
         {
            long nextDigit;

            switch (roundingMode)
            {
            case ROUND_UP:
               if (rem > 0)
                  if(intOut >= 0)
                     intOut += 1;
                  else
                     intOut -= 1;
               if (rem < 0)
                  if(intOut <= 0)
                     intOut -= 1;
                  else
                     intOut += 1;
               break;
            case ROUND_DOWN:
               break;
            case ROUND_CEILING:
               if (rem > 0)
                  intOut += 1;
               if (rem < 0 )
                  if(intOut > 0)
                     intOut +=1;
               break;
            case ROUND_FLOOR:
               if (rem < 0)
                  if(intOut <= 0)
                     intOut -= 1;
               break;
            case ROUND_HALF_UP:
               nextDigit  = rem*10/divisor.intLong;
               if (nextDigit>=5)
                  intOut += 1;
               if (nextDigit<=-5)
                  intOut -= 1;
               break;
            case ROUND_HALF_DOWN:
               if (intOut > 0)
               {
                  if (rem > 0)
                  {
                     if (rem * 2 > divisor.intLong)
                        intOut += 1;
                     break;
                  }
                  else
                     /*ibm@52082 - broke this calculation in ibm@51550, fixed now */
                     if (rem * -2 < divisor.intLong)
                        intOut += 1;

                  break;
               }
               else
               {
                  if (rem > 0)
                  {
                     if (rem * 2 > divisor.intLong)
                        intOut += 1;            //ibm@51550 rem is +ive so round positively
                     break;
                  }
                  else
                     if (rem * -2 > divisor.intLong)
                        intOut -= 1;
                  break;
               }
            case ROUND_HALF_EVEN:
               if (intOut%2==0)
               {
                  //treat this as ROUND_HALF_DOWN
                  if (intOut > 0)
                  {
                     if (rem > 0)
                     {
                        if (rem * 2 > divisor.intLong)
                           intOut += 1;
                        break;
                     }
                     else
                        if (rem * 2 < divisor.intLong)
                           intOut += 1;

                        break;
                  }
                  else
                  {
                     if (rem > 0)
                     {
                        if (rem * 2 > divisor.intLong)
                           intOut += 1;            //ibm@51550 rem is +ive so round positively
                        break;
                     }
                     else
                        if (rem * -2 > divisor.intLong)
                           intOut -= 1;
                        break;
                  }
               }
               else
               {
                  //treat this as ROUND_HALF_UP
                  nextDigit  = rem*10/divisor.intLong;
                  if (nextDigit>=5)
                  {
                     intOut += 1;
                     break;
                  }

                  if (nextDigit<=-5)
                  {
                     intOut -= 1;
                     break;
                  }
                  break;
               }
            case ROUND_UNNECESSARY:
               if (rem != 0)
                  throw new ArithmeticException("BigDecimal: Rounding necessary");
               break;
            default:
               throw new IllegalArgumentException("BigDecimal: Invalid rounding mode");
            }
         }


         return new BigDecimal(intOut,scaleIn);
      }

      }
      else
      {

	if (scale + val.scale >= this.scale) {
	    dividend = this.setScale(scale + val.scale);
	    divisor = val;
	} else {
	    dividend = this;
	    divisor = val.setScale(this.scale - scale);
	}
      }
      if (dividend.intVal == null)
         dividend.intVal = BigInteger.valueOf(dividend.intLong);
      if (divisor.intVal == null) {
         try {
            final long lTemp = divisor.intLong;
            BigInteger biTemp = (BigInteger)
               (java.security.AccessController.doPrivileged(
                  new java.security.PrivilegedExceptionAction() {
                    public Object run() {
                       return BigInteger.valueOf(lTemp);
                    }
                 }
               ));
            divisor.intVal = biTemp;
         } catch (java.security.PrivilegedActionException pae) {
            divisor.intVal = BigInteger.valueOf(divisor.intLong);
         }
      }


      //End intLong support

      /* Do the division and return result if it's exact */
      BigInteger i[] = dividend.intVal.divideAndRemainder(divisor.intVal);
      BigInteger q = i[0], r = i[1];
      if (r.signum() == 0)
         return new BigDecimal(q, scale);

      else if (roundingMode == ROUND_UNNECESSARY) /* Rounding prohibited */
         throw new ArithmeticException("Rounding necessary");

      /* Round as appropriate */
      int signum = dividend.signum() * divisor.signum(); /* Sign of result */
      boolean increment;
      if (roundingMode == ROUND_UP)
      {         /* Away from zero */
         increment = true;
      }
      else if (roundingMode == ROUND_DOWN)
      {    /* Towards zero */
         increment = false;
      }
      else if (roundingMode == ROUND_CEILING)
      { /* Towards +infinity */
         increment = (signum > 0);
      }
      else if (roundingMode == ROUND_FLOOR)
      {   /* Towards -infinity */
         increment = (signum < 0);
      }
      else
      { /* Remaining modes based on nearest-neighbor determination */
         int cmpFracHalf = r.abs().multiply(BigInteger.valueOf(2)).
                           compareTo(divisor.intVal.abs());
         if (cmpFracHalf < 0)
         {    /* We're closer to higher digit */
            increment = false;
         }
         else if (cmpFracHalf > 0)
         {  /* We're closer to lower digit */
            increment = true;
         }
         else
         {            /* We're dead-center */
            if (roundingMode == ROUND_HALF_UP)
               increment = true;
            else if (roundingMode == ROUND_HALF_DOWN)
               increment = false;
            else  /* roundingMode == ROUND_HALF_EVEN */
               increment = q.testBit(0); /* true iff q is odd */
         }
      }
      return(increment
             ? new BigDecimal(q.add(BigInteger.valueOf(signum)), scale)
             : new BigDecimal(q, scale));

   }

   /**
    * Returns a BigDecimal whose value is <tt>(this / val)</tt>, and whose
    * scale is <tt>this.scale()</tt>.  If rounding must be performed to
    * generate a result with the given scale, the specified rounding mode is
    * applied.
    *
    * @param  val value by which this BigDecimal is to be divided.
    * @param  roundingMode rounding mode to apply.
    * @return <tt>this / val</tt>
    * @throws ArithmeticException <tt>val==0</tt>, or
    * 	       <tt>roundingMode==ROUND_UNNECESSARY</tt> and
    *	       <tt>this.scale()</tt> is insufficient to represent the result
    *	       of the division exactly.
    * @throws IllegalArgumentException <tt>roundingMode</tt> does not
    *	       represent a valid rounding mode.
    * @see    #ROUND_UP
    * @see    #ROUND_DOWN
    * @see    #ROUND_CEILING
    * @see    #ROUND_FLOOR
    * @see    #ROUND_HALF_UP
    * @see    #ROUND_HALF_DOWN
    * @see    #ROUND_HALF_EVEN
    * @see    #ROUND_UNNECESSARY
    */
   public BigDecimal divide(BigDecimal val, int roundingMode)
   {
      return this.divide(val, scale, roundingMode);
   }

   /**
    * Returns a BigDecimal whose value is the absolute value of this
    * BigDecimal, and whose scale is <tt>this.scale()</tt>.
    *
    * @return <tt>abs(this)</tt>
    */
   public BigDecimal abs()  //We get intLong support in signum and negate here
   {
      if (this.intLong != NO_LONG)
      {
         if (this.intLong < 0)
            return new BigDecimal(-intLong, scale);
         else
            return this;
      }

      return(signum() < 0 ? negate() : this);
   }

   /**
    * Returns a BigDecimal whose value is <tt>(-this)</tt>, and whose scale
    * is <tt>this.scale()</tt>.
    *
    * @return <tt>-this</tt>
    */
   public BigDecimal negate()
   {
      //Begin intLong support
      if (intLong != NO_LONG)
      {
         return new BigDecimal(-intLong, scale);
      }
      //End intLong support
      return new BigDecimal(intVal.negate(), scale);
   }

   /**
    * Returns the signum function of this BigDecimal.
    *
    * @return -1, 0 or 1 as the value of this BigDecimal is negative, zero or
    *	       positive.
    */
   public int signum()
   {
      //Begin intLong support
      if (intLong != NO_LONG)
      {
         if (intLong > 0)
            return 1;
         else if (intLong < 0)
            return -1;
         else
            return 0;
      }
      //End intLong support

      return intVal.signum();
   }

   /**
    * Returns the <i>scale</i> of this BigDecimal.  (The scale is the number
    * of digits to the right of the decimal point.)
    *
    * @return the scale of this BigDecimal.
    */
   public int scale()
   {
      return scale;
   }

   /**
    * Returns a BigInteger whose value is the <i>unscaled value</i> of this
    * BigDecimal.  (Computes <tt>(this * 10<sup>this.scale()</sup>)</tt>.)
    *
    * @return the unscaled value of this BigDecimal.
    * @since   1.2
    */
   public BigInteger unscaledValue()
   {
      //Begin intLong support
      if (intLong != NO_LONG)
      {
         if (intVal == null)
            this.intVal = BigInteger.valueOf(intLong);
      }
      //End intLong support
      return intVal;
   }

   // Rounding Modes

   /**
    * Rounding mode to round away from zero.  Always increments the
    * digit prior to a non-zero discarded fraction.  Note that this rounding
    * mode never decreases the magnitude of the calculated value.
    */
   public final static int ROUND_UP =        0;

   /**
    * Rounding mode to round towards zero.  Never increments the digit
    * prior to a discarded fraction (i.e., truncates).  Note that this
    * rounding mode never increases the magnitude of the calculated value.
    */
   public final static int ROUND_DOWN =   1;

   /**
    * Rounding mode to round towards positive infinity.  If the
    * BigDecimal is positive, behaves as for <tt>ROUND_UP</tt>; if negative,
    * behaves as for <tt>ROUND_DOWN</tt>.  Note that this rounding mode never
    * decreases the calculated value.
    */
   public final static int ROUND_CEILING =   2;

   /**
    * Rounding mode to round towards negative infinity.  If the
    * BigDecimal is positive, behave as for <tt>ROUND_DOWN</tt>; if negative,
    * behave as for <tt>ROUND_UP</tt>.  Note that this rounding mode never
    * increases the calculated value.
    */
   public final static int ROUND_FLOOR =     3;

   /**
    * Rounding mode to round towards "nearest neighbor" unless both
    * neighbors are equidistant, in which case round up.
    * Behaves as for <tt>ROUND_UP</tt> if the discarded fraction is &gt;= .5;
    * otherwise, behaves as for <tt>ROUND_DOWN</tt>.  Note that this is the
    * rounding mode that most of us were taught in grade school.
    */
   public final static int ROUND_HALF_UP =   4;

   /**
    * Rounding mode to round towards "nearest neighbor" unless both
    * neighbors are equidistant, in which case round down.
    * Behaves as for <tt>ROUND_UP</tt> if the discarded fraction is &gt; .5;
    * otherwise, behaves as for <tt>ROUND_DOWN</tt>.
    */
   public final static int ROUND_HALF_DOWN =    5;

   /**
    * Rounding mode to round towards the "nearest neighbor" unless both
    * neighbors are equidistant, in which case, round towards the even
    * neighbor.  Behaves as for ROUND_HALF_UP if the digit to the left of the
    * discarded fraction is odd; behaves as for ROUND_HALF_DOWN if it's even.
    * Note that this is the rounding mode that minimizes cumulative error
    * when applied repeatedly over a sequence of calculations.
    */
   public final static int ROUND_HALF_EVEN =    6;

   /**
    * Rounding mode to assert that the requested operation has an exact
    * result, hence no rounding is necessary.  If this rounding mode is
    * specified on an operation that yields an inexact result, an
    * <tt>ArithmeticException</tt> is thrown.
    */
   public final static int ROUND_UNNECESSARY =  7;


   // Scaling/Rounding Operations

   /**
    * Returns a BigDecimal whose scale is the specified value, and whose
    * unscaled value is determined by multiplying or dividing this
    * BigDecimal's unscaled value by the appropriate power of ten to maintain
    * its overall value.  If the scale is reduced by the operation, the
    * unscaled value must be divided (rather than multiplied), and the value
    * may be changed; in this case, the specified rounding mode is applied to
    * the division.
    * <p>
    * Note that since BigDecimal objects are immutable, calls of this
    * method do <i>not</i> result in the original object being
    * modified, contrary to the usual convention of having methods
    * named <code>set<i>X</i></code> mutate field
    * <code><i>X</i></code>.  Instead, <code>setScale</code> returns
    * an object with the proper scale; the returned object may or may
    * not be newly allocated.
    *
    * @param  scale scale of the BigDecimal value to be returned.
    * @param  roundingMode The rounding mode to apply.
    * @return a BigDecimal whose scale is the specified value, and whose
    *	       unscaled value is determined by multiplying or dividing this
    * 	       BigDecimal's unscaled value by the appropriate power of ten to
    *	       maintain its overall value.
    * @throws ArithmeticException <tt>scale</tt> is negative, or
    * 	       <tt>roundingMode==ROUND_UNNECESSARY</tt> and the specified
    *	       scaling operation would require rounding.
    * @throws IllegalArgumentException <tt>roundingMode</tt> does not
    *	       represent a valid rounding mode.
    * @see    #ROUND_UP
    * @see    #ROUND_DOWN
    * @see    #ROUND_CEILING
    * @see    #ROUND_FLOOR
    * @see    #ROUND_HALF_UP
    * @see    #ROUND_HALF_DOWN
    * @see    #ROUND_HALF_EVEN
    * @see    #ROUND_UNNECESSARY
    */
   public BigDecimal setScale(int scale, int roundingMode)
   {

      if (scale < 0)
         throw new ArithmeticException("Negative scale");
      if (roundingMode < ROUND_UP || roundingMode > ROUND_UNNECESSARY)
         throw new IllegalArgumentException("Invalid rounding mode");

      /* Handle the easy cases */
      if (scale == this.scale)
         return this;


      if (scale > this.scale)
         return scaleUp(this, scale-this.scale);
      else if (scale < MAX_DIGITS)/* scale < this.scale */
         return divide(scaleUp(valueOf(1), scale), scale, roundingMode);
      else
         return divide(valueOf(1),scale,roundingMode);

   }

   /**
    * Returns a BigDecimal whose scale is the specified value, and whose
    * value is numerically equal to this BigDecimal's.  Throws an
    * ArithmeticException if this is not possible.  This call is typically
    * used to increase the scale, in which case it is guaranteed that there
    * exists a BigDecimal of the specified scale and the correct value.  The
    * call can also be used to reduce the scale if the caller knows that the
    * BigDecimal has sufficiently many zeros at the end of its fractional
    * part (i.e., factors of ten in its integer value) to allow for the
    * rescaling without loss of precision.
    * <p>
    * This method returns the same result as the two argument version
    * of setScale, but saves the caller the trouble of specifying a
    * rounding mode in cases where it is irrelevant.
    * <p>
    * Note that since BigDecimal objects are immutable, calls of this
    * method do <i>not</i> result in the original object being
    * modified, contrary to the usual convention of having methods
    * named <code>set<i>X</i></code> mutate field
    * <code><i>X</i></code>.  Instead, <code>setScale</code> returns
    * an object with the proper scale; the returned object may or may
    * not be newly allocated.
    *
    * @param  scale scale of the BigDecimal value to be returned.
    * @return a BigDecimal whose scale is the specified value, and whose
    *	       unscaled value is determined by multiplying or dividing this
    * 	       BigDecimal's unscaled value by the appropriate power of ten to
    *	       maintain its overall value.
    * @throws ArithmeticException <tt>scale</tt> is negative, or
    * 	       the specified scaling operation would require rounding.
    * @see    #setScale(int, int)
    */
   public BigDecimal setScale(int scale)
   {
      return setScale(scale, ROUND_UNNECESSARY);
   }


   // Decimal Point Motion Operations

   /**
    * Returns a BigDecimal which is equivalent to this one with the decimal
    * point moved n places to the left.  If n is non-negative, the call merely
    * adds n to the scale.  If n is negative, the call is equivalent to
    * movePointRight(-n).  (The BigDecimal returned by this call has value
    * <tt>(this * 10<sup>-n</sup>)</tt> and scale
    * <tt>max(this.scale()+n, 0)</tt>.)
    *
    * @param  n number of places to move the decimal point to the left.
    * @return a BigDecimal which is equivalent to this one with the decimal
    *	       point moved <tt>n</tt> places to the left.
    */
   public BigDecimal movePointLeft(int n)
   {
      //Begin intLong support
      if (intLong != NO_LONG)
         return(n >= 0 ? new BigDecimal(intLong, scale+n) : movePointRight(-n));
      //End intLong support
      return(n>=0 ? new BigDecimal(intVal, scale+n) : movePointRight(-n));
   }

   /**
    * Moves the decimal point the specified number of places to the right.
    * If this BigDecimal's scale is &gt;= <tt>n</tt>, the call merely
    * subtracts <tt>n</tt> from the scale; otherwise, it sets the scale to
    * zero, and multiplies the integer value by
    * <tt>10<sup>(n - this.scale)</sup></tt>.  If <tt>n</tt>
    * is negative, the call is equivalent to <tt>movePointLeft(-n)</tt>. (The
    * BigDecimal returned by this call has value
    * <tt>(this * 10<sup>n</sup>)</tt> and scale
    * <tt>max(this.scale()-n, 0)</tt>.)
    *
    * @param  n number of places to move the decimal point to the right.
    * @return a BigDecimal which is equivalent to this one with the decimal
    *         point moved <tt>n</tt> places to the right.
    */
   public BigDecimal movePointRight(int n)
   {

      //Begin intLong support
      if (intLong != NO_LONG)
      {
         if (scale >= n)
            return new BigDecimal(intLong, scale-n);
         if (n<MAX_DIGITS)
         {
            long multval=1;
            for (int i=scale; i<n; i++)                            /*ibm@64701*/
               multval *=10L;
            long max = MAX_LONG / multval;
            if (((intLong<0) & (-intLong<max)) || ((intLong>=0) & (intLong<max)))
               return new BigDecimal(intLong*multval, 0);          /*ibm@64701*/
            else
            {
               return new BigDecimal(timesTenToThe(BigInteger.valueOf(intLong), n-scale),0);
            }

         }
      }
      if (intVal == null)
         intVal = BigInteger.valueOf(intLong);
      //End intLong support
      return(scale >= n ? new BigDecimal(intVal, scale-n)
             : new BigDecimal(timesTenToThe(intVal, n-scale),0));
   }

   // Comparison Operations

   /**
    * Compares this BigDecimal with the specified BigDecimal.   Two
    * BigDecimals that are equal in value but have a different scale (like
    * 2.0 and 2.00) are considered equal by this method.  This method is
    * provided in preference to individual methods for each of the six
    * boolean comparison operators (&lt;, ==, &gt;, &gt;=, !=, &lt;=).  The
    * suggested idiom for performing these comparisons is:
    * <tt>(x.compareTo(y)</tt> &lt;<i>op</i>&gt; <tt>0)</tt>,
    * where &lt;<i>op</i>&gt; is one of the six comparison operators.
    *
    * @param  val BigDecimal to which this BigDecimal is to be compared.
    * @return -1, 0 or 1 as this BigDecimal is numerically less than, equal
    *         to, or greater than <tt>val</tt>.
    */
   public int compareTo(BigDecimal val)
   {


      BigDecimal arg[] = new BigDecimal[2];
      arg[0] = this; arg[1] = val;


      if (arg[0].scale != arg[1].scale)
      {
         matchScale(arg);
      }
      //Begin intLong support
      if (arg[0].intLong != NO_LONG && arg[1].intLong != NO_LONG)
      {
         if (arg[0].intLong==arg[1].intLong)
            return 0;
         if (arg[0].intLong<arg[1].intLong)
            return -1;
         return 1;
      }

      if(arg[0].intVal == null)
         arg[0].intVal = BigInteger.valueOf(arg[0].intLong);
      if(arg[1].intVal == null) {
         try {
            final long lTemp = arg[1].intLong;
            BigInteger biTemp = (BigInteger)
               (java.security.AccessController.doPrivileged(
                  new java.security.PrivilegedExceptionAction() {
                     public Object run() {
                        return BigInteger.valueOf(lTemp);
                     }
                  }
               ));
            arg[1].intVal = biTemp;
         } catch (java.security.PrivilegedActionException pae) {
            arg[1].intVal = BigInteger.valueOf(arg[1].intLong);
         }
      }
      //End intLong support

      /* Optimization: would run fine without the next three lines */
      int sigDiff = signum() - val.signum();
      if (sigDiff != 0)
         return(sigDiff > 0 ? 1 : -1);


      /* If signs match, scale and compare intVals */

      return arg[0].intVal.compareTo(arg[1].intVal);
   }

   /**
    * Compares this BigDecimal with the specified Object.  If the Object is a
    * BigDecimal, this method behaves like {@link #compareTo compareTo}.
    * Otherwise, it throws a <tt>ClassCastException</tt> (as BigDecimals are
    * comparable only to other BigDecimals).
    *
    * @param  o Object to which this BigDecimal is to be compared.
    * @return a negative number, zero, or a positive number as this
    *	       BigDecimal is numerically less than, equal to, or greater
    *	       than <tt>o</tt>, which must be a BigDecimal.
    * @throws ClassCastException <tt>o</tt> is not a BigDecimal.
    * @see    #compareTo(java.math.BigDecimal)
    * @see    Comparable
    * @since  1.2
    */
   public int compareTo(Object o)
   {
      return compareTo((BigDecimal)o);
   }

   /**
    * Compares this BigDecimal with the specified Object for
    * equality.  Unlike {@link #compareTo compareTo}, this method
    * considers two BigDecimals equal only if they are equal in value
    * and scale (thus 2.0 is not equal to 2.00 when compared by this
    * method).
    *
    * @param  x Object to which this BigDecimal is to be compared.
    * @return <tt>true</tt> if and only if the specified Object is a
    *	       BigDecimal whose value and scale are equal to this BigDecimal's.
    * @see    #compareTo(java.math.BigDecimal)
    */
   public boolean equals(Object x)
   {
      if (!(x instanceof BigDecimal))
         return false;
      BigDecimal xDec = (BigDecimal) x;

      if(intVal == null)
         intVal = BigInteger.valueOf(intLong);
      if(xDec.intVal == null) {
         try {
            final long lTemp = xDec.intLong;
            BigInteger biTemp = (BigInteger)
               (java.security.AccessController.doPrivileged(
                  new java.security.PrivilegedExceptionAction() {
                     public Object run() {
                        return BigInteger.valueOf(lTemp);
                     }
                  }
               ));
            xDec.intVal = biTemp;
         } catch (java.security.PrivilegedActionException pae) {
            xDec.intVal = BigInteger.valueOf(xDec.intLong);
         }
      }

      //End of intLong support

      return scale == xDec.scale && intVal.equals(xDec.intVal);
   }

   /**
    * Returns the minimum of this BigDecimal and <tt>val</tt>.
    *
    * @param  val value with which the minimum is to be computed.
    * @return the BigDecimal whose value is the lesser of this BigDecimal and
    *	       <tt>val</tt>.  If they are equal, as defined by the
    * 	       {@link #compareTo compareTo} method, either may be returned.
    * @see    #compareTo(java.math.BigDecimal)
    */
   public BigDecimal min(BigDecimal val)
   {
      return(compareTo(val)<0 ? this : val);
   }

   /**
    * Returns the maximum of this BigDecimal and <tt>val</tt>.
    *
    * @param  val value with which the maximum is to be computed.
    * @return the BigDecimal whose value is the greater of this BigDecimal
    *	       and <tt>val</tt>.  If they are equal, as defined by the
    * 	       {@link #compareTo compareTo} method, either may be returned.
    * @see    #compareTo(java.math.BigDecimal)
    */
   public BigDecimal max(BigDecimal val)
   {
      return(compareTo(val)>0 ? this : val);
   }


   // Hash Function

   /**
    * Returns the hash code for this BigDecimal.  Note that two BigDecimals
    * that are numerically equal but differ in scale (like 2.0 and 2.00)
    * will generally <i>not</i> have the same hash code.
    *
    * @return hash code for this BigDecimal.
    */
   public int hashCode()
   {
      //Begin intLong support
      if (intLong != NO_LONG)
      {
         /* ibm@51972.1 revised this calculation */
         BigInteger intTemp = BigInteger.valueOf(intLong);
         return 31*intTemp.hashCode() + scale;
         /* ibm@51972.1 ends */
      }
      //End intLong support
      return 31*intVal.hashCode() + scale;
   }

    //
    // add one to the least significant digit.
    // in the unlikely event there is a carry out,
    // deal with it.
    //
    private String
    roundup(String val){
        int i;
        char[] digits = val.toCharArray();
        int nDigits = digits.length;

        int q = digits[ i = (nDigits-1)];
        if ( q == '9' ){
            while ( q == '9' && i > 0 ){
                digits[i] = '0';
                q = digits[--i];
            }
            if ( q == '9' ){
                // carryout! High-order 1, rest 0s, larger exp.
                digits[0] = '0';
                return "1" + String.valueOf(digits);
            }
            // else fall through.
        }
        digits[i] = (char)(q+1);
        return String.valueOf(digits);
    }



   // Format Converters

   /**
    * Returns the string representation of this BigDecimal.  The digit-to-
    * character mapping provided by {@link Character#forDigit} is used.
    * A leading minus sign is used to indicate sign, and the number of digits
    * to the right of the decimal point is used to indicate scale.  (This
    * representation is compatible with the (String) constructor.)
    *
    * @return String representation of this BigDecimal.
    * @see    Character#forDigit
    * @see    #BigDecimal(java.lang.String)
    */
   public String toString()
   {
      String intString;


       //Begin intLong support
      if (intLong != NO_LONG)
      {
         if (scale == 0)
            return Long.toString(intLong);

         if(scale <= FASTTOSTRING_MAXSCALE)
             return LongtoString(intLong);

         intString = Long.toString(Math.abs(intLong));
      }
      else
      {
         if (scale == 0)   // No decimal point
            return intVal.toString();
         intString = intVal.abs().toString();
      }

      //Insert decimal point
      StringBuffer buf;
      int signum = signum();
      int insertionPoint = intString.length() - scale;
      if (insertionPoint == 0)
      {  // Point goes right before intVal
         return(signum<0 ? "-0." : "0.") + intString;
      }
      else if (insertionPoint > 0)
      { //Point goes inside intVal
         buf = new StringBuffer(intString);
         buf.insert(insertionPoint, '.');
         if (signum < 0)
            buf.insert(0, '-');
      }
      else
      { //We must insert zeros between point and intVal
         buf = new StringBuffer(3-insertionPoint + intString.length());
         buf.append(signum<0 ? "-0." : "0.");
         for (int i=0; i<-insertionPoint; i++)
            buf.append('0');
         buf.append(intString);
      }
      return buf.toString();

   }

   // Per-thread buffer for string/stringbuffer conversion
   private static ThreadLocal perThreadBuffer = new ThreadLocal() {
      protected synchronized Object initialValue() {
         return new char[CHARBUFFER_SIZE];
      }
   };   
   
   private String LongtoString(long i) {
   //if (i == Long.MIN_VALUE)             // This cannot happen as Long.MIN_VALUE == NO_LONG
   //   return "-9223372036854775808";    // and this is an intLong only codepath  
      char[] buf = (char[])(perThreadBuffer.get());
      long q;
      int r;
      int charPos = CHARBUFFER_SIZE;
      int insertionPoint = charPos - scale;
      char sign = 0;

      if (i < 0) {
         sign = '-';
         i = -i;
      }

      // Get 2 digits/iteration using longs until quotient fits into an int
      while (i > Integer.MAX_VALUE) { 
         q = i / 100;
         // really: r = i - (q * 100);
         r = (int)(i - ((q << 6) + (q << 5) + (q << 2)));
         i = q;
         if (insertionPoint == charPos)
             buf[--charPos] = '.';
         buf[--charPos] = DigitOnes[r];
         if(insertionPoint == charPos)
             buf[--charPos] = '.';
         buf[--charPos] = DigitTens[r];
      }

      // Get 2 digits/iteration using ints
      int q2;
      int i2 = (int)i;
      while (i2 >= 65536) {
         q2 = i2 / 100;
         // really: r = i2 - (q * 100);
         r = i2 - ((q2 << 6) + (q2 << 5) + (q2 << 2));
         i2 = q2;
         if (insertionPoint == charPos)
             buf[--charPos] = '.';
         buf[--charPos] = DigitOnes[r];
         if (insertionPoint == charPos)
             buf[--charPos] = '.';
         buf[--charPos] = DigitTens[r];
      }

      // Fall thru to fast mode for smaller numbers
      // assert(i2 <= 65536, i2);
      for (;;) {
         if (insertionPoint == charPos)
             buf[--charPos] = '.';

         q2 = (i2 * 52429) >>> (16+3);
         r = i2 - ((q2 << 3) + (q2 << 1));  // r = i2-(q2*10) ...
         buf[--charPos] = DigitOnes[r];
         i2 = q2;
         if (i2 == 0)
             break;
      }

      // check for padding 0s
      if(insertionPoint <= charPos) {

         while ((charPos - insertionPoint) >= 1) {
            buf[--charPos] = '0';
         }

         buf[--charPos] = '.';
         buf[--charPos] = '0';
      }

      // check for negative values
      if (sign != 0) {
         buf[--charPos] = sign;
      }

      return new String(buf, charPos, (CHARBUFFER_SIZE - charPos));
   }

   /**
    * Converts this BigDecimal to a BigInteger.  This conversion is
    * analogous to a <a
    * href="http://java.sun.com/docs/books/jls/second_edition/html/conversions.doc.html#25363"><i>narrowing
    * primitive conversion</i></a> from <code>double</code> to
    * <code>long</code> as defined in the <a
    * href="http://java.sun.com/docs/books/jls/html/">Java Language
    * Specification</a>: any fractional part of this BigDecimal will
    * be discarded.  Note that this conversion can lose information
    * about the precision of the BigDecimal value.
    *
    * @return this BigDecimal converted to a BigInteger.
    */
   public BigInteger toBigInteger()
   {
      //Begin intLong support
      if (intVal == null)
         intVal = BigInteger.valueOf(intLong);
      //End intLong support
      return(scale==0 ? intVal
             : intVal.divide(BigInteger.valueOf(10).pow(scale)));
   }

    /**
     * Converts this BigDecimal to an <code>int</code>.  This
     * conversion is analogous to a <a
     * href="http://java.sun.com/docs/books/jls/second_edition/html/conversions.doc.html#25363"><i>narrowing
     * primitive conversion</i></a> from <code>double</code> to
     * <code>short</code> as defined in the <a
     * href="http://java.sun.com/docs/books/jls/html/">Java Language
     * Specification</a>: any fractional part of this BigDecimal will
     * be discarded, and if the resulting &quot;BigInteger&quot; is
     * too big to fit in an <code>int</code>, only the low-order 32
     * bits are returned.  Note that this conversion can lose
     * information about the overall magnitude and precision of the
     * BigDecimal value as well as return a result with the opposite
     * sign.
     *
     * @return this BigDecimal converted to an <code>int</code>.
     */
   public int intValue()
   {
      //Begin intLong support
      if (intLong != NO_LONG && scale == 0)
         return(int)intLong;
      //End intLong support (intVal == null handled in toBigInteger)
      return toBigInteger().intValue();
   }

    /**
     * Converts this BigDecimal to a <code>long</code>.  This
     * conversion is analogous to a <a
     * href="http://java.sun.com/docs/books/jls/second_edition/html/conversions.doc.html#25363"><i>narrowing
     * primitive conversion</i></a> from <code>double</code> to
     * <code>short</code> as defined in the <a
     * href="http://java.sun.com/docs/books/jls/html/">Java Language
     * Specification</a>: any fractional part of this BigDecimal will
     * be discarded, and if the resulting &quot;BigInteger&quot; is
     * too big to fit in a <code>long</code>, only the low-order 64
     * bits are returned.  Note that this conversion can lose
     * information about the overall magnitude and precision of the
     * BigDecimal value as well as return a result with the opposite
     * sign.
     *
     * @return this BigDecimal converted to an <code>long</code>.
     */
   public long longValue()
   {
      //Begin intLong support
      if (intLong != NO_LONG && scale == 0)
         return intLong;
      //End intLong support
      return toBigInteger().longValue();
   }

   /**
    * Converts this BigDecimal to a <code>float</code>.  This
    * conversion is similar to the <a
    * href="http://java.sun.com/docs/books/jls/second_edition/html/conversions.doc.html#25363"><i>narrowing
    * primitive conversion</i></a> from <code>double</code> to
    * <code>float</code> defined in the <a
    * href="http://java.sun.com/docs/books/jls/html/">Java Language
    * Specification</a>: if this BigDecimal has too great a magnitude
    * to represent as a <code>float</code>, it will be converted to
    * {@link Float#NEGATIVE_INFINITY} or {@link
    * Float#POSITIVE_INFINITY} as appropriate.  Note that even when
    * the return value is finite, this conversion can lose
    * information about the precision of the BigDecimal value.
    *
    * @return this BigDecimal converted to a <code>float</code>.
    */
   public float floatValue()
   {
      /* Somewhat inefficient, but guaranteed to work. */
      return Float.valueOf(this.toString()).floatValue();
   }

   /**
    * Converts this BigDecimal to a <code>double</code>.  This
    * conversion is similar to the <a
    * href="http://java.sun.com/docs/books/jls/second_edition/html/conversions.doc.html#25363"><i>narrowing
    * primitive conversion</i></a> from <code>double</code> to
    * <code>float</code> as defined in the <a
    * href="http://java.sun.com/docs/books/jls/html/">Java Language
    * Specification</a>: if this BigDecimal has too great a magnitude
    * represent as a <code>double</code>, it will be converted to
    * {@link Double#NEGATIVE_INFINITY} or {@link
    * Double#POSITIVE_INFINITY} as appropriate.  Note that even when
    * the return value is finite, this conversion can lose
    * information about the precision of the BigDecimal value.
    *
    * @return this BigDecimal converted to a <code>double</code>.
    */
   public double doubleValue()
   {
      /* Somewhat inefficient, but guaranteed to work. */
      return Double.valueOf(this.toString()).doubleValue();
   }


   // Private "Helper" Methods

    /* Returns a digit.digit string */
    private String getValueString(int signum, String intString, int scale) {
        /* Insert decimal point */
        StringBuffer buf;
        int insertionPoint = intString.length() - scale;
        if (insertionPoint == 0) {  /* Point goes right before intVal */
            return (signum<0 ? "-0." : "0.") + intString;
        } else if (insertionPoint > 0) { /* Point goes inside intVal */
            buf = new StringBuffer(intString);
            buf.insert(insertionPoint, '.');
            if (signum < 0)
                buf.insert(0, '-');
        } else { /* We must insert zeros between point and intVal */
            buf = new StringBuffer(3-insertionPoint + intString.length());
            buf.append(signum<0 ? "-0." : "0.");
            for (int i=0; i<-insertionPoint; i++)
                buf.append('0');
            buf.append(intString);
        }
        return buf.toString();
    }

   // START Special methods for intLong
   private void multIntLong(long m)
   {

      long max;
      if (m < 0)
         max = MIN_LONG/m;
      else
         max = MAX_LONG/m;
      if (((intLong<0) & (-intLong<max)) || ((intLong>=0) & (intLong<max)))
      {

         intLong *= m;
      }
      else
      {       // handle as Big Integer
         intVal = BigInteger.valueOf(intLong).multiply(BigInteger.valueOf(m));
         intLong = NO_LONG;                                                 /*ibm@41155*/
      }
   }

   private BigDecimal scaleUp(BigDecimal a, int b )
   {
      if(a.intLong == NO_LONG)
      {
         BigDecimal t = new BigDecimal(timesTenToThe(a.intVal,b),a.scale+b);
         return new BigDecimal(timesTenToThe(a.intVal,b),a.scale+b);
      }
      else
      {
         long iLong = a.intLong;

         if (b<MAX_DIGITS)
         {
            long multval=1;

            for (int i=0; i<b; i++)
               multval *=10L;

            long max = MAX_LONG / multval;
            if (((iLong<0) & (-iLong<max)) || ((iLong>=0) & (iLong<max)))
               return new BigDecimal(iLong*multval, a.scale+b);
            else
            {
               return new BigDecimal(timesTenToThe(BigInteger.valueOf(iLong),b), a.scale+b);
            }

         }
         else
            return new BigDecimal(timesTenToThe(BigInteger.valueOf(iLong),b),a.scale+b);

      }

   }

   /* Returns (a * 10^b) */
   private static BigInteger timesTenToThe(BigInteger a, int b)
   {
      return a.multiply(BigInteger.valueOf(10).pow(b));
   }


   /*
    * If the scales of val[0] and val[1] differ, rescale (non-destructively)
    * the lower-scaled BigDecimal so they match.
    */
   private static void matchScale(BigDecimal[] val)
   {
      if (val[0].scale < val[1].scale)
         val[0] = val[0].setScale(val[1].scale);
      else if (val[1].scale < val[0].scale)
         val[1] = val[1].setScale(val[0].scale);
   }

   /**
    * Reconstitute the <tt>BigDecimal</tt> instance from a stream (that is,
    * deserialize it).
    */
   private synchronized void readObject(java.io.ObjectInputStream s)
   throws java.io.IOException, ClassNotFoundException {
      intLong = NO_LONG;                                            /*ibm@40825.1*/
      // Read in all fields
      s.defaultReadObject();

      // Validate scale factor
      if (scale < 0)
         throw new java.io.StreamCorruptedException(
                                                   "BigDecimal: Negative scale");
   }

   /* ibm@40919 */
   /**
    * Serialize this <tt>BigDecimal</tt> instance.
    */
   private synchronized void writeObject(java.io.ObjectOutputStream s)
   throws java.io.IOException {
      /* Synchronise intVal for compatibility with earlier implementations
       * of java.math.BigDecimal
       */
      if (intLong != NO_LONG) {
          intVal = BigInteger.valueOf(intLong);
      }
      /* Write all fields */
      s.defaultWriteObject();
   }
}
