/*
 * @(#)src/classes/sov/java/lang/ClassLoader.java, jvm, hs142, 20050519 1.55
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v1.4.2
 * (C) Copyright IBM Corp. 1998, 2004. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * ===========================================================================
 */




/*
 *
 * Change activity:
 *
 * Reason  Date   Origin  Description
 * ------  ----   ------  ----------------------------------------------------
 * 009184  250400 hdmaw   ScJVM: Shared Classes Phase I, Copy private fields
 * 009183  270400 hdmaw   ScJVM: Shared Classes Phase I, Register & Reset
 * 009768  110500 hdmaw   ScJVM: Shared Classes Phase I, Policy loader hierarchy
 * 009455  150500 hdmaw   ScJVM: Split class loader cache
 * 005655  220500 hdclw   ScJVM: Mutable statics dirty checks
 * 011675  220500 hdmaw   ScJVM: Reset application class loaders
 * 013848  190700 hdspp   ScJVM: getSystemClassLoader shouldn't be dirty
 * 013768  080800 hdmaw   Overuse of reflection API during ResetJavaVM
 * 012551  110800 hdmaw   ScJVM,PERF: Store trustedness in java.lang.ClassLoader
 * 025627  241000 hdejs:  ScJVM: Add parameter to SetJVMUnresettable...
 * 027157  270201 hdejs   ScJVM: Shared classes 2
 * 029092  140301 hdejs   ScJVM: Add identifier to know if sharing is needed
 * 036549  051001 hdrkm   ScJVM: Fix package support for shared classes
 * 037733  020102 hdseb   Police calls to assertion API
 * 041561  100402 hutchig Change getDefaultDomain as part of 1.4.0 merge
 * 051363.1 050602 vkumara UnsatisfiedLinkError message is unhelpful
 * 053847  300802 seb     Allow registering of shiraz allocation policy
 * 057755  190203 chamlain Race in definePackage for shared classes
 * 059131  260303 seb     Lock heap before allocating codesource buffer
 * 061061  090503 hutchig ClassLoader$Finalizer first field must be classloader
 * 063132  140803 nichanir CICS: RTE initialize the Vector classes to size 100
 * 063520  180803 riclau  Error message from loadLibrary can be misleading
 * 064405  311003 kiranram  Support AIX shared library archives
 * 064582  031103 shankar  142 Sun Code Merge
 * 078731  281004 Vikram   System.loadlibrary() does not match filename
 * 083748  020305 premehta Condition checking for AIX platform
 * 088153  050505 madison  Close zAAP bootclasspath exposure
 * ===========================================================================
 * Module Information:
 *
 * DESCRIPTION: ClassLoader           
 * ===========================================================================
 */


package java.lang;

import java.lang.reflect.Field;                                     /*ibm@9768*/
import java.io.InputStream;
import java.io.ByteArrayInputStream;                                /*ibm@27157*/
import java.io.ByteArrayOutputStream;                               /*ibm@27157*/
import java.io.ObjectInputStream;                                   /*ibm@27157*/
import java.io.ObjectOutputStream;                                  /*ibm@27157*/
import java.io.IOException;
import java.io.StreamCorruptedException;                            /*ibm@27157*/
import java.io.StringWriter;
import java.io.PrintWriter;
import java.io.File;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;
import java.util.Stack;
import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.jar.Manifest;
import java.util.StringTokenizer;
import java.net.URL;
import java.net.MalformedURLException;
import java.security.AccessController;
import java.security.AccessControlContext;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.security.ProtectionDomain;
import java.security.PermissionCollection;
import java.security.CodeSource;
import java.security.Policy;
import sun.misc.URLClassPath;
import sun.misc.Resource;
import sun.misc.CompoundEnumeration;
import sun.misc.ClassFileTransformer;
import sun.reflect.Reflection;
import sun.security.action.GetPropertyAction;

import com.ibm.jvm.InvalidClassLoaderParentException;               /*ibm@9768*/
import com.ibm.jvm.ExtendedSystem;                                    /*SHIRAZ*/
import com.ibm.jvm.ShareableClassLoaderSetAssertException;          /*ibm@37733*/
import com.ibm.jvm.AllocationPolicy;          			    /*ibm@53847*/

import sun.security.util.SecurityConstants;                         /*ibm@64582*/

/**
 * A class loader is an object that is responsible for loading classes. The
 * class <tt>ClassLoader</tt> is an abstract class.  Given the name of a
 * class, a class loader should attempt to locate or generate data that
 * constitutes a definition for the class.  A typical strategy is to transform
 * the name into a file name and then read a "class file" of that name
 * from a file system.
 *
 * <p> Every {@link Class <tt>Class</tt>} object contains a {@link
 * Class#getClassLoader() reference} to the <tt>ClassLoader</tt> that defined
 * it.
 *
 * <p> <tt>Class</tt> objects for array classes are not created by class
 * loaders, but are created automatically as required by the Java runtime.
 * The class loader for an array class, as returned by {@link
 * Class#getClassLoader()} is the same as the class loader for its element
 * type; if the element type is a primitive type, then the array class has no
 * class loader.
 *
 * <p> Applications implement subclasses of <tt>ClassLoader</tt> in order to
 * extend the manner in which the Java virtual machine dynamically loads
 * classes.
 *
 * <p> Class loaders may typically be used by security managers to indicate
 * security domains.
 *
 * <p> The <tt>ClassLoader</tt> class uses a delegation model to search for
 * classes and resources.  Each instance of <tt>ClassLoader</tt> has an
 * associated parent class loader.  When requested to find a class or
 * resource, a <tt>ClassLoader</tt> instance will delegate the search for the
 * class or resource to its parent class loader before attempting to find the
 * class or resource itself.  The virtual machine's built-in class loader,
 * called the "bootstrap class loader", does not itself have a parent but may
 * serve as the parent of a <tt>ClassLoader</tt> instance.
 *
 * <p> Normally, the Java virtual machine loads classes from the local file
 * system in a platform-dependent manner.  For example, on UNIX systems, the
 * virtual machine loads classes from the directory defined by the
 * <tt>CLASSPATH</tt> environment variable.
 *
 * <p> However, some classes may not originate from a file; they may originate
 * from other sources, such as the network, or they could be constructed by an
 * application.  The method {@link #defineClass(String, byte[], int, int)
 * <tt>defineClass</tt>} converts an array of bytes into an instance of class
 * <tt>Class</tt>. Instances of this newly defined class can be created using
 * {@link Class#newInstance <tt>Class.newInstance</tt>}.
 *
 * <p> The methods and constructors of objects created by a class loader may
 * reference other classes.  To determine the class(es) referred to, the Java
 * virtual machine invokes the {@link #loadClass <tt>loadClass</tt>} method of
 * the class loader that originally created the class.
 *
 * <p> For example, an application could create a network class loader to
 * download class files from a server.  Sample code might look like:
 *
 * <blockquote><pre>
 *   ClassLoader loader&nbsp;= new NetworkClassLoader(host,&nbsp;port);
 *   Object main&nbsp;= loader.loadClass("Main", true).newInstance();
 *	 &nbsp;.&nbsp;.&nbsp;.
 * </pre></blockquote>
 *
 * <p> The network class loader subclass must define the methods {@link
 * #findClass <tt>findClass</tt>} and <tt>loadClassData</tt> to load a class
 * from the network.  Once it has downloaded the bytes that make up the class,
 * it should use the method {@link #defineClass <tt>defineClass</tt>} to
 * create a class instance.  A sample implementation is:
 *
 * <blockquote><pre>
 *     class NetworkClassLoader extends ClassLoader {
 *         String host;
 *         int port;
 *
 *         public Class findClass(String name) {
 *             byte[] b = loadClassData(name);
 *             return defineClass(name, b, 0, b.length);
 *         }
 *
 *         private byte[] loadClassData(String name) {
 *             // load the class data from the connection
 *             &nbsp;.&nbsp;.&nbsp;.
 *         }
 *     }
 * </pre></blockquote>
 *
 * @version  1.169, 01/23/03
 * @see      #resolveClass(Class)
 * @since    1.0
 */
public abstract class ClassLoader {

    private static native void registerNatives();
    static {
        registerNatives();
    }


    /*
     * This field is will be set to an internal pointer value by the JVM
     * to anchor various tables associated with the class.  This is defined
     * as a long value to allow sufficient space for 64-bit pointers.  This
     * variable should not be touched by any other code.
     */
    /*ibm@9455*/
    private long loaderCachePointer = 0;

    /*
     * Identifies if this class loader is a trusted loader.  This is only
     * Used by the internal JVM code.
     */
    /*ibm@12551*/
    private boolean isTrusted;

    /*
     * Identifies if this class loader needs to be concerned with sharing
     * Used by the internal JVM code.
     */
    /*ibm@29092*/
    private boolean isShared;

    /*
     * an object to drive class loader finalization without having to add
     * a public finalize() method to the object.
     */
    /*ibm@9455*/
    private Finalizer finalizer;

    /*
     * If initialization succeed this is set to true and security checks will
     * succeed. Otherwise the object is not initialized and the object is
     * useless.
     */
    private boolean initialized = false;

    /*
     * The parent class loader for delegation.
     */
    private ClassLoader parent;

    /*
     * Hashtable that maps packages to certs
     */
    private Hashtable package2certs = new Hashtable(11);

    /*
     * Hashtable that maps CodeSource objects to serialized buffers ibm@27157
     */
    private Hashtable codesource2buffer = new Hashtable(11);    /*ibm@27157*/

    /*
     * shared among all packages with unsigned classes
     */
    java.security.cert.Certificate[] nocerts;

    /*
     * The classes loaded by this class loader. The only purpose of this
     * table is to keep the classes from being GC'ed until the loader
     * is GC'ed.
     */
    private Vector classes;                                         /*ibm@63132*/

    /*
     * Namespace name for shared classes
     */
    private String namespace;                                       /*ibm@9184*/

    /*
     * Indicates an application class loader that has been "deleted" by a
     * ResetJavaVM
     */
    private boolean deleted = false;                               /*ibm@11675*/

    /* Cached private field references */
    /* ibm@13768 */
    private static Field launcherExtLoader = null;


	private static int check_path_len = 0;							/*ibm@78731*/
	static int special_err = 1;										/*ibm@78731*/
	static int count = 0;											/*ibm@78731*/
	static UnsatisfiedLinkError old[];								/*ibm@78731*/
	static int path_size = 0;										/*ibm@78731*/

    /* Cached private field references */
    private static Field packageSealBase = null;                   /*ibm@36549*/

    /*
     * Indicates which heap instances of classes loaded by this
     * classloader should go into
     */
    private AllocationPolicy allocationPolicy = null;		   /*ibm@53847*/
    

    /*ibm@53847*/
    private void registerAllocationPolicy(AllocationPolicy policy)
    {
	allocationPolicy = policy;
    }

    /*
     * Called by the VM to record every loaded class with this loader.
     */
    int addClass(Class c) {					   /*ibm@53847*/
        classes.addElement(c);
	/*ibm@53847 start*/
	if(allocationPolicy == null) {
	    return(AllocationPolicy.DEFAULT);
	} else {
	    return(allocationPolicy.getAllocationPolicy(this, c));
	}
	/*ibm@53847 end*/
    }

    /*
     * The packages defined in this class loader. Each package name is
     * mapped to its corresponding Package object.
     */
    private HashMap packages = new HashMap();

    /**
     * Creates a new class loader using the specified parent class loader
     * for delegation.
     * <p>
     * If there is a security manager, its <code>checkCreateClassLoader</code>
     * method is called. This may result in a security exception.
     *
     * @param parent the parent class loader
     *
     * @throws  SecurityException if a security manager exists and its
     * <code>checkCreateClassLoader</code> method doesn't allow creation of a
     * new class loader.
     * @see       java.lang.SecurityException
     * @see       java.lang.SecurityManager#checkCreateClassLoader()
     * @since     1.2
     */
    protected ClassLoader(ClassLoader parent) {
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkCreateClassLoader();
        }

        setParent(parent);                                          /*ibm@9768*/
        initialize();                                               /*ibm@9183*/
    }

    /**
     * Creates a new class loader using the <code>ClassLoader</code>
     * returned by the method <code>getSystemClassLoader()</code> as the
     * parent class loader.
     * <p>
     * If there is a security manager, its <code>checkCreateClassLoader</code>
     * method is called. This may result in a security exception.
     *
     * @throws  SecurityException
     *    if a security manager exists and its <code>checkCreateClassLoader</code>
     *    method doesn't allow creation of a new class loader.
     *
     * @see       java.lang.SecurityException
     * @see       java.lang.SecurityManager#checkCreateClassLoader()
     */
    protected ClassLoader() {
        SecurityManager security = System.getSecurityManager();
        if (security != null) {
            security.checkCreateClassLoader();
        }

        setParent(getSystemClassLoader());                          /*ibm@9768*/
        initialize();                                               /*ibm@9183*/
    }

    /*
     * Complete initialization of class loader
     */
    /*ibm@9183*/
    private void initialize ()
    {

        /*ibm@63132 start*/
        if (com.ibm.jvm.ExtendedSystem.isResettableJVM()) {
            classes = new Vector(100);
        } else {
            classes = new Vector();
        }
        /*ibm@63132 end*/

        /*
         * Get the backing control blocks created for this class loader
         * and create a finalizer to do the clean up after a GC.
         */
        /*ibm@9455*/
        initialize0();
        finalizer = new Finalizer(this);

        /*
         * Shareable class loaders are not initialized until registered
         * with a namespace
         */
        if (!(this instanceof com.ibm.jvm.classloader.Shareable)) {
            initialized = true;
        }
    }

    native void initialize0();                                      /*ibm@9455*/

    /*
     * Check and set class loader parent
     */
    /*ibm@9768*/
    private void setParent (ClassLoader parent)
    {
        /*
         * Check that the parent is valid:
         * 1. For application must be shareable or bootstrap
         * 2. For middleware must be middleware, extensions or bootstrap
         */
        if (this instanceof com.ibm.jvm.classloader.Shareable) {
            if (!(parent instanceof com.ibm.jvm.classloader.Shareable)
                && (null != parent)) {
                throw new InvalidClassLoaderParentException(
                    "Parent not shareable class loader: " + parent);
            }
        }
        if (this instanceof com.ibm.jvm.classloader.Middleware) {
            if (!(parent instanceof com.ibm.jvm.classloader.Middleware)
                && (null != parent) && (getExtClassLoader() != parent)) {
                throw new InvalidClassLoaderParentException(
                    "Parent not middleware class loader: " + parent);
            }
        }
        this.parent = parent;
    }

    /*
     * Get the standard extensions class loader object
     */
    /*ibm@9768*/
    private static ClassLoader getExtClassLoader ()
    {
        try {
            sun.misc.Launcher launcher = sun.misc.Launcher.getLauncher();

            /*ibm@13768*/
            if (null == launcherExtLoader) {
                launcherExtLoader = getPrivateField(launcher.getClass(),"extLoader");
            }
            return (ClassLoader)launcherExtLoader.get(launcher);
        }
        catch (IllegalAccessException ex) {
            throw new InternalError(ex.toString());
        }
        catch (IllegalArgumentException ex) {
            throw new InternalError(ex.toString());
        }
        catch (SecurityException ex) {
            throw new InternalError(ex.toString());
        }
    }
    /*
     * The sun.misc.Laucher "extLoader" field are all private so we need to get
     * it using reflection.
     */
    /*ibm@9768*/
    private static Field getPrivateField (final Class clazz, final String name)
    {
        Field field2 = (Field)
            AccessController.doPrivileged(new PrivilegedAction() {
            public Object run () {
                Field field1 = null;
                try {
                    field1 = clazz.getDeclaredField(name);
                    if (null != field1) field1.setAccessible(true);
                }
                catch (NoSuchFieldException ex) {
                    throw new InternalError(ex.toString());
                }
                return field1;
            }
        });
        return field2;
    }

    /**
     * Loads the class with the specified name. This method searches for
     * classes in the same manner as the {@link #loadClass(String, boolean)}
     * method. It is called by the Java virtual machine to resolve class
     * references. Calling this method is equivalent to calling
     * <code>loadClass(name, false)</code>.
     *
     * @param     name the name of the class
     * @return    the resulting <code>Class</code> object
     * @exception ClassNotFoundException if the class was not found
     */
    public Class loadClass(String name) throws ClassNotFoundException
    {
        return loadClass(name, false);
    }

                                                                     /*ibm@8389*/
    /**
     * Validates that a given protection domain is allowed to load a
     * class.  This is only called by the internal class loader to
     * verify that it is acceptable to resolve an already defined class
     * within another domain.
     * @param     name     The package name of the class
     * @param     pd       The required protection domain
     */
    private static void checkPackageAccess(final String name, ProtectionDomain pd) {
        final SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            AccessController.doPrivileged(new PrivilegedAction() {
                public Object run() {
                    sm.checkPackageAccess(name);
                    return null;
                }
            }, new AccessControlContext(new ProtectionDomain[] {pd}));
        }
    }                                                                /*ibm@8389*/


    /**
     * Loads the class with the specified name.  The default implementation of
     * this method searches for classes in the following order:<p>
     *
     * <ol>
     * <li> Call {@link #findLoadedClass(String)} to check if the class has
     *      already been loaded. <p>
     * <li> Call the <code>loadClass</code> method on the parent class
     *      loader.  If the parent is <code>null</code> the class loader
     *      built-in to the virtual machine is used, instead. <p>
     * <li> Call the {@link #findClass(String)} method to find the class. <p>
     * </ol>
     *
     * If the class was found using the above steps, and the
     * <code>resolve</code> flag is true, this method will then call the
     * {@link #resolveClass(Class)} method on the resulting class object.
     * <p>
     * From the Java 2 SDK, v1.2, subclasses of ClassLoader are
     * encouraged to override
     * {@link #findClass(String)}, rather than this method.<p>
     *
     * @param     name the name of the class
     * @param     resolve if <code>true</code> then resolve the class
     * @return    the resulting <code>Class</code> object
     * @exception ClassNotFoundException if the class could not be found
     */
    protected synchronized Class loadClass(String name, boolean resolve)
        throws ClassNotFoundException
    {
        // First, check if the class has already been loaded
        Class c = findLoadedClass(name);
        if (c == null) {
            try {
                if (parent != null) {
                    c = parent.loadClass(name, false);
                } else {
                    c = findBootstrapClass0(name);                   /*ibm@8389*/
                }
            } catch (ClassNotFoundException e) {
            }
            if (c == null)
            // If still not found, then call findClass in order
            // to find the class.
                c = findClass(name);
        }
        if (resolve) {
            resolveClass(c);
        }
        return c;
    }

    /*
     * This method is called by the virtual machine to load
     * a class.
     */
    private synchronized Class loadClassInternal(String name)
        throws ClassNotFoundException {

        return loadClass(name);
    }

    /**
     * Finds the specified class. This method should be overridden
     * by class loader implementations that follow the new delegation model
     * for loading classes, and will be called by the <code>loadClass</code>
     * method after checking the parent class loader for the requested class.
     * The default implementation throws <code>ClassNotFoundException</code>.
     *
     * @param  name the name of the class
     * @return the resulting <code>Class</code> object
     * @exception ClassNotFoundException if the class could not be found
     * @since  1.2
     */
    protected Class findClass(String name) throws ClassNotFoundException {
        throw new ClassNotFoundException(name);
    }

    /**
     * Converts an array of bytes into an instance of class <tt>Class</tt>.
     * Before the <tt>Class</tt> can be used it must be resolved.  This method
     * is deprecated in favor of the version that takes the class name as its
     * first argument, and is more secure.
     *
     * @param  b
     *         The bytes that make up the class data.  The bytes in positions
     *         <tt>off</tt> through <tt>off+len-1</tt> should have the format
     *         of a valid class file as defined by the <a
     *         href="http://java.sun.com/docs/books/vmspec/">Java Virtual
     *         Machine Specification</a>.
     *
     * @param  off
     *         The start offset in <tt>b</tt> of the class data
     *
     * @param  len
     *         The length of the class data
     *
     * @return  The <tt>Class</tt> object that was created from the specified
     *          class data
     *
     * @throws  ClassFormatError
     *          If the data did not contain a valid class
     *
     * @throws  IndexOutOfBoundsException
     *          If either <tt>off</tt> or <tt>len</tt> is negative, or if
     *          <tt>off+len</tt> is greater than <tt>b.length</tt>.
     *
     * @see  #loadClass(String, boolean)
     * @see  #resolveClass(Class)
     *
     * @deprecated  Replaced by {@link #defineClass(String, byte[], int, int)
     * defineClass(String, byte[], int, int)}
     */
    protected final Class defineClass(byte[] b, int off, int len)
        throws ClassFormatError
    {
        return defineClass(null, b, off, len, null);
    }

    /**
     * Converts an array of bytes into an instance of class <tt>Class</tt>.
     * Before the <tt>Class</tt> can be used it must be resolved.
     *
     * <p> This method assigns a default {@link java.security.ProtectionDomain
     * <tt>ProtectionDomain</tt>} to the newly defined class.  The
     * <tt>ProtectionDomain</tt> is effectively granted the same set of
     * permissions returned when {@link
     * java.security.Policy#getPermissions(java.security.CodeSource)
     * <tt>Policy.getPolicy().getPermissions(new CodeSource(null, null))</tt>}
     * is invoked.  The default domain is created on the first invocation of
     * {@link #defineClass(String, byte[], int, int) <tt>defineClass</tt>},
     * and re-used on subsequent invocations.
     *
     * <p> To assign a specific <tt>ProtectionDomain</tt> to the class, use
     * the {@link #defineClass(String, byte[], int, int,
     * java.security.ProtectionDomain) <tt>defineClass</tt>} method that takes a
     * <tt>ProtectionDomain</tt> as one of its arguments.  </p>
     *
     * @param  name
     *         The expected name of the class, or <tt>null</tt>
     *         if not known, using '<tt>.</tt>' and not '<tt>/</tt>' as the
     *         separator and without a trailing <tt>.class</tt> suffix.
     *
     * @param  b
     *         The bytes that make up the class data.  The bytes in positions
     *         <tt>off</tt> through <tt>off+len-1</tt> should have the format
     *         of a valid class file as defined by the <a
     *         href="http://java.sun.com/docs/books/vmspec/">Java Virtual
     *         Machine Specification</a>.
     *
     * @param  off
     *         The start offset in <tt>b</tt> of the class data
     *
     * @param  len
     *         The length of the class data
     *
     * @return  The <tt>Class</tt> object that was created from the specified
     *          class data.
     *
     * @throws  ClassFormatError
     *          If the data did not contain a valid class
     *
     * @throws  IndexOutOfBoundsException
     *          If either <tt>off</tt> or <tt>len</tt> is negative, or if
     *          <tt>off+len</tt> is greater than <tt>b.length</tt>.
     *
     * @throws  SecurityException
     *          If an attempt is made to add this class to a package that
     *          contains classes that were signed by a different set of
     *          certificates than this class (which is unsigned), or if the
     *          class name begins with "<tt>java.</tt>".
     *
     * @see  #loadClass(String, boolean)
     * @see  #resolveClass(Class)
     * @see  java.security.CodeSource
     * @see  java.security.SecureClassLoader
     *
     * @since  1.1
     */
    protected final Class defineClass(String name, byte[] b, int off, int len)
        throws ClassFormatError
    {
        return defineClass(name, b, off, len, null);
    }

    /**
     * Converts an array of bytes into an instance of class <tt>Class</tt>,
     * with an optional <tt>ProtectionDomain</tt>.  If the domain is
     * <tt>null</tt>, then a default domain will be assigned to the class as
     * specified in the documentation for {@link #defineClass(String, byte[],
     * int, int)}.  Before the class can be used it must be resolved.
     *
     * <p> The first class defined in a package determines the exact set of
     * certificates that all subsequent classes defined in that package must
     * contain.  The set of certificates for a class is obtained from the
     * {@link java.security.CodeSource <tt>CodeSource</tt>} within the
     * <tt>ProtectionDomain</tt> of the class.  Any classes added to that
     * package must contain the same set of certificates or a
     * <tt>SecurityException</tt> will be thrown.  Note that if the
     * <tt>name</tt> argument is <tt>null</tt>, this check is not performed.
     * You should always pass in the name of the class you are defining as
     * well as the bytes.  This ensures that the class you are defining is
     * indeed the class you think it is.
     *
     * <p> The specified class name cannot begin with "<tt>java.</tt>", since
     * all classes in the "<tt>java.*</tt> packages can only be defined by the
     * bootstrap class loader. If the name parameter is not <tt>null</tt>, it
     * must be equal to the name of the class specified by the byte array
     * "<tt>b</tt>", otherwise a {@link <tt>NoClassDefFoundError</tt>} will be
     * thrown.  </p>
     *
     * @param  name
     *         The expected name of the class, or <tt>null</tt> if not known,
     *         using '<tt>.</tt>' and not '<tt>/</tt>' as the separator and
     *         without a trailing "<tt>.class</tt>" suffix.
     *
     * @param  b
     *         The bytes that make up the class data. The bytes in positions
     *         <tt>off</tt> through <tt>off+len-1</tt> should have the format
     *         of a valid class file as defined by the <a
     *         href="http://java.sun.com/docs/books/vmspec/">Java Virtual
     *         Machine Specification</a>.
     *
     * @param  off
     *         The start offset in <tt>b</tt> of the class data
     *
     * @param  len
     *         The length of the class data
     *
     * @param  protectionDomain
     *         The ProtectionDomain of the class
     *
     * @return  The <tt>Class</tt> object created from the data,
     *          and optional <tt>ProtectionDomain</tt>.
     *
     * @throws  ClassFormatError
     *          If the data did not contain a valid class
     *
     * @throws  NoClassDefFoundError
     *          If <tt>name</tt> is not equal to the name of the class
     *          specified by <tt>b</tt>
     *
     * @throws  IndexOutOfBoundsException
     *          If either <tt>off</tt> or <tt>len</tt> is negative, or if
     *          <tt>off+len</tt> is greater than <tt>b.length</tt>.
     *
     * @throws  SecurityException
     *          If an attempt is made to add this class to a package that
     *          contains classes that were signed by a different set of
     *          certificates than this class, or if the class name begins with
     *          "<tt>java.</tt>".
     */
    protected final Class defineClass(String name, byte[] b, int off, int len,
                                      ProtectionDomain protectionDomain)
        throws ClassFormatError
    {
        check();
        if ((name != null) && name.startsWith("java.")) {
            throw new SecurityException("Prohibited package name: " +
                                        name.substring(0, name.lastIndexOf('.')));
        }
        if (protectionDomain == null) {
            protectionDomain = getDefaultDomain();
        }
        /*ibm@27157 begin*/
        CodeSource source = protectionDomain.getCodeSource();
        java.security.cert.Certificate certs[] = null;
        byte flatsource[] = null;

	if (source != null) {
	    certs = source.getCertificates();
	    if (true == isShared)  /*ibm@29092*/
 		flatsource = resolveSharedCodeSource(source);
	}

        if (name != null)
	    checkCerts(name, source);

        Class c = null;

        try
        {
            c = defineClass0(name, b, off, len, protectionDomain, certs, flatsource);
        }
        catch (ClassFormatError cfe)
        {
            // Class format error - try to transform the bytecode and
            // define the class again
            //
            Object[] transformers = ClassFileTransformer.getTransformers();

            for (int i=0; transformers != null && i < transformers.length; i++)
            {
                try
                {
                    // Transform byte code using transformer
                    byte[] tb = ((ClassFileTransformer) transformers[i]).transform(b, off, len);
                    c = defineClass0(name, tb, 0, tb.length, protectionDomain, certs, flatsource);
                    break;
                }
                catch (ClassFormatError cfe2)
                {
                    // If ClassFormatError occurs, try next transformer
                }
            }

            // Rethrow original ClassFormatError if unable to transform
            // bytecode to well-formed
            //
            if (c == null)
                throw cfe;
        }

        if (certs != null)
            setSigners(c, certs);
        return c;
    }

    /**
     * Converts a CodeSource object into a serialized byte array object
     * allocated from the system heap.  This special byte array object can
     * be reinflated into a CodeSource object for a class in another JVM.
     * Since multiple classes can be associated with a single CodeSource,
     * we cache the CodeSource objects in the class loader to optimize
     * the memory usage.
     *
     * @param source the CodeSource object we're flattening
     * @return a byte-array object created by serializing the CodeSource
     *         object.  Wherever possible, multiple requests for a given
     *         CodeSource will be mapped to the same byte-array object.
     */
    private byte [] resolveSharedCodeSource(CodeSource source)
    {
	/*ibm@59131 Lock the system heap here to avoid race conditions between JVMs*/
	ExtendedSystem.getGlobalLock();				/*ibm@59131*/
        byte buffer[] = (byte[])codesource2buffer.get(source);
        if (buffer == null) {
            try {
                ByteArrayOutputStream stream = new ByteArrayOutputStream();
                ObjectOutputStream ostream = new ObjectOutputStream(stream);
                ostream.writeObject(source);
                buffer = stream.toByteArray();
                buffer = getSystemHeapArray(buffer);
                codesource2buffer.put(source, buffer);
            } catch (IOException e) {}
        }
	ExtendedSystem.releaseGlobalLock();			/*ibm@59131*/
        return buffer;
    }

    private native byte[] getSystemHeapArray(byte[] target);
    /*ibm@27157 end*/

    private synchronized void checkCerts(String name, CodeSource cs)
    {
        int i = name.lastIndexOf('.');
        String pname = (i == -1) ? "" : name.substring(0,i);
        java.security.cert.Certificate[] pcerts =
            (java.security.cert.Certificate[]) package2certs.get(pname);
        if (pcerts == null) {
            // first class in this package gets to define which
            // certificates must be the same for all other classes
            // in this package
            if (cs != null) {
                pcerts = cs.getCertificates();
            }
            if (pcerts == null) {
                if (nocerts == null)
                    nocerts = new java.security.cert.Certificate[0];
                pcerts = nocerts;
            }
            package2certs.put(pname, pcerts);
        } else {
            java.security.cert.Certificate[] certs = null;
            if (cs != null) {
                certs = cs.getCertificates();
            }

            if (!compareCerts(pcerts,certs)) {
                throw new SecurityException("class \""+ name+
                                            "\"'s signer information does not match signer information of other classes in the same package");
            }
        }
    }

    /**
     * check to make sure the certs for the new class (certs) are
     * the same as the certs for the first class inserted
     * in the package (pcerts)
     */
    private boolean compareCerts(java.security.cert.Certificate[] pcerts,
                                 java.security.cert.Certificate[] certs)
    {
        // certs can be null, indicating no certs.
        if ((certs == null) || (certs.length == 0)) {
            return pcerts.length == 0;
        }

        // the length must be the same at this point
        if (certs.length != pcerts.length)
            return false;

        // go through and make sure all the certs in one array
        // are in the other and vice-versa.
        boolean match;
        for (int i=0; i < certs.length; i++) {
            match = false;
            for (int j=0; j < pcerts.length; j++) {
                if (certs[i].equals(pcerts[j])) {
                    match = true;
                    break;
                }
            }
            if (!match) return false;
        }

        // now do the same for pcerts
        for (int i=0; i < pcerts.length; i++) {
            match = false;
            for (int j=0; j < certs.length; j++) {
                if (pcerts[i].equals(certs[j])) {
                    match = true;
                    break;
                }
            }
            if (!match) return false;
        }

        return true;
    }

    /**
     * Links the specified class.
     * This (misleadingly named) method may be used by a class loader to
     * link a class. If the class <code>c</code> has already been linked,
     * then this method simply returns. Otherwise, the class is linked
     * as described in the "Execution" chapter of the <i>Java Language
     * Specification</i>.
     *
     * @param c the class to link
     * @exception NullPointerException if <code>c</code> is <code>null</code>.
     * @see   java.lang.ClassLoader#defineClass(java.lang.String,byte[],int,int)
     */
    protected final void resolveClass(Class c) {
        check();
        resolveClass0(c);
    }

    /**
     * Finds a class with the specified name, loading it if necessary.
     *
     * <p> This method loads the class through the system class loader (see
     * {@link #getSystemClassLoader()}).  The <tt>Class</tt> object returned
     * might have more than one <tt>ClassLoader</tt> associated with it.
     * Subclasses of <tt>ClassLoader</tt> need not usually invoke this method,
     * because most class loaders need to override just {@link
     * #findClass(String)}.  </p>
     *
     * @param  name
     *         The name of the class that is to be found
     *
     * @return  The <tt>Class</tt> object for the specified <tt>name</tt>
     *
     * @throws  ClassNotFoundException
     *          If the class could not be found
     *
     * @see  #ClassLoader(ClassLoader)
     * @see  #getParent()
     */
    protected final Class findSystemClass(String name)
        throws ClassNotFoundException
    {
        check();
        ClassLoader system = getSystemClassLoader();
        if (system == null) {
            return findBootstrapClass(name);
        }
        return system.loadClass(name);
    }

    /**
     * Returns the parent class loader for delegation. Some implementations
     * may use <code>null</code> to represent the bootstrap class
     * loader. This method will return <code>null</code> in such
     * implementations if this class loader's parent is the bootstrap
     * class loader.
     * <p>
     * If a security manager is present, and the caller's class loader is
     * not null and is not an ancestor of this class loader, then
     * this method calls the security manager's <code>checkPermission</code>
     * method with a <code>RuntimePermission("getClassLoader")</code>
     * permission to ensure it's ok to access the parent class loader.
     * If not, a <code>SecurityException</code> will be thrown.
     *
     * @return the parent <code>ClassLoader</code>
     * @throws SecurityException
     *    if a security manager exists and its
     *    <code>checkPermission</code> method doesn't allow
     *    access to this class loader's parent class loader.
     *
     * @see SecurityManager#checkPermission
     * @see java.lang.RuntimePermission
     *
     * @since 1.2
     */
    public final ClassLoader getParent() {
        if (parent == null)
            return null;
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            ClassLoader ccl = getCallerClassLoader();
            if (ccl != null && !isAncestor(ccl)) {
                sm.checkPermission(SecurityConstants.GET_CLASSLOADER_PERMISSION);  /*ibm@64582*/ 
            }
        }
        return parent;
    }

    /**
     * Sets the signers of a class. This should be called after defining a
     * class.
     *
     * @param c the <code>Class</code> object
     * @param signers the signers for the class
     * @since JDK1.1
     */
    protected final void setSigners(Class c, Object[] signers) {
        check();
        c.setSigners(signers);
    }

    private Class findBootstrapClass0(String name)                   /*ibm@8389*/
        throws ClassNotFoundException {                              /*ibm@8389*/
        check();                                                     /*ibm@8389*/
        return findBootstrapClass(name);                             /*ibm@8389*/
    }                                                                /*ibm@8389*/

    private native Class defineClass0(String name, byte[] b, int off, int len,
        ProtectionDomain pd, java.security.cert.Certificate[] certs, byte[] flatsource); /*ibm@27157*/
    private native void resolveClass0(Class c);
    private native Class findBootstrapClass(String name)
        throws ClassNotFoundException;

    /*
     * Check to make sure the class loader has been initialized.
     */
    private void check() {
        if (!initialized) {
            /*
             * Is this an application class loader that was used on a
             * previous run of the JVM?
             */
            /*ibm@11752*/
            if (deleted) {
               ExtendedSystem.setJVMUnresettableConditionally(
                    ExtendedSystem.SCJVM_MIDDLEWARE_RETAINING_APPLICATION_CLASSLOADER,
                    new String("Application class loader retained across reset"));/*ibm@25627*/
            }

            /*ibm@9183*/
            if (!(this instanceof com.ibm.jvm.classloader.Shareable)) {
                throw new SecurityException("ClassLoader object not initialized");
            }
            else {
                throw new SecurityException("ClassLoader object not registered: " + this);
            }
        }
    }

    /**
     * Finds the class with the given name if it had been previously loaded
     * through this class loader.
     *
     * @param  name the class name
     * @return the <code>Class</code> object, or <code>null</code> if
     *         the class has not been loaded
     * @since  JDK1.1
     */
    protected native final Class findLoadedClass(String name);

    /**
     * Finds the resource with the given name.  A resource is some data
     * (images, audio, text, etc) that can be accessed by class code in a way
     * that is independent of the location of the code.
     *
     * <p> The name of a resource is a '<tt>/</tt>'-separated path name that
     * identifies the resource.
     *
     * <p> This method will first search the parent class loader for the
     * resource; if the parent is <tt>null</tt> the path of the class loader
     * built-in to the virtual machine is searched.  That failing, this method
     * will invoke {@link #findResource(String)} to find the resource.  </p>
     *
     * @param  name
     *         The resource name
     *
     * @return  A <tt>URL</tt> object for reading the resource, or
     *          <tt>null</tt> if the resource could not be found or the invoker
     *          doesn't have adequate  privileges to get the resource.
     *
     * @since  1.1
     */
    public URL getResource(String name) {
        URL url;
        if (parent != null) {
            url = parent.getResource(name);
        } else {
            url = getBootstrapResource(name);
        }
        if (url == null) {
            url = findResource(name);
        }
        return url;
    }

    /**
     * Finds all the resources with the given name. A resource is some data
     * (images, audio, text, etc) that can be accessed by class code in a way
     * that is independent of the location of the code.
     *
     * <p>The name of a resource is a <tt>/</tt>-separated path name that
     * identifies the resource.
     *
     * <p> The search order is described in the documentation for {@link
     * #getResource(String)}.  </p>
     *
     * @param  name
     *         The resource name
     *
     * @return  An enumeration of {@link java.net.URL <tt>URL</tt>} objects for
     *          the resource.  If no resources could  be found, the enumeration
     *          will be empty.  Resources that the class loader doesn't have
     *          access to will not be in the enumeration.
     *
     * @throws  IOException
     *          If I/O errors occur
     *
     * @see  #findResources(String)
     *
     * @since  1.2
     */
    public final Enumeration getResources(String name) throws IOException {
        Enumeration[] tmp = new Enumeration[2];
        if (parent != null) {
            tmp[0] = parent.getResources(name);
        } else {
            tmp[0] = getBootstrapResources(name);
        }
        tmp[1] = findResources(name);

        return new CompoundEnumeration(tmp);
    }

    /**
     * Returns an Enumeration of URLs representing all the resources with
     * the given name. Class loader implementations should override this
     * method to specify where to load resources from.
     *
     * @param  name the resource name
     * @return an Enumeration of URLs for the resources
     * @throws IOException if I/O errors occur
     * @since  1.2
     */
    protected Enumeration findResources(String name) throws IOException {
        return new CompoundEnumeration(new Enumeration[0]);
    }

    /**
     * Finds the resource with the given name. Class loader
     * implementations should override this method to specify where to
     * find resources.
     *
     * @param  name the resource name
     * @return a URL for reading the resource, or <code>null</code>
     *         if the resource could not be found
     * @since  1.2
     */
    protected URL findResource(String name) {
        return null;
    }

    /**
     * Find a resource of the specified name from the search path used to load
     * classes.  This method locates the resource through the system class
     * loader (see {@link #getSystemClassLoader()}).  </p>
     *
     * @param  name
     *         The resource name
     *
     * @return  A {@link java.net.URL <tt>URL</tt>} object for reading the
     *          resource, or <tt>null</tt> if the resource could not be found
     *
     * @since  1.1
     */
    public static URL getSystemResource(String name) {
        ClassLoader system = getSystemClassLoader();
        if (system == null) {
            return getBootstrapResource(name);
        }
        return system.getResource(name);
    }

    /**
     * Find resources from the VM's built-in classloader.
     */
    private static URL getBootstrapResource(String name) {
        URLClassPath ucp = getBootstrapClassPath();
        Resource res = ucp.getResource(name);
        return res != null ? res.getURL() : null;
    }

    /**
     * Finds all resources of the specified name from the search path used to
     * load classes.  The resources thus found are returned as an
     * {@link java.util.Enumeration <tt>Enumeration</tt>} of {@link
     * java.net.URL <tt>URL</tt>} objects.
     *
     * <p> The search order is described in the documentation for {@link
     * #getSystemResource(String)}.  </p>
     *
     * @param  name
     *         The resource name
     *
     * @return  An enumeration of resource {@link java.net.URL <tt>URL</tt>}
     *          objects
     *
     * @throws  IOException
     *          If I/O errors occur

     * @since  1.2
     */
    public static Enumeration getSystemResources(String name)
        throws IOException
    {
        ClassLoader system = getSystemClassLoader();
        if (system == null) {
            return getBootstrapResources(name);
        }
        return system.getResources(name);
    }

    /**
     * Find resources from the VM's built-in classloader.
     */
    private static Enumeration getBootstrapResources(String name)
        throws IOException
    {
        final Enumeration e = getBootstrapClassPath().getResources(name);
        return new Enumeration () {
            public Object nextElement() {
                return ((Resource)e.nextElement()).getURL();
            }
            public boolean hasMoreElements() {
                return e.hasMoreElements();
            }
        };
    }

    /*
     * Returns the URLClassPath that is used for finding system resources.
     */
    static URLClassPath getBootstrapClassPath() {
        if (bootstrapClassPath == null) {
            bootstrapClassPath = sun.misc.Launcher.getBootstrapClassPath();
        }
        return bootstrapClassPath;
    }

    private static URLClassPath bootstrapClassPath;

    /**
     * Returns an input stream for reading the specified resource.
     *
     * <p> The search order is described in the documentation for {@link
     * #getResource(String)}.  </p>
     *
     * @param  name
     *         The resource name
     *
     * @return  An input stream for reading the resource, or <tt>null</tt>
     *          if the resource could not be found
     *
     * @since  1.1
     */
    public InputStream getResourceAsStream(String name) {
        URL url = getResource(name);
        try {
            return url != null ? url.openStream() : null;
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * Open for reading, a resource of the specified name from the search path
     * used to load classes.  This method locates the resource through the
     * system class loader (see {@link #getSystemClassLoader()}).  </p>
     *
     * @param  name
     *         The resource name
     *
     * @return  An input stream for reading the resource, or <tt>null</tt>
     * 	        if the resource could not be found
     *
     * @since  1.1
     */
    public static InputStream getSystemResourceAsStream(String name) {
        URL url = getSystemResource(name);
        try {
            return url != null ? url.openStream() : null;
        } catch (IOException e) {
            return null;
        }
    }

    /**
     * Returns the system class loader for delegation.  This is the default
     * delegation parent for new <tt>ClassLoader</tt> instances, and is
     * typically the class loader used to start the application.
     *
     * <p> This method is first invoked early in the runtime's startup
     * sequence, at which point it creates the system class loader and sets it
     * as the context class loader of the invoking <tt>Thread</tt>.
     *
     * <p> The default system class loader is an implementation-dependent
     * instance of this class.
     *
     * <p> If the system property "<tt>java.system.class.loader</tt>" is defined
     * when this method is first invoked then the value of that property is
     * taken to be the name of a class that will be returned as the system
     * class loader.  The class is loaded using the default system class loader
     * and must define a public constructor that takes a single parameter of
     * type <tt>ClassLoader</tt> which is used as the delegation parent.  An
     * instance is then created using this constructor with the default system
     * class loader as the parameter.  The resulting class loader is defined
     * to be the system class loader.
     *
     * <p> If a security manager is present, and the invoker's class loader is
     * not <tt>null</tt> and the invoker's class loader is not the same as or
     * an ancestor of the system class loader, then this method invokes the
     * security manager's {@link
     * SecurityManager#checkPermission(java.security.Permission)
     * <tt>checkPermission</tt>} method with a {@link
     * RuntimePermission#RuntimePermission(String)
     * <tt>RuntimePermission("getClassLoader")</tt>} permission to verify
     * access to the system class loader.  If not, a
     * <tt>SecurityException</tt> will be thrown.  </p>
     *
     * @return  The system <tt>ClassLoader</tt> for delegation, or
     *          <tt>null</tt> if none
     *
     * @throws  SecurityException
     *          If a security manager exists and its <tt>checkPermission</tt>
     *          method doesn't allow access to the system class loader.
     *
     * @throws  IllegalStateException
     *          If invoked recursively during the construction of the class
     *          loader specified by the "<tt>java.system.class.loader</tt>"
     *          property.
     *
     * @throws  Error
     *          If the system property "<tt>java.system.class.loader</tt>"
     *          is defined but the named class could not be loaded, the
     *          provider class does not define the required constructor, or an
     *          exception is thrown by that constructor when it is invoked. The
     *          underlying cause of the error can be retrieved via the
     *          {@link Throwable#getCause()} method.
     *
     * @revised  1.4
     */
    public static ClassLoader getSystemClassLoader() {
        initSystemClassLoader();
        if (scl == null) {
            return null;
        }
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            ClassLoader ccl = getCallerClassLoader();
            if (ccl != null && ccl != scl && !scl.isAncestor(ccl)) {
                sm.checkPermission(SecurityConstants.GET_CLASSLOADER_PERMISSION);  /*ibm@64582*/
            }
        }
        return scl;
    }

    private static synchronized void initSystemClassLoader() {
        if (!sclSet) {
            if (scl != null)
                throw new IllegalStateException("recursive call");
            sun.misc.Launcher l = sun.misc.Launcher.getLauncher();
            if (l != null) {
                Throwable oops = null;
                scl = l.getClassLoader();
                try {
                    PrivilegedExceptionAction a;
                    a = new SystemClassLoaderAction(scl);
                    scl = (ClassLoader) AccessController.doPrivileged(a);
                } catch (PrivilegedActionException pae) {
                    oops = pae.getCause();
                    if (oops instanceof InvocationTargetException) {
                        oops = oops.getCause();
                    }
                }
                if (oops != null) {
                    if (oops instanceof Error) {
                        throw (Error) oops;
                    } else {
                        // wrap the exception
                        throw new Error(oops);
                    }
                }
            }
            sclSet = true;
        }
    }

    /* ibm@5603*/
    private static void removeSystemClassLoader() {
        scl = null;
        sclSet = false;
    }

    /*
     * Phase I. During registerShareableClassLoader() copy private fields
     * from the class loader currently registered with a namespace.
     * Don't copy "initialized" or "parent" as this is set by constructor.
     * No need to copy "classes" as we cleared the Vector at ResetJavaVM.
     */
    /*ibm@9184*/
    private void copyFrom (ClassLoader oldLoader)
    {
        package2certs   = oldLoader.package2certs;
        nocerts         = oldLoader.nocerts;
        packages        = oldLoader.packages;
        nativeLibraries = oldLoader.nativeLibraries;
        defaultDomain   = oldLoader.defaultDomain;
        codesource2buffer = oldLoader.codesource2buffer; /*ibm@27157*/

        /* Set old refs to null to aid GC */
        oldLoader.package2certs   = null;
        oldLoader.nocerts         = null;
        oldLoader.packages        = null;
        oldLoader.nativeLibraries = null;
        oldLoader.defaultDomain   = null;
        oldLoader.codesource2buffer  = null;             /*ibm@27157*/
    }

    // Returns true if the specified class loader can be found
    // in this class loader's delegation chain.
    boolean isAncestor(ClassLoader cl) {
        ClassLoader acl = this;
        do {
            acl = acl.parent;
            if (cl == acl) {
                return true;
            }
        } while (acl != null);
        return false;
    }


    // Returns the caller's class loader, or null if none.
    // NOTE this must always be called when there is exactly one
    // intervening frame from the core libraries on the stack between
    // this method's invocation and the desired caller.
    static ClassLoader getCallerClassLoader() {
        // NOTE use of more generic Reflection.getCallerClass()
        Class caller = Reflection.getCallerClass(3);
        // This can be null if the VM is requesting it
        if (caller == null) {
            return null;
        }
        // Circumvent security check since this is package-private
        return caller.getClassLoader0();
    }


    // The class loader for the system
    private static ClassLoader scl;

    // Set to true once the system class loader has been set
    private static boolean sclSet;

    // Permission to access system or parent class loader
    private static RuntimePermission getClassLoaderPerm = null;

    static RuntimePermission getGetClassLoaderPerm()
    {
        if (getClassLoaderPerm == null)
            getClassLoaderPerm = new RuntimePermission("getClassLoader");
        return getClassLoaderPerm;
    }

    /**
     * Defines a package by name in this <tt>ClassLoader</tt>.  This allows
     * class loaders to define the packages for their classes. Packages must
     * be created before the class is defined, and package names must be
     * unique within a class loader and cannot be redefined or changed once
     * created.  </p>
     *
     * @param  name
     *         The package name
     *
     * @param  specTitle
     *         The specification title
     *
     * @param  specVersion
     *         The specification version
     *
     * @param  specVendor
     *         The specification vendor
     *
     * @param  implTitle
     *         The implementation title
     *
     * @param  implVersion
     *         The implementation version
     *
     * @param  implVendor
     *         The implementation vendor
     *
     * @param  sealBase
     *         If not <tt>null</tt>, then this package is sealed with
     *         respect to the given code source {@link java.net.URL
     *         <tt>URL</tt>}  object.  Otherwise, the package is not sealed.
     *
     * @return  The newly defined <tt>Package</tt> object
     *
     * @throws  IllegalArgumentException
     *          If package name duplicates an existing package either in this
     *          class loader or one of its ancestors
     *
     * @since  1.2
     */
    protected Package definePackage(String name, String specTitle,
                                    String specVersion, String specVendor,
                                    String implTitle, String implVersion,
                                    String implVendor, URL sealBase)
        throws IllegalArgumentException
    {
        synchronized (packages) {
            Package pkg = getPackage(name);
            if (pkg != null) {
                /*ibm@57755 begin*/
                /* this packet is already defined, normally an exception */
                if (isShared) {
                     /* shared classloader, was package defined in this JVM already? */
                     if (pkg.isLocallyDefined()) {
                          throw new IllegalArgumentException(name);
                     } else { 
                         /* OK this time, package was defined by another JVM */
                         pkg.setLocallyDefined(true);
                         return pkg;
                     }
                } else {
                /*ibm@57755 end*/
                     throw new IllegalArgumentException(name);
                }
                                                                        
            }
            pkg = new Package(name, specTitle, specVersion, specVendor,
                              implTitle, implVersion, implVendor,
                              sealBase);
            packages.put(name, pkg);
            /*ibm@27157 begin*/
            /* if this is a shared class loader, we need to */
            /* squirrel this away in the internal tables so the */
            /* other JVMs can use the information. */
            if (true == isShared) { /*ibm@29092*/
                try {
                    ByteArrayOutputStream stream = new ByteArrayOutputStream();
                    ObjectOutputStream ostream = new ObjectOutputStream(stream);
                    /*ibm@36549 begin*/
                    ostream.writeObject(pkg.getName());
                    ostream.writeObject(pkg.getSpecificationTitle());
                    ostream.writeObject(pkg.getSpecificationVersion());
                    ostream.writeObject(pkg.getSpecificationVendor());
                    ostream.writeObject(pkg.getImplementationTitle());
                    ostream.writeObject(pkg.getImplementationVersion());
                    ostream.writeObject(pkg.getImplementationVendor());
                    if (null == packageSealBase) {
                        packageSealBase = getPrivateField(pkg.getClass(),"sealBase");
                    }
                    ostream.writeObject(packageSealBase.get(pkg));
                    byte[] buffer = stream.toByteArray();
                    addNameSpacePackage(name.intern(), buffer);
                } catch (IOException e) {}
                  catch (IllegalAccessException e) {}
                    /*ibm@36549 end*/
            }
            /*ibm@27157 end*/
            return pkg;
        }
    }

    /**
     * Returns a <tt>Package</tt> that has been defined by this class loader
     * or any of its ancestors.  </p>
     *
     * @param  name
     *         The package name
     *
     * @return  The <tt>Package</tt> corresponding to the given name, or
     *          <tt>null</tt> if not found
     *
     * @since  1.2
     */
    protected Package getPackage(String name) {
        synchronized (packages) {
            Package pkg = (Package)packages.get(name);
            /*ibm@27157 begin*/
            if (pkg == null && true == isShared) { /*ibm@29092*/
                /* get this as an interned string so that we can */
                /* look this up from the internal tables. */
                name = name.intern();
                /* now check this internally */
                byte[] flatpackage = getNameSpacePackage(name);
                if (flatpackage != null) {
                    try {
                        ObjectInputStream stream = new ObjectInputStream(new ByteArrayInputStream(flatpackage));
                        /*ibm@36549 begin*/
                        String pkgName = (String)stream.readObject();
                        String specTitle = (String)stream.readObject();
                        String specVersion = (String)stream.readObject();
                        String specVendor = (String)stream.readObject();
                        String implTitle = (String)stream.readObject();
                        String implVersion = (String)stream.readObject();
                        String implVendor = (String)stream.readObject();
                        URL sealBase = (URL)stream.readObject();
                        pkg = new Package(pkgName, specTitle, specVersion, specVendor,
                            implTitle, implVersion, implVendor, sealBase);
                        /* indicate package got from shared internal tables ibm@57755*/
                        pkg.setLocallyDefined(false);                     /*ibm@57755*/
                        /*ibm@36549 end*/
                        packages.put(name, pkg);
                    }
                    catch (IOException e) {}
                    catch (ClassNotFoundException e) {}
                }
            }
            /*ibm@27157 end*/
            if (pkg == null) {
                if (parent != null) {
                    pkg = parent.getPackage(name);
                } else {
                    pkg = Package.getSystemPackage(name);
                }
                if (pkg != null) {
                    packages.put(name, pkg);
                }
            }
            return pkg;
        }
    }

    /**
     * Returns all of the <tt>Packages</tt> defined by this class loader and
     * its ancestors.  </p>
     *
     * @return  The array of <tt>Package</tt> objects defined by this
     *          <tt>ClassLoader</tt>
     *
     * @since  1.2
     */
    protected Package[] getPackages() {
        Map map;
        synchronized (packages) {
            /*ibm@27157 begin*/
            /* if this is a shared name space, we need to */
            /* synchronize the package list with the set of */
            /* packages that have been defined for the namespace. */
            /* We only inflate new entries that might have appeared */
            /* since the last time we checked.  If this is the */
            /* first call, then we'll be picking up the entire set. */
            if (true == isShared) { /*ibm@29092*/
                String[] names = getNameSpacePackageNames();
                for (int i = 0; i < names.length; i++) {
                    String key = names[i];
                    if (key != null && packages.get(key) == null) {
                        try {
                            byte[] flatpackage = getNameSpacePackage(key);
                            ObjectInputStream stream = new ObjectInputStream(new ByteArrayInputStream(flatpackage));
                            /*ibm@36549 begin*/
                            String pkgName = (String)stream.readObject();
                            String specTitle = (String)stream.readObject();
                            String specVersion = (String)stream.readObject();
                            String specVendor = (String)stream.readObject();
                            String implTitle = (String)stream.readObject();
                            String implVersion = (String)stream.readObject();
                            String implVendor = (String)stream.readObject();
                            URL sealBase = (URL)stream.readObject();
                            Package pkg = new Package(pkgName, specTitle, specVersion, specVendor,
                                implTitle, implVersion, implVendor, sealBase);
                            /* indicate package got from shared internal tables ibm@57755*/
                            pkg.setLocallyDefined(false);                     /*ibm@57755*/
                            /*ibm@36549 end*/
                            packages.put(key, pkg);
                        }
                        catch (IOException e) {}
                        catch (ClassNotFoundException e) {}
                    }
                }
            }
            /*ibm@27157 end*/
            map = (Map)packages.clone();
        }


        Package[] pkgs;
        if (parent != null) {
            pkgs = parent.getPackages();
        } else {
            pkgs = Package.getSystemPackages();
        }
        if (pkgs != null) {
            for (int i = 0; i < pkgs.length; i++) {
                String pkgName = pkgs[i].getName();
                if (map.get(pkgName) == null) {
                    map.put(pkgName, pkgs[i]);
                }
            }
        }
        return (Package[])map.values().toArray(new Package[map.size()]);
    }

    private native void addNameSpacePackage(String name, byte[] pkg);/*ibm@27157*/
    private native byte[] getNameSpacePackage(String name);          /*ibm@27157*/
    private native String[] getNameSpacePackageNames();              /*ibm@27157*/

    /**
     * Returns the absolute path name of a native library.  The VM invokes this
     * method to locate the native libraries that belong to classes loaded with
     * this class loader. If this method returns <tt>null</tt>, the VM
     * searches the library along the path specified as the
     * "<tt>java.library.path</tt>" property.  </p>
     *
     * @param  libname
     *         The library name
     *
     * @return  The absolute path of the native library
     *
     * @see  System#loadLibrary(String)
     * @see  System#mapLibraryName(String)
     *
     * @since  1.2
     */
    protected String findLibrary(String libname) {
        return null;
    }

    /**
     * The inner class Finalizer drives finalization of tables created
     * by a class loader.  Once the finalization object is garbage
     * collected, its finalizer will force cleanup of the tables associated
     * with its bound class loader.
     */
    /*ibm@9455*/
    static class Finalizer {

        /* ibm@61061: Do not change the fields of this class without 
         * referring to the changes made under defect 60474 or 61061
         */
        private ClassLoader loader;

        Finalizer(ClassLoader loader) {
            this.loader = loader;
        }

        native void finalize0 (ClassLoader loader);

        protected void finalize() {
            finalize0 (loader);
        }
    }

    /**
     * The inner class NativeLibrary denotes a loaded native library instance.
     * Every classloader contains a vector of loaded native libraries in the
     * private field <tt>nativeLibraries</tt>.  The native libraries loaded
     * into the system are entered into the <tt>systemNativeLibraries</tt>
     * vector.
     *
     * <p> Every native library reuqires a particular version of JNI. This is
     * denoted by the private <tt>jniVersion</tt> field.  This field is set by
     * the VM when it loads the library, and used by the VM to pass the correct
     * version of JNI to the native methods.  </p>
     *
     * @version  1.169 01/23/03
     * @see      ClassLoader
     * @since    1.2
     */
    static class NativeLibrary {
        /* opaque handle to native library, used in native code. */
        long handle;
        /* the version of JNI environment the native library requires. */
        private int jniVersion;
        /* the class from which the library is loaded, also indicates
           the loader this native library belongs. */
        private Class fromClass;
        /* the actual name of the library  - ibm@64405 */
        String actualName;
        /* the canonicalized name of the native library. */
        String name;

        /* Library is "tagged" - set by NativeLibrary.load   ibm@88153 */
        boolean JVM_Tagged;

        native void load(String name, String actualName); /* ibm@64405 */
        native long find(String name);
        native void unload();

        public NativeLibrary(Class fromClass, String name) {
            this.name = name;
            this.fromClass = fromClass;
        }

        /* A new constructor to accomadate the support for AIX shared library archives - ibm@64405 */
        public NativeLibrary(Class fromClass, String name, String actualName) {
            this.actualName = actualName;   
            this.name = name;
            this.fromClass = fromClass;
        }

        protected void finalize() {
            synchronized (loadedLibraryNames) {
                if (fromClass.getClassLoader() != null && handle != 0) {
                    /* remove the native library name */
                    int size = loadedLibraryNames.size();
                    for (int i = 0; i < size; i++) {
                        if (name.equals(loadedLibraryNames.elementAt(i))) {
                            loadedLibraryNames.removeElementAt(i);
                            break;
                        }
                    }
                    /* unload the library. */
                    ClassLoader.nativeLibraryContext.push(this);
                    try {
                        unload();
                    } finally {
                        ClassLoader.nativeLibraryContext.pop();
                    }
                }
            }
        }
        /* Called in the VM to determine the context class in
           JNI_Load/JNI_Unload */
        static Class getFromClass() {
            return ((NativeLibrary)
                    (ClassLoader.nativeLibraryContext.peek())).fromClass;
        }
    }

    /* the "default" domain. Set as the default ProtectionDomain
     * on newly created classses.
     */
    private ProtectionDomain defaultDomain = null;

    /* the "default" permissions, shared by the default domains.
     */
    private static PermissionCollection defaultPermissions = null;

    /*
     * returns (and initializes) the default domain.
     * ibm@41561 function implementation replaced in 1.4.0 merge
     */
    private synchronized ProtectionDomain getDefaultDomain() {
        if (defaultDomain == null) {
            CodeSource cs = new CodeSource(null, null);
            defaultDomain = new ProtectionDomain(cs, null, this, null);
        }
        return defaultDomain;
    }

    /* All native library names we've loaded. */
    private static Vector loadedLibraryNames = new Vector();
    /* Native libraries belonging to system classes. */
    private static Vector systemNativeLibraries = new Vector();
    /* Native libraries associated with the class loader. */
    private Vector nativeLibraries = new Vector();

    /* native libraries being loaded/unloaded. */
    private static Stack nativeLibraryContext = new Stack();

    /* The paths searched for libraries */
    static private String usr_paths[];
    static private String sys_paths[];

    private static String[] initializePath(String propname) {
        String ldpath = System.getProperty(propname, "");
        String ps = File.pathSeparator;
        int ldlen = ldpath.length();
        int i, j, n;
        // Count the separators in the path
        i = ldpath.indexOf(ps);
        n = 0;
        while (i >= 0) {
            n++;
            i = ldpath.indexOf(ps, i+1);
        }

        // allocate the array of paths - n :'s = n + 1 path elements
        String[] paths = new String[n + 1];

        // Fill the array with paths from the ldpath
        n = i = 0;
        j = ldpath.indexOf(ps);
        while (j >= 0) {
            if (j - i > 0) {
                paths[n++] = ldpath.substring(i, j);
            } else if (j - i == 0) {
                paths[n++] = ".";
            }
            i = j + 1;
            j = ldpath.indexOf(ps, i);
        }
        paths[n] = ldpath.substring(i, ldlen);
        return paths;
    }


    /* Called in the java.lang.Runtime class to implement load
     * and loadLibrary.
     */
    static void loadLibrary(Class fromClass, String actualName,        /*ibm@64405*/
                            boolean isAbsolute) {
        ClassLoader loader =
            (fromClass == null) ? null : fromClass.getClassLoader();
        if (sys_paths == null) {
            usr_paths = initializePath("java.library.path");
            sys_paths = initializePath("sun.boot.library.path");
        }
       
        /*
        * The library name has to be extracted from the actual name. This
        * specific to AIX, in case any shared library archives are loaded. - ibm@64405 
        */
	/*start ibm@83748*/
        String name;
	if (System.getProperty("os.name").equals("AIX")) {
	    StringTokenizer firstToken = new StringTokenizer(actualName, "(");
	    name = firstToken.nextToken();
	} else {
            name=actualName;
	}
        /* end ibm@83748*/

        if (isAbsolute) {
            if (loadLibrary0(fromClass, new File(name), actualName)) {   /*ibm@64405*/
                return;
            }
            throw new UnsatisfiedLinkError("Can't load library: " + name);
        }
        if (loader != null) {
            String libfilename = loader.findLibrary(name);
            if (libfilename != null) {
                File libfile = new File(libfilename);
                if (!libfile.isAbsolute()) {
                    throw new UnsatisfiedLinkError(
    "ClassLoader.findLibrary failed to return an absolute path: " + libfilename);
                }
                if (loadLibrary0(fromClass, libfile, actualName)) {   /*ibm@64405*/
                    return;
                }
                throw new UnsatisfiedLinkError ("Can't load " + libfilename);
            }
        }

        /*ibm@4143: Change to library load process to search for library
         *          without exec suffix (eg "_g") if the properly named
         *          library cannot be found first. A warning is issued if
         *          the renamed library is loaded.
         */
        String mappedName = System.mapLibraryName(name);
        String suffixName = new String(mappedName);
        String suffix = System.getProperty("java.execsuffix", ""); /*ibm@8952*/
        for (int j = 0; j < 2; j++) {
            for (int i = 0 ; i < sys_paths.length ; i++) {
                File libfile = new File(sys_paths[i], mappedName);
                if (loadLibrary0(fromClass, libfile, actualName)) {    /*ibm@64405*/
                    if (j == 1) {
                        System.err.println("Warning: Loaded " +
                                           mappedName + " instead of " +
                                           suffixName);
                    }
                    return;
                }
            }

			old = new UnsatisfiedLinkError[256]; /*ibm@78731*/
			path_size = (usr_paths.length -1 ); /*ibm@78731*/

            if (loader != null) {
                for (int i = 0 ; i < usr_paths.length ; i++) {
                    File libfile = new File(usr_paths[i], mappedName);
					
					check_path_len = i;									    /*ibm@78731*/	
				
                    if (loadLibrary0(fromClass, libfile, actualName)) {    /*ibm@64405*/
                        if (j == 1) {
                            System.err.println("Warning: Loaded " +
                                               mappedName + " instead of " +
                                               suffixName);
                        }
                        return;
                    }
                }
            }
            /* If this is a suffix build, try again without the suffix */
            if ((suffix == "") || (mappedName.lastIndexOf(suffix) < 0)) {
                break;
            }
            mappedName = suffixName.substring(0,
                         suffixName.lastIndexOf(suffix)) +
                         suffixName.substring(suffixName.lastIndexOf(suffix) +
                         suffix.length());
        }
        /* end of ibm@4143 */

        // Oops, it failed
        throw new UnsatisfiedLinkError("Can't find library " +
                                       name +
                                       "  (" +
                                       suffixName +
                                       (System.getProperty("os.name").equals("AIX") ? " or .so"
                                                                                    : "") +
                                       (suffix != "" ? ", or " +
                                                       mappedName +
                                                       (System.getProperty("os.name").equals("AIX") ? " or .so"
                                                                                                    : "")
                                                     : "") +
                                       /*ibm@63520 starts*/
                                       ") in sun.boot.library.path" +
                                       (loader != null ? " or java.library.path" : "") +
                                       "\nsun.boot.library.path="+
                                       System.getProperty("sun.boot.library.path") + 
                                       (loader != null ? "\njava.library.path="+
                                                         System.getProperty("java.library.path")
                                                       : "")
                                       /*ibm@63520 ends*/
                                       ); /*ibm@7656
                                                                    ibm@8952
                                                                    ibm@51363.1*/
    }

    private static boolean loadLibrary0(Class fromClass, final File file, String actualName) {
                                                                    /*ibm@2743, ibm@64405*/
/*
    The following test is commented out, as it forces an assumption on the
    library file actually existing under this name.  This prevents the HPI
    from implementing a more sophisticated search order that allows both .a
    and .so library files to be implemented.  The canonical name seems to
    map properly, so it still can be used for a uniqueness test.
        Boolean exists = (Boolean)
            AccessController.doPrivileged(new PrivilegedAction() {
                public Object run() {
                    return new Boolean(file.exists());
                }
            });
        if (!exists.booleanValue()) {
            return false;
        }
 */
        String name;
        try {
            name = file.getCanonicalPath();
        } catch (IOException e) {
            return false;
        }
        ClassLoader loader =
            (fromClass == null) ? null : fromClass.getClassLoader();
        Vector libs =
            loader != null ? loader.nativeLibraries : systemNativeLibraries;
        synchronized (libs) {
            int size = libs.size();
            for (int i = 0; i < size; i++) {
                NativeLibrary lib = (NativeLibrary)libs.elementAt(i);
                if (name.equals(lib.name)) {
                    return true;
                }
            }

            synchronized (loadedLibraryNames) {
                if (loadedLibraryNames.contains(name)) {
                    throw new UnsatisfiedLinkError
                        ("Native Library " +
                         name +
                         " already loaded in another classloader");
                }
                /* If the library is being loaded (must be by
                 * the same thread, because Runtime.load and
                 * Runtime.loadLibrary are synchronous). The
                 * reason is can occur is that the JNI_OnLoad
                 * function can cause another loadLibrary call.
                 *
                 * Thus we can use a static stack to hold the list
                 * of libraries we are loading.
                 *
                 * If there is a pending load operation for the
                 * library, we immediately return success; otherwise,
                 * we raise UnsatisfiedLinkError.
                 */
                int n = nativeLibraryContext.size();
                for (int i = 0; i < n; i++) {
                    NativeLibrary lib = (NativeLibrary)
                        nativeLibraryContext.elementAt(i);
                    if (name.equals(lib.name)) {
                        if (loader == lib.fromClass.getClassLoader()) {
                            return true;
                        } else {
                            throw new UnsatisfiedLinkError
                                ("Native Library " +
                                 name +
                                 " is being loaded in another classloader");
                        }
                    }
                }
                NativeLibrary lib = new NativeLibrary(fromClass, name);
                nativeLibraryContext.push(lib);
                try {
                    lib.load(name, actualName);		            /*ibm@64405*/
                } catch (UnsatisfiedLinkError e) {                  /*ibm@3435*/
                    /* Now we perform the test up */
                    /* above.  If we had a load */
                    /* failure, and the file does not */
                    /* exist, we need to continue */
                    /* searching.  If it does exist, we */
                    /* had a dependency problem, and */
                    /* need to rethrow the exception. */
                    Boolean exists = (Boolean)                      /*ibm@3435*/
                    AccessController.doPrivileged(new PrivilegedAction() {
                        public Object run() {                       /*ibm@3435*/
                        return new Boolean(file.exists());          /*ibm@3435*/
                        }                                           /*ibm@3435*/
                    });                                             /*ibm@3435*/

					if(special_err==1)								/*ibm@78731*/
					{	
						if (!exists.booleanValue()) {               /*ibm@3435*/
								return false;                       /*ibm@3435*/
						}                                           /*ibm@3435*/

					}												/*ibm@78731*/
					special_err = 0;								/*ibm@78731*/
					old[count++] = e;								/*ibm@78731*/

					if(check_path_len<path_size)					/*ibm@78731*/
					{
						return false;								/*ibm@78731*/
					}												/*ibm@78731*/
					  throw old[0];									/*ibm@78731*/	
//                    throw e;                                      /*ibm@3434*/

                } finally {
                    nativeLibraryContext.pop();
                }
                if (lib.handle != 0) {
                    loadedLibraryNames.addElement(name);
                    libs.addElement(lib);
                    return true;
                }
                return false;
            }
        }
    }

    /* Called in the VM class linking code. */
    static long findNative(ClassLoader loader, String name) {
        Vector libs =
            loader != null ? loader.nativeLibraries : systemNativeLibraries;
        synchronized (libs) {
            int size = libs.size();
            for (int i = 0; i < size; i++) {
                NativeLibrary lib = (NativeLibrary)libs.elementAt(i);
                long entry = lib.find(name);
                if (entry != 0)
                    return entry;
            }
        }
        return 0;
    }

    /* Return the value of NativeLibrary.JVM_Tagged for the         */
    /* library in which native method "name" is located.  ibm@88153 */
    static boolean isNativeTagged(ClassLoader loader, String name) {
        Vector libs =
            loader != null ? loader.nativeLibraries : systemNativeLibraries;
        synchronized (libs) {
            int size = libs.size();
            for (int i = 0; i < size; i++) {
                NativeLibrary lib = (NativeLibrary)libs.elementAt(i);
                long entry = lib.find(name);
                if (entry != 0) {
                    return lib.JVM_Tagged;
                }
            }
        }
        return false;
    }       

    /*
     * The default toggle for assertion checking.
     */
    private boolean defaultAssertionStatus = false;

    /*
     * Maps String packageName to Boolean package default assertion status
     * Note that the default package is placed under a null map key.
     * If this field is null then we are delegating assertion status queries
     * to the VM, i.e., none of this ClassLoader's assertion status
     * modification methods have been called.
     */
    private Map packageAssertionStatus = null;

    /*
     * Maps String fullyQualifiedClassName to Boolean assertionStatus
     * If this field is null then we are delegating assertion status queries
     * to the VM, i.e., none of this ClassLoader's assertion status
     * modification methods have been called.
     */
    Map classAssertionStatus = null;

    /**
     * Sets the default assertion status for this class loader.  This setting
     * determines whether classes loaded by this class loader and initialized
     * in the future will have assertions enabled or disabled by default.
     * This setting may be overridden on a per-package or per-class basis by
     * invoking {@link #setPackageAssertionStatus(String,boolean)} or {@link
     * #setClassAssertionStatus(String,boolean)}.
     *
     * @param enabled <tt>true</tt> if classes loaded by this class loader
     *        will henceforth have assertions enabled by default,
     *        <tt>false</tt> if they will have assertions disabled by default.
     * @since 1.4
     */
    public synchronized void setDefaultAssertionStatus(boolean enabled) {
        if(isShared) {
            throw new ShareableClassLoaderSetAssertException(
                "Cannot set default assertion status in a shared class loader");        /*ibm@37733*/
        } else {
            if (classAssertionStatus == null)
                initializeJavaAssertionMaps();

            defaultAssertionStatus = enabled;
        }
    }

    /**
     * Sets the package default assertion status for the named package.  The
     * package default assertion status determines the assertion status for
     * classes initialized in the future that belong to the named package or
     * any of its "subpackages".
     *
     * <p> A subpackage of a package named p is any package whose name begins
     * with "<tt>p.</tt>".  For example, <tt>javax.swing.text</tt> is a
     * subpackage of <tt>javax.swing</tt>, and both <tt>java.util</tt> and
     * <tt>java.lang.reflect</tt> are subpackages of <tt>java</tt>.
     *
     * <p> In the event that multiple package defaults apply to a given class,
     * the package default pertaining to the most specific package takes
     * precedence over the others.  For example, if <tt>javax.lang</tt> and
     * <tt>javax.lang.reflect</tt> both have package defaults associated with
     * them, the latter package default applies to classes in
     * <tt>javax.lang.reflect</tt>.
     *
     * <p> Package defaults take precedence over the class loader's default
     * assertion status, and may be overridden on a per-class basis by invoking
     * {@link #setClassAssertionStatus(String, boolean)}.  </p>
     *
     * @param  packageName
     *         The name of the package whose package default assertion status
     *         is to be set. A <tt>null</tt> value indicates the unnamed
     *         package that is "current" (<a *
     *         href="http://java.sun.com/docs/books/jls/">Java Language
     *         Specification</a>, section 7.4.2).
     *
     * @param  enabled
     *         <tt>true</tt> if classes loaded by this classloader and
     *         belonging to the named package or any of its subpackages will
     *         have assertions enabled by default, <tt>false</tt> if they will
     *         have assertions disabled by default.
     *
     * @since  1.4
     */
    public synchronized void setPackageAssertionStatus(String packageName,
                                                       boolean enabled)
    {
        if(isShared) {
            throw new ShareableClassLoaderSetAssertException(
                "Cannot set package assertion status in a shared class loader");        /*ibm@37733*/
        } else {
            if (packageAssertionStatus == null)
                initializeJavaAssertionMaps();

            packageAssertionStatus.put(packageName, Boolean.valueOf(enabled));
        }
    }

    /**
     * Sets the desired assertion status for the named top-level class in this
     * class loader and any nested classes contained therein.  This setting
     * takes precedence over the class loader's default assertion status, and
     * over any applicable per-package default.  This method has no effect if
     * the named class has already been initialized.  (Once a class is
     * initialized, its assertion status cannot change.)
     *
     * <p> If the named class is not a top-level class, this invocation will
     * have no effect on the actual assertion status of any class, and its
     * return value is undefined.  </p>
     *
     * @param  className
     *         The fully qualified class name of the top-level class whose
     *         assertion status is to be set.
     *
     * @param  enabled
     *         <tt>true</tt> if the named class is to have assertions
     *         enabled when (and if) it is initialized, <tt>false</tt> if the
     *         class is to have assertions disabled.
     *
     * @since  1.4
     */
    public synchronized void setClassAssertionStatus(String className,
                                                     boolean enabled)
    {
        if(isShared) {
            throw new ShareableClassLoaderSetAssertException(
                "Cannot set class assertion status in a shared class loader");      /*ibm@37733*/
        } else {
            if (classAssertionStatus == null)
                initializeJavaAssertionMaps();

            classAssertionStatus.put(className, Boolean.valueOf(enabled));
        }
    }

    /**
     * Sets the default assertion status for this class loader to
     * <tt>false</tt> and discards any package defaults or class assertion
     * status settings associated with the class loader.  This method is
     * provided so that class loaders can be made to ignore any command line or
     * persistent assertion status settings and "start with a clean slate."
     * </p>
     *
     * @since  1.4
     */
    public synchronized void clearAssertionStatus() {
        /*
         * Whether or not "Java assertion maps" are initialized, set
         * them to empty maps, effectively ignoring any present settings.
         */
        if(isShared) {
            throw new ShareableClassLoaderSetAssertException(
                "Cannot clear assertion status in a shareable class loader");       /*ibm@37733*/
        } else {
            classAssertionStatus = new HashMap();
            packageAssertionStatus = new HashMap();

            defaultAssertionStatus = false;
        }
    }

    /**
     * Returns the assertion status that would be assigned to the specified
     * class if it were to be initialized at the time this method is invoked.
     * If the named class has had its assertion status set, the most recent
     * setting will be returned; otherwise, if any package default assertion
     * status pertains to this class, the most recent setting for the most
     * specific pertinent package default assertion status is returned;
     * otherwise, this class loader's default assertion status is returned.
     * </p>
     *
     * @param  className
     *         The fully qualified class name of the class whose desired
     *         assertion status is being queried.
     *
     * @return  The desired assertion status of the specified class.
     *
     * @see  #setClassAssertionStatus(String, boolean)
     * @see  #setPackageAssertionStatus(String, boolean)
     * @see  #setDefaultAssertionStatus(boolean)
     *
     * @since  1.4
     */
    synchronized boolean desiredAssertionStatus(String className) {
        Boolean result;

        // assert classAssertionStatus   != null;
        // assert packageAssertionStatus != null;

        // Check for a class entry
        result = (Boolean)classAssertionStatus.get(className);
        if (result != null)
            return result.booleanValue();

        // Check for most specific package entry
        int dotIndex = className.lastIndexOf(".");
        if (dotIndex < 0) { // default package
            result = (Boolean)packageAssertionStatus.get(null);
            if (result != null)
                return result.booleanValue();
        }
        while(dotIndex > 0) {
            className = className.substring(0, dotIndex);
            result = (Boolean)packageAssertionStatus.get(className);
            if (result != null)
                return result.booleanValue();
            dotIndex = className.lastIndexOf(".", dotIndex-1);
        }

        // Return the classloader default
        return defaultAssertionStatus;
    }

    // Set up the assertions with information provided by the VM.
    private void initializeJavaAssertionMaps() {
        // assert Thread.holdsLock(this);

        classAssertionStatus = new HashMap();
        packageAssertionStatus = new HashMap();
        AssertionStatusDirectives directives = retrieveDirectives();

        for(int i=0; i<directives.classes.length; i++)
            classAssertionStatus.put(directives.classes[i],
                              Boolean.valueOf(directives.classEnabled[i]));

        for(int i=0; i<directives.packages.length; i++)
            packageAssertionStatus.put(directives.packages[i],
                              Boolean.valueOf(directives.packageEnabled[i]));

        defaultAssertionStatus = directives.deflt;
    }

    // Retrieves the assertion directives from the VM.
    private static native AssertionStatusDirectives retrieveDirectives();

}

class SystemClassLoaderAction implements PrivilegedExceptionAction {
    private ClassLoader parent;

    SystemClassLoaderAction(ClassLoader parent) {
        this.parent = parent;
    }

    public Object run() throws Exception {
        ClassLoader sys;
        Constructor ctor;
        Class c;
        Class cp[] = { ClassLoader.class };
        Object params[] = { parent };

        String cls = System.getProperty("java.system.class.loader");
        if (cls == null) {
            return parent;
        }

        c = Class.forName(cls, true, parent);
        ctor = c.getDeclaredConstructor(cp);
        sys = (ClassLoader) ctor.newInstance(params);
        Thread.currentThread().setContextClassLoader(sys);
        return sys;
    }
}
