/*
 * @(#)src/classes/sov/java/io/InternalSHA.java, io, as142, 20050517 1.2.2.1
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v1.4.2
 * (C) Copyright IBM Corp. 1998, 2003. All Rights Reserved
 * ===========================================================================
 */



/*
 * ===========================================================================
 * Change activity:
 *
 * Reason  Date   Origin   Description
 * ------  ----   ------   ---------------------------------------------------
 * 060727  290503 nichanir Serialization loads up too many security classes.
 *                         This class is to be used by ObjectStreamClass as an
 *                         alternative mechanism to MessageDigest using the SHA
 *                         algorithm and will be private to the java.io package
 * 
 * ===========================================================================
 */

package java.io;

/**
 * This class implements the Secure Hash Algorithm (SHA) developed by
 * the National Institute of Standards and Technology along with the
 * National Security Agency.
 * This internal implementation is only used by the ObjectStreamClass code and
 * is based on the InternalSHA class that was originally created by pabbott in
 * defect 56238 to create a bootstrap provider
 *
 */
class InternalSHA {

    /* Buffer of int's and count of characters accumulated
     * 64 bytes are included in each hash block so the low order
     * bits of count are used to know how to pack the bytes into ints
     * and to know when to compute the block and start the next one.
     */
    private int W[] = new int[80];
    private long count = 0;
    private final int countmax = 64;
    private final int countmask = (countmax - 1);

    private int AA, BB, CC, DD, EE;

    /**
     * Creates a new SHA object.
     */
    InternalSHA() {
        reset();
    }

    /**
     * Update a byte.
     *
     * @param b	the byte
     */
    void update(byte b) {
        int word;
        int offset;

        /* 
         * compute word offset and bit offset within word the low bits
         * of count are inverted to make put the bytes in the write order 
         */
        word = ((int) count & countmask) >>> 2;
        offset = (~(int) count & 3) << 3;

        W[word] = (W[word] & ~(0xff << offset)) | ((b & 0xff) << offset);

        /* If this is the last byte of a block, compute the partial hash */
        if (((int) count & countmask) == countmask) {
            computeBlock();
        }
        count++;
    }


    /**
     * Update a buffer.
     *
     * @param b    the data to be updated.
     * @param off  the start offset in the data
     * @param len  the number of bytes to be updated.
     */
    void update(byte b[], int off, int len) {
        int word;

        if ((off < 0) || (len < 0) || (off + len > b.length))
            throw new ArrayIndexOutOfBoundsException();

        // Use single writes until integer aligned
        while ((len > 0) && ((int) count & 3) != 0) {
            update(b[off]);
            off++;
            len--;
        }

        /* Assemble groups of 4 bytes to be inserted in integer array */
        for (; len >= 4; len -= 4, off += 4) {

            word = ((int) count & countmask) >> 2;

            W[word] =
                ((b[off] & 0xff) << 24)
                    | ((b[off + 1] & 0xff) << 16)
                    | ((b[off + 2] & 0xff) << 8)
                    | ((b[off + 3] & 0xff));

            count += 4;
            if (((int) count & countmask) == 0) {
                computeBlock();
            }
        }

        /* Use single writes for last few bytes */
        for (; len > 0; len--, off++) {
            update(b[off]);
        }
    }

    /**
     * Resets the buffers and hash value to start a new hash.
     */
    void reset() {
        AA = 0x67452301;
        BB = 0xefcdab89;
        CC = 0x98badcfe;
        DD = 0x10325476;
        EE = 0xc3d2e1f0;

        for (int i = 0; i < 80; i++)
            W[i] = 0;
        count = 0;
    }

    /**
     * Update the buffer and compute the final hash and return a 
     * byte[20] array
     */
    
    byte[] digest(byte[] input) {
	update(input,0,input.length);
	return digest();
    }

    /**
     * Computes the final hash and returns the final value as a
     * byte[20] array. The object is reset to be ready for further
     * use, as specified in the JavaSecurity MessageDigest
     * specification.  
     */
    byte[] digest() {
        byte hashvalue[] = new byte[20];

        /* The number of bits before padding occurs */
        long bits = count << 3;

        update((byte) 0x80);

        /* Pad with zeros until length is a multiple of 448 (the last two
           32 ints are used a holder for bits (see above). */
        while ((int) (count & countmask) != 56) {
            update((byte) 0);
        }

        W[14] = (int) (bits >>> 32);
        W[15] = (int) (bits & 0xffffffff);

        count += 8;
        computeBlock();

        // Copy out the result
        hashvalue[0] = (byte) (AA >>> 24);
        hashvalue[1] = (byte) (AA >>> 16);
        hashvalue[2] = (byte) (AA >>> 8);
        hashvalue[3] = (byte) (AA >>> 0);

        hashvalue[4] = (byte) (BB >>> 24);
        hashvalue[5] = (byte) (BB >>> 16);
        hashvalue[6] = (byte) (BB >>> 8);
        hashvalue[7] = (byte) (BB >>> 0);

        hashvalue[8] = (byte) (CC >>> 24);
        hashvalue[9] = (byte) (CC >>> 16);
        hashvalue[10] = (byte) (CC >>> 8);
        hashvalue[11] = (byte) (CC >>> 0);

        hashvalue[12] = (byte) (DD >>> 24);
        hashvalue[13] = (byte) (DD >>> 16);
        hashvalue[14] = (byte) (DD >>> 8);
        hashvalue[15] = (byte) (DD >>> 0);

        hashvalue[16] = (byte) (EE >>> 24);
        hashvalue[17] = (byte) (EE >>> 16);
        hashvalue[18] = (byte) (EE >>> 8);
        hashvalue[19] = (byte) (EE >>> 0);

        reset(); // remove the evidence

        return hashvalue;
    }

    // Constants for each round
    private final int round1_kt = 0x5a827999;
    private final int round2_kt = 0x6ed9eba1;
    private final int round3_kt = 0x8f1bbcdc;
    private final int round4_kt = 0xca62c1d6;

    /**
     * Compute a the hash for the current block.
     *
     * This is in the same vein as Peter Gutmann's algorithm listed in
     * the back of Applied Cryptography, Compact implementation of
     * "old" NIST Secure Hash Algorithm.
     *
     */
    private void computeBlock() {
        int temp, a, b, c, d, e;

        // The first 16 ints have the byte stream, compute the rest of
        // the buffer
        for (int t = 16; t <= 79; t++) {
            temp = W[t - 3] ^ W[t - 8] ^ W[t - 14] ^ W[t - 16];
            W[t] = ((temp << 1) | (temp >>> (32 - 1)));
        }

        a = AA;
        b = BB;
        c = CC;
        d = DD;
        e = EE;

        // Round 1
        for (int i = 0; i < 20; i++) {
            temp =
                ((a << 5) | (a >>> (32 - 5)))
                    + ((b & c) | ((~b) & d))
                    + e
                    + W[i]
                    + round1_kt;
            e = d;
            d = c;
            c = ((b << 30) | (b >>> (32 - 30)));
            b = a;
            a = temp;
        }

        // Round 2
        for (int i = 20; i < 40; i++) {
            temp = ((a << 5) | (a >>> (32 - 5))) + (b ^ c ^ d) + e + W[i] + round2_kt;
            e = d;
            d = c;
            c = ((b << 30) | (b >>> (32 - 30)));
            b = a;
            a = temp;
        }

        // Round 3
        for (int i = 40; i < 60; i++) {
            temp =
                ((a << 5) | (a >>> (32 - 5)))
                    + ((b & c) | (b & d) | (c & d))
                    + e
                    + W[i]
                    + round3_kt;
            e = d;
            d = c;
            c = ((b << 30) | (b >>> (32 - 30)));
            b = a;
            a = temp;
        }

        // Round 4
        for (int i = 60; i < 80; i++) {
            temp = ((a << 5) | (a >>> (32 - 5))) + (b ^ c ^ d) + e + W[i] + round4_kt;
            e = d;
            d = c;
            c = ((b << 30) | (b >>> (32 - 30)));
            b = a;
            a = temp;
        }
        AA += a;
        BB += b;
        CC += c;
        DD += d;
        EE += e;
    }

}
