
package java.lang;

import java.io.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * This class is the superclass of all classes which
 * can be thrown by the virtual machine. The two direct
 * subclasses represent recoverable exceptions (Exception)
 * and unrecoverable errors (Error). This class provides
 * common methods for accessing a string message which
 * provides extra information about the circumstances in
 * which the throwable was created, and for filling in a
 * walkback (i.e. a record of the call stack at a
 * particular point in time) which can be printed later.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Error
 * @see			Exception
 * @see			RuntimeException
 */
public class Throwable implements java.io.Serializable {
	static final long serialVersionUID = -3042686055658047285L;

	/**
	 * The message provided when the exception was created.
	 */
	private String detailMessage;

	/**
	 * An object which describes the walkback.
	 */
	private transient Object walkback;

	/**
	 * The cause of this Throwable. Null when there is
	 * no cause.
	 */
	private Throwable cause = this;
	private StackTraceElement[] stackTrace = null;

/**
 * Constructs a new instance of this class with its
 * walkback filled in.
 *
 * @author		OTI
 * @version		initial
 */
public Throwable () {
	super ();
	fillInStackTrace();
}

/**
 * Constructs a new instance of this class with its
 * walkback and message filled in.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		detailMessage String
 *				The detail message for the exception.
 */
public Throwable (String detailMessage) {
	this ();
	this.detailMessage = detailMessage;
}

/**
 * Constructs a new instance of this class with its
 * walkback, message and cause filled in.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		detailMessage String
 *				The detail message for the exception.
 * @param		throwable The cause of this Throwable
 */
public Throwable (String detailMessage, Throwable throwable) {
	this ();
	this.detailMessage = detailMessage;
	cause = throwable;
}

/**
 * Constructs a new instance of this class with its
 * walkback and cause filled in.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		throwable The cause of this Throwable
 */
public Throwable (Throwable throwable) {
	this ();
	this.detailMessage = throwable==null ? null : throwable.toString();
	cause = throwable;
}

/**
 * Record in the receiver a walkback from the point
 * where this message was sent. The message is
 * public so that code which catches a throwable and
 * then <em>re-throws</em> it can adjust the walkback
 * to represent the location where the exception was
 * re-thrown.
 */
public native Throwable fillInStackTrace();

/**
 * Answers the extra information message which was provided
 * when the throwable was created. If no message was provided
 * at creation time, then answer null.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *				The receiver's message.
 */
public String getMessage() {
	return detailMessage;
}

/**
 * Answers the extra information message which was provided
 * when the throwable was created. If no message was provided
 * at creation time, then answer null. Subclasses may override
 * this method to answer localized text for the message.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *				The receiver's message.
 */
public String getLocalizedMessage() {
	return getMessage();
}

/**
 * Answers an array of StackTraceElement. Each StackTraceElement represents
 * a entry on the stack.
 */
private native StackTraceElement[] getStackTraceImpl();

/**
 * Answers an array of StackTraceElement. Each StackTraceElement represents
 * a entry on the stack.
 *
 * @author		OTI
 * @version		initial
 */
public StackTraceElement[] getStackTrace() {
	if (stackTrace == null)
		stackTrace = getStackTraceImpl();
	return (StackTraceElement[])stackTrace.clone();
}

/**
 * Sets the array of StackTraceElements. Each StackTraceElement represents
 * a entry on the stack. A copy of this array will be returned by getStackTrace()
 * and printed by printStackTrace().
 *
 * @author		OTI
 * @version		initial
 *
 * @param	trace The array of StackTraceElement
 */
public void setStackTrace(StackTraceElement[] trace) {
	StackTraceElement[] newTrace = (StackTraceElement[])trace.clone();
	for (int i=0; i<newTrace.length; i++)
		if (newTrace[i] == null) throw new NullPointerException();
	stackTrace = newTrace;
}

/**
 * Outputs a printable representation of the receiver's
 * walkback on the System.err stream.
 *
 * @author		OTI
 * @version		initial
 */
public void printStackTrace () {
	printStackTrace(System.err);
}

/**
 * Outputs a printable representation of the receiver's
 * walkback on the stream specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		err PrintStream
 *				The stream to write the walkback on.
 */
public void printStackTrace (PrintStream err) {
	err.println(toString());
	if (stackTrace == null)
		stackTrace = getStackTraceImpl();
	// Don't use getStackTrace() as it calls clone()
	// Get stackTrace, in case stackTrace is reassigned
	StackTraceElement[] stack = stackTrace;
	for (int i=0; i<stack.length; i++)
		err.println(stack[i]);

	Throwable throwable = getCause();
	if (throwable != null) {
		err.print("Caused by: ");
		throwable.printStackTrace(err);
	}
}

/**
 * Outputs a printable representation of the receiver's
 * walkback on the writer specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		err PrintWriter
 *				The writer to write the walkback on.
 */
public void printStackTrace(PrintWriter err) {
	err.println(toString());
	if (stackTrace == null)
		stackTrace = getStackTraceImpl();
	// Don't use getStackTrace() as it calls clone()
	// Get stackTrace, in case stackTrace is reassigned
	StackTraceElement[] stack = stackTrace;
	for (int i=0; i<stack.length; i++)
		err.println(stack[i]);

	Throwable throwable = getCause();
	if (throwable != null) {
		err.print("Caused by: ");
		throwable.printStackTrace(err);
	}
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *				a printable representation for the receiver.
 */
public String toString () {
	String msg = getLocalizedMessage();
	String name = getClass().getName();
	if (msg == null) {
		return name;
	} else {
		return new StringBuffer(name.length() + 2 + msg.length())
			.append(name).append(": ").append(msg).toString();
	}
}

/**
 * Initialize the cause of the receiver. The cause cannot be
 * reassigned.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		throwable The cause of this Throwable
 *
 * @exception	IllegalArgumentException when the cause is the receiver
 * @exception	IllegalStateException when the cause has already been initialized
 *
 * @return		the receiver.
 */
public synchronized Throwable initCause(Throwable throwable) {
	if (cause == this) {
		if (throwable != this) {
			cause = throwable;
			return this;
		} else throw new IllegalArgumentException("Cause cannot be the receiver");
	} else throw new IllegalStateException("Cause already initialized");
}

/**
 * Answers the cause of this Throwable, or null if there
 * is no cause.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		Throwable
 *				The receiver's cause.
 */
public Throwable getCause() {
	if (cause == this) return null;
	return cause;
}

private void writeObject(ObjectOutputStream s) throws IOException {
	// ensure the stackTrace field is initialized
	if (stackTrace == null)
		stackTrace = getStackTraceImpl();
	s.defaultWriteObject();
}

}
