package com.ibm.oti.vm;

import java.io.*;

/**
 * Provides an InputStream interface over OS memory.
 *
 * <pre>
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corp. 1999, 2004  All Rights Reserved
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 * </pre>
 */
class MemInputStream extends InputStream implements OSMemoryAccessor {
	private long	pointer;
	private int	size;
	private int	offset;
	private int	mark;
	private Jxe	jxe;

/**
 * Answer a new InputStream over the specified OS memory object.
 */
MemInputStream(long pointer, int size, Jxe jxe) {
	this.pointer = pointer;
	this.size = size;
	this.offset = 0;
	this.jxe = jxe;
}

/**
 * Answers the number of bytes left in this input stream.
 */
public int available() throws IOException {
	if (pointer == 0)
		throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
	if (offset >= size) return 0;

	return size - offset;
}
/**
 * Closes the input stream.
 */
public void close() throws IOException {
	pointer = 0;
}

/**
 * Get the pointer to the OS memory.
 */
public long getPointer() {
	return pointer;
}

/**
 * Get the size of this stream.
 */
public int getSize() {
	return size;
}

/**
 * Sets a mark in the input stream.
 */
public synchronized void mark(int readlimit) {
	mark = offset;
}
/**
 * Answers true; this class supports mark/reset.
 */
public boolean markSupported() {
	return true;
}
/**
 * Answers the next byte in the input stream.
 */
public int read() throws IOException {
	if (pointer == 0)
		throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
	if (offset >= size) return -1;

	if (jxe != null && jxe.getJxePointer() == 0)
		throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
	return nativeGetByte(pointer, offset++) & 0xFF;
}
/**
 * Reads up to len bytes from the input stream.  Answers the
 * number of bytes actually read.
 */
public int read(byte[] b, int off, int len) throws IOException {
	if (pointer == 0)
		throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
	if (off >= 0 && len >= 0 && len <= b.length - off) {
		if (offset >= size) return -1;

		int read = min(len,size-offset);
		if (jxe != null && jxe.getJxePointer() == 0)
			throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
		nativeMemcpy(b, off, pointer+offset, read);
		offset += read;
		return read;
	} else throw new ArrayIndexOutOfBoundsException();
}
/**
 * Resets the input stream to the location last mark()ed.
 */
public synchronized void reset() {
	offset = mark;
}
/**
 * Skips over n bytes in the input stream.
 */
public long skip(long n) throws IOException {
	int skipped;

	if (n <= 0) return 0;
	if (pointer == 0)
		throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));

	skipped = min((int)n,size-offset);
	offset += skipped;

	return skipped;
}

/**
 * Calculate the minimum of two ints.
 */
static int min(int x, int y) {
	return x < y ? x : y;
}

private native byte nativeGetByte(long pointer, int offset);
private native void nativeMemcpy(byte[] bytes, int offset, long pointer, int length);
}
