package com.ibm.oti.vm;

import java.io.*;
import java.util.*;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;

/**
 * This class is a model of the meta data available in a loaded jxe.
 * Note that some of the accessors may return null if the information
 * is not available in the jxe.  Generally, the data available through
 * these accessors is calculated when the jxe is created, or explicitly
 * set by the options used to build the jxe.
 *
 * <pre>
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corp. 1999, 2004  All Rights Reserved
 * US Government Users Restricted Rights - Use, duplication or disclosure
 * restricted by GSA ADP Schedule Contract with IBM Corp.
 * </pre>
 */
public class JxeMetaData {
	private static final String META_NAME = "META-INF/JXE.MF";
	private static final boolean useNative = com.ibm.oti.vm.VM.useNatives();

	private Hashtable table;
	private boolean   bigEndian;
	private int       intSize;
	private String    jxeName;
	private String    startupClass;
	private String    uuid;
	private String    version;
	private String    description;
	private String    target;
	private Vector    prereqs;
	private int       ramClassSize;
	private boolean   interpretable = true;
	private boolean   posIndependent = false;
	private char[]    buf;

/**
 * Construct a new JxeMetaData from a Jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
protected JxeMetaData(Jxe jxe) {
	super();

	InputStream stream = jxe.internalGetResourceAsStream(META_NAME);
	if (null != stream)
		initializeTable(stream);
}

private JxeMetaData() {
}

/**
 * Construct a new JxeMetaData from a File which is a JXE.
 */
public static JxeMetaData fromFile(File file) throws IOException, JxeException {
	ZipFile zip = new ZipFile(file);
	ZipEntry entry = zip.getEntry(META_NAME);
	if (entry == null) throw new JxeException(com.ibm.oti.util.Msg.getString("K01c5", file));
	JxeMetaData meta = new JxeMetaData();
	meta.initializeTable(zip.getInputStream(entry));
	return meta;
}

/**
 * Convert UTF8 bytes to a String.
 */
private String convertToString(byte[] bytes, int offset, int length) {
	try {
		if (useNative)
			return com.ibm.oti.util.Util.convertFromUTF8(bytes, offset, length);
		else {
			if (length > buf.length) buf = new char[length];
			return com.ibm.oti.util.Util.convertUTF8WithBuf(bytes, buf, offset, length);
		}
	} catch (UTFDataFormatException e) {
		return "";
	}
}

/**
 * Initialize the table.
 */
private void initializeTable(InputStream stream) {
	byte[]     bytes;
	int        offset;
	int        length;
	String     key;
	String     val;
	Vector     vals;

	table = new Hashtable();

	try {
		bytes = new byte[stream.available()];
		stream.read(bytes, 0, bytes.length);
		stream.close();
	} catch (IOException e) { return; }
	offset = 0;

	if (!useNative);
		buf = new char[256];
	while (offset < bytes.length) {
		length = 0;
		while (0 != bytes[offset+length]) length++;
		key = convertToString(bytes,offset,length);

		offset += length + 1;

		length = 0;
		while (0 != bytes[offset+length]) length++;
		val = convertToString(bytes,offset,length);

		setLocalVal(key,val);

		offset += length + 2; // remember to skip x'0A' !!

		vals = (Vector) table.get(key);
		if (null == vals) {
			vals = new Vector();
			table.put(key,vals);
		}

		vals.addElement(val);
	}
	buf = null;
}

/**
 * Parse an int, return 0 on error.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
private int parseInt(String value) {
	try {
		return Integer.parseInt(value);
	}
	catch (NumberFormatException e) {
		return 0;
	}
}

/**
 * Handle all the keys we want to remember.
 */
private void setLocalVal(String key, String val) {
	int number;

	if (key.equals("bigEndian")) {
		number = parseInt(val);
		bigEndian = (number != 0);
	}

	else if (key.equals("intSize"))
		intSize = parseInt(val);

	else if (key.equals("jxeName"))
		jxeName = val;

	else if (key.equals("startupClass")) {
		if (null != val) {
			val = val.replace('/','.');
		}
		startupClass = val;
	}

	else if (key.equals("uuid"))
		uuid = val;

	else if (key.equals("version"))
		version = val;

	else if (key.equals("description"))
		description = val;

	else if (key.equals("ramClassSize"))
		ramClassSize = parseInt(val);

	else if (key.equals("prereq")) {
		if (null == prereqs) prereqs = new Vector();
		prereqs.addElement(val);

	} else if (key.equals("interpretable")) {
		number = parseInt(val);
		interpretable = (number != 0);

	} else if (key.equals("posIndependent")) {
		number = parseInt(val);
		posIndependent = (number != 0);

	} else if (key.equals("target"))
		target = val;
}

/**
 * Get the values of a key.  Returns null if no values are available for the key, or
 * else a Vector of Strings which are the values associated with the keys.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public Vector getValues(String key) {
	return (Vector) table.get(key);
}

/**
 * Answer whether this jxe is for big endian architectures or not.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public boolean isBigEndian() { return bigEndian; }

/**
 * Answer the int size this jxe was built for (32 or 64).
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public int getIntSize() { return intSize; }

/**
 * Answer the name of the basic name of jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public String getJxeName() { return jxeName; }

/**
 * Answer the startupClass of the jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public String getStartupClass() { return startupClass; }

/**
 * Answer the uuid of the jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public String getUuid() { return uuid; }

/**
 * Answer the version of the jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public String getVersion() { return version; }

/**
 * Answer the description of the jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public String getDescription() { return description; }

/**
 * Answer the ram class size of the jxe.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public int getRamClassSize() { return ramClassSize; }

/**
 * Answer the basic names of prereq jxes.
 *
 * <pre>(c) Copyright IBM Corp 1999</pre>
 */
public Vector getPrereqs() { return prereqs; }

/**
 * Answer whether bytecode is present for all methods in this jxe.
 */
public boolean isInterpretable() { return interpretable; }

/**
 * Answer whether this jxe can be located at any address without relocation.
 */
public boolean isPositionIndependent() { return posIndependent; }

/**
 * Answer the rom image version.
 */
public int getRomImageVersion() {
	if (version != null) {
		int index = version.lastIndexOf('.');
		if (index > 0) {
			String romImageVersion = version.substring(index);
			return parseInt(romImageVersion);
		}
	}
	return 0;
}

/**
 * Answer the target platform identifier of the ahead-of-time
 * compiled code in the jxe, or null if none.
 */
public String getTarget() { return target; }
}
