/*
 * @(#)AtomicLong.java	1.12 06/06/15
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package java.util.concurrent.atomic;
import sun.misc.Unsafe;

/**
 * A {@code long} value that may be updated atomically.  See the
 * {@link java.util.concurrent.atomic} package specification for
 * description of the properties of atomic variables. An
 * {@code AtomicLong} is used in applications such as atomically
 * incremented sequence numbers, and cannot be used as a replacement
 * for a {@link java.lang.Long}. However, this class does extend
 * {@code Number} to allow uniform access by tools and utilities that
 * deal with numerically-based classes.
 *
 * @since 1.5
 * @author Doug Lea
 */
                                                                                //IBM-perf_AtomicLong
/*                                                                              //IBM-perf_AtomicLong
 *  Date     Work-Item Author         Comment                                   //IBM-perf_AtomicLong
 *  ======== ========= ============== ==================================        //IBM-perf_AtomicLong
 *  20061206   JSE-769 Mark Bluemel   Performance changes for PPC               //IBM-perf_AtomicLong
 *                                    (which uses a single lock for             //IBM-perf_AtomicLong
 *                                     volatile longs, so a synchronized        //IBM-perf_AtomicLong
 *                                     version of this class is prefered)       //IBM-perf_AtomicLong
 *                                    Performs acceptably elsewhere, so is      //IBM-perf_AtomicLong
 *                                    to be generally applied.                  //IBM-perf_AtomicLong
 */                                                                             //IBM-perf_AtomicLong
public class AtomicLong extends Number implements java.io.Serializable {
    private static final long serialVersionUID = 1927816293512124184L;

    // setup to use Unsafe.compareAndSwapLong for updates
    private static final Unsafe unsafe = Unsafe.getUnsafe();
    private static final long valueOffset;

    /**
     * Records whether the underlying JVM supports lockless
     * CompareAndSet for longs. While the unsafe.CompareAndSetLong
     * method works in either case, some constructions should be
     * handled at Java level to avoid locking user-visible locks.
     */
    static final boolean VM_SUPPORTS_LONG_CAS = VMSupportsCS8();

    /**
     * Returns whether underlying JVM supports lockless CompareAndSet
     * for longs. Called only once and cached in VM_SUPPORTS_LONG_CAS.
     */
    private static native boolean VMSupportsCS8();

    static {
      try {
        valueOffset = unsafe.objectFieldOffset
            (AtomicLong.class.getDeclaredField("value"));
      } catch (Exception ex) { throw new Error(ex); }
    }

    private long value;                                                         //IBM-perf_AtomicLong

    /**
     * Creates a new AtomicLong with the given initial value.
     *
     * @param initialValue the initial value
     */
    public AtomicLong(long initialValue) {
        value = initialValue;
    }

    /**
     * Creates a new AtomicLong with initial value {@code 0}.
     */
    public AtomicLong() {
    }

    /**
     * Gets the current value.
     *
     * @return the current value
     */
    public final synchronized long get() {                                      //IBM-perf_AtomicLong
        return value;
    }

    /**
     * Sets to the given value.
     *
     * @param newValue the new value
     */
    public final synchronized void set(long newValue) {                         //IBM-perf_AtomicLong
        value = newValue;
    }

    /**
     * Eventually sets to the given value.
     *
     * @param newValue the new value
     * @since 1.6
     */
    public final synchronized void lazySet(long newValue) {                     //IBM-perf_AtomicLong
        value = newValue;                                                       //IBM-perf_AtomicLong
    }

    /**
     * Atomically sets to the given value and returns the old value.
     *
     * @param newValue the new value
     * @return the previous value
     */
    public final synchronized long getAndSet(long newValue) {                   //IBM-perf_AtomicLong
       long current = value;                                                    //IBM-perf_AtomicLong
       value = newValue;                                                        //IBM-perf_AtomicLong
       return current;                                                          //IBM-perf_AtomicLong
    }

    /**
     * Atomically sets the value to the given updated value
     * if the current value {@code ==} the expected value.
     *
     * @param expect the expected value
     * @param update the new value
     * @return true if successful. False return indicates that
     * the actual value was not equal to the expected value.
     */
    public final synchronized boolean compareAndSet(long expect, long update) { //IBM-perf_AtomicLong
        if (value == expect) {                                                  //IBM-perf_AtomicLong
            value = update;                                                     //IBM-perf_AtomicLong
            return true;                                                        //IBM-perf_AtomicLong
        } else {                                                                //IBM-perf_AtomicLong
            return false;                                                       //IBM-perf_AtomicLong
        }                                                                       //IBM-perf_AtomicLong
    }

    /**
     * Atomically sets the value to the given updated value
     * if the current value {@code ==} the expected value.
     *
     * <p>May <a href="package-summary.html#Spurious">fail spuriously</a>
     * and does not provide ordering guarantees, so is only rarely an
     * appropriate alternative to {@code compareAndSet}.
     *
     * @param expect the expected value
     * @param update the new value
     * @return true if successful.
     */
    public final boolean weakCompareAndSet(long expect, long update) {
      return compareAndSet(expect,update);                                      //IBM-perf_AtomicLong
    }

    /**
     * Atomically increments by one the current value.
     *
     * @return the previous value
     */
    public final synchronized long getAndIncrement() {                          //IBM-perf_AtomicLong
       long current = value;                                                    //IBM-perf_AtomicLong
       ++value;                                                                 //IBM-perf_AtomicLong
       return current;                                                          //IBM-perf_AtomicLong
    }

    /**
     * Atomically decrements by one the current value.
     *
     * @return the previous value
     */
    public final synchronized long getAndDecrement() {                          //IBM-perf_AtomicLong
       long current = value;                                                    //IBM-perf_AtomicLong
       --value;                                                                 //IBM-perf_AtomicLong
       return current;                                                          //IBM-perf_AtomicLong
    }                                                                           //IBM-perf_AtomicLong

    /**
     * Atomically adds the given value to the current value.
     *
     * @param delta the value to add
     * @return the previous value
     */
    public final synchronized long getAndAdd(long delta) {                      //IBM-perf_AtomicLong
       long current = value;                                                    //IBM-perf_AtomicLong
       value += delta;                                                          //IBM-perf_AtomicLong
       return current;                                                          //IBM-perf_AtomicLong
    }

    /**
     * Atomically increments by one the current value.
     *
     * @return the updated value
     */
    public final synchronized long incrementAndGet() {                          //IBM-perf_AtomicLong
       ++value;                                                                 //IBM-perf_AtomicLong
       return value;                                                            //IBM-perf_AtomicLong
    }

    /**
     * Atomically decrements by one the current value.
     *
     * @return the updated value
     */
    public final synchronized long decrementAndGet() {                          //IBM-perf_AtomicLong
       --value;                                                                 //IBM-perf_AtomicLong
       return value;                                                            //IBM-perf_AtomicLong
    }

    /**
     * Atomically adds the given value to the current value.
     *
     * @param delta the value to add
     * @return the updated value
     */
    public final synchronized long addAndGet(long delta) {                      //IBM-perf_AtomicLong
       value += delta;                                                          //IBM-perf_AtomicLong
       return value;                                                            //IBM-perf_AtomicLong
    }

    /**
     * Returns the String representation of the current value.
     * @return the String representation of the current value.
     */
    public String toString() {
        return Long.toString(get());
    }


    public int intValue() {
	return (int)get();
    }

    public long longValue() {
	return (long)get();
    }

    public float floatValue() {
	return (float)get();
    }

    public double doubleValue() {
	return (double)get();
    }

}
//IBM-perf_AtomicLong
