package java.lang;

/*
 * Licensed Materials - Property of IBM,
 *     Copyright IBM Corp. 1998, 2007  All Rights Reserved
 */

import java.io.*;
import java.util.Map;
import java.util.Properties;
import java.util.PropertyPermission;
import java.security.*;

/**
 * Class System provides a standard place for programs
 * to find system related information. All System API
 * is static.
 *
 * @author		OTI
 * @version		initial
 */
public final class System {

	// The standard input, output, and error streams.
	// Typically, these are connected to the shell which
	// ran the Java program.
	/**
	 * Default input stream
	 */
	public static final InputStream in;
	/**
	 * Default output stream
	 */
	public static final PrintStream out;
	/**
	 * Default error output stream
	 */
	public static final PrintStream err;

	//Get a ref to the Runtime instance for faster lookup
	private static final Runtime RUNTIME = Runtime.getRuntime();
	/**
	 * The System Properties table.
	 */
	private static Properties systemProperties;

	/**
	 * The System default SecurityManager.
	 */
	private static SecurityManager security;

	private static final int NANO_RETURN = 0;
	private static final int NANO_DIVIDE = 1;
	private static final int NANO_INT_DIVIDE = 2;
	private static final int NANO_MULTIPLY = 3;
	private static final int NANO_INT_MULTIPLY = 4;
	/**
	 * The type of adjustment required.
	 */
	private static final int nanoType;
	/**
	 * Non-integer nano adjustment value.
	 */
	private static final double nanoAdjust;
	/**
	 * Integer nano adjustment value.
	 */
	private static final long nanoIntAdjust;

	private static volatile Console console = null;
	private static volatile boolean consoleInitialized = false;

	// Initialize all the slots in System on first use.
	static {
		long hiresFrequency = hiresFrequencyImpl();
		if (hiresFrequency == 1000000000L) {
			nanoType = NANO_RETURN;
			nanoAdjust = 0;
			nanoIntAdjust = 0;
		} else if (hiresFrequency < 1000000000L) {
			nanoAdjust = (double)1000000000L / hiresFrequency;
			if (Math.floor(nanoAdjust) == nanoAdjust) {
				nanoType = NANO_INT_MULTIPLY;
				nanoIntAdjust = (long)nanoAdjust;
			} else {
				nanoType = NANO_MULTIPLY;
				nanoIntAdjust = 0;
			}
		} else {
			nanoAdjust = hiresFrequency / (double)1000000000L;
			if (Math.floor(nanoAdjust) == nanoAdjust) {
				nanoType = NANO_DIVIDE;
				nanoIntAdjust = (long)nanoAdjust;
			} else {
				nanoType = NANO_DIVIDE;
				nanoIntAdjust = 0;
			}
		}

		// Fill in the properties from the VM information.
		ensureProperties();
		// Set up standard in, out, and err.
		// Check the default encoding
		StringCoding.encode(new char[1], 0, 1);
		err = com.ibm.jvm.io.ConsolePrintStream.localize(new BufferedOutputStream(new FileOutputStream(FileDescriptor.err)), true);
		out = com.ibm.jvm.io.ConsolePrintStream.localize(new BufferedOutputStream(new FileOutputStream(FileDescriptor.out)), true);
		in = null;
		com.ibm.misc.SystemIntialization.firstChanceHook();
	}

static void completeInitialization() {
	setIn(com.ibm.jvm.io.ConsoleInputStream.localize(new BufferedInputStream(new FileInputStream(FileDescriptor.in))));
	Terminator.setup();
	Shutdown.add(ApplicationShutdownHooks.hook());
	com.ibm.misc.SystemIntialization.lastChanceHook();

}

/**
 * Sets the value of the static slot "in" in the receiver
 * to the passed in argument.
 *
 * @param		newIn 		the new value for in.
 */
public static void setIn(InputStream newIn) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToSetIO);
	setFieldImpl("in", newIn); //$NON-NLS-1$
}

/**
 * Sets the value of the static slot "out" in the receiver
 * to the passed in argument.
 *
 * @param		newOut 		the new value for out.
 */
public static void setOut(java.io.PrintStream newOut) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToSetIO);
	setFieldImpl("out", newOut); //$NON-NLS-1$
}

/**
 * Sets the value of the static slot "err" in the receiver
 * to the passed in argument.
 *
 * @param		newErr  	the new value for err.
 */
public static void setErr(java.io.PrintStream newErr) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToSetIO);
	setFieldImpl("err", newErr); //$NON-NLS-1$
}

/**
 * Prevents this class from being instantiated.
 */
private System() {
}

/**
 * Copies the contents of <code>array1</code> starting at offset <code>start1</code>
 * into <code>array2</code> starting at offset <code>start2</code> for
 * <code>length</code> elements.
 *
 * @param		array1 		the array to copy out of
 * @param		start1 		the starting index in array1
 * @param		array2 		the array to copy into
 * @param		start2 		the starting index in array2
 * @param		length 		the number of elements in the array to copy
 */
public static native void arraycopy(Object array1, int start1, Object array2, int start2, int length);

/**
 * Private version of the arraycopy method used by the jit for reference arraycopies
 */
private static void arraycopy(Object[] A1, int offset1, Object[] A2, int offset2, int length) {
	if (A1 == null || A2 == null) throw new NullPointerException();
	if (offset1 >= 0 && offset2 >= 0 && length >= 0 &&
		length <= A1.length - offset1 && length <= A2.length - offset2)
	{
		// Check if this is a forward or backwards arraycopy
		if (A1 != A2 || offset1 > offset2 || offset1+length <= offset2) {
			for (int i = 0; i < length; ++i) {
				A2[offset2+i] = A1[offset1+i];
			}
		} else {
			for (int i = length-1; i >= 0; --i) {
				A2[offset2+i] = A1[offset1+i];
			}
		}
	} else throw new ArrayIndexOutOfBoundsException();
}

/**
 * Answers the current time expressed as milliseconds since
 * the time 00:00:00 UTC on January 1, 1970.
 *
 * @return		the time in milliseconds.
 */
public static native long currentTimeMillis();

private static final int InitLocale = 0;
private static final int PlatformEncoding = 1;
private static final int FileEncoding = 2;
private static final int OSEncoding = 3;

/**
 * If systemProperties is unset, then create a new one based on the values
 * provided by the virtual machine.
 */
@SuppressWarnings("nls")
private static void ensureProperties() {
	systemProperties = new Properties();

	String platformEncoding = getEncoding(PlatformEncoding);
	String fileEncoding, osEncoding = null;
	String definedFileEncoding = getEncoding(FileEncoding);
	String definedOSEncoding = getEncoding(OSEncoding);
	if (definedFileEncoding != null) {
		fileEncoding = definedFileEncoding;
		// if file.encoding is defined, and os.encoding is not, use the detected
		// platform encoding for os.encoding
		if (definedOSEncoding == null) {
			osEncoding = platformEncoding;
		}
	} else {
		fileEncoding = platformEncoding;
	}
	// if os.encoding is not defined, file.encoding will be used
	if (osEncoding == null)
		osEncoding = definedOSEncoding;
	if (osEncoding != null)
		systemProperties.put("os.encoding", osEncoding);
	systemProperties.put("ibm.system.encoding", platformEncoding);
	systemProperties.put("sun.jnu.encoding", platformEncoding);

	systemProperties.put("file.encoding", fileEncoding);

	systemProperties.put("java.version", "1.6.0");
	systemProperties.put("java.specification.version", "1.6");

	systemProperties.put("java.specification.vendor", "Sun Microsystems Inc.");
	systemProperties.put("java.specification.name", "Java Platform API Specification");

	systemProperties.put("com.ibm.oti.configuration", "scar");

	String[] list = getPropertyList();
	for (int i=0; i<list.length; i+=2) {
		String key = list[i];
		if (key == null) break;
		systemProperties.put(key, list[i+1]);
	}

	systemProperties.put("com.ibm.oti.jcl.build", "20080613_1526");
}
/**
 * Causes the virtual machine to stop running, and the
 * program to exit. If runFinalizersOnExit(true) has been
 * invoked, then all finalizers will be run first.
 *
 * @param		code		the return code.
 *
 * @throws		SecurityException 	if the running thread is not allowed to cause the vm to exit.
 *
 * @see			SecurityManager#checkExit
 */
public static void exit(int code) {
	RUNTIME.exit(code);
}

/**
 * Indicate to the virtual machine that it would be a
 * good time to collect available memory. Note that, this
 * is a hint only.
 */
public static void gc() {
	RUNTIME.gc();
}

/**
 * Returns an environment variable.
 *
 * @param 		var			the name of the environment variable
 * @return		the value of the specified environment variable
 */
@SuppressWarnings("dep-ann")
public static String getenv(String var) {
	if (var == null) throw new NullPointerException();
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(new RuntimePermission("getenv." + var)); //$NON-NLS-1$
	return ProcessEnvironment.getenv(var);

}

/**
 * Answers the system properties. Note that this is
 * not a copy, so that changes made to the returned
 * Properties object will be reflected in subsequent
 * calls to getProperty and getProperties.
 * <p>
 * Security managers should restrict access to this
 * API if possible.
 *
 * @return		the system properties
 */
public static Properties getProperties() {

	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPropertiesAccess();

	return systemProperties;
}

/**
 * Answers the system properties without any security
 * checks. This is used for access from within java.lang.
 *
 * @return		the system properties
 */
static Properties internalGetProperties() {
	return systemProperties;
}
/**
 * Answers the value of a particular system property.
 * Answers null if no such property exists,
 * <p>
 * The properties currently provided by the virtual
 * machine are:
 * <pre>
 *     java.vendor.url
 *     java.class.path
 *     user.home
 *     java.class.version
 *     os.version
 *     java.vendor
 *     user.dir
 *     user.timezone
 *     path.separator
 *     os.name
 *     os.arch
 *     line.separator
 *     file.separator
 *     user.name
 *     java.version
 *     java.home
 * </pre>
 *
 * @param		prop		the system property to look up
 * @return		the value of the specified system property, or null if the property doesn't exist
 */
@SuppressWarnings("nls")
public static String getProperty(String prop) {
	return getProperty(prop, null);
}

/**
 * Answers the value of a particular system property.
 * If no such property is found, answers the defaultValue.
 *
 * @param		prop			the system property to look up
 * @param		defaultValue	return value if system property is not found
 * @return		the value of the specified system property, or defaultValue if the property doesn't exist
 */
public static String getProperty(String prop, String defaultValue) {
	if (prop.length() == 0) throw new IllegalArgumentException();

	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPropertyAccess(prop);

	String result = systemProperties.getProperty(prop, defaultValue);

	return result;
}

/**
 * Sets the value of a particular system property.
 *
 * @param		prop		the system property to change
 * @param		value		the value to associate with prop
 * @return		the old value of the property, or null
 */
public static String setProperty(String prop, String value) {
	if (prop.length() == 0) throw new IllegalArgumentException();

	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			new PropertyPermission(prop, "write")); //$NON-NLS-1$
	return (String)systemProperties.setProperty(prop, value);
}

/**
 * Answers an array of Strings containing key..value pairs
 * (in consecutive array elements) which represent the
 * starting values for the system properties as provided
 * by the virtual machine.
 *
 * @return		the default values for the system properties.
 */
private static native String [] getPropertyList();

/**
 * Return the requested encoding.
 * 		0 - initialize locale
 * 		1 - detected platform encoding
 * 		2 - command line defined file.encoding
 * 		3 - command line defined os.encoding
 */
private static native String getEncoding(int type);

/**
 * Answers the active security manager.
 *
 * @return		the system security manager object.
 */
public static SecurityManager getSecurityManager() {
	return security;
}

/**
 * Answers an integer hash code for the parameter.
 * The hash code returned is the same one that would
 * be returned by java.lang.Object.hashCode(), whether
 * or not the object's class has overridden hashCode().
 * The hash code for null is 0.
 *
 * @param		anObject	the object
 * @return		the hash code for the object
 *
 * @see			java.lang.Object#hashCode
 */
public static native int identityHashCode(Object anObject);

/**
 * Loads the specified file as a dynamic library.
 *
 * @param 		pathName	the path of the file to be loaded
 */
public static void load(String pathName) {
	SecurityManager smngr = System.getSecurityManager();
	if (smngr != null)
		smngr.checkLink(pathName);
	ClassLoader.loadLibraryWithPath(pathName, ClassLoader.callerClassLoader(), null);
}

/**
 * Loads and links the library specified by the argument.
 *
 * @param		libName		the name of the library to load
 *
 * @throws		UnsatisfiedLinkError	if the library could not be loaded
 * @throws		SecurityException 		if the library was not allowed to be loaded
 */
public static void loadLibrary(String libName) {
	ClassLoader.loadLibraryWithClassLoader(libName, ClassLoader.callerClassLoader());
}

/**
 * Provides a hint to the virtual machine that it would
 * be useful to attempt to perform any outstanding
 * object finalizations.
 */
public static void runFinalization() {
	RUNTIME.runFinalization();
}

/**
 * Ensure that, when the virtual machine is about to exit,
 * all objects are finalized. Note that all finalization
 * which occurs when the system is exiting is performed
 * after all running threads have been terminated.
 *
 * @param 		flag 		true means finalize all on exit.
 *
 * @deprecated 	This method is unsafe.
 */
@Deprecated
public static void runFinalizersOnExit(boolean flag) {
	Runtime.runFinalizersOnExit(flag);
}

/**
 * Answers the system properties. Note that the object
 * which is passed in not copied, so that subsequent
 * changes made to the object will be reflected in
 * calls to getProperty and getProperties.
 * <p>
 * Security managers should restrict access to this
 * API if possible.
 *
 * @param		p			the property to set
 */
public static void setProperties(Properties p) {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPropertiesAccess();
	if (p == null) {
		ensureProperties();
	} else {
		systemProperties = p;
	}
}

/**
 * Sets the active security manager. Note that once
 * the security manager has been set, it can not be
 * changed. Attempts to do so will cause a security
 * exception.
 *
 * @param		s			the new security manager
 *
 * @throws		SecurityException 	if the security manager has already been set.
 */
public static void setSecurityManager(final SecurityManager s) {
	final SecurityManager currentSecurity = security;
	if (s != null) {
		try {
			// Preload classes used for checkPackageAccess(),
			// otherwise we could go recursive
			s.checkPackageAccess("java.lang"); //$NON-NLS-1$
		} catch (Exception e) {}
		try {
			AccessController.doPrivileged(new PrivilegedAction() {
				public Object run() {
					ProtectionDomain oldDomain = currentSecurity == null ?
						System.class.getPDImpl() : currentSecurity.getClass().getPDImpl();
						ProtectionDomain newDomain = s.getClass().getPDImpl();
					if (oldDomain != newDomain) {
						// initialize external messages
						com.ibm.oti.util.Msg.getString("K002c"); //$NON-NLS-1$
						// initialize the protection domain, which may include preloading the
						// dynamic permissions from the policy before installing
						newDomain.implies(new AllPermission());
					}
					return null;
				}});
		} catch (Exception e) {}
	}
	if (currentSecurity != null) {
		currentSecurity.checkPermission(
			RuntimePermission.permissionToSetSecurityManager);
	}
	security = s;
}

/**
 * Answers the platform specific file name format for the shared
 * library named by the argument.
 *
 * @param		userLibName 	the name of the library to look up.
 * @return		the platform specific filename for the library
 */
public static native String mapLibraryName(String userLibName);

/**
 * Sets the value of the named static field in the receiver
 * to the passed in argument.
 *
 * @param		fieldName 	the name of the field to set, one of in, out, or err
 * @param		stream 		the new value of the field
 */
private static native void setFieldImpl(String fieldName, Object stream);

static Class getCallerClass() {
	return Class.getStackClass(2);
}

/**
 * Return the channel inherited by the invocation of the running vm. The channel is
 * obtained by calling SelectorProvider.inheritedChannel().
 *
 * @return the inherited channel or null
 *
 * @throws IOException if an IO error occurs getting the inherited channel
 */
public static java.nio.channels.Channel inheritedChannel() throws IOException {
	return java.nio.channels.spi.SelectorProvider.provider().inheritedChannel();
}

/**
 * Returns the number of high resolution timer ticks per second.
 */
private static native long hiresFrequencyImpl();

/**
 * Returns the current tick count of the high resolution timer.
 */
private static native long hiresClockImpl();

/**
 * Returns the current tick count in nanoseconds. The tick count
 * only reflects elapsed time.
 *
 * @return		the current nanosecond tick count, which may be negative
 */
public static long nanoTime() {
	final long hiresClock = hiresClockImpl();
	switch (nanoType) {
		case NANO_DIVIDE:
			return (long)(hiresClock / nanoAdjust);
		case NANO_INT_DIVIDE:
			return hiresClock / nanoIntAdjust;
		case NANO_MULTIPLY:
			return (long)(hiresClock * nanoAdjust);
		case NANO_INT_MULTIPLY:
			return hiresClock * nanoIntAdjust;
		default:
			return hiresClock;
	}
}

/**
 * Removes the property.
 *
 * @param 		prop		the name of the property to remove
 * @return		the value of the property removed
 */
public static String clearProperty(String prop) {
	if (prop.length() == 0) throw new IllegalArgumentException();
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(new PropertyPermission(prop, "write")); //$NON-NLS-1$
	return (String)systemProperties.remove(prop);
}

/**
 * Returns all of the system environment variables in an unmodifiable Map.
 *
 * @return	an unmodifiable Map containing all of the system environment variables.
 */
public static Map<String, String> getenv() {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(new RuntimePermission("getenv.*")); //$NON-NLS-1$
	return ProcessEnvironment.getenv();
}

/**
 * Return the Console or null.
 *
 * @return the Console or null
 */
public static Console console() {
	if (consoleInitialized) return console;
	synchronized(System.class) {
		if (consoleInitialized) return console;
		console = sun.misc.SharedSecrets.getJavaIOAccess().console();
		consoleInitialized = true;
	}
	return console;
}
}
