/*
 *  2008-08-17: this file was modified by International Business Machines Corporation.
 *  Modifications Copyright 2008 IBM Corporation.
 */

/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package java.beans;

import java.util.HashMap;
import java.util.Map;

public class PropertyEditorManager {

    private static String[] path = { "org.apache.harmony.beans.editors" }; //$NON-NLS-1$

    private static final Map<Class<?>, Class<?>> registeredEditors = new HashMap<Class<?>, Class<?>>();

    static{
        //init default editors for primitive types
        registeredEditors.put(Long.TYPE, getDefaultEditorClass("LongEditor"));
        registeredEditors.put(Boolean.TYPE, getDefaultEditorClass("BooleanEditor"));
        registeredEditors.put(Short.TYPE, getDefaultEditorClass("ShortEditor"));
        registeredEditors.put(Float.TYPE, getDefaultEditorClass("FloatEditor"));
        registeredEditors.put(Double.TYPE, getDefaultEditorClass("DoubleEditor"));
        registeredEditors.put(Byte.TYPE, getDefaultEditorClass("ByteEditor"));
        registeredEditors.put(Integer.TYPE, getDefaultEditorClass("IntEditor"));

    }

    static Class getDefaultEditorClass(String className){
        for(String apath: path){
            String fullName = apath + "." + className;
            try{
                Class c = Class.forName(fullName);
                c.asSubclass(PropertyEditor.class);
                return c;
            }catch(Exception e){
                //ignore
            }
        }
        return null;
    }

    public PropertyEditorManager() {
        // default constructor, do nothing
    }

    public static void registerEditor(Class<?> targetType, Class<?> editorClass) {
        if (targetType == null) {
            throw new NullPointerException();
        }

        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            sm.checkPropertiesAccess();
        }
        if (editorClass != null) {
            registeredEditors.put(targetType, editorClass);
        } else {
            registeredEditors.remove(targetType);
        }
    }

    public static synchronized PropertyEditor findEditor(Class<?> targetType) {
        if (targetType == null) {
            throw new NullPointerException();
        }

        Class<?> editorClass = null;
        PropertyEditor editor = null;

        editorClass = registeredEditors.get(targetType);
        if (editorClass != null) {
            try {
                editor = (PropertyEditor) editorClass.newInstance();
            } catch (Throwable e) {
                // ignore
            }
        }
        
        if (editor == null) {
            //TODO: need more test here to ensure the behavior here matches RI, 
            //especially on the sequence of the editor loading and class loader usage
            String editorClassName = targetType.getName() + "Editor"; //$NON-NLS-1$
            editor = loadEditor(targetType, editorClassName);
            if(editor == null){
                String shortEditorClassName = editorClassName
                        .substring(editorClassName.lastIndexOf(".") + 1); //$NON-NLS-1$

                if (targetType.isPrimitive()) {
                    shortEditorClassName = shortEditorClassName.substring(0, 1)
                            .toUpperCase()
                            + shortEditorClassName.substring(1);
                }
                
                for (String element : path) {
                    editorClassName = element + "." + shortEditorClassName; //$NON-NLS-1$
                    editor = loadEditor(targetType, editorClassName);
                    if(editor != null){
                        break;
                    }
                }
            }
        }
        return editor;
    }

    private static PropertyEditor loadEditor(Class<?> targetType, String editorClassName){
        Class editorClass = null;
        ClassLoader loader = targetType.getClassLoader();
        if (loader != null) {
            try{
                editorClass = Class.forName(editorClassName, true, loader);
                editorClass.asSubclass(PropertyEditor.class);
                return (PropertyEditor) editorClass.newInstance();
            }catch(Exception e){//ignore
            }
        }

        loader=ClassLoader.getSystemClassLoader();
        try{
            editorClass = Class.forName(editorClassName, true, loader);
            editorClass.asSubclass(PropertyEditor.class);
            return (PropertyEditor) editorClass.newInstance();
        }catch(Exception e){//ignore
        }

        loader = Thread.currentThread().getContextClassLoader();
        try{
            editorClass = Class.forName(editorClassName, true, loader);
            editorClass.asSubclass(PropertyEditor.class);
            return (PropertyEditor) editorClass.newInstance();
        }catch(Exception e){//ignore
        }

        return null;
    }

    public static void setEditorSearchPath(String[] apath) {
        SecurityManager sm = System.getSecurityManager();
        if (sm != null) {
            sm.checkPropertiesAccess();
        }
        if(apath == null){
            apath = new String[0];
        }
        synchronized(PropertyEditorManager.class){
            path = apath;
        }
    }

    public static synchronized String[] getEditorSearchPath() {
        return path.clone();
    }
}
