/*
 * @(#)URLClassLoader.java	1.91 07/04/06
 *
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package java.net;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.io.File;
import java.io.FilePermission;
import java.io.InputStream;
import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandlerFactory;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.jar.Manifest;
import java.util.jar.Attributes;
import java.util.jar.Attributes.Name;
import java.security.CodeSigner;
import java.util.Vector;                                                        //IBM-shared_classes
import java.util.regex.Pattern;                                                 //IBM-shared_classes
import java.util.regex.Matcher;                                                 //IBM-shared_classes
import java.util.regex.PatternSyntaxException;                                  //IBM-shared_classes
import java.security.PrivilegedAction;
import java.security.PrivilegedExceptionAction;
import java.security.AccessController;
import java.security.AccessControlContext;
import java.security.SecureClassLoader;
import java.security.CodeSource;
import java.security.Permission;
import java.security.PermissionCollection;
import sun.misc.Resource;
import sun.misc.URLClassPath;
import sun.net.www.ParseUtil;
import sun.security.util.SecurityConstants;
import com.ibm.jvm.ClassLoaderDiagnosticsHelper;                                //IBM-shared_classes
import com.ibm.oti.shared.Shared;                                               //IBM-shared_classes
import com.ibm.oti.shared.SharedClassHelperFactory;                             //IBM-shared_classes
import com.ibm.oti.shared.SharedClassURLClasspathHelper;                        //IBM-shared_classes
import com.ibm.oti.shared.HelperAlreadyDefinedException;                        //IBM-shared_classes
/**
 * This class loader is used to load classes and resources from a search
 * path of URLs referring to both JAR files and directories. Any URL that
 * ends with a '/' is assumed to refer to a directory. Otherwise, the URL
 * is assumed to refer to a JAR file which will be opened as needed.
 * <p>
 * The AccessControlContext of the thread that created the instance of
 * URLClassLoader will be used when subsequently loading classes and
 * resources.
 * <p>
 * The classes that are loaded are by default granted permission only to
 * access the URLs specified when the URLClassLoader was created.
 *
 * @author  David Connelly
 * @version 1.91, 04/06/07
 * @since   1.2
 */
public class URLClassLoader extends SecureClassLoader {
    /* The search path for classes and resources */
    URLClassPath ucp;
    /* Syncronizer object for the packege hash */                               //IBM-shared_classes
    private static final Object packageSyncronizer = new Object();              //IBM-shared_classes
    /* The context to be used when loading classes and resources */
    private AccessControlContext acc;
    /* Private member fields used for Shared classes*/                          //IBM-shared_classes
    private SharedClassURLClasspathHelper sharedClassURLClasspathHelper;        //IBM-shared_classes
    private SharedClassMetaDataCache sharedClassMetaDataCache;                  //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Wrapper class for maintaining the index of where the metadata (codesource and manifest) //IBM-shared_classes
     * is found - used only in Shared classes context.                          //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private class SharedClassIndexHolder implements SharedClassURLClasspathHelper.IndexHolder { //IBM-shared_classes
        int index;                                                              //IBM-shared_classes
                                                                                //IBM-shared_classes
        public void setIndex(int index) {                                       //IBM-shared_classes
            this.index = index;                                                 //IBM-shared_classes
        }                                                                       //IBM-shared_classes
    }                                                                           //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Wrapper class for internal storage of metadata (codesource and manifest) associated with  //IBM-shared_classes
     * shared class - used only in Shared classes context.                      //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private class SharedClassMetaData {                                         //IBM-shared_classes
        private CodeSource codeSource;                                          //IBM-shared_classes
        private Manifest manifest;                                              //IBM-shared_classes
                                                                                //IBM-shared_classes
        SharedClassMetaData(CodeSource codeSource, Manifest manifest) {         //IBM-shared_classes
            this.codeSource = codeSource;                                       //IBM-shared_classes
            this.manifest = manifest;                                           //IBM-shared_classes
        }                                                                       //IBM-shared_classes
        public CodeSource getCodeSource() { return codeSource; }                //IBM-shared_classes
        public Manifest getManifest() { return manifest; }                      //IBM-shared_classes
    }                                                                           //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Represents a collection of SharedClassMetaData objects retrievable by    //IBM-shared_classes
     * index.                                                                   //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private class SharedClassMetaDataCache {                                    //IBM-shared_classes
        private final static int BLOCKSIZE = 10;                                //IBM-shared_classes
        private SharedClassMetaData[] store;                                    //IBM-shared_classes
                                                                                //IBM-shared_classes
        public SharedClassMetaDataCache(int initialSize) {                      //IBM-shared_classes
            /* Allocate space for an initial amount of metadata entries */      //IBM-shared_classes
            store = new SharedClassMetaData[initialSize];                       //IBM-shared_classes
        }                                                                       //IBM-shared_classes
                                                                                //IBM-shared_classes
        /**                                                                     //IBM-shared_classes
         * Retrieves the SharedClassMetaData stored at the given index, or null //IBM-shared_classes
         * if no SharedClassMetaData was previously stored at the given index   //IBM-shared_classes
         * or the index is out of range.                                        //IBM-shared_classes
         */                                                                     //IBM-shared_classes
        public synchronized SharedClassMetaData getSharedClassMetaData(int index) { //IBM-shared_classes
            if (index < 0 || store.length < (index+1)) {                        //IBM-shared_classes
                return null;                                                    //IBM-shared_classes
            }                                                                   //IBM-shared_classes
            return store[index];                                                //IBM-shared_classes
        }                                                                       //IBM-shared_classes
                                                                                //IBM-shared_classes
        /**                                                                     //IBM-shared_classes
         * Stores the supplied SharedClassMetaData at the given index in the    //IBM-shared_classes
         * store. The store will be grown to contain the index if necessary.    //IBM-shared_classes
         */                                                                     //IBM-shared_classes
        public synchronized void setSharedClassMetaData(int index,              //IBM-shared_classes
                                                     SharedClassMetaData data) { //IBM-shared_classes
            ensureSize(index);                                                  //IBM-shared_classes
            store[index] = data;                                                //IBM-shared_classes
        }                                                                       //IBM-shared_classes
                                                                                //IBM-shared_classes
        /* Ensure that the store can hold at least index number of entries */   //IBM-shared_classes
        private synchronized void ensureSize(int index) {                       //IBM-shared_classes
            if (store.length < (index+1)) {                                     //IBM-shared_classes
                int newSize = (index+BLOCKSIZE);                                //IBM-shared_classes
                SharedClassMetaData[] newSCMDS = new SharedClassMetaData[newSize]; //IBM-shared_classes
                System.arraycopy(store, 0, newSCMDS, 0, store.length);          //IBM-shared_classes
                store = newSCMDS;                                               //IBM-shared_classes
            }                                                                   //IBM-shared_classes
        }                                                                       //IBM-shared_classes
    }                                                                           //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Return true if shared classes support is active, otherwise false.        //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private boolean usingSharedClasses() {                                      //IBM-shared_classes
        return (sharedClassURLClasspathHelper != null);                         //IBM-shared_classes
    }                                                                           //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Initialize support for shared classes.                                   //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private void initializeSharedClassesSupport(URL[] initialClassPath) {       //IBM-shared_classes
        /* get the Shared class helper and initialize the metadata store if we are sharing */ //IBM-shared_classes
        SharedClassHelperFactory sharedClassHelperFactory = Shared.getSharedClassHelperFactory(); //IBM-shared_classes
                                                                                //IBM-shared_classes
        if (sharedClassHelperFactory != null) {                                 //IBM-shared_classes
            try {                                                               //IBM-shared_classes
                this.sharedClassURLClasspathHelper = sharedClassHelperFactory.getURLClasspathHelper(this, initialClassPath); //IBM-shared_classes
            } catch (HelperAlreadyDefinedException ex) { // thrown if we get 2 types of helper for the same classloader //IBM-shared_classes
                ex.printStackTrace();                                           //IBM-shared_classes
            }                                                                   //IBM-shared_classes
            /* Only need to create a meta data cache if using shared classes */ //IBM-shared_classes
            if (usingSharedClasses()) {                                         //IBM-shared_classes
                /* Create a metadata cache */                                   //IBM-shared_classes
                this.sharedClassMetaDataCache = new SharedClassMetaDataCache(initialClassPath.length); //IBM-shared_classes
            }                                                                   //IBM-shared_classes
        }                                                                       //IBM-shared_classes
    }                                                                           //IBM-shared_classes

    /**
     * Constructs a new URLClassLoader for the given URLs. The URLs will be
     * searched in the order specified for classes and resources after first
     * searching in the specified parent class loader. Any URL that ends with
     * a '/' is assumed to refer to a directory. Otherwise, the URL is assumed
     * to refer to a JAR file which will be downloaded and opened as needed.
     *
     * <p>If there is a security manager, this method first
     * calls the security manager's <code>checkCreateClassLoader</code> method
     * to ensure creation of a class loader is allowed.
     * 
     * @param urls the URLs from which to load classes and resources
     * @param parent the parent class loader for delegation
     * @exception  SecurityException  if a security manager exists and its  
     *             <code>checkCreateClassLoader</code> method doesn't allow 
     *             creation of a class loader.
     * @see SecurityManager#checkCreateClassLoader
     */
    public URLClassLoader(URL[] urls, ClassLoader parent) {
	super(parent);
	// this is to make the stack depth consistent with 1.1
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
	    security.checkCreateClassLoader();
	}
	initializeSharedClassesSupport(urls);                                   //IBM-shared_classes
	ucp = new URLClassPath(urls, null, sharedClassURLClasspathHelper);      //IBM-shared_classes
	acc = AccessController.getContext();
    }

    /**
     * Constructs a new URLClassLoader for the specified URLs using the
     * default delegation parent <code>ClassLoader</code>. The URLs will
     * be searched in the order specified for classes and resources after
     * first searching in the parent class loader. Any URL that ends with
     * a '/' is assumed to refer to a directory. Otherwise, the URL is
     * assumed to refer to a JAR file which will be downloaded and opened
     * as needed.
     *
     * <p>If there is a security manager, this method first
     * calls the security manager's <code>checkCreateClassLoader</code> method
     * to ensure creation of a class loader is allowed.
     * 
     * @param urls the URLs from which to load classes and resources
     *
     * @exception  SecurityException  if a security manager exists and its  
     *             <code>checkCreateClassLoader</code> method doesn't allow 
     *             creation of a class loader.
     * @see SecurityManager#checkCreateClassLoader
     */
    public URLClassLoader(URL[] urls) {
	super();
	// this is to make the stack depth consistent with 1.1
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
	    security.checkCreateClassLoader();
	}
	initializeSharedClassesSupport(urls);                                   //IBM-shared_classes
	ucp = new URLClassPath(urls, null, sharedClassURLClasspathHelper);      //IBM-shared_classes
	acc = AccessController.getContext();
    }

    /**
     * Constructs a new URLClassLoader for the specified URLs, parent
     * class loader, and URLStreamHandlerFactory. The parent argument
     * will be used as the parent class loader for delegation. The
     * factory argument will be used as the stream handler factory to
     * obtain protocol handlers when creating new jar URLs.
     *
     * <p>If there is a security manager, this method first
     * calls the security manager's <code>checkCreateClassLoader</code> method
     * to ensure creation of a class loader is allowed.
     *
     * @param urls the URLs from which to load classes and resources
     * @param parent the parent class loader for delegation
     * @param factory the URLStreamHandlerFactory to use when creating URLs
     *
     * @exception  SecurityException  if a security manager exists and its  
     *             <code>checkCreateClassLoader</code> method doesn't allow 
     *             creation of a class loader.
     * @see SecurityManager#checkCreateClassLoader
     */
    public URLClassLoader(URL[] urls, ClassLoader parent,
			  URLStreamHandlerFactory factory) {
	super(parent);
	// this is to make the stack depth consistent with 1.1
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
	    security.checkCreateClassLoader();
	}
	initializeSharedClassesSupport(urls);                                   //IBM-shared_classes
	ucp = new URLClassPath(urls, factory, sharedClassURLClasspathHelper);   //IBM-shared_classes
	acc = AccessController.getContext();
    }

    /**
     * Appends the specified URL to the list of URLs to search for
     * classes and resources.
     *
     * @param url the URL to be added to the search path of URLs
     */
    protected void addURL(URL url) {
	ucp.addURL(url);
    }

    /**
     * Returns the search path of URLs for loading classes and resources.
     * This includes the original list of URLs specified to the constructor,
     * along with any URLs subsequently appended by the addURL() method.
     * @return the search path of URLs for loading classes and resources.
     */
    public URL[] getURLs() {
	return ucp.getURLs();
    }

    /* Stores a list of classes for which is show detailed class loading */     //IBM-shared_classes
    private static Vector showClassLoadingFor = null;                           //IBM-shared_classes
    /* Caches whether the vector showClassLoadingFor is empty */                //IBM-shared_classes
    private static boolean showLoadingMessages = false;                         //IBM-shared_classes
    /* Property to use to setup the detailed classloading output */             //IBM-shared_classes
    private final static String showClassLoadingProperty = "ibm.cl.verbose";    //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Initializes the showClassLoadingFor vector. All expressions are precompiled //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    static                                                                      //IBM-shared_classes
    {                                                                           //IBM-shared_classes
        Vector showClassLoadingFor = new Vector();                              //IBM-shared_classes
        String classes = System.getProperty(showClassLoadingProperty);          //IBM-shared_classes
        /* If the property exists then process the supplied file expressions */ //IBM-shared_classes
        if (classes != null) {                                                  //IBM-shared_classes
            StringTokenizer classMatchers = new StringTokenizer(classes, ",");  //IBM-shared_classes
            while (classMatchers.hasMoreTokens()) {                             //IBM-shared_classes
                String classMatcher = classMatchers.nextToken();                //IBM-shared_classes
                /* Do the replacements to allow more readable expressions to be supplied */ //IBM-shared_classes
                String classMatcherExp = classMatcher.replaceAll("\\.", "\\."); //IBM-shared_classes
                classMatcherExp = classMatcherExp.replaceAll("\\*", ".*");      //IBM-shared_classes
                Pattern pattern;                                                //IBM-shared_classes
                /* Add the compiled pattern to the vector */                    //IBM-shared_classes
                try {                                                           //IBM-shared_classes
                    pattern = Pattern.compile(classMatcherExp);                 //IBM-shared_classes
                    showClassLoadingFor.addElement(pattern);                    //IBM-shared_classes
                } catch (PatternSyntaxException e) {                            //IBM-shared_classes
                    /*                                                          //IBM-shared_classes
                     * The user has supplied something which has is not         //IBM-shared_classes
                     * a legal regular expression (or isn't now that it has been //IBM-shared_classes
                     * transformed!) Warn the user and ignore this expression   //IBM-shared_classes
                     */                                                         //IBM-shared_classes
                    System.err.println("Illegal class matching expression \"" + classMatcher + //IBM-shared_classes
                        "\" supplied by property " + showClassLoadingProperty); //IBM-shared_classes
                }                                                               //IBM-shared_classes
            }                                                                   //IBM-shared_classes
        }                                                                       //IBM-shared_classes
        /*                                                                      //IBM-shared_classes
         * Cache whether a check should be made to see whether to show loading messages for //IBM-shared_classes
         * a particular class                                                   //IBM-shared_classes
         */                                                                     //IBM-shared_classes
        if (!showClassLoadingFor.isEmpty()) {                                   //IBM-shared_classes
            showLoadingMessages = true;                                         //IBM-shared_classes
        }                                                                       //IBM-shared_classes
        URLClassLoader.showClassLoadingFor = showClassLoadingFor;               //IBM-shared_classes
    }                                                                           //IBM-shared_classes
                                                                                //IBM-shared_classes
                                                                                //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Returns whether the class loading should be explicitly shown for a       //IBM-shared_classes
     * particular class. This is determined by the system property ibm.cl.verbose //IBM-shared_classes
     * which contains a comma separated list of file expressions.               //IBM-shared_classes
     * A file expression being a regular expression but with .* substituted for * //IBM-shared_classes
     * and \. substituted for . to allow a more readable form to be used        //IBM-shared_classes
     * If no property exists or contains no expressions then showClassLoadingFor //IBM-shared_classes
     * will be an empty vector and this will be the only test each time this function //IBM-shared_classes
     * is called. Otherwise name will be matched against each expression in the vector //IBM-shared_classes
     * and if a match is found true is returned, otherwise false                //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private boolean showClassLoading(String name)                               //IBM-shared_classes
    {                                                                           //IBM-shared_classes
        /* If there are supplied expressions try and match this class name against them */ //IBM-shared_classes
        if (URLClassLoader.showLoadingMessages) {                               //IBM-shared_classes
            Enumeration patterns = URLClassLoader.showClassLoadingFor.elements(); //IBM-shared_classes
            while (patterns.hasMoreElements()) {                                //IBM-shared_classes
                Pattern pattern = (Pattern)patterns.nextElement();              //IBM-shared_classes
                Matcher matcher = pattern.matcher(name);                        //IBM-shared_classes
                if (matcher.matches()) {                                        //IBM-shared_classes
                    return true;                                                //IBM-shared_classes
                }                                                               //IBM-shared_classes
            }                                                                   //IBM-shared_classes
        }                                                                       //IBM-shared_classes
        /* Either no expressions or no matches */                               //IBM-shared_classes
        return false;                                                           //IBM-shared_classes
    }                                                                           //IBM-shared_classes
    /**
     * Finds and loads the class with the specified name from the URL search
     * path. Any URLs referring to JAR files are loaded and opened as needed
     * until the class is found.
     *
     * @param name the name of the class
     * @return the resulting class
     * @exception ClassNotFoundException if the class could not be found
     */
    protected Class<?> findClass(final String name)
	 throws ClassNotFoundException
    {
	try {
            boolean scl = showClassLoading(name);                               //IBM-shared_classes
            if (scl) {                                                           //IBM-shared_classes
                ClassLoaderDiagnosticsHelper.attemptingToLoadClass(this, name); //IBM-shared_classes
            }                                                                    //IBM-shared_classes
            /* Try to find the class from the shared cache using the class name.  If we found the class  //IBM-shared_classes
             * and if we have its corresponding metadata (codesource and manifest entry) already cached,  //IBM-shared_classes
             * then we define the class passing in these parameters.  If however, we do not have the  //IBM-shared_classes
             * metadata cached, then we define the class as normal.  Also, if we do not find the class //IBM-shared_classes
             * from the shared class cache, we define the class as normal.      //IBM-shared_classes
             */                                                                 //IBM-shared_classes
            if (usingSharedClasses()) {                                         //IBM-shared_classes
                SharedClassIndexHolder sharedClassIndexHolder = new SharedClassIndexHolder(); //IBM-shared_classes
                byte[] sharedClazz = sharedClassURLClasspathHelper.findSharedClass(name, sharedClassIndexHolder); //IBM-shared_classes
                                                                                //IBM-shared_classes
                if (sharedClazz != null) {                                      //IBM-shared_classes
                    int indexFoundData = sharedClassIndexHolder.index;          //IBM-shared_classes
                    SharedClassMetaData metadata = sharedClassMetaDataCache.getSharedClassMetaData(indexFoundData);  //IBM-shared_classes
                    if (metadata != null) {                                     //IBM-shared_classes
                        try {                                                   //IBM-shared_classes
                            Class clazz = defineClass(name,sharedClazz,         //IBM-shared_classes
                                               metadata.getCodeSource(),        //IBM-shared_classes
                                               metadata.getManifest());         //IBM-shared_classes
                            if (scl) {                                          //IBM-shared_classes
                                ClassLoaderDiagnosticsHelper.loadedClass(this, name); //IBM-shared_classes
                            }                                                   //IBM-shared_classes
                            return clazz;                                       //IBM-shared_classes
                        } catch (IOException e) {                               //IBM-shared_classes
                            e.printStackTrace();                                //IBM-shared_classes
                        }                                                       //IBM-shared_classes
		    }
		}                                                               //IBM-shared_classes
   	    }	                                                        //IBM-shared_classes
            ClassFinder loader = new ClassFinder(name, this);    /*ibm@80916.1*/ //IBM-shared_classes
            Class clazz = (Class)AccessController.doPrivileged(loader, acc);    //IBM-shared_classes
            if (clazz == null) {                                     /*ibm@802*/ //IBM-shared_classes
                if (scl) {                                                       //IBM-shared_classes
                    ClassLoaderDiagnosticsHelper.failedToLoadClass(this, name);  //IBM-shared_classes
                }                                                                //IBM-shared_classes
               	throw new ClassNotFoundException(name);              /*ibm@802*/ //IBM-shared_classes
            }                                                                   //IBM-shared_classes
            if (scl) {                                                           //IBM-shared_classes
                ClassLoaderDiagnosticsHelper.loadedClass(this, name);            //IBM-shared_classes
            }                                                                   //IBM-shared_classes
            return clazz;                                            /*ibm@802*/ //IBM-shared_classes
	} catch (java.security.PrivilegedActionException pae) {
	    throw (ClassNotFoundException) pae.getException();
	}
    }

    /*
     * Defines a Class using the class bytes obtained from the specified
     * Resource. The resulting Class must be resolved before it can be
     * used.
     */
    private Class defineClass(String name, Resource res) throws IOException {
        Class clazz = null;                                                     //IBM-shared_classes
        CodeSource cs = null;                                                   //IBM-shared_classes
        Manifest man = null;                                                    //IBM-shared_classes
	int i = name.lastIndexOf('.');
	URL url = res.getCodeSourceURL();
	if (i != -1) {
	    String pkgname = name.substring(0, i);
	    // Check if package already loaded.
	    Package pkg = getPackage(pkgname);
	    man = res.getManifest();                                            //IBM-shared_classes
	    if(pkg==null){                                                      //IBM-shared_classes
		synchronized(packageSyncronizer) {                              //IBM-shared_classes
		    //again check if package already loaded inside sync block   //IBM-shared_classes
		    pkg = getPackage(pkgname);                                  //IBM-shared_classes
		    if(pkg == null) {                                           //IBM-shared_classes
			if (man != null) {                                      //IBM-shared_classes
			    definePackage(pkgname, man, url);                   //IBM-shared_classes
			} else {                                                //IBM-shared_classes
			    definePackage(pkgname, null, null, null, null, null,  //IBM-shared_classes
				    		null, null);                //IBM-shared_classes
			}                                                       //IBM-shared_classes
		    }                                                           //IBM-shared_classes
		} // end synchronized                                           //IBM-shared_classes
	    }                                                                   //IBM-shared_classes
	    // We are not using else here as we want following check to happen  //IBM-shared_classes
	    // before we proceed                                                //IBM-shared_classes
	    if (pkg != null) {
		// Package found, so check package sealing.
		if (pkg.isSealed()) {
		    // Verify that code source URL is the same.
		    if (!pkg.isSealed(url)) {
			throw new SecurityException(
			    "sealing violation: package " + pkgname + " is sealed");
		    }
		} else {
		    // Make sure we are not attempting to seal the package
		    // at this code source URL.
		    if ((man != null) && isSealed(pkgname, man)) {
			throw new SecurityException(
			    "sealing violation: can't seal package " + pkgname + 
			    ": already loaded");
		    }
		}
	    }
	}
	// Now read the class bytes and define the class
	java.nio.ByteBuffer bb = res.getByteBuffer();
	if (bb != null) {
	    // Use (direct) ByteBuffer:
	    CodeSigner[] signers = res.getCodeSigners();
	    cs = new CodeSource(url, signers);                                  //IBM-shared_classes
            clazz = defineClass(name, bb, cs);                                  //IBM-shared_classes
	} else {
	    byte[] b = res.getBytes();
	    // must read certificates AFTER reading bytes.
	    CodeSigner[] signers = res.getCodeSigners();
	    cs = new CodeSource(url, signers);                                  //IBM-shared_classes
	    clazz = defineClass(name, b, 0, b.length, cs);                      //IBM-shared_classes
	}                                                                       //IBM-shared_classes
        /*                                                                      //IBM-shared_classes
         * Since we have already stored the class path index (of where this resource came from), we can retrieve  //IBM-shared_classes
         * it here.  The storing is done in getResource() in URLClassPath.java.  The index is the specified  //IBM-shared_classes
         * position in the URL search path (see getLoader()).  The storeSharedClass() call below, stores the  //IBM-shared_classes
         * class in the shared class cache for future use.                      //IBM-shared_classes
         */                                                                     //IBM-shared_classes
        if (usingSharedClasses()) {                                             //IBM-shared_classes
                                                                                //IBM-shared_classes
            /* Determine the index into the search path for this class */       //IBM-shared_classes
            int index = res.getClasspathLoadIndex();                            //IBM-shared_classes
            /* Check to see if we have already cached metadata for this index */ //IBM-shared_classes
            SharedClassMetaData metadata = sharedClassMetaDataCache.getSharedClassMetaData(index);  //IBM-shared_classes
            /* If we have not already cached the metadata for this index... */  //IBM-shared_classes
            if (metadata == null) {                                             //IBM-shared_classes
                /* ... create a new metadata entry */                           //IBM-shared_classes
                metadata = new SharedClassMetaData(cs, man);                    //IBM-shared_classes
                /* Cache the metadata for this index for future use */          //IBM-shared_classes
                sharedClassMetaDataCache.setSharedClassMetaData(index, metadata); //IBM-shared_classes
                                                                                //IBM-shared_classes
            }                                                                   //IBM-shared_classes
            boolean storeSuccessful = false;                                    //IBM-shared_classes
            try {                                                               //IBM-shared_classes
                /* Store class in shared class cache for future use */          //IBM-shared_classes
                storeSuccessful =                                               //IBM-shared_classes
                  sharedClassURLClasspathHelper.storeSharedClass(clazz, index); //IBM-shared_classes
            } catch (Exception e) {                                             //IBM-shared_classes
                e.printStackTrace();                                            //IBM-shared_classes
            }                                                                   //IBM-shared_classes
        }                                                                       //IBM-shared_classes
                                                                                //IBM-shared_classes
        return clazz;                                                           //IBM-shared_classes
    }                                                                           //IBM-shared_classes
    /*                                                                          //IBM-shared_classes
     * Defines a class using the class bytes, codesource and manifest           //IBM-shared_classes
     * obtained from the specified shared class cache. The resulting            //IBM-shared_classes
     * class must be resolved before it can be used.  This method is            //IBM-shared_classes
     * used only in a Shared classes context.                                   //IBM-shared_classes
     */                                                                         //IBM-shared_classes
    private Class defineClass(String name, byte[] sharedClazz, CodeSource codesource, Manifest man) throws IOException { //IBM-shared_classes
	int i = name.lastIndexOf('.');                                          //IBM-shared_classes
	URL url = codesource.getLocation();                                     //IBM-shared_classes
	if (i != -1) {                                                          //IBM-shared_classes
	    String pkgname = name.substring(0, i);                              //IBM-shared_classes
	    // Check if package already loaded.                                 //IBM-shared_classes
	    Package pkg = getPackage(pkgname);                                  //IBM-shared_classes
	    if(pkg==null){                                                      //IBM-shared_classes
		synchronized(packageSyncronizer) {                              //IBM-shared_classes
		    //again check if package already loaded inside sync block   //IBM-shared_classes
		    pkg = getPackage(pkgname);                                  //IBM-shared_classes
		    if(pkg == null) {                                           //IBM-shared_classes
			if (man != null) {                                      //IBM-shared_classes
			    definePackage(pkgname, man, url);                   //IBM-shared_classes
			} else {                                                //IBM-shared_classes
			    definePackage(pkgname, null, null, null, null, null,  //IBM-shared_classes
				    		null, null);                //IBM-shared_classes
			}                                                       //IBM-shared_classes
		    }                                                           //IBM-shared_classes
		} // end synchronized                                           //IBM-shared_classes
	    }                                                                   //IBM-shared_classes
	    // We are not using else here as we want following check to happen  //IBM-shared_classes
	    // before we proceed                                                //IBM-shared_classes
            if (pkg != null) {                                                  //IBM-shared_classes
		// Package found, so check package sealing.                     //IBM-shared_classes
		if (pkg.isSealed()) {                                           //IBM-shared_classes
		    // Verify that code source URL is the same.                 //IBM-shared_classes
		    if (!pkg.isSealed(url)) {                                   //IBM-shared_classes
			throw new SecurityException(                            //IBM-shared_classes
			    "sealing violation: package " + pkgname + " is sealed"); //IBM-shared_classes
		    }                                                           //IBM-shared_classes
		} else {                                                        //IBM-shared_classes
		    // Make sure we are not attempting to seal the package      //IBM-shared_classes
		    // at this code source URL.                                 //IBM-shared_classes
		    if ((man != null) && isSealed(pkgname, man)) {              //IBM-shared_classes
			throw new SecurityException(                            //IBM-shared_classes
			    "sealing violation: can't seal package " + pkgname +  //IBM-shared_classes
			    ": already loaded");                                //IBM-shared_classes
		    }                                                           //IBM-shared_classes
		}                                                               //IBM-shared_classes
	    }                                                                   //IBM-shared_classes
	}
	/*                                                                      //IBM-shared_classes
         * Now read the class bytes and define the class.  We don't need to call  //IBM-shared_classes
         * storeSharedClass(), since its already in our shared class cache.     //IBM-shared_classes
         */                                                                     //IBM-shared_classes
        return defineClass(name, sharedClazz, 0, sharedClazz.length, codesource); //IBM-shared_classes
    }

    /**
     * Defines a new package by name in this ClassLoader. The attributes
     * contained in the specified Manifest will be used to obtain package
     * version and sealing information. For sealed packages, the additional
     * URL specifies the code source URL from which the package was loaded.
     *
     * @param name  the package name
     * @param man   the Manifest containing package version and sealing
     *              information
     * @param url   the code source url for the package, or null if none
     * @exception   IllegalArgumentException if the package name duplicates
     *              an existing package either in this class loader or one
     *              of its ancestors
     * @return the newly defined Package object
     */
    protected Package definePackage(String name, Manifest man, URL url)
	throws IllegalArgumentException
    {
	String path = name.replace('.', '/').concat("/");
	String specTitle = null, specVersion = null, specVendor = null;
	String implTitle = null, implVersion = null, implVendor = null;
	String sealed = null;
	URL sealBase = null;

	Attributes attr = man.getAttributes(path);
	if (attr != null) {
	    specTitle   = attr.getValue(Name.SPECIFICATION_TITLE);
	    specVersion = attr.getValue(Name.SPECIFICATION_VERSION);
	    specVendor  = attr.getValue(Name.SPECIFICATION_VENDOR);
	    implTitle   = attr.getValue(Name.IMPLEMENTATION_TITLE);
	    implVersion = attr.getValue(Name.IMPLEMENTATION_VERSION);
	    implVendor  = attr.getValue(Name.IMPLEMENTATION_VENDOR);
	    sealed      = attr.getValue(Name.SEALED);
	}
	attr = man.getMainAttributes();
	if (attr != null) {
	    if (specTitle == null) {
		specTitle = attr.getValue(Name.SPECIFICATION_TITLE);
	    }
	    if (specVersion == null) {
		specVersion = attr.getValue(Name.SPECIFICATION_VERSION);
	    }
	    if (specVendor == null) {
		specVendor = attr.getValue(Name.SPECIFICATION_VENDOR);
	    }
	    if (implTitle == null) {
		implTitle = attr.getValue(Name.IMPLEMENTATION_TITLE);
	    }
	    if (implVersion == null) {
		implVersion = attr.getValue(Name.IMPLEMENTATION_VERSION);
	    }
	    if (implVendor == null) {
		implVendor = attr.getValue(Name.IMPLEMENTATION_VENDOR);
	    }
	    if (sealed == null) {
		sealed = attr.getValue(Name.SEALED);
	    }
	}
	if ("true".equalsIgnoreCase(sealed)) {
	    sealBase = url;
	}
	return definePackage(name, specTitle, specVersion, specVendor,
			     implTitle, implVersion, implVendor, sealBase);
    }

    /*
     * Returns true if the specified package name is sealed according to the
     * given manifest.
     */
    private boolean isSealed(String name, Manifest man) {
	String path = name.replace('.', '/').concat("/");
	Attributes attr = man.getAttributes(path);
	String sealed = null;
	if (attr != null) {
	    sealed = attr.getValue(Name.SEALED);
	}
	if (sealed == null) {
	    if ((attr = man.getMainAttributes()) != null) {
		sealed = attr.getValue(Name.SEALED);
	    }
	}
	return "true".equalsIgnoreCase(sealed);
    }

    /**
     * Finds the resource with the specified name on the URL search path.
     *
     * @param name the name of the resource
     * @return a <code>URL</code> for the resource, or <code>null</code> 
     * if the resource could not be found.
     */
    public URL findResource(final String name) {
	/*
	 * The same restriction to finding classes applies to resources
	 */
	URL url = 
	    (URL) AccessController.doPrivileged(new PrivilegedAction() {
                public Object run() {
                    return ucp.findResource(name, true);
                }
            }, acc);

	return url != null ? ucp.checkURL(url) : null;
    }

    /**
     * Returns an Enumeration of URLs representing all of the resources
     * on the URL search path having the specified name.
     *
     * @param name the resource name
     * @exception IOException if an I/O exception occurs
     * @return an <code>Enumeration</code> of <code>URL</code>s
     */
    public Enumeration<URL> findResources(final String name)
	throws IOException
    {
        final Enumeration e = ucp.findResources(name, true);

	return new Enumeration<URL>() {
	    private URL url = null;

	    private boolean next() {
		if (url != null) {
		    return true;
		}
		do {
		    URL u = (URL)
			AccessController.doPrivileged(new PrivilegedAction() {
			    public Object run() {
				if (!e.hasMoreElements())
                               	    return null;
                            	return e.nextElement();
			    }
			}, acc);
		    if (u == null) 
			break;
		    url = ucp.checkURL(u);
		} while (url == null);
		return url != null;
	    }

	    public URL nextElement() {
		if (!next()) {
		    throw new NoSuchElementException();
		}
		URL u = url;
		url = null;
		return u;
	    }

	    public boolean hasMoreElements() {
		return next();
	    }
	};
    }

    /**
     * Returns the permissions for the given codesource object.
     * The implementation of this method first calls super.getPermissions
     * and then adds permissions based on the URL of the codesource.
     * <p>
     * If the protocol of this URL is "jar", then the permission granted 
     * is based on the permission that is required by the URL of the Jar 
     * file.
     * <p>
     * If the protocol is "file"
     * and the path specifies a file, then permission to read that
     * file is granted. If protocol is "file" and the path is
     * a directory, permission is granted to read all files
     * and (recursively) all files and subdirectories contained in
     * that directory.
     * <p>
     * If the protocol is not "file", then
     * to connect to and accept connections from the URL's host is granted.
     * @param codesource the codesource
     * @return the permissions granted to the codesource
     */
    protected PermissionCollection getPermissions(CodeSource codesource)
    {
	PermissionCollection perms = super.getPermissions(codesource);

	URL url = codesource.getLocation();

	Permission p;
	URLConnection urlConnection;

	try {
	    urlConnection = url.openConnection();
	    p = urlConnection.getPermission();
	} catch (java.io.IOException ioe) {
	    p = null;
	    urlConnection = null;
	}

	if (p instanceof FilePermission) {
	    // if the permission has a separator char on the end,
	    // it means the codebase is a directory, and we need
	    // to add an additional permission to read recursively
	    String path = p.getName();
	    if (path.endsWith(File.separator)) {
		path += "-";
		p = new FilePermission(path, SecurityConstants.FILE_READ_ACTION);
	    }
	} else if ((p == null) && (url.getProtocol().equals("file"))) {
	    String path = url.getFile().replace('/', File.separatorChar);
            path = ParseUtil.decode(path);
	    if (path.endsWith(File.separator))
		path += "-";
	    p =  new FilePermission(path, SecurityConstants.FILE_READ_ACTION);
	} else {
	    URL locUrl = url;
	    if (urlConnection instanceof JarURLConnection) {
		locUrl = ((JarURLConnection)urlConnection).getJarFileURL();
	    }
	    String host = locUrl.getHost();
	    if (host != null && (host.length() > 0))
		p = new SocketPermission(host,
					 SecurityConstants.SOCKET_CONNECT_ACCEPT_ACTION);
	} 
	// make sure the person that created this class loader
	// would have this permission

	if (p != null) {
	    final SecurityManager sm = System.getSecurityManager();
	    if (sm != null) {
		final Permission fp = p;
		AccessController.doPrivileged(new PrivilegedAction() {
		    public Object run() throws SecurityException {
			sm.checkPermission(fp);
			return null;
		    }
		}, acc);
	    }
	    perms.add(p);
	}
	return perms;
    }

    /**
     * Creates a new instance of URLClassLoader for the specified
     * URLs and parent class loader. If a security manager is
     * installed, the <code>loadClass</code> method of the URLClassLoader
     * returned by this method will invoke the
     * <code>SecurityManager.checkPackageAccess</code> method before
     * loading the class.
     *
     * @param urls the URLs to search for classes and resources
     * @param parent the parent class loader for delegation
     * @return the resulting class loader
     */
    public static URLClassLoader newInstance(final URL[] urls,
					     final ClassLoader parent) {
	// Save the caller's context
	AccessControlContext acc = AccessController.getContext();
	// Need a privileged block to create the class loader
	URLClassLoader ucl =
	    (URLClassLoader) AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
		    return new FactoryURLClassLoader(urls, parent);
		}
	    });
	// Now set the context on the loader using the one we saved,
	// not the one inside the privileged block...
	ucl.acc = acc;
	return ucl;
    }

    /**
     * Creates a new instance of URLClassLoader for the specified
     * URLs and default parent class loader. If a security manager is
     * installed, the <code>loadClass</code> method of the URLClassLoader
     * returned by this method will invoke the
     * <code>SecurityManager.checkPackageAccess</code> before
     * loading the class.
     *
     * @param urls the URLs to search for classes and resources
     * @return the resulting class loader
     */
    public static URLClassLoader newInstance(final URL[] urls) {
	// Save the caller's context
	AccessControlContext acc = AccessController.getContext();
	// Need a privileged block to create the class loader
	URLClassLoader ucl = (URLClassLoader)
	    AccessController.doPrivileged(new PrivilegedAction() {
		public Object run() {
		    return new FactoryURLClassLoader(urls);
		}
	    });

	// Now set the context on the loader using the one we saved,
	// not the one inside the privileged block...
	ucl.acc = acc;
	return ucl;
    }

  final class ClassFinder implements PrivilegedExceptionAction                  //IBM-shared_classes
  {                                                                             //IBM-shared_classes
     private String name;                                                       //IBM-shared_classes
     private ClassLoader classloader;                                           //IBM-shared_classes
                                                                                //IBM-shared_classes
     public ClassFinder(String name, ClassLoader loader) {                      //IBM-shared_classes
        this.name = name;                                                       //IBM-shared_classes
        this.classloader = loader;                                              //IBM-shared_classes
     }                                                                          //IBM-shared_classes
                                                                                //IBM-shared_classes
     public Object run() throws ClassNotFoundException {                        //IBM-shared_classes
	String path = name.replace('.', '/').concat(".class");                  //IBM-shared_classes
        try {                                                                   //IBM-shared_classes
            Resource res = ucp.getResource(path, false, classloader, showClassLoading(name));  //IBM-shared_classes
            if (res != null)                                                    //IBM-shared_classes
                return defineClass(name, res);                                  //IBM-shared_classes
        } catch (IOException e) {                                               //IBM-shared_classes
                throw new ClassNotFoundException(name, e);                      //IBM-shared_classes
        }                                                                       //IBM-shared_classes
        return null;                                                            //IBM-shared_classes
     }                                                                          //IBM-shared_classes
  }                                                                             //IBM-shared_classes
}

final class FactoryURLClassLoader extends URLClassLoader {

    FactoryURLClassLoader(URL[] urls, ClassLoader parent) {
	super(urls, parent);
    }

    FactoryURLClassLoader(URL[] urls) {
	super(urls);
    }

    public final synchronized Class loadClass(String name, boolean resolve)
	throws ClassNotFoundException
    {
	// First check if we have permission to access the package. This
	// should go away once we've added support for exported packages.
	SecurityManager sm = System.getSecurityManager();
	if (sm != null) {
	    int i = name.lastIndexOf('.');
	    if (i != -1) {
		sm.checkPackageAccess(name.substring(0, i));
	    }
	}
	return super.loadClass(name, resolve);
    }
}
//IBM-shared_classes
