/*
 * @(#)src/classes/sov/java/lang/Math.java, lang, asdev, 20060428 1.26
 * ===========================================================================
 * Licensed Materials - Property of IBM
 * "Restricted Materials of IBM"
 *
 * IBM SDK, Java(tm) 2 Technology Edition, v5.0
 * (C) Copyright IBM Corp. 1998, 2005. All Rights Reserved
 * ===========================================================================
 */

/*
 * ===========================================================================
 (C) Copyright Sun Microsystems Inc, 1992, 2004. All rights reserved.
 * ===========================================================================
 */

/*
 *
 * Change activity:
 *
 * Reason  Date   Origin  Description
 * ------  ----   ------  ----------------------------------------------------
 * 071120  260304 preece  Replaced by java implemetation
 * 071184  290304 preece  Fix JCK signature test and exp bug.
 * 071257  010404 preece  Fix JCK IEEEremainderTests
 * 071504  080404 andyt   Separate implementations: java for j9, native for sov
 * 080007  171104 corbin  Put back into one file and implement new math functions for 5.0
 * 082866  090205 andyt   Add missing log10 method
 * 085110  050405 stalleyj Correct the new definitions for 5.0
 * 089395  100605 andyt    FVT: Tan tests fails
 * 092073  070705 andyt    FVT: Math.rint returns wrong value
 * 095269  120905 preece  Set E and PI consistenlty with Sun. 
 * 099559  170106 preece  Optionally use Strict versions ( enable for realtime ).
 * 099802  310106 preece  JIT performance problem using static block
 *
 * ===========================================================================
 */
 
package java.lang;

import java.util.Random;
import com.ibm.jvm.Util;                        /*ibm@99559*/ 


/*
 * Module Information:
 *
 * DESCRIPTION:
 * This is version of Math.java that avoids JNI calls to fdlibm. 
 *
 * ===========================================================================
 */

// Referenced classes of package java.lang:
//            Object, Double, Float

public final class Math
{

    private static final boolean useStrict = Util.isRealTime();  /*ibm@99802*/

    private Math()
    {
    }

    public static double abs(double d)
    {
        long bits = Double.doubleToLongBits(d);
        bits &= 0x7fffffffffffffffL;
        return Double.longBitsToDouble(bits);
    }

    public static float abs(float f)
    {
        int bits = Float.floatToIntBits(f);
        bits &= 0x7fffffff;
        return Float.intBitsToFloat(bits);
    }

    public static int abs(int i)
    {
        return i < 0 ? -i : i;
    }

    public static long abs(long l)
    {
        return l < 0L ? -l : l;
    }

    //public static native double acos(double d);

    //public static native double asin(double d);

    //public static native double atan(double d);

    //public static native double atan2(double d, double d1);

    //public static native double ceil(double d);

    //public static native double cos(double d);

    //public static native double exp(double d);

    //public static native double floor(double d);

    //public static native double IEEEremainder(double d, double d1);

    //public static native double log(double d);

    public static double max(double a, double b) {
        if (a != a) return a;	// a is NaN
	if ((a == 0.0d) && (b == 0.0d)) {
	    return a+b;
	}
	return (a >= b) ? a : b;
    }

    private static double max_old(double d1, double d2)
    {
        if(d1 > d2)
            return d1;
        if(d1 < d2)
            return d2;
        if(d1 != d2)
            return (0.0D / 0.0D);
        if(d1 == 0.0D && (Double.doubleToLongBits(d1) & Double.doubleToLongBits(d2) & 0x8000000000000000L) == 0L)
            return 0.0D;
        else
            return d1;
    }

    public static float max(float a, float b) {
        if (a != a) return a;	// a is NaN
	if ((a == 0.0f) && (b == 0.0f)) {
            return a+b;
	}
	return (a >= b) ? a : b;
    }

    private static float max_old(float f1, float f2)
    {
        if(f1 > f2)
            return f1;
        if(f1 < f2)
            return f2;
        if(f1 != f2)
            return (0.0F / 0.0F);
        if(f1 == 0.0F && (Float.floatToIntBits(f1) & Float.floatToIntBits(f2) & 0x80000000) == 0)
            return 0.0F;
        else
            return f1;
    }

    public static int max(int i1, int i2)
    {
        return i1 <= i2 ? i2 : i1;
    }

    public static long max(long l1, long l2)
    {
        return l1 <= l2 ? l2 : l1;
    }

    public static double min(double a, double b) {
        if (a != a) return a;	// a is NaN
	if ((a == 0.0d) && (b == 0.0d)) {
	    return -(-a-b);
	}
	return (a <= b) ? a : b;
    }
    
    private static double min_old(double d1, double d2)
    {
        if(d1 > d2)
            return d2;
        if(d1 < d2)
            return d1;
        if(d1 != d2)
            return (0.0D / 0.0D);
        if(d1 == 0.0D && ((Double.doubleToLongBits(d1) | Double.doubleToLongBits(d2)) & 0x8000000000000000L) != 0L)
            return 0D;
        else
            return d1;
    }

    public static float min(float a, float b) {
        if (a != a) return a;	// a is NaN
	if ((a == 0.0f) && (b == 0.0f)) {
	    return -(-a-b);
	}
	return (a <= b) ? a : b;
    }

    private static float min_old(float f1, float f2)
    {
        if(f1 > f2)
            return f2;
        if(f1 < f2)
            return f1;
        if(f1 != f2)
            return (0.0F / 0.0F);
        if(f1 == 0.0F && ((Float.floatToIntBits(f1) | Float.floatToIntBits(f2)) & 0x80000000) != 0)
            return 0F;
        else
            return f1;
    }

    public static int min(int i1, int i2)
    {
        return i1 >= i2 ? i2 : i1;
    }

    public static long min(long l1, long l2)
    {
        return l1 >= l2 ? l2 : l1;
    }

    public static native double pow(double d, double d1);

    //public static native double rint(double d);

    public static long round(double d)
    {
        if(d != d)
            return 0L;
        else
            return (long)floor(d + 0.5D);
    }

    public static int round(float f)
    {
        if(f != f)
            return 0;
        else
            return (int)floor(f + 0.5F);
    }

    //public static native double sin(double d);

    public static native double sqrt(double d);

    /**
     * Returns the base 10 logarithm of a <code>double</code> value.
     * Special cases:
     *
     * <ul><li>If the argument is NaN or less than zero, then the result 
     * is NaN.
     * <li>If the argument is positive infinity, then the result is 
     * positive infinity.
     * <li>If the argument is positive zero or negative zero, then the 
     * result is negative infinity.
     * <li> If the argument is equal to 10<sup><i>n</i></sup> for
     * integer <i>n</i>, then the result is <i>n</i>.
     * </ul>
     * 
     * <p>The computed result must be within 1 ulp of the exact result.
     * Results must be semi-monotonic.
     *
     * @param   a   a value
     * @return  the base 10 logarithm of  <code>a</code>.
     * @since 1.5
     */
    public static double log10(double a) {
	return StrictMath.log10(a); // default impl. delegates to StrictMath
    }

    /**
     * Returns the cube root of a <code>double</code> value.  For
     * positive finite <code>x</code>, <code>cbrt(-x) ==
     * -cbrt(x)</code>; that is, the cube root of a negative value is
     * the negative of the cube root of that value's magnitude.
     * 
     * Special cases: 
     *
     * <ul>
     * 
     * <li>If the argument is NaN, then the result is NaN.
     *
     * <li>If the argument is infinite, then the result is an infinity
     * with the same sign as the argument.
     *
     * <li>If the argument is zero, then the result is a zero with the
     * same sign as the argument.
     * 
     * </ul>
     *
     * <p>The computed result must be within 1 ulp of the exact result.
     * 
     * @param   a   a value.
     * @return  the cube root of <code>a</code>.
     * @since 1.5
     */
    public static double cbrt(double a) {
	return StrictMath.cbrt(a);
    }

    /**
     * Returns the size of an ulp of the argument.  An ulp of a
     * <code>double</code> value is the positive distance between this
     * floating-point value and the <code>double</code> value next
     * larger in magnitude.  Note that for non-NaN <i>x</i>,
     * <code>ulp(-<i>x</i>) == ulp(<i>x</i>)</code>.
     * 
     * <p>Special Cases:
     * <ul>
     * <li> If the argument is NaN, then the result is NaN.
     * <li> If the argument is positive or negative infinity, then the
     * result is positive infinity.
     * <li> If the argument is positive or negative zero, then the result is
     * <code>Double.MIN_VALUE</code>.
     * <li> If the argument is &plusmn;<code>Double.MAX_VALUE</code>, then
     * the result is equal to 2<sup>971</sup>.
     * </ul>
     *
     * @param d the floating-point value whose ulp is to be returned
     * @return the size of an ulp of the argument
     * @author Joseph D. Darcy
     * @since 1.5
     */
    public static double ulp(double d) {
	return sun.misc.FpUtils.ulp(d);
    }

    /**
     * Returns the size of an ulp of the argument.  An ulp of a
     * <code>float</code> value is the positive distance between this
     * floating-point value and the <code>float</code> value next
     * larger in magnitude.  Note that for non-NaN <i>x</i>,
     * <code>ulp(-<i>x</i>) == ulp(<i>x</i>)</code>.
     * 
     * <p>Special Cases:
     * <ul>
     * <li> If the argument is NaN, then the result is NaN.
     * <li> If the argument is positive or negative infinity, then the
     * result is positive infinity.
     * <li> If the argument is positive or negative zero, then the result is
     * <code>Float.MIN_VALUE</code>.
     * <li> If the argument is &plusmn;<code>Float.MAX_VALUE</code>, then
     * the result is equal to 2<sup>104</sup>.
     * </ul>
     *
     * @param f the floating-point value whose ulp is to be returned
     * @return the size of an ulp of the argument
     * @author Joseph D. Darcy
     * @since 1.5
     */
    public static float ulp(float f) {
	return sun.misc.FpUtils.ulp(f);
    }

    /**
     * Returns the signum function of the argument; zero if the argument
     * is zero, 1.0 if the argument is greater than zero, -1.0 if the
     * argument is less than zero.
     *
     * <p>Special Cases:
     * <ul>
     * <li> If the argument is NaN, then the result is NaN.
     * <li> If the argument is positive zero or negative zero, then the
     *      result is the same as the argument.
     * </ul>
     *
     * @param d the floating-point value whose signum is to be returned
     * @return the signum function of the argument
     * @author Joseph D. Darcy
     * @since 1.5
     */
    public static double signum(double d) {
	return sun.misc.FpUtils.signum(d);
    }

    /**
     * Returns the signum function of the argument; zero if the argument
     * is zero, 1.0f if the argument is greater than zero, -1.0f if the
     * argument is less than zero.
     *
     * <p>Special Cases:
     * <ul>
     * <li> If the argument is NaN, then the result is NaN.
     * <li> If the argument is positive zero or negative zero, then the
     *      result is the same as the argument.
     * </ul>
     *
     * @param f the floating-point value whose signum is to be returned
     * @return the signum function of the argument
     * @author Joseph D. Darcy
     * @since 1.5
     */
    public static float signum(float f) {
	return sun.misc.FpUtils.signum(f);
    }

    /**
     * Returns the hyperbolic sine of a <code>double</code> value.
     * The hyperbolic sine of <i>x</i> is defined to be
     * (<i>e<sup>x</sup>&nbsp;-&nbsp;e<sup>-x</sup></i>)/2
     * where <i>e</i> is {@linkplain Math#E Euler's number}.
     *
     * <p>Special cases:
     * <ul>
     *
     * <li>If the argument is NaN, then the result is NaN.
     *
     * <li>If the argument is infinite, then the result is an infinity
     * with the same sign as the argument.
     *
     * <li>If the argument is zero, then the result is a zero with the
     * same sign as the argument.
     *
     * </ul>
     *
     * <p>The computed result must be within 2.5 ulps of the exact result.
     *
     * @param   x The number whose hyperbolic sine is to be returned.
     * @return  The hyperbolic sine of <code>x</code>.
     * @since 1.5
     */
    public static double sinh(double x) {
	return StrictMath.sinh(x);
    }

    /**
     * Returns the hyperbolic cosine of a <code>double</code> value.
     * The hyperbolic cosine of <i>x</i> is defined to be
     * (<i>e<sup>x</sup>&nbsp;+&nbsp;e<sup>-x</sup></i>)/2
     * where <i>e</i> is {@linkplain Math#E Euler's number}.
     *
     * <p>Special cases:
     * <ul>
     *
     * <li>If the argument is NaN, then the result is NaN.
     *
     * <li>If the argument is infinite, then the result is positive
     * infinity.
     *
     * <li>If the argument is zero, then the result is <code>1.0</code>.
     *
     * </ul>
     *
     * <p>The computed result must be within 2.5 ulps of the exact result.
     *
     * @param   x The number whose hyperbolic cosine is to be returned.
     * @return  The hyperbolic cosine of <code>x</code>.
     * @since 1.5
     */
    public static double cosh(double x) {
	return StrictMath.cosh(x);
    }

    /**
     * Returns the hyperbolic tangent of a <code>double</code> value.
     * The hyperbolic tangent of <i>x</i> is defined to be
     * (<i>e<sup>x</sup>&nbsp;-&nbsp;e<sup>-x</sup></i>)/(<i>e<sup>x</sup>&nbsp;+&nbsp;e<sup>-x</sup></i>),
     * in other words, {@linkplain Math#sinh
     * sinh(<i>x</i>)}/{@linkplain Math#cosh cosh(<i>x</i>)}.  Note
     * that the absolute value of the exact tanh is always less than
     * 1.
     *
     * <p>Special cases:
     * <ul>
     *
     * <li>If the argument is NaN, then the result is NaN.
     *
     * <li>If the argument is zero, then the result is a zero with the
     * same sign as the argument.
     *
     * <li>If the argument is positive infinity, then the result is
     * <code>+1.0</code>.
     *
     * <li>If the argument is negative infinity, then the result is
     * <code>-1.0</code>.
     *  
     * </ul>
     *
     * <p>The computed result must be within 2.5 ulps of the exact result.
     * The result of <code>tanh</code> for any finite input must have
     * an absolute value less than or equal to 1.  Note that once the
     * exact result of tanh is within 1/2 of an ulp of the limit value
     * of &plusmn;1, correctly signed &plusmn;<code>1.0</code> should
     * be returned.
     *
     * @param   x The number whose hyperbolic tangent is to be returned.
     * @return  The hyperbolic tangent of <code>x</code>.
     * @since 1.5
     */
    public static double tanh(double x) {
	return StrictMath.tanh(x);
    }

    /**
     * Returns sqrt(<i>x</i><sup>2</sup>&nbsp;+<i>y</i><sup>2</sup>)
     * without intermediate overflow or underflow.
     *
     * <p>Special cases:
     * <ul>
     *
     * <li> If either argument is infinite, then the result
     * is positive infinity.
     *
     * <li> If either argument is NaN and neither argument is infinite,
     * then the result is NaN.
     *
     * </ul>
     *
     * <p>The computed result must be within 1 ulp of the exact
     * result.  If one parameter is held constant, the results must be
     * semi-monotonic in the other parameter.
     *
     * @param x a value
     * @param y a value
     * @return sqrt(<i>x</i><sup>2</sup>&nbsp;+<i>y</i><sup>2</sup>)
     * without intermediate overflow or underflow
     * @since 1.5
     */
    public static double hypot(double x, double y) {
	return StrictMath.hypot(x, y);
    }

    /**
     * Returns <i>e</i><sup>x</sup>&nbsp;-1.  Note that for values of
     * <i>x</i> near 0, the exact sum of
     * <code>expm1(x)</code>&nbsp;+&nbsp;1 is much closer to the true
     * result of <i>e</i><sup>x</sup> than <code>exp(x)</code>.
     *
     * <p>Special cases:
     * <ul>
     * <li>If the argument is NaN, the result is NaN.
     *
     * <li>If the argument is positive infinity, then the result is
     * positive infinity.
     *
     * <li>If the argument is negative infinity, then the result is
     * -1.0.
     *
     * <li>If the argument is zero, then the result is a zero with the
     * same sign as the argument.
     *
     * </ul>
     *
     * <p>The computed result must be within 1 ulp of the exact result.
     * Results must be semi-monotonic.  The result of
     * <code>expm1</code> for any finite input must be greater than or
     * equal to <code>-1.0</code>.  Note that once the exact result of
     * <i>e</i><sup><code>x</code></sup>&nbsp;-&nbsp;1 is within 1/2
     * ulp of the limit value -1, <code>-1.0</code> should be
     * returned.
     *
     * @param   x   the exponent to raise <i>e</i> to in the computation of
     *              <i>e</i><sup><code>x</code></sup>&nbsp;-1.
     * @return  the value <i>e</i><sup><code>x</code></sup>&nbsp;-&nbsp;1.
     */
    public static double expm1(double x) {
	return StrictMath.expm1(x);
    }

    /**
     * Returns the natural logarithm of the sum of the argument and 1.
     * Note that for small values <code>x</code>, the result of
     * <code>log1p(x)</code> is much closer to the true result of ln(1
     * + <code>x</code>) than the floating-point evaluation of
     * <code>log(1.0+x)</code>.
     *
     * <p>Special cases:
     *
     * <ul>
     *
     * <li>If the argument is NaN or less than -1, then the result is
     * NaN.
     *
     * <li>If the argument is positive infinity, then the result is
     * positive infinity.
     *
     * <li>If the argument is negative one, then the result is
     * negative infinity.
     *
     * <li>If the argument is zero, then the result is a zero with the
     * same sign as the argument.
     *
     * </ul>
     *
     * <p>The computed result must be within 1 ulp of the exact result.
     * Results must be semi-monotonic.
     *
     * @param   x   a value
     * @return the value ln(<code>x</code>&nbsp;+&nbsp;1), the natural
     * log of <code>x</code>&nbsp;+&nbsp;1
     */
    public static double log1p(double x) {
	return StrictMath.log1p(x);
    }
    
    //public static native double tan(double d);

    public static double random()
    {
        if(random == null)
            random = new Random();
        return random.nextDouble();
    }

    public static double toRadians(double angdeg)
    {
        return (angdeg / 180D) * PI;
    }

    public static double toDegrees(double angrad)
    {
        return (angrad * 180D) / PI;
    }

    /** 
     * The <code>double</code> value that is closer than any other to 
     * <i>e</i>, the base of the natural logarithms. 
     */ 
    public static final double E = 2.7182818284590452354;               /*ibm@95269*/
 
    /** 
     * The <code>double</code> value that is closer than any other to 
     * <i>pi</i>, the ratio of the circumference of a circle to its 
     * diameter. 
     */ 
    public static final double PI = 3.14159265358979323846;             /*ibm@95269*/ 

    private static Random random;


   // ****** CUSTOM FUNCTIONS ****** 

    private static double __HI__LO(int a, int b) {
        return Double.longBitsToDouble((long)a << 32 | (b & 0xffffffffL));
    }
    private static int __HI(double d) {
        return (int)(Double.doubleToRawLongBits(d) >> 32);
    }
    private static int __LO(double d) {
        return (int)Double.doubleToRawLongBits(d);
    }
    private static double __HI(double d, int x) {
        return __HI__LO(x, __LO(d));
    }
    private static double __HI__PLUS(double d, int x) {
        return __HI__LO(__HI(d)+x, __LO(d));
    }
    private static double __LO(double d, int x) {
        return __HI__LO(__HI(d), x);
    }
    
    private final static double one = 1.0;
    private static final double tiny  = 1.0e-300;
    private static final double zero  = 0.0;
    private static final double pi_o_4  = 7.8539816339744827900E-01; /* 0x3FE921FB, 0x54442D18 */
    private static final double pi_o_2  = 1.5707963267948965580E+00; /* 0x3FF921FB, 0x54442D18 */
    private static final double pi      = 3.1415926535897931160E+00; /* 0x400921FB, 0x54442D18 */
    private static final double pi_lo   = 1.2246467991473531772E-16; /* 0x3CA1A626, 0x33145C07 */

    /* __kernel_sin( x, y, iy)
     * kernel sin function on [-pi/4, pi/4], pi/4 ~ 0.7854
     * Input x is assumed to be bounded by ~pi/4 in magnitude.
     * Input y is the tail of x.
     * Input iy indicates whether y is 0. (if iy=0, y assume to be 0).
     *
     * Algorithm
     *	1. Since sin(-x) = -sin(x), we need only to consider positive x.
     *	2. if x < 2^-27 (hx<0x3e400000 0), return x with inexact if x!=0.
     *	3. sin(x) is approximated by a polynomial of degree 13 on
     *	   [0,pi/4]
     *		  	         3            13
     *	   	sin(x) ~ x + S1*x + ... + S6*x
     *	   where
     *
     * 	|sin(x)         2     4     6     8     10     12  |     -58
     * 	|----- - (1+S1*x +S2*x +S3*x +S4*x +S5*x  +S6*x   )| <= 2
     * 	|  x 					           |
     *
     *	4. sin(x+y) = sin(x) + sin'(x')*y
     *		    ~ sin(x) + (1-x*x/2)*y
     *	   For better accuracy, let
     *		     3      2      2      2      2
     *		r = x *(S2+x *(S3+x *(S4+x *(S5+x *S6))))
     *	   then                   3    2
     *		sin(x) = x + (S1*x + (x *(r-y/2)+y))
     */

    private static final double
    half =  5.00000000000000000000e-01, /* 0x3FE00000, 0x00000000 */
    S1  = -1.66666666666666324348e-01, /* 0xBFC55555, 0x55555549 */
    S2  =  8.33333333332248946124e-03, /* 0x3F811111, 0x1110F8A6 */
    S3  = -1.98412698298579493134e-04, /* 0xBF2A01A0, 0x19C161D5 */
    S4  =  2.75573137070700676789e-06, /* 0x3EC71DE3, 0x57B1FE7D */
    S5  = -2.50507602534068634195e-08, /* 0xBE5AE5E6, 0x8A2B9CEB */
    S6  =  1.58969099521155010221e-10; /* 0x3DE5D93A, 0x5ACFD57C */

    private static double __kernel_sin(double x, double y, int iy) {
            double z,r,v;
            int ix;
            ix = __HI(x)&0x7fffffff;	/* high word of x */
            if(ix<0x3e400000)			/* |x| < 2**-27 */
               {if((int)x==0) return x;}		/* generate inexact */
            z	=  x*x;
            v	=  z*x;
            r	=  S2+z*(S3+z*(S4+z*(S5+z*S6)));
            if(iy==0) return x+v*(S1+z*r);
            else      return x-((z*(half*y-v*r)-y)-v*S1);
    }

    private final static double C1  =  4.16666666666666019037e-02, /* 0x3FA55555, 0x5555554C */
    C2  = -1.38888888888741095749e-03, /* 0xBF56C16C, 0x16C15177 */
    C3  =  2.48015872894767294178e-05, /* 0x3EFA01A0, 0x19CB1590 */
    C4  = -2.75573143513906633035e-07, /* 0xBE927E4F, 0x809C52AD */
    C5  =  2.08757232129817482790e-09, /* 0x3E21EE9E, 0xBDB4B1C4 */
    C6  = -1.13596475577881948265e-11; /* 0xBDA8FAE9, 0xBE8838D4 */
    private static double __kernel_cos(double x, double y) {
        double a,hz,z,r,qx;
        int ix;
        ix = __HI(x)&0x7fffffff;        /* ix = |x|'s high word*/
        if(ix<0x3e400000) {                     /* if x < 2**27 */
            if(((int)x)==0) return one;         /* generate inexact */
        }
        z  = x*x;
        r  = z*(C1+z*(C2+z*(C3+z*(C4+z*(C5+z*C6)))));
        if(ix < 0x3FD33333)                     /* if |x| < 0.3 */
            return one - (0.5*z - (z*r - x*y));
        else {
            if(ix > 0x3fe90000) {               /* x > 0.78125 */
                qx = 0.28125;
            } else {
                //__HI(qx) = ix-0x00200000;       /* x/4 */
                //__LO(qx) = 0;
                qx = __HI__LO(ix-0x00200000, 0);       /* x/4 */
            }
            hz = 0.5*z-qx;
            a  = one-qx;
            return a - (hz - (z*r-x*y));
        }
    }


/* __kernel_tan( x, y, k )
 * kernel tan function on [-pi/4, pi/4], pi/4 ~ 0.7854
 * Input x is assumed to be bounded by ~pi/4 in magnitude.
 * Input y is the tail of x.
 * Input k indicates whether tan (if k=1) or
 * -1/tan (if k= -1) is returned.
 *
 * Algorithm
 *	1. Since tan(-x) = -tan(x), we need only to consider positive x.
 *	2. if x < 2^-28 (hx<0x3e300000 0), return x with inexact if x!=0.
 *	3. tan(x) is approximated by a odd polynomial of degree 27 on
 *	   [0,0.67434]
 *		  	         3             27
 *	   	tan(x) ~ x + T1*x + ... + T13*x
 *	   where
 *
 * 	        |tan(x)         2     4            26   |     -59.2
 * 	        |----- - (1+T1*x +T2*x +.... +T13*x    )| <= 2
 * 	        |  x 					|
 *
 *	   Note: tan(x+y) = tan(x) + tan'(x)*y
 *		          ~ tan(x) + (1+x*x)*y
 *	   Therefore, for better accuracy in computing tan(x+y), let
 *		     3      2      2       2       2
 *		r = x *(T2+x *(T3+x *(...+x *(T12+x *T13))))
 *	   then
 *		 		    3    2
 *		tan(x+y) = x + (T1*x + (x *(r+y)+y))
 *
 *      4. For x in [0.67434,pi/4],  let y = pi/4 - x, then
 *		tan(x) = tan(pi/4-y) = (1-tan(y))/(1+tan(y))
 *		       = 1 - 2*(tan(y) - (tan(y)^2)/(1+tan(y)))
 */

    static final double pio4   =  7.85398163397448278999e-01; /* 0x3FE921FB, 0x54442D18 */
    static final double pio4lo =  3.06161699786838301793e-17; /* 0x3C81A626, 0x33145C07 */
    static final double T[]    =  {
      3.33333333333334091986e-01, /* 0x3FD55555, 0x55555563 */
      1.33333333333201242699e-01, /* 0x3FC11111, 0x1110FE7A */
      5.39682539762260521377e-02, /* 0x3FABA1BA, 0x1BB341FE */
      2.18694882948595424599e-02, /* 0x3F9664F4, 0x8406D637 */
      8.86323982359930005737e-03, /* 0x3F8226E3, 0xE96E8493 */
      3.59207910759131235356e-03, /* 0x3F6D6D22, 0xC9560328 */
      1.45620945432529025516e-03, /* 0x3F57DBC8, 0xFEE08315 */
      5.88041240820264096874e-04, /* 0x3F4344D8, 0xF2F26501 */
      2.46463134818469906812e-04, /* 0x3F3026F7, 0x1A8D1068 */
      7.81794442939557092300e-05, /* 0x3F147E88, 0xA03792A6 */
      7.14072491382608190305e-05, /* 0x3F12B80F, 0x32F0A7E9 */
     -1.85586374855275456654e-05, /* 0xBEF375CB, 0xDB605373 */
      2.59073051863633712884e-05, /* 0x3EFB2A70, 0x74BF7AD4 */
    };

    private static double __kernel_tan(double x, double y, int iy) {
	double z,r,v,w,s;
	int ix,hx;
	hx = __HI(x);	/* high word of x */
	ix = hx&0x7fffffff;	/* high word of |x| */
	if(ix<0x3e300000) {			/* x < 2**-28 */
	  if((int)x==0) {			/* generate inexact */
	    if (((ix | __LO(x)) | (iy + 1)) == 0)
	      return one / abs(x);
            /*ibm@89395...*/
            else { 
              if (iy == 1) 
                return x; 
              else {    /* compute -1 / (x+y) carefully */ 
                double a, t; 
                 
                z = w = x + y; 
//                __LO(z) = 0; 
                z = __HI__LO(__HI(z), 0);
                v = y - (z - x); 
                t = a = -one / w; 
//                __LO(t) = 0; 
                t = __HI__LO(__HI(t), 0);
                s = one + t * z; 
                return t + a * (s + t * v); 
              } 
            } 
            /*...ibm@89395*/
          }
        }
	if(ix>=0x3FE59428) { 			/* |x|>=0.6744 */
	    if(hx<0) {x = -x; y = -y;}
	    z = pio4-x;
	    w = pio4lo-y;
	    x = z+w; y = 0.0;
	}
	z	=  x*x;
	w 	=  z*z;
    /* Break x^5*(T[1]+x^2*T[2]+...) into
     *	  x^5(T[1]+x^4*T[3]+...+x^20*T[11]) +
     *	  x^5(x^2*(T[2]+x^4*T[4]+...+x^22*[T12]))
     */
	r = T[1]+w*(T[3]+w*(T[5]+w*(T[7]+w*(T[9]+w*T[11]))));
	v = z*(T[2]+w*(T[4]+w*(T[6]+w*(T[8]+w*(T[10]+w*T[12])))));
	s = z*x;
	r = y + z*(s*(r+v)+y);
	r += T[0]*s;
	w = x+r;
	if(ix>=0x3FE59428) {
	    v = (double)iy;
	    return (double)(1-((hx>>30)&2))*(v-2.0*(x-(w*w/(w+v)-r)));
	}
	if(iy==1) return w;
	else {		/* if allow error up to 2 ulp,
			   simply return -1.0/(x+r) here */
     /*  compute -1.0/(x+r) accurately */
	    double a,t;
	    z  = w;
	    //__LO(z) = 0;
	    z = __HI__LO(__HI(z), 0);
	    v  = r-(z - x); 	/* z+v = r+x */
	    t = a  = -1.0/w;	/* a = -1.0/w */
	    //__LO(t) = 0;
	    t = __HI__LO(__HI(t), 0);
	    s  = 1.0+t*z;
	    return t+a*(s+t*v);
	}
    }

    public static double sin(double x) {
        if (useStrict) return StrictMath.sin(x);                            /*ibm@99559*/
        double z,r,v;
        int ix;
        ix = __HI(x)&0x7fffffff;	/* high word of x */
        if (ix<0x3e400000)			/* |x| < 2**-27 */
           {if((int)x==0) return x;}		/* generate inexact */
	if(ix > 0x3fe921fb) return sin_full(x, ix);

        z = x*x;
        v = z*x;
        r = S2+z*(S3+z*(S4+z*(S5+z*S6)));
        return x+v*(S1+z*r);
    }
    
    private static double sin_full(double x, int ix) {
	double y[] = new double[2],z=0.0;
	int n;


    /* sin(Inf or NaN) is NaN */
	if (ix>=0x7ff00000) return x+(-x);    /* ibm@7221 - to fool the optimization so we don't get 0 */
	/* ibm@7221 else if (ix>=0x7ff00000) return x-x; */

    /* argument reduction needed */
	else {
	    n = __ieee754_rem_pio2(x,y);
	    switch(n&3) {
		case 0: return  __kernel_sin(y[0],y[1],1);
		case 1: return  __kernel_cos(y[0],y[1]);
		case 2: return -__kernel_sin(y[0],y[1],1);
		default:
			return -__kernel_cos(y[0],y[1]);
	    }
	}
    }

    public static double cos(double x) {
        if (useStrict) return StrictMath.cos(x);                            /*ibm@99559*/
        double a,hz,z,r,qx;
        int ix;
        ix = __HI(x)&0x7fffffff;        /* ix = |x|'s high word*/
        if(ix<0x3e400000) {                     /* if x < 2**27 */
            if(((int)x)==0) return one;         /* generate inexact */
        }

        if(ix > 0x3fe921fb) return cos_full(x, ix);

        z  = x*x;
        r  = z*(C1+z*(C2+z*(C3+z*(C4+z*(C5+z*C6)))));
        if(ix < 0x3FD33333)                     /* if |x| < 0.3 */
            return one - (0.5*z - z*r);
        else {
            if(ix > 0x3fe90000) {               /* x > 0.78125 */
                qx = 0.28125;
            } else {
                //__HI(qx) = ix-0x00200000;       /* x/4 */
                //__LO(qx) = 0;
                qx = __HI__LO(ix-0x00200000, 0);       /* x/4 */
            }
            hz = 0.5*z-qx;
            a  = one-qx;
            return a - (hz - z*r);
        }
    }

    private static double cos_full(double x, int ix) {
        double y[] = new double[2],z=0.0;
        int n;

        /* cos(Inf or NaN) is NaN */
        if(ix>=0x7ff00000) return x + (-x);  /* ibm@7221 - don't let this be optimized to 0 */
        /* ibm@7221 if(ix>=0x7ff00000) return x-x; */

	/* argument reduction needed */
	else {
	    n = __ieee754_rem_pio2(x,y);
	    switch(n&3) {
		case 0: return  __kernel_cos(y[0],y[1]);
		case 1: return -__kernel_sin(y[0],y[1],1);
		case 2: return -__kernel_cos(y[0],y[1]);
		default:
		        return  __kernel_sin(y[0],y[1],1);
	    }
	}
    }

    public static double tan(double x) {
        if (useStrict) return StrictMath.tan(x);                            /*ibm@99559*/
	double z,r,v,w,s;
	int ix,hx;
	hx = __HI(x);	/* high word of x */
	ix = hx&0x7fffffff;	/* high word of |x| */

	if(ix > 0x3fe921fb) return tan_full(x, ix);

	if(ix<0x3e300000) {			/* x < 2**-28 */
	    	if((int)x==0) {			/* generate inexact */
			if(((ix|__LO(x))|2)==0) 
				return one/abs(x);
			else 
				return x;
	    	}
        }
	
	if(ix>=0x3FE59428) { 			/* |x|>=0.6744 */
	    if(hx<0) {
	    	x = -x; 
	    }
	    
	    z = pio4-x;
	    w = pio4lo;
	    x = z+w; 
	}
	
	z =  x*x;
	w =  z*z;
    /* Break x^5*(T[1]+x^2*T[2]+...) into
     *	  x^5(T[1]+x^4*T[3]+...+x^20*T[11]) +
     *	  x^5(x^2*(T[2]+x^4*T[4]+...+x^22*[T12]))
     */
	r = T[1]+w*(T[3]+w*(T[5]+w*(T[7]+w*(T[9]+w*T[11]))));
	v = z*(T[2]+w*(T[4]+w*(T[6]+w*(T[8]+w*(T[10]+w*T[12])))));
	s = z*x;
	r = z*(s*(r+v));
	r += T[0]*s;
	w = x+r;
	if(ix>=0x3FE59428) {
	    v = 1.0d;
	    return (double)(1-((hx>>30)&2))*(v-2.0*(x-(w*w/(w+v)-r)));
	}

	return w;
    }

    private static double tan_full(double x, int ix) {
	double y[] = new double[2],z=0.0;
	int n;

    /* High word of x. */
	ix = __HI(x);

    /* |x| ~< pi/4 */
	ix &= 0x7fffffff;
	//if(ix <= 0x3fe921fb) return __kernel_tan(x,z,1);

    /* tan(Inf or NaN) is NaN */
	//else
	if (ix>=0x7ff00000) return x+(-x);		/* ibm@7221 *//* NaN */
    /* ibm@7221 else if (ix>=0x7ff00000) return x-x;		*//* NaN */

    /* argument reduction needed */
	else {
	    n = __ieee754_rem_pio2(x,y);
	    return __kernel_tan(y[0],y[1],1-((n&1)<<1)); /*   1 -- n even
							-1 -- n odd */
	}
    }

    public static double floor(double x) {
        if (useStrict) return StrictMath.floor(x);                          /*ibm@99559*/
	int i0,i1,j0;
	//unsigned i,j;
	int i,j;
	i0 =  __HI(x);
	i1 =  __LO(x);
	j0 = ((i0>>20)&0x7ff)-0x3ff;
	if(j0<20) {
	    if(j0<0) { 	/* raise inexact if x != 0 */
		if(huge+x>0.0) {/* return 0*sign(x) if |x|<1 */
		    if(i0>=0) {i0=i1=0;}
		    else if(((i0&0x7fffffff)|i1)!=0)
			{ i0=0xbff00000;i1=0;}
		}
	    } else {
		i = (0x000fffff)>>j0;
		if(((i0&i)|i1)==0) return x; /* x is integral */
		if(huge+x>0.0) {	/* raise inexact flag */
		    if(i0<0) i0 += (0x00100000)>>j0;
		    i0 &= (~i); i1=0;
		}
	    }
	} else if (j0>51) {
	    if(j0==0x400) return x+x;	/* inf or NaN */
	    else return x;		/* x is integral */
	} else {
	    //i = ((unsigned)(0xffffffff))>>(j0-20);
	    i = 0xffffffff >>>(j0-20);
	    if((i1&i)==0) return x;	/* x is integral */
	    if(huge+x>0.0) { 		/* raise inexact flag */
		if(i0<0) {
		    if(j0==20) i0+=1;
		    else {
			j = i1+(1<<(52-j0));
			//if(j<(unsigned)i1) i0 +=1 ; 	/* got a carry */
			if((j & 0xffffffffL)<(i1 & 0xffffffffl)) i0 +=1 ; 	/* got a carry */
			i1=j;
		    }
		}
		i1 &= (~i);
	    }
	}
	//__HI(x) = i0;
	//__LO(x) = i1;
	x = __HI__LO(i0, i1);
	return x;
    }

    public static double ceil(double x) {
        if (useStrict) return StrictMath.ceil(x);                          /*ibm@99559*/
	int i0,i1,j0;
	//unsigned i,j;
	int i,j;
	i0 =  __HI(x);
	i1 =  __LO(x);
	j0 = ((i0>>20)&0x7ff)-0x3ff;
	if(j0<20) {
	    if(j0<0) { 	/* raise inexact if x != 0 */
		if(huge+x>0.0) {/* return 0*sign(x) if |x|<1 */
		    if(i0<0) {i0=0x80000000;i1=0;}
		    else if((i0|i1)!=0) { i0=0x3ff00000;i1=0;}
		}
	    } else {
		i = (0x000fffff)>>j0;
		if(((i0&i)|i1)==0) return x; /* x is integral */
		if(huge+x>0.0) {	/* raise inexact flag */
		    if(i0>0) i0 += (0x00100000)>>j0;
		    i0 &= (~i); i1=0;
		}
	    }
	} else if (j0>51) {
	    if(j0==0x400) return x+x;	/* inf or NaN */
	    else return x;		/* x is integral */
	} else {
	    //i = ((unsigned)(0xffffffff))>>(j0-20);
	    i = 0xffffffff>>>(j0-20);
	    if((i1&i)==0) return x;	/* x is integral */
	    if(huge+x>0.0) { 		/* raise inexact flag */
		if(i0>0) {
		    if(j0==20) i0+=1;
		    else {
			j = i1 + (1<<(52-j0));
			//if(j<(unsigned)i1) i0+=1;	/* got a carry */
			if((j&0xffffffffL)<(i1&0xffffffffL)) i0+=1;	/* got a carry */
			i1 = j;
		    }
		}
		i1 &= (~i);
	    }
	}
	//__HI(x) = i0;
	//__LO(x) = i1;
        x = __HI__LO(i0, i1);
	return x;
    }

/* __ieee754_exp(x)
 * Returns the exponential of x.
 *
 * Method
 *   1. Argument reduction:
 *      Reduce x to an r so that |r| <= 0.5*ln2 ~ 0.34658.
 *	Given x, find r and integer k such that
 *
 *               x = k*ln2 + r,  |r| <= 0.5*ln2.
 *
 *      Here r will be represented as r = hi-lo for better
 *	accuracy.
 *
 *   2. Approximation of exp(r) by a special rational function on
 *	the interval [0,0.34658]:
 *	Write
 *	    R(r**2) = r*(exp(r)+1)/(exp(r)-1) = 2 + r*r/6 - r**4/360 + ...
 *      We use a special Reme algorithm on [0,0.34658] to generate
 * 	a polynomial of degree 5 to approximate R. The maximum error
 *	of this polynomial approximation is bounded by 2**-59. In
 *	other words,
 *	    R(z) ~ 2.0 + P1*z + P2*z**2 + P3*z**3 + P4*z**4 + P5*z**5
 *  	(where z=r*r, and the values of P1 to P5 are listed below)
 *	and
 *	    |                  5          |     -59
 *	    | 2.0+P1*z+...+P5*z   -  R(z) | <= 2
 *	    |                             |
 *	The computation of exp(r) thus becomes
 *                             2*r
 *		exp(r) = 1 + -------
 *		              R - r
 *                                 r*R1(r)
 *		       = 1 + r + ----------- (for better accuracy)
 *		                  2 - R1(r)
 *	where
 *			         2       4             10
 *		R1(r) = r - (P1*r  + P2*r  + ... + P5*r   ).
 *
 *   3. Scale back to obtain exp(x):
 *	From step 1, we have
 *	   exp(x) = 2^k * exp(r)
 *
 * Special cases:
 *	exp(INF) is INF, exp(NaN) is NaN;
 *	exp(-INF) is 0, and
 *	for finite argument, only exp(0)=1 is exact.
 *
 * Accuracy:
 *	according to an error analysis, the error is always less than
 *	1 ulp (unit in the last place).
 *
 * Misc. info.
 *	For IEEE double
 *	    if x >  7.09782712893383973096e+02 then exp(x) overflow
 *	    if x < -7.45133219101941108420e+02 then exp(x) underflow
 *
 * Constants:
 * The hexadecimal values are the intended ones for the following
 * constants. The decimal values may be used, provided that the
 * compiler will convert from decimal to binary accurately enough
 * to produce the hexadecimal values shown.
 */

    private static final double halF[]	    = { 0.5,-0.5,};
    private static final double twom1000    =   9.33263618503218878990e-302;    /* 2**-1000=0x01700000,0*/
    private static final double o_threshold =   7.09782712893383973096e+02;     /* 0x40862E42, 0xFEFA39EF */
    private static final double u_threshold =  -7.45133219101941108420e+02;     /* 0xc0874910, 0xD52D3051 */
    private static final double ln2HI[]     = { 6.93147180369123816490e-01,     /* 0x3fe62e42, 0xfee00000 */
	                                       -6.93147180369123816490e-01,};   /* 0xbfe62e42, 0xfee00000 */
    private static final double ln2LO[]     = { 1.90821492927058770002e-10,     /* 0x3dea39ef, 0x35793c76 */
	                                       -1.90821492927058770002e-10,};   /* 0xbdea39ef, 0x35793c76 */
    private static final double invln2      =  1.44269504088896338700e+00;      /* 0x3ff71547, 0x652b82fe */
    private static final double P1          =  1.66666666666666019037e-01;      /* 0x3FC55555, 0x5555553E */
    private static final double P2          = -2.77777777770155933842e-03;      /* 0xBF66C16C, 0x16BEBD93 */
    private static final double P3          =  6.61375632143793436117e-05;      /* 0x3F11566A, 0xAF25DE2C */
    private static final double P4          = -1.65339022054652515390e-06;      /* 0xBEBBBD41, 0xC5D26BF1 */
    private static final double P5          =  4.13813679705723846039e-08;      /* 0x3E663769, 0x72BEA4D0 */


    public static double exp(double x)	/* default IEEE double exp */
    {
        if (useStrict) return StrictMath.exp(x);                            /*ibm@99559*/
	double y,hi=0,lo=0,c,t;
	int k=0,xsb;
	int hx;

	hx  = __HI(x);	/* high word of x */
	//xsb = (hx>>31)&1;		/* sign bit of x */
	xsb = (hx>>>31)&1;		/* sign bit of x */
	hx &= 0x7fffffff;		/* high word of |x| */

	if (hx >= 0x40862E42) return exp_full(x, hx, xsb);

	if (hx <0x3e300000)
	    if(huge+x>one) return one+x;/* trigger inexact */

    /* argument reduction */
	if(hx > 0x3fd62e42) {		/* if  |x| > 0.5 ln2 */
	    if(hx < 0x3FF0A2B2) {	/* and |x| < 1.5 ln2 */
		if (xsb == 1)
		    {
		    hi = x - (-6.93147180369123816490e-01); k = -1;
                    lo = -1.90821492927058770002e-10;
		    x  = hi - lo;
		    t  = x*x;
		    c  = x - t*(P1+t*(P2+t*(P3+t*(P4+t*P5))));
		    y = one-((-1.90821492927058770002e-10 -(x*c)/(2.0-c))-hi);
		    y = __HI__PLUS(y, (k<<20));	/* add k to y's exponent */
		    return y;
		    }
		else if (xsb == 0)
		    {
		    hi = x - (6.93147180369123816490e-01); k = 1;
                    lo = 1.90821492927058770002e-10;
		    x  = hi - lo;
		    t  = x*x;
		    c  = x - t*(P1+t*(P2+t*(P3+t*(P4+t*P5))));
		    y = one-((1.90821492927058770002e-10 -(x*c)/(2.0-c))-hi);
		    y = __HI__PLUS(y, (k<<20));	/* add k to y's exponent */
		    return y;
		    }
		else 
		    {
		    hi = x-ln2HI[xsb]; lo=ln2LO[xsb]; k = 1-xsb-xsb;
		    }
	    } else {
		k  = (int) (invln2*x+halF[xsb]);
		t  = k;
		hi = x - t*ln2HI[0];	/* t*ln2HI is exact here */
		lo = t*ln2LO[0];
	    }
	    x  = hi - lo;
	}
	else {
	    t  = x*x;
    	    c  = x - t*(P1+t*(P2+t*(P3+t*(P4+t*P5))));

	    return one-((x*c)/(c-2.0)-x);
	}


	t  = x*x;
	c  = x - t*(P1+t*(P2+t*(P3+t*(P4+t*P5))));
	y = one-((lo-(x*c)/(2.0-c))-hi);
	if(k >= -1021) {
	    //__HI(y) += (k<<20);	/* add k to y's exponent */
	    y = __HI__PLUS(y, (k<<20));	/* add k to y's exponent */
	    return y;
	} else {
	    //__HI(y) += ((k+1000)<<20);/* add k to y's exponent */
	    y = __HI__PLUS(y, (k+1000)<<20);/* add k to y's exponent */
	    return y*twom1000;
	}
    }

    private static double exp_full(double x, int hx, int xsb)	/* default IEEE double exp */
    {
	double y,hi=0,lo=0,c,t;
	int k=0;

    /* filter out non-finite argument */
	if(hx >= 0x40862E42) {			/* if |x|>=709.78... */
            if(hx>=0x7ff00000) {
		if(((hx&0xfffff)|__LO(x))!=0)
		     return x+x; 		/* NaN */
		else return (xsb==0)? x:0.0;	/* exp(+-inf)={inf,0} */
	    }
	    if(x > o_threshold) return huge*huge; /* overflow */
	    if(x < u_threshold) return twom1000*twom1000; /* underflow */
	}

    /* argument reduction */
	if(hx > 0x3fd62e42) {		/* if  |x| > 0.5 ln2 */
	    if(hx < 0x3FF0A2B2) {	/* and |x| < 1.5 ln2 */
		hi = x-ln2HI[xsb]; lo=ln2LO[xsb]; k = 1-xsb-xsb;
	    } else {
		k  = (int) (invln2*x+halF[xsb]);
		t  = k;
		hi = x - t*ln2HI[0];	/* t*ln2HI is exact here */
		lo = t*ln2LO[0];
	    }
	    x  = hi - lo;
	}
	else {
	    k = 0;
	}


    /* x is now in primary range */
	t  = x*x;
	c  = x - t*(P1+t*(P2+t*(P3+t*(P4+t*P5))));
	if(k==0) 	return one-((x*c)/(c-2.0)-x);
	else 		y = one-((lo-(x*c)/(2.0-c))-hi);
	if(k >= -1021) {
	    //__HI(y) += (k<<20);	/* add k to y's exponent */
	    y = __HI__PLUS(y, (k<<20));	/* add k to y's exponent */
	    return y;
	} else {
	    //__HI(y) += ((k+1000)<<20);/* add k to y's exponent */
	    y = __HI__PLUS(y, (k+1000)<<20);/* add k to y's exponent */
	    return y*twom1000;
	}
    }

    private static final double ln2_hi  =  6.93147180369123816490e-01;  /* 3fe62e42 fee00000 */
    private static final double ln2_lo  =  1.90821492927058770002e-10;  /* 3dea39ef 35793c76 */
    private static final double Lg1     = 6.666666666666735130e-01;     /* 3FE55555 55555593 */
    private static final double Lg2     = 3.999999999940941908e-01;     /* 3FD99999 9997FA04 */
    private static final double Lg3     = 2.857142874366239149e-01;     /* 3FD24924 94229359 */
    private static final double Lg4     = 2.222219843214978396e-01;     /* 3FCC71C5 1D8E78AF */
    private static final double Lg5     = 1.818357216161805012e-01;     /* 3FC74664 96CB03DE */
    private static final double Lg6     = 1.531383769920937332e-01;     /* 3FC39A09 D078C69F */
    private static final double Lg7     = 1.479819860511658591e-01;     /* 3FC2F112 DF3E5244 */

    public static double log(double x) {
        if (useStrict) return StrictMath.log(x);                            /*ibm@99559*/
        double hfsq,f,s,z,R,w,t1,t2,dk;
        int k,hx,i,j;
        //unsigned lx;
        int lx;
        hx = __HI(x);           /* high word of x */
        lx = __LO(x);           /* low  word of x */

        k=0;
        if (hx < 0x00100000) {                  /* x < 2**-1022  */
            if (((hx&0x7fffffff)|lx)==0)
                return -two54/zero;             /* log(+-0)=-inf */
            if (hx<0) return (x-x)/zero;        /* log(-#) = NaN */
            k -= 54; x *= two54; /* subnormal number, scale up x */
            hx = __HI(x);               /* high word of x */
        }
        if (hx >= 0x7ff00000) return x+x;
        k += (hx>>20)-1023;
        hx &= 0x000fffff;
        i = (hx+0x95f64)&0x100000;
        //__HI(x) = hx|(i^0x3ff00000);    /* normalize x or x/2 */
        x = __HI(x, hx|(i^0x3ff00000));    /* normalize x or x/2 */
        k += (i>>20);
        f = x-1.0;
        if((0x000fffff&(2+hx))<3) {     /* |f| < 2**-20 */
            if(f==zero) {
                if (k==0) return zero;
                else {dk=(double)k; return (dk*ln2_hi)+(dk*ln2_lo);} /*ibm@24433*/
            }
            R = f*f*(0.5-0.33333333333333333*f);
            if(k==0) return f-R; else {dk=(double)k;
                     return dk*ln2_hi-((R-dk*ln2_lo)-f);}
        }
        s = f/(2.0+f);
        dk = (double)k;
        z = s*s;
        i = hx-0x6147a;
        w = z*z;
        j = 0x6b851-hx;
        t1= w*(Lg2+w*(Lg4+w*Lg6));
        t2= z*(Lg1+w*(Lg3+w*(Lg5+w*Lg7)));
        i |= j;
        R = t2+t1;
        if(i>0) {
            hfsq=0.5*f*f;
            if(k==0) return f-(hfsq-s*(hfsq+R)); else
                     return dk*ln2_hi-((hfsq-(s*(hfsq+R)+dk*ln2_lo))-f);
        } else {
            if(k==0) return f-s*(f-R); else
                     return dk*ln2_hi-((s*(f-R)-dk*ln2_lo)-f);
        }
    }

    private static final double atanhi0 = 4.63647609000806093515e-01; /* atan(0.5)hi 0x3FDDAC67, 0x0561BB4F */
    private static final double atanlo0 = 2.26987774529616870924e-17; /* atan(0.5)lo 0x3C7A2B7F, 0x222F65E2 */

    private static final double atanhi[] = {
      4.63647609000806093515e-01, /* atan(0.5)hi 0x3FDDAC67, 0x0561BB4F */
      7.85398163397448278999e-01, /* atan(1.0)hi 0x3FE921FB, 0x54442D18 */
      9.82793723247329054082e-01, /* atan(1.5)hi 0x3FEF730B, 0xD281F69B */
      1.57079632679489655800e+00, /* atan(inf)hi 0x3FF921FB, 0x54442D18 */
    };

    private static final double atanlo[] = {
      2.26987774529616870924e-17, /* atan(0.5)lo 0x3C7A2B7F, 0x222F65E2 */
      3.06161699786838301793e-17, /* atan(1.0)lo 0x3C81A626, 0x33145C07 */
      1.39033110312309984516e-17, /* atan(1.5)lo 0x3C700788, 0x7AF0CBBD */
      6.12323399573676603587e-17, /* atan(inf)lo 0x3C91A626, 0x33145C07 */
    };

    private static final double aT[] = {
      3.33333333333329318027e-01, /* 0x3FD55555, 0x5555550D */
     -1.99999999998764832476e-01, /* 0xBFC99999, 0x9998EBC4 */
      1.42857142725034663711e-01, /* 0x3FC24924, 0x920083FF */
     -1.11111104054623557880e-01, /* 0xBFBC71C6, 0xFE231671 */
      9.09088713343650656196e-02, /* 0x3FB745CD, 0xC54C206E */
     -7.69187620504482999495e-02, /* 0xBFB3B0F2, 0xAF749A6D */
      6.66107313738753120669e-02, /* 0x3FB10D66, 0xA0D03D51 */
     -5.83357013379057348645e-02, /* 0xBFADDE2D, 0x52DEFD9A */
      4.97687799461593236017e-02, /* 0x3FA97B4B, 0x24760DEB */
     -3.65315727442169155270e-02, /* 0xBFA2B444, 0x2C6A6C2F */
      1.62858201153657823623e-02, /* 0x3F90AD3A, 0xE322DA11 */
    };	

    public static double atan(double x) {
        if (useStrict) return StrictMath.atan(x);                           /*ibm@99559*/
	double w,s1,s2,z;
	int ix,hx;

	hx = __HI(x);
	ix = hx&0x7fffffff;
	if (ix >= 0x3e200000) return atan_full(x);

	if(huge+x>one) return x;	/* raise inexact */

	z = x*x;
	w = z*z;
    /* break sum from i=0 to 10 aT[i]z**(i+1) into odd and even poly */
	s1 = z*(aT[0]+w*(aT[2]+w*(aT[4]+w*(aT[6]+w*(aT[8]+w*aT[10])))));
	s2 = w*(aT[1]+w*(aT[3]+w*(aT[5]+w*(aT[7]+w*aT[9]))));
	
	return x - x*(s1+s2);
    }

    private static double atan_full(double x) {
	double w,s1,s2,z;
	int ix,hx,id;

	hx = __HI(x);
	ix = hx&0x7fffffff;
	if(ix>=0x44100000) {	/* if |x| >= 2^66 */
	    //System.out.println("1");
	    if(ix>0x7ff00000||
		(ix==0x7ff00000&&(__LO(x)!=0)))
		return x+x;		/* NaN */
	    if(hx>0) return  atanhi[3]+atanlo[3];
	    else     return -atanhi[3]-atanlo[3];
	} if (ix < 0x3fdc0000) {	/* |x| < 0.4375 */
	    //System.out.println("2");
	    if (ix < 0x3e200000) {	/* |x| < 2^-29 */
		if(huge+x>one) return x;	/* raise inexact */
	    }
	    id = -1;
	} else {
	    //x = fabs(x);
	    x = abs(x);
	    if (ix < 0x3ff30000) {		/* |x| < 1.1875 */
		//System.out.println("3");
		if (ix < 0x3fe60000) {	/* 7/16 <=|x|<11/16 */
		    id = 0; x = (2.0*x-one)/(2.0+x);
		} else {			/* 11/16<=|x|< 19/16 */
		    id = 1; x  = (x-one)/(x+one);
		}
	    } else {
		if (ix < 0x40038000) {	/* |x| < 2.4375 */
		    //System.out.println("4");
		    id = 2; x  = (x-1.5)/(one+1.5*x);
		} else {			/* 2.4375 <= |x| < 2^66 */
		    //System.out.println("5");
		    id = 3; x  = -1.0/x;
		}
	    }
	}
    /* end of argument reduction */
	z = x*x;
	w = z*z;
    /* break sum from i=0 to 10 aT[i]z**(i+1) into odd and even poly */
	s1 = z*(aT[0]+w*(aT[2]+w*(aT[4]+w*(aT[6]+w*(aT[8]+w*aT[10])))));
	s2 = w*(aT[1]+w*(aT[3]+w*(aT[5]+w*(aT[7]+w*aT[9]))));
	if (id<0) return x - x*(s1+s2);
	else {
	    z = atanhi[id] - ((x*(s1+s2) - atanlo[id]) - x);
	    return (hx<0)? -z:z;
	}
    }
    
    private static final double
    pio2_hi =  1.57079632679489655800e+00, /* 0x3FF921FB, 0x54442D18 */
    pio2_lo =  6.12323399573676603587e-17, /* 0x3C91A626, 0x33145C07 */
    pio4_hi =  7.85398163397448278999e-01, /* 0x3FE921FB, 0x54442D18 */
        /* coefficient for R(x^2) */
    pS0 =  1.66666666666666657415e-01, /* 0x3FC55555, 0x55555555 */
    pS1 = -3.25565818622400915405e-01, /* 0xBFD4D612, 0x03EB6F7D */
    pS2 =  2.01212532134862925881e-01, /* 0x3FC9C155, 0x0E884455 */
    pS3 = -4.00555345006794114027e-02, /* 0xBFA48228, 0xB5688F3B */
    pS4 =  7.91534994289814532176e-04, /* 0x3F49EFE0, 0x7501B288 */
    pS5 =  3.47933107596021167570e-05, /* 0x3F023DE1, 0x0DFDF709 */
    qS1 = -2.40339491173441421878e+00, /* 0xC0033A27, 0x1C8A2D4B */
    qS2 =  2.02094576023350569471e+00, /* 0x40002AE5, 0x9C598AC8 */
    qS3 = -6.88283971605453293030e-01, /* 0xBFE6066C, 0x1B8D0159 */
    qS4 =  7.70381505559019352791e-02; /* 0x3FB3B8C5, 0xB12E9282 */

    public static double asin(double x) {
        if (useStrict) return StrictMath.asin(x);                          /*ibm@99559*/
        double t=0,w,p,q,c,r,s;
        int hx,ix;

        hx = __HI(x);
        ix = hx&0x7fffffff;
        if(ix>= 0x3ff00000) {           /* |x|>= 1 */
            if(((ix-0x3ff00000)|__LO(x))==0)
                    /* asin(1)=+-pi/2 with inexact */
                return x*pio2_hi+x*pio2_lo;
            return (x+(-x))/(x+(-x));           /* ibm@7221 *//* asin(|x|>1) is NaN */
            /* return (x+(-x))/(x-x);           ibm@7221 *//* asin(|x|>1) is NaN */
        } else if (ix<0x3fe00000) {     /* |x|<0.5 */
            if(ix<0x3e400000) {         /* if |x| < 2**-27 */
                if(huge+x>one) return x;/* return x with inexact if x!=0*/
            } else
                t = x*x;
                p = t*(pS0+t*(pS1+t*(pS2+t*(pS3+t*(pS4+t*pS5)))));
                q = one+t*(qS1+t*(qS2+t*(qS3+t*qS4)));
                w = p/q;
                return x+(x*w); /* ibm@7221 */
                /* return x+x*w; ibm@7221 */
        }
        /* 1> |x|>= 0.5 */
        w = one-abs(x);
        t = w*0.5;
        p = t*(pS0+t*(pS1+t*(pS2+t*(pS3+t*(pS4+t*pS5)))));
        q = one+t*(qS1+t*(qS2+t*(qS3+t*qS4)));
        //s = sqrt(t);
        s = Math.sqrt(t);
        if(ix>=0x3FEF3333) {    /* if |x| > 0.975 */
            w = p/q;
            t = pio2_hi-(2.0*(s+(s*w))-pio2_lo);                   /*IBM@15563*/
        } else {
            w  = s;
            //__LO(w) = 0;
            w = __HI__LO(__HI(w), 0);
            c  = (t-w*w)/(s+w);
            r  = p/q;
            p  = 2.0*s*r-(pio2_lo-2.0*c);
            q  = pio4_hi-2.0*w;
            t  = pio4_hi-(p-q);
        }
        if(hx>0) return t; else return -t;
    }
    
    public static double acos(double x) {
        if (useStrict) return StrictMath.acos(x);                           /*ibm@99559*/
	double z,p,q,r,w,s,c,df;
	int hx,ix;
	hx = __HI(x);
	ix = hx&0x7fffffff;
	if(ix>=0x3ff00000) {	/* |x| >= 1 */
	    if(((ix-0x3ff00000)|__LO(x))==0) {	/* |x|==1 */
		if(hx>0) return 0.0;		/* acos(1) = 0  */
		else return pi+2.0*pio2_lo;	/* acos(-1)= pi */
	    }
	    
	    return (x+(-x))/(x+(-x));    /* ibm@7948 */
	}
	
	if(ix<0x3fe00000) {	/* |x| < 0.5 */
	    if(ix<=0x3c600000) return pio2_hi+pio2_lo;/*if|x|<2**-57*/
	    z = x*x;
	    p = z*(pS0+z*(pS1+z*(pS2+z*(pS3+z*(pS4+z*pS5)))));
	    q = one+z*(qS1+z*(qS2+z*(qS3+z*qS4)));
	    r = p/q;
	    return pio2_hi - (x - (pio2_lo-x*r));
	} else  if (hx<0) {		/* x < -0.5 */
	    z = (one+x)*0.5;
	    p = z*(pS0+z*(pS1+z*(pS2+z*(pS3+z*(pS4+z*pS5)))));
	    q = one+z*(qS1+z*(qS2+z*(qS3+z*qS4)));
	    //s = sqrt(z);
	    s = Math.sqrt(z);
	    r = p/q;
	    w = r*s-pio2_lo;
	    return pi - 2.0*(s+w);
	} else {			/* x > 0.5 */
	    z = (one-x)*0.5;
	    //s = sqrt(z);
	    s = Math.sqrt(z);
	    df = s;
	    //__LO(df) = 0;
	    df = __HI__LO(__HI(df), 0);
	    c  = (z-df*df)/(s+df);
	    p = z*(pS0+z*(pS1+z*(pS2+z*(pS3+z*(pS4+z*pS5)))));
	    q = one+z*(qS1+z*(qS2+z*(qS3+z*qS4)));
	    r = p/q;
	    w = r*s+c;
	    return 2.0*(df+w);
	}
    }    
    
    public static double atan2(double y, double x) {
        if (useStrict) return StrictMath.atan2(y,x);                        /*ibm@99559*/
	double z;
	int k,m,hx,hy,ix,iy;
	//unsigned lx,ly;
	int lx,ly;

	hx = __HI(x); ix = hx&0x7fffffff;
	lx = __LO(x);
	hy = __HI(y); iy = hy&0x7fffffff;
	ly = __LO(y);
	//if(((ix|((lx|(0-lx))>>31))>0x7ff00000)||
	//   ((iy|((ly|(0-ly))>>31))>0x7ff00000))	/* x or y is NaN */
	if(((long)(ix|((lx|(0-lx))>>>31))>0x7ff00000)||
	   ((long)(iy|((ly|(0-ly))>>>31))>0x7ff00000))	/* x or y is NaN */
	   return x+y;
	if(((hx-0x3ff00000)|lx)==0) return atan(y);   /* x=1.0 */
	m = ((hy>>31)&1)|((hx>>30)&2);	/* 2*sign(x)+sign(y) */

    /* when y = 0 */
	if((iy|ly)==0) {
	    switch(m) {
		case 0:
		case 1: return y; 	/* atan(+-0,+anything)=+-0 */
		case 2: return  pi+tiny;/* atan(+0,-anything) = pi */
		case 3: return -pi-tiny;/* atan(-0,-anything) =-pi */
	    }
	}
    /* when x = 0 */
	if((ix|lx)==0) return (hy<0)?  -pi_o_2-tiny: pi_o_2+tiny;

    /* when x is INF */
	if(ix==0x7ff00000) {
	    if(iy==0x7ff00000) {
		switch(m) {
		    case 0: return  pi_o_4+tiny;/* atan(+INF,+INF) */
		    case 1: return -pi_o_4-tiny;/* atan(-INF,+INF) */
		    case 2: return  3.0*pi_o_4+tiny;/*atan(+INF,-INF)*/
		    case 3: return -3.0*pi_o_4-tiny;/*atan(-INF,-INF)*/
		}
	    } else {
		switch(m) {
		    case 0: return  zero  ;	/* atan(+...,+INF) */
		    case 1: return -zero  ;	/* atan(-...,+INF) */
		    case 2: return  pi+tiny  ;	/* atan(+...,-INF) */
		    case 3: return -pi-tiny  ;	/* atan(-...,-INF) */
		}
	    }
	}
    /* when y is INF */
	if(iy==0x7ff00000) return (hy<0)? -pi_o_2-tiny: pi_o_2+tiny;

    /* compute y/x */
	k = (iy-ix)>>20;
	if(k > 60) z=pi_o_2+0.5*pi_lo; 	/* |y/x| >  2**60 */
	else if(hx<0&&k<-60) z=0.0; 	/* |y|/x < -2**60 */
	//else z=atan(fabs(y/x));		/* safe to do y/x */
	else z=atan(abs(y/x));		/* safe to do y/x */
	switch (m) {
	    case 0: return       z  ;	/* atan(+,+) */
	    //case 1: __HI(z) ^= 0x80000000;
	    case 1: z = __HI__LO(__HI(z)^0x80000000, __LO(z));
		    return       z  ;	/* atan(-,+) */
	    case 2: return  pi-(z-pi_lo);/* atan(+,-) */
	    default: /* case 3 */
	    	    return  (z-pi_lo)-pi;/* atan(-,-) */
	}
    }    

    private static final double
       TWO52[]={
       4.50359962737049600000e+15, /* 0x43300000, 0x00000000 */
      -4.50359962737049600000e+15, /* 0xC3300000, 0x00000000 */
    };

    /**
     * Returns the <code>double</code> value that is closest in value
     * to the argument and is equal to a mathematical integer. If two
     * <code>double</code> values that are mathematical integers are
     * equally close, the result is the integer value that is
     * even. Special cases:
     * <ul><li>If the argument value is already equal to a mathematical 
     * integer, then the result is the same as the argument. 
     * <li>If the argument is NaN or an infinity or positive zero or negative 
     * zero, then the result is the same as the argument.</ul>
     *
     * @param   a   a <code>double</code> value.
     * @return  the closest floating-point value to <code>a</code> that is
     *          equal to a mathematical integer.
     */
    public static double rint(double a) {
        return StrictMath.rint(a); // default impl. delegates to StrictMath
    }

    public static double IEEEremainder(double x, double p) {
        if (useStrict) return StrictMath.IEEEremainder(x,p);                /*ibm@99559*/
	int hx,hp;
	//unsigned sx,lx,lp;
	int sx,lx,lp;
	double p_half;

	hx = __HI(x);		/* high word of x */
	lx = __LO(x);		/* low  word of x */
	hp = __HI(p);		/* high word of p */
	lp = __LO(p);		/* low  word of p */
	sx = hx&0x80000000;
	hp &= 0x7fffffff;
	hx &= 0x7fffffff;

    /* purge off exception values */
	if((hp|lp)==0) return (x*p)/(x*p); 	/* p = 0 */
	if((hx>=0x7ff00000)||			/* x not finite */
	  ((hp>=0x7ff00000)&&			/* p is NaN */
	  (((hp-0x7ff00000)|lp)!=0)))
	    return (x*p)/(x*p);

	//System.out.print("x: "+x);

	if (hp<=0x7fdfffff) x = __ieee754_fmod(x,p+p);	/* now x < 2p */
	if (((hx-hp)|(lx-lp))==0) return zero*x;
	//x  = fabs(x);

	x  = abs(x);
	//System.out.print("x: "+x);
	//p  = fabs(p);
	p  = abs(p);
	if (hp<0x00200000) {
	    if(x+x>p) {
		x-=p;
		if(x+x>=p) x -= p;
	    }
	} else {
	    p_half = 0.5*p;
	    if(x>p_half) {
		x-=p;
		if(x>=p_half) x -= p;
	    }

	   // System.out.println("** x:"+x +" p:"+p);

	}
	//__HI(x) ^= sx;
	x = __HI__LO(__HI(x)^sx, __LO(x));
	return x;
}

/*
 * __ieee754_fmod(x,y)
 * Return x mod y in exact arithmetic
 * Method: shift and subtract
 */

    private static final double Zero[] = {0.0, -0.0,};
    private static double __ieee754_fmod(double x, double y) {
	int n,hx,hy,hz,ix,iy,sx,i;
	//unsigned lx,ly,lz;
	int lx,ly,lz;

	hx = __HI(x);		/* high word of x */
	lx = __LO(x);		/* low  word of x */
	hy = __HI(y);		/* high word of y */
	ly = __LO(y);		/* low  word of y */
	sx = hx&0x80000000;		/* sign of x */
	hx ^=sx;		/* |x| */
	hy &= 0x7fffffff;	/* |y| */

    /* purge off exception values */
	if((hy|ly)==0||(hx>=0x7ff00000)||	/* y=0,or x not finite */
	  //((hy|((ly|(0-ly))>>31))>0x7ff00000))	/* or y is NaN */
	  ((hy|((ly|(0-ly))>>>31))>0x7ff00000))	/* or y is NaN */
	    return (x*y)/(x*y);
	if(hx<=hy) {
	    //if((hx<hy)||(lx<ly)) return x;	/* |x|<|y| return x */
	    if((hx<hy)||((lx&0xffffffffL)<(ly&0xffffffffL))) return x;	/* |x|<|y| return x */
	    if(lx==ly)
		//return Zero[(unsigned)sx>>31];	/* |x|=|y| return x*0*/
		return Zero[sx>>>31];	/* |x|=|y| return x*0*/
	}

    /* determine ix = ilogb(x) */
	if(hx<0x00100000) {	/* subnormal x */
	    if(hx==0) {
		for (ix = -1043, i=lx; i>0; i<<=1) ix -=1;
	    } else {
		for (ix = -1022,i=(hx<<11); i>0; i<<=1) ix -=1;
	    }
	} else ix = (hx>>20)-1023;

    /* determine iy = ilogb(y) */
	if(hy<0x00100000) {	/* subnormal y */
	    if(hy==0) {
		for (iy = -1043, i=ly; i>0; i<<=1) iy -=1;
	    } else {
		for (iy = -1022,i=(hy<<11); i>0; i<<=1) iy -=1;
	    }
	} else iy = (hy>>20)-1023;

    /* set up {hx,lx}, {hy,ly} and align y to x */
	if(ix >= -1022)
	    hx = 0x00100000|(0x000fffff&hx);
	else {		/* subnormal x, shift x to normal */
	    n = -1022-ix;
	    if(n<=31) {
	        //hx = (hx<<n)|(lx>>(32-n));
	        hx = (hx<<n)|(lx>>>(32-n));
	        lx <<= n;
	    } else {
		hx = lx<<(n-32);
		lx = 0;
	    }
	}

	if(iy >= -1022)
	    hy = 0x00100000|(0x000fffff&hy);
	else {		/* subnormal y, shift y to normal */
	    n = -1022-iy;
	    if(n<=31) {
	        //hy = (hy<<n)|(ly>>(32-n));
	        hy = (hy<<n)|(ly>>>(32-n));
	        ly <<= n;
	    } else {
		hy = ly<<(n-32);
		ly = 0;
	    }
	}

    /* fix point fmod */
	n = ix - iy;
	//while(n--) {
	while(n--!=0) {
	    //hz=hx-hy;lz=lx-ly; if(lx<ly) hz -= 1;
	    hz=hx-hy;lz=lx-ly; if((lx&0xffffffffL)<(ly&0xffffffffL)) hz -= 1;
	    if(hz<0){
	    	//hx = hx+hx+(lx>>31); 
	    	hx = hx+hx+(lx>>>31); 
	    	lx = lx+lx;
	    }
	    else {
	    	if((hz|lz)==0) 		/* return sign(x)*0 */
		    //return Zero[(unsigned)sx>>31];
		    return Zero[sx>>>31];
	    	//hx = hz+hz+(lz>>31); lx = lz+lz;
	    	hx = hz+hz+(lz>>>31); lx = lz+lz;
	    }
	}
	//hz=hx-hy;lz=lx-ly; if(lx<ly) hz -= 1;
	hz=hx-hy;lz=lx-ly; if((lx&0xffffffffL)<(ly&0xffffffffL)) hz -= 1;
	if(hz>=0) {hx=hz;lx=lz;}

	//System.out.println(" >hx: " + hx + ", lx: " + lx + ", hy: " + hy +  ", ly: " + ly + ", hz: " + hz +  ", lz: " + lz +"< ");

    /* convert back to floating value and restore the sign */
	if((hx|lx)==0) 			/* return sign(x)*0 */
	    //return Zero[(unsigned)sx>>31];
	    return Zero[sx>>>31];
	while(hx<0x00100000) {		/* normalize x */
	    //hx = hx+hx+(lx>>31); lx = lx+lx;
	    hx = hx+hx+(lx>>>31); lx = lx+lx;
	    iy -= 1;
	}
	if(iy>= -1022) {	/* normalize output */
	    hx = ((hx-0x00100000)|((iy+1023)<<20));
	    //__HI(x) = hx|sx;
	    //__LO(x) = lx;
	    x = __HI__LO(hx|sx, lx);
	} else {		/* subnormal output */
	    n = -1022 - iy;
	    if(n<=20) {
		//lx = (lx>>n)|((unsigned)hx<<(32-n));
		lx = (lx>>>n)|(hx<<(32-n));
		hx >>= n;
	    } else if (n<=31) {
		//lx = (hx<<(32-n))|(lx>>n); hx = sx;
		lx = (hx<<(32-n))|(lx>>>n); hx = sx;
	    } else {
		lx = hx>>(n-32); hx = sx;
	    }
	    //__HI(x) = hx|sx;
	    //__LO(x) = lx;
	    x = __HI__LO(hx|sx, lx);
	    x *= one;		/* create necessary signal */
	}
	return x;		/* exact output */
    }

// **** HELPER FUNCTIONS ****


/* __ieee754_rem_pio2(x,y)
 *
 * return the remainder of x rem pi/2 in y[0]+y[1]
 * use __kernel_rem_pio2()
 */


/*
 * Table of constants for 2/pi, 396 Hex digits (476 decimal) of 2/pi
 */
private static final int two_over_pi[] = {
0xA2F983, 0x6E4E44, 0x1529FC, 0x2757D1, 0xF534DD, 0xC0DB62,
0x95993C, 0x439041, 0xFE5163, 0xABDEBB, 0xC561B7, 0x246E3A,
0x424DD2, 0xE00649, 0x2EEA09, 0xD1921C, 0xFE1DEB, 0x1CB129,
0xA73EE8, 0x8235F5, 0x2EBB44, 0x84E99C, 0x7026B4, 0x5F7E41,
0x3991D6, 0x398353, 0x39F49C, 0x845F8B, 0xBDF928, 0x3B1FF8,
0x97FFDE, 0x05980F, 0xEF2F11, 0x8B5A0A, 0x6D1F6D, 0x367ECF,
0x27CB09, 0xB74F46, 0x3F669E, 0x5FEA2D, 0x7527BA, 0xC7EBE5,
0xF17B3D, 0x0739F7, 0x8A5292, 0xEA6BFB, 0x5FB11F, 0x8D5D08,
0x560330, 0x46FC7B, 0x6BABF0, 0xCFBC20, 0x9AF436, 0x1DA9E3,
0x91615E, 0xE61B08, 0x659985, 0x5F14A0, 0x68408D, 0xFFD880,
0x4D7327, 0x310606, 0x1556CA, 0x73A8C9, 0x60E27B, 0xC08C6B,
};

private static final int npio2_hw[] = {
0x3FF921FB, 0x400921FB, 0x4012D97C, 0x401921FB, 0x401F6A7A, 0x4022D97C,
0x4025FDBB, 0x402921FB, 0x402C463A, 0x402F6A7A, 0x4031475C, 0x4032D97C,
0x40346B9C, 0x4035FDBB, 0x40378FDB, 0x403921FB, 0x403AB41B, 0x403C463A,
0x403DD85A, 0x403F6A7A, 0x40407E4C, 0x4041475C, 0x4042106C, 0x4042D97C,
0x4043A28C, 0x40446B9C, 0x404534AC, 0x4045FDBB, 0x4046C6CB, 0x40478FDB,
0x404858EB, 0x404921FB,
};

/*
 * invpio2:  53 bits of 2/pi
 * pio2_1:   first  33 bit of pi/2
 * pio2_1t:  pi/2 - pio2_1
 * pio2_2:   second 33 bit of pi/2
 * pio2_2t:  pi/2 - (pio2_1+pio2_2)
 * pio2_3:   third  33 bit of pi/2
 * pio2_3t:  pi/2 - (pio2_1+pio2_2+pio2_3)
 */

private static final double
//zero =  0.00000000000000000000e+00, /* 0x00000000, 0x00000000 */
//half =  5.00000000000000000000e-01, /* 0x3FE00000, 0x00000000 */
two24 =  1.67772160000000000000e+07, /* 0x41700000, 0x00000000 */
invpio2 =  6.36619772367581382433e-01, /* 0x3FE45F30, 0x6DC9C883 */
pio2_1  =  1.57079632673412561417e+00, /* 0x3FF921FB, 0x54400000 */
pio2_1t =  6.07710050650619224932e-11, /* 0x3DD0B461, 0x1A626331 */
pio2_2  =  6.07710050630396597660e-11, /* 0x3DD0B461, 0x1A600000 */
pio2_2t =  2.02226624879595063154e-21, /* 0x3BA3198A, 0x2E037073 */
pio2_3  =  2.02226624871116645580e-21, /* 0x3BA3198A, 0x2E000000 */
pio2_3t =  8.47842766036889956997e-32; /* 0x397B839A, 0x252049C1 */

    private static int __ieee754_rem_pio2(double x, double y[]) {
	double z,w,t,r,fn;
	double tx[] = new double[3];
	int e0,i,j,nx,n,ix,hx;

	hx = __HI(x);		/* high word of x */
	ix = hx&0x7fffffff;
	if(ix<=0x3fe921fb)   /* |x| ~<= pi/4 , no need for reduction */
	    {y[0] = x; y[1] = 0; return 0;}
	if(ix<0x4002d97c) {  /* |x| < 3pi/4, special case with n=+-1 */
	    if(hx>0) {
		z = x - pio2_1;
		if(ix!=0x3ff921fb) { 	/* 33+53 bit pi is good enough */
		    y[0] = z - pio2_1t;
		    y[1] = (z-y[0])-pio2_1t;
		} else {		/* near pi/2, use 33+33+53 bit pi */
		    z -= pio2_2;
		    y[0] = z - pio2_2t;
		    y[1] = (z-y[0])-pio2_2t;
		}
		return 1;
	    } else {	/* negative x */
		z = x + pio2_1;
		if(ix!=0x3ff921fb) { 	/* 33+53 bit pi is good enough */
		    y[0] = z + pio2_1t;
		    y[1] = (z-y[0])+pio2_1t;
		} else {		/* near pi/2, use 33+33+53 bit pi */
		    z += pio2_2;
		    y[0] = z + pio2_2t;
		    y[1] = (z-y[0])+pio2_2t;
		}
		return -1;
	    }
	}
	if(ix<=0x413921fb) { /* |x| ~<= 2^19*(pi/2), medium size */
	    t  = abs(x);
	    n  = (int) (t*invpio2+half);
	    fn = (double)n;
	    r  = t-fn*pio2_1;
	    w  = fn*pio2_1t;	/* 1st round good to 85 bit */
	    if(n<32&&ix!=npio2_hw[n-1]) {
		y[0] = r-w;	/* quick check no cancellation */
	    } else {
	        j  = ix>>20;
	        y[0] = r-w;
	        i = j-(((__HI(y[0]))>>20)&0x7ff);
	        if(i>16) {  /* 2nd iteration needed, good to 118 */
		    t  = r;
		    w  = fn*pio2_2;
		    r  = t-w;
		    w  = fn*pio2_2t-((t-r)-w);
		    y[0] = r-w;
		    i = j-(((__HI(y[0]))>>20)&0x7ff);
		    if(i>49)  {	/* 3rd iteration need, 151 bits acc */
		    	t  = r;	/* will cover all possible cases */
		    	w  = fn*pio2_3;
		    	r  = t-w;
		    	w  = fn*pio2_3t-((t-r)-w);
		    	y[0] = r-w;
		    }
		}
	    }
	    y[1] = (r-y[0])-w;
	    if(hx<0) 	{y[0] = -y[0]; y[1] = -y[1]; return -n;}
	    else	 return n;
	}
    /*
     * all other (large) arguments
     */
	if(ix>=0x7ff00000) {		/* x is inf or NaN */
	    y[0]=y[1]=x-x; return 0;
	}
    /* set z = scalbn(|x|,ilogb(x)-23) */
	//__LO(z) = __LO(x);
	e0 	= (ix>>20)-1046;	/* e0 = ilogb(z)-23; */
	//__HI(z) = ix - (e0<<20);
        z = __HI__LO(ix - (e0<<20), __LO(x));
	for(i=0;i<2;i++) {
		tx[i] = (double)((int)(z));
		z     = (z-tx[i])*two24;
	}
	tx[2] = z;
	nx = 3;
	while(tx[nx-1]==zero) nx--;	/* skip zero term */
	n  =  __kernel_rem_pio2(tx,y,e0,nx,2,two_over_pi);
	if(hx<0) {y[0] = -y[0]; y[1] = -y[1]; return -n;}
	return n;
    }

/*
 * __kernel_rem_pio2(x,y,e0,nx,prec,ipio2)
 * double x[],y[]; int e0,nx,prec; int ipio2[];
 *
 * __kernel_rem_pio2 return the last three digits of N with
 *		y = x - N*pi/2
 * so that |y| < pi/2.
 *
 * The method is to compute the integer (mod 8) and fraction parts of
 * (2/pi)*x without doing the full multiplication. In general we
 * skip the part of the product that are known to be a huge integer (
 * more accurately, = 0 mod 8 ). Thus the number of operations are
 * independent of the exponent of the input.
 *
 * (2/pi) is represented by an array of 24-bit integers in ipio2[].
 *
 * Input parameters:
 * 	x[]	The input value (must be positive) is broken into nx
 *		pieces of 24-bit integers in double precision format.
 *		x[i] will be the i-th 24 bit of x. The scaled exponent
 *		of x[0] is given in input parameter e0 (i.e., x[0]*2^e0
 *		match x's up to 24 bits.
 *
 *		Example of breaking a double positive z into x[0]+x[1]+x[2]:
 *			e0 = ilogb(z)-23
 *			z  = scalbn(z,-e0)
 *		for i = 0,1,2
 *			x[i] = floor(z)
 *			z    = (z-x[i])*2**24
 *
 *
 *	y[]	ouput result in an array of double precision numbers.
 *		The dimension of y[] is:
 *			24-bit  precision	1
 *			53-bit  precision	2
 *			64-bit  precision	2
 *			113-bit precision	3
 *		The actual value is the sum of them. Thus for 113-bit
 *		precison, one may have to do something like:
 *
 *		long double t,w,r_head, r_tail;
 *		t = (long double)y[2] + (long double)y[1];
 *		w = (long double)y[0];
 *		r_head = t+w;
 *		r_tail = w - (r_head - t);
 *
 *	e0	The exponent of x[0]
 *
 *	nx	dimension of x[]
 *
 *  	prec	an integer indicating the precision:
 *			0	24  bits (single)
 *			1	53  bits (double)
 *			2	64  bits (extended)
 *			3	113 bits (quad)
 *
 *	ipio2[]
 *		integer array, contains the (24*i)-th to (24*i+23)-th
 *		bit of 2/pi after binary point. The corresponding
 *		floating value is
 *
 *			ipio2[i] * 2^(-24(i+1)).
 *
 * External function:
 *	double scalbn(), floor();
 *
 *
 * Here is the description of some local variables:
 *
 * 	jk	jk+1 is the initial number of terms of ipio2[] needed
 *		in the computation. The recommended value is 2,3,4,
 *		6 for single, double, extended,and quad.
 *
 * 	jz	local integer variable indicating the number of
 *		terms of ipio2[] used.
 *
 *	jx	nx - 1
 *
 *	jv	index for pointing to the suitable ipio2[] for the
 *		computation. In general, we want
 *			( 2^e0*x[0] * ipio2[jv-1]*2^(-24jv) )/8
 *		is an integer. Thus
 *			e0-3-24*jv >= 0 or (e0-3)/24 >= jv
 *		Hence jv = max(0,(e0-3)/24).
 *
 *	jp	jp+1 is the number of terms in PIo2[] needed, jp = jk.
 *
 * 	q[]	double array with integral value, representing the
 *		24-bits chunk of the product of x and 2/pi.
 *
 *	q0	the corresponding exponent of q[0]. Note that the
 *		exponent for q[i] would be q0-24*i.
 *
 *	PIo2[]	double precision array, obtained by cutting pi/2
 *		into 24 bits chunks.
 *
 *	f[]	ipio2[] in floating point
 *
 *	iq[]	integer array by breaking up q[] in 24-bits chunk.
 *
 *	fq[]	final product of x*(2/pi) in fq[0],..,fq[jk]
 *
 *	ih	integer. If >0 it indicates q[] is >= 0.5, hence
 *		it also indicates the *sign* of the result.
 *
 */


/*
 * Constants:
 * The hexadecimal values are the intended ones for the following
 * constants. The decimal values may be used, provided that the
 * compiler will convert from decimal to binary accurately enough
 * to produce the hexadecimal values shown.
 */

private static final int init_jk[] = {2,3,4,6}; /* initial value for jk */
private static final double PIo2[] = {
  1.57079625129699707031e+00, /* 0x3FF921FB, 0x40000000 */
  7.54978941586159635335e-08, /* 0x3E74442D, 0x00000000 */
  5.39030252995776476554e-15, /* 0x3CF84698, 0x80000000 */
  3.28200341580791294123e-22, /* 0x3B78CC51, 0x60000000 */
  1.27065575308067607349e-29, /* 0x39F01B83, 0x80000000 */
  1.22933308981111328932e-36, /* 0x387A2520, 0x40000000 */
  2.73370053816464559624e-44, /* 0x36E38222, 0x80000000 */
  2.16741683877804819444e-51, /* 0x3569F31D, 0x00000000 */
};

private static final double
//zero   = 0.0,
//one    = 1.0,
//two24   =  1.67772160000000000000e+07, /* 0x41700000, 0x00000000 */
twon24  =  5.96046447753906250000e-08; /* 0x3E700000, 0x00000000 */

    private static int __kernel_rem_pio2(double x[], double y[], int e0, int nx, int prec, final int ipio2[]) {
	int jz,jx,jv,jp,jk,carry,n,iq[] = new int[20],i,j,k,m,q0,ih;
	double z,fw,f[] = new double[20],fq[] = new double[20],q[] = new double[20];

    /* initialize jk*/
	jk = init_jk[prec];
	jp = jk;

    /* determine jx,jv,q0, note that 3>q0 */
	jx =  nx-1;
	jv = (e0-3)/24; if(jv<0) jv=0;
	q0 =  e0-24*(jv+1);

    /* set up f[0] to f[jx+jk] where f[jx+jk] = ipio2[jv+jk] */
	j = jv-jx; m = jx+jk;
	for(i=0;i<=m;i++,j++) f[i] = (j<0)? zero : (double) ipio2[j];

    /* compute q[0],q[1],...q[jk] */
	for (i=0;i<=jk;i++) {
	    for(j=0,fw=0.0;j<=jx;j++) fw += x[j]*f[jx+i-j]; q[i] = fw;
	}

	jz = jk;
recompute: do {
    /* distill q[] into iq[] reversingly */
	for(i=0,j=jz,z=q[jz];j>0;i++,j--) {
	    fw    =  (double)((int)(twon24* z));
	    iq[i] =  (int)(z-two24*fw);
	    z     =  q[j-1]+fw;
	}

    /* compute n */
	z  = scalbn(z,q0);		/* actual value of z */
	z -= 8.0*floor(z*0.125);		/* trim off integer >= 8 */
	n  = (int) z;
	z -= (double)n;
	ih = 0;
	if(q0>0) {	/* need iq[jz-1] to determine n */
	    i  = (iq[jz-1]>>(24-q0)); n += i;
	    iq[jz-1] -= i<<(24-q0);
	    ih = iq[jz-1]>>(23-q0);
	}
	else if(q0==0) ih = iq[jz-1]>>23;
	else if(z>=0.5) ih=2;

	if(ih>0) {	/* q > 0.5 */
	    n += 1; carry = 0;
	    for(i=0;i<jz ;i++) {	/* compute 1-q */
		j = iq[i];
		if(carry==0) {
		    if(j!=0) {
			carry = 1; iq[i] = 0x1000000- j;
		    }
		} else  iq[i] = 0xffffff - j;
	    }
	    if(q0>0) {		/* rare case: chance is 1 in 12 */
	        switch(q0) {
	        case 1:
	    	   iq[jz-1] &= 0x7fffff; break;
	    	case 2:
	    	   iq[jz-1] &= 0x3fffff; break;
	        }
	    }
	    if(ih==2) {
		z = one - z;
		if(carry!=0) z -= scalbn(one,q0);
	    }
	}

    /* check if recomputation is needed */
	if(z==zero) {
	    j = 0;
	    for (i=jz-1;i>=jk;i--) j |= iq[i];
	    if(j==0) { /* need recomputation */
		for(k=1;iq[jk-k]==0;k++);   /* k = no. of terms needed */

		for(i=jz+1;i<=jz+k;i++) {   /* add q[jz+1] to q[jz+k] */
		    f[jx+i] = (double) ipio2[jv+i];
		    for(j=0,fw=0.0;j<=jx;j++) fw += x[j]*f[jx+i-j];
		    q[i] = fw;
		}
		jz += k;
		//goto recompute;
                continue recompute;
	    }
	}

        break; } while (true);
    /* chop off zero terms */
	if(z==0.0) {
	    jz -= 1; q0 -= 24;
	    while(iq[jz]==0) { jz--; q0-=24;}
	} else { /* break z into 24-bit if necessary */
	    z = scalbn(z,-q0);
	    if(z>=two24) {
		fw = (double)((int)(twon24*z));
		iq[jz] = (int)(z-two24*fw);
		jz += 1; q0 += 24;
		iq[jz] = (int) fw;
	    } else iq[jz] = (int) z ;
	}

    /* convert integer "bit" chunk to floating-point value */
	fw = scalbn(one,q0);
	for(i=jz;i>=0;i--) {
	    q[i] = fw*(double)iq[i]; fw*=twon24;
	}

    /* compute PIo2[0,...,jp]*q[jz,...,0] */
	for(i=jz;i>=0;i--) {
	    for(fw=0.0,k=0;k<=jp&&k<=jz-i;k++) fw += PIo2[k]*q[i+k];
	    fq[jz-i] = fw;
	}

    /* compress fq[] into y[] */
	switch(prec) {
	    case 0:
		fw = 0.0;
		for (i=jz;i>=0;i--) fw += fq[i];
		y[0] = (ih==0)? fw: -fw;
		break;
	    case 1:
	    case 2:
		fw = 0.0;
		for (i=jz;i>=0;i--) fw += fq[i];
		y[0] = (ih==0)? fw: -fw;
		fw = fq[0]-fw;
		for (i=1;i<=jz;i++) fw += fq[i];
		y[1] = (ih==0)? fw: -fw;
		break;
	    case 3:	/* painful */
		for (i=jz;i>0;i--) {
		    fw      = fq[i-1]+fq[i];
		    fq[i]  += fq[i-1]-fw;
		    fq[i-1] = fw;
		}
		for (i=jz;i>1;i--) {
		    fw      = fq[i-1]+fq[i];
		    fq[i]  += fq[i-1]-fw;
		    fq[i-1] = fw;
		}
		for (fw=0.0,i=jz;i>=2;i--) fw += fq[i];
		if(ih==0) {
		    y[0] =  fq[0]; y[1] =  fq[1]; y[2] =  fw;
		} else {
		    y[0] = -fq[0]; y[1] = -fq[1]; y[2] = -fw;
		}
	}
	return n&7;
}

/*
 * scalbn (double x, int n)
 * scalbn(x,n) returns x* 2**n  computed by  exponent
 * manipulation rather than by actually performing an
 * exponentiation or a multiplication.
 */

private static final double
two54   =  1.80143985094819840000e+16, /* 0x43500000, 0x00000000 */
twom54  =  5.55111512312578270212e-17, /* 0x3C900000, 0x00000000 */
//huge   = 1.0e+300,
huge   = 1.0e+300;
//tiny   = 1.0e-300;

    private static double scalbn (double x, int n) {
	int  k,hx,lx;
	hx = __HI(x);
	lx = __LO(x);
        k = (hx&0x7ff00000)>>20;		/* extract exponent */
        if (k==0) {				/* 0 or subnormal x */
            if ((lx|(hx&0x7fffffff))==0) return x; /* +-0 */
	    x *= two54;
	    hx = __HI(x);
	    k = ((hx&0x7ff00000)>>20) - 54;
            if (n< -50000) return tiny*x; 	/*underflow*/
	    }
        if (k==0x7ff) return x+x;		/* NaN or Inf */
        k = k+n;
        if (k >  0x7fe) return huge*copysign(huge,x); /* overflow  */
        if (k > 0) 				/* normal result */
	    //{__HI(x) = (hx&0x800fffff)|(k<<20); return x;}
	    {x = __HI__LO((hx&0x800fffff)|(k<<20), __LO(x)); return x;}
        if (k <= -54) {
            if (n > 50000) 	/* in case integer overflow in n+k */
		return huge*copysign(huge,x);	/*overflow*/
	    else return tiny*copysign(tiny,x); 	/*underflow*/
	}
        k += 54;				/* subnormal result */
        //__HI(x) = (hx&0x800fffff)|(k<<20);
        x = __HI__LO((hx&0x800fffff)|(k<<20), __LO(x));
        return x*twom54;
    }

/*
 * copysign(double x, double y)
 * copysign(x,y) returns a value with the magnitude of x and
 * with the sign bit of y.
 */

    private static double copysign(double x, double y) {
	x = __HI__LO((__HI(x)&0x7fffffff)|(__HI(y)&0x80000000), __LO(x));
        return x;
    }

}




























































































































































































































































































































































































































































































































































































































































































































































































































































































