
package java.lang;

import java.util.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

class J9VMInternals {
	private static final int UNINITIALIZED = 0;
	private static final int INITIALIZED = 1;
	private static final int FAILED = 2;
	private static final int UNVERIFIED = 3;

	// cannot create any instances in <clinit> in this special class
	private static Map exceptions;

	/*
	 * Called by the vm after everything else is initialized.
	 */
	private static void completeInitialization() {
		exceptions = new WeakHashMap();
		ClassLoader.completeInitialization();
		Thread.currentThread().completeInitialization();
	}

	private static native void sendClassPrepareEvent(Class clazz);

	/**
	 * Verify the specified Class using the VM byte code verifier.
	 *
	 * @param clazz the Class to verify.
	 *
	 * @throws VerifyError if the Class cannot be verified
	 */
	static void verify(Class clazz) {
		while (true) {
			switch (getInitStatus(clazz)) {
				case INITIALIZED:
				case UNINITIALIZED:
				case FAILED:
					return;

				case UNVERIFIED: {
					Class superclass;

					synchronized (clazz) {
						if (getInitStatus(clazz) != UNVERIFIED) break;
						setInitThread(clazz);
					}
					superclass = clazz.getSuperclass();
					try {
						// verify the superclass
						if (superclass != null)
							verify(superclass);
						// verify this class
						verifyImpl(clazz);
					} catch (Error e) {
						setInitStatus(clazz, UNVERIFIED);
						throw e;
					} catch (Throwable e) {
						setInitStatus(clazz, UNVERIFIED);
						throw new VerifyError(e.toString());
					}
					synchronized (clazz) {
						if (getInitThread(clazz) && ((getInitStatus(clazz) & 0x3) == UNVERIFIED)) {
							setInitStatus(clazz, UNINITIALIZED);
						}
					}
					return;
				}

				default: // INPROGRESS
					synchronized (clazz) {
						int status = getInitStatus(clazz);

						if (((status & ~0x3) == 0)) break;
						if ((status & 0x3) != UNVERIFIED) return;
						if (!getInitThread(clazz)) {
							try { clazz.wait(); } catch (InterruptedException e) {}
							break;
						}
					}

					try {
						verifyImpl(clazz);
					} catch (Error e) {
						setInitStatus(clazz, UNVERIFIED);
						throw e;
					} catch (Throwable e) {
						setInitStatus(clazz, UNVERIFIED);
						throw new VerifyError(e.toString());
					}
					synchronized (clazz) {
						if (getInitThread(clazz) && ((getInitStatus(clazz) & 0x3) == UNVERIFIED)) {
							setInitStatus(clazz, UNINITIALIZED);
						}
					}
			}
		}
	}

	private static native void verifyImpl(Class clazz) throws Throwable;

	/**
	 * Sent internally by the VM to initiatiate
	 * initialization of the receiver.  See chapter
	 * 2.17.5 of the JVM Specification (2nd ed)
	 *
	 * @throws		Throwable
	 */
	private static void initialize(Class clazz) throws Throwable {
		while (true) {
			switch (getInitStatus(clazz)) {
				case INITIALIZED:
					return;
				case UNVERIFIED:
					verify(clazz);
					break;
				case FAILED:
					NoClassDefFoundError notFound = new NoClassDefFoundError(clazz.getName() + " (initialization failure)"); //$NON-NLS-1$
					// if exceptions is null, we're initializing and running single threaded
					if (exceptions != null) {
						synchronized(exceptions) {
							Throwable reason = (Throwable)exceptions.get(clazz);
							if (reason != null) notFound.initCause(reason);
						}
					}
					throw notFound;
				case UNINITIALIZED: {
					Class superclass;

					synchronized (clazz) {
						if (getInitStatus(clazz) != UNINITIALIZED) break;
						setInitThread(clazz);
					}

					// initialize the superclass
					superclass = clazz.getSuperclass();
					if (superclass != null) {
						try {
							initialize(superclass);
						} catch (Error err) {
							setInitStatus(clazz, FAILED);
							// if exceptions is null, we're initializing and running single threaded
							if (exceptions == null)
								exceptions = new WeakHashMap();
							synchronized(exceptions) {
								Throwable cause = err;
								if (err instanceof ExceptionInInitializerError) {
									cause = ((ExceptionInInitializerError)err).getException();
								}
								exceptions.put(clazz, copyThrowable(cause));
							}
							throw err;
						}
					}

					sendClassPrepareEvent(clazz);

					// initialize this class
					try {
						/* When we are initializing the statics of the class
						 * we want to be in the correct memory space for the class loader.
						 * If the class loader does not have a memory space, then we want
						 * to initialize it in the base memory space.  In the situation where
						 * we are not in mutli-memory space mode, we will get null back when
						 * we request the memory space for the class loader, and we won't need
						 * to switch memory spaces.  If we have the system class loader then we
						 * want to initialize in the base memory space.  Their are problems treating
						 * the system class loader the same as others because not everything required
						 * by MemorySpace.getMemorySpaceForClassLoader has been initialized the first
						 * time the method is called.
						 */
						initializeImpl(clazz);
					} catch (Error err) {
						setInitStatus(clazz, FAILED);
						// if exceptions is null, we're initializing and running single threaded
						if (exceptions == null)
							exceptions = new WeakHashMap();
						synchronized(exceptions) {
							exceptions.put(clazz, copyThrowable(err));
						}
						throw err;
					} catch (Throwable t) {
						setInitStatus(clazz, FAILED);
						// if exceptions is null, we're initializing and running single threaded
						if (exceptions == null)
							exceptions = new WeakHashMap();
						synchronized(exceptions) {
							exceptions.put(clazz, copyThrowable(t));
						}
						throw new ExceptionInInitializerError(t);
					}

					setInitStatus(clazz, INITIALIZED);
					return;
				}

				default: // INPROGRESS
					synchronized (clazz) {
						int status = getInitStatus(clazz);

						if ((status & ~0x3) == 0) break;
						if ((status & 0x3) == UNINITIALIZED) {
							if (getInitThread(clazz)) return;
							try { clazz.wait(); } catch (InterruptedException e) {}
							break;
						}
					}
					verify(clazz);
			}
		}
	}

	/**
	 * Copy the specified Throwable, wrapping the stack trace for each
	 * Throwable. Check for loops so we don't go infinite.
	 *
	 * @param throwable the Throwable to copy
	 *
	 * @return a copy of the Throwable
	 */
	private static Throwable copyThrowable(Throwable throwable) {
		Throwable root = new Throwable(throwable.toString());
		root.setStackTrace(throwable.getStackTrace());
		Throwable parent = root;
		Throwable cause = throwable.getCause();
		HashMap found = new HashMap();
		found.put(throwable, throwable);
		while (cause != null && found.get(cause) == null) {
			found.put(cause, cause);
			Throwable child = new Throwable(cause.toString());
			child.setStackTrace(cause.getStackTrace());
			parent.initCause(child);
			parent = child;
			cause = cause.getCause();
		}
		return root;
	}

	/**
	 * Used to indicate the end of class initialization.
	 * Sets the initialization status and notifies any
	 * threads which may have been waiting for
	 * initialization to complete
	 *
	 * @param		status
	 *					INITIALIZED (1)
	 *					FAILED (2)
	 *
	 */
	private static void setInitStatus(Class clazz, int status) {
		synchronized(clazz) {
			setInitStatusImpl(clazz, status);
			clazz.notifyAll();
		}
	}

	/**
	 * Answers the receiver's initialization status
	 *
	 * @return		status
	 *					UNINITIALIZED (0)
	 *					INITIALIZED (1)
	 *					FAILED (2)
	 *					INPROGRESS (0xnnnnnnn[048C])
	 *
	 */
	private static native int getInitStatus(Class clazz);

	/**
	 * Set the receiver's initialization status
	 *
	 * @param		status
	 *					INITIALIZED (1)
	 *					FAILED (2)
	 *
	 */
	private static native void setInitStatusImpl(Class clazz, int status);

	/**
	 * Run the receiver's <clinit> method and initialize
	 * any static variables
	 *
	 * @throws		Throwable Any exception may be thrown
	 */
	private static native void initializeImpl(Class clazz) throws Throwable;

	/**
	 * Answers true if the current thread is currently
	 * initializing this class
	 *
	 * @return		true if the current thread is initializing the receiver
	 *
	 */
	private static native boolean getInitThread(Class clazz);

	/**
	 * Set the receiver's initialize status to be 'in
	 * progress' and save the current thread as the
	 * initializing thread.
	 */
	private static native void setInitThread(Class clazz);

	/**
	 * Private method to be called by the VM after a Threads dies and throws ThreadDeath
	 * It has to <code>notifyAll()</code> so that <code>join</code> can work properly.
	 * However, it has to be done when the Thread is "thought of" as being dead by other
	 * observer Threads (<code>isAlive()</code> has to return false for the Thread
	 * running this method or <code>join</code> may never return)
	 *
	 * @author		OTI
	 * @version		initial
	 */
	private static void threadCleanup(Thread thread) {
		// don't synchronize the remove! Otherwise deadlock may occur
		try {
			thread.threadGroup.remove(thread);		// Leave the ThreadGroup. This is why remove can't be private
		}
		finally {

			thread.threadGroup = null;

			synchronized(thread.lock) {
				thread.threadRef = Thread.NO_REF;				// So that isAlive() can work
			}
			synchronized(thread) {
				thread.notifyAll();
			}
		}
	}

	private static void checkPackageAccess(Class clazz) {
		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
			sm.checkPackageAccess(clazz.getPackageName());
		}
	}
}
