package java.lang;

import java.io.InputStream;
import java.security.ProtectionDomain;
import java.security.AllPermission;
import java.security.Permissions;
import java.lang.reflect.*;
import java.net.URL;

import com.ibm.oti.reflect.AnnotationHelper;
import java.lang.annotation.*;
import java.util.HashMap;
import java.util.Map;
import java.util.Collections;
import java.util.WeakHashMap;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import sun.reflect.generics.repository.ClassRepository;
import sun.reflect.generics.factory.CoreReflectionFactory;
import sun.reflect.generics.scope.ClassScope;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * An instance of class Class is the in-image representation
 * of a Java class. There are three basic types of Classes
 * <dl>
 * <dt><em>Classes representing object types (classes or interfaces)</em></dt>
 * <dd>These are Classes which represent the class of a
 *     simple instance as found in the class hierarchy.
 *     The name of one of these Classes is simply the
 *     fully qualified class name of the class or interface
 *     that it represents. Its <em>signature</em> is
 *     the letter "L", followed by its name, followed
 *     by a semi-colon (";").</dd>
 * <dt><em>Classes representing base types</em></dt>
 * <dd>These Classes represent the standard Java base types.
 *     Although it is not possible to create new instances
 *     of these Classes, they are still useful for providing
 *     reflection information, and as the component type
 *     of array classes. There is one of these Classes for
 *     each base type, and their signatures are:
 *     <ul>
 *     <li><code>B</code> representing the <code>byte</code> base type</li>
 *     <li><code>S</code> representing the <code>short</code> base type</li>
 *     <li><code>I</code> representing the <code>int</code> base type</li>
 *     <li><code>J</code> representing the <code>long</code> base type</li>
 *     <li><code>F</code> representing the <code>float</code> base type</li>
 *     <li><code>D</code> representing the <code>double</code> base type</li>
 *     <li><code>C</code> representing the <code>char</code> base type</li>
 *     <li><code>Z</code> representing the <code>boolean</code> base type</li>
 *     <li><code>V</code> representing void function return values</li>
 *     </ul>
 *     The name of a Class representing a base type
 *     is the keyword which is used to represent the
 *     type in Java source code (i.e. "int" for the
 *     <code>int</code> base type.</dd>
 * <dt><em>Classes representing array classes</em></dt>
 * <dd>These are Classes which represent the classes of
 *     Java arrays. There is one such Class for all array
 *     instances of a given arity (number of dimensions)
 *     and leaf component type. In this case, the name of the
 *     class is one or more left square brackets (one per
 *     dimension in the array) followed by the signature ofP
 *     the class representing the leaf component type, which
 *     can be either an object type or a base type. The
 *     signature of a Class representing an array type
 *     is the same as its name.</dd>
 * </dl>
 *
 * @author		OTI
 * @version		initial
 */
public final class Class<T> implements java.io.Serializable, GenericDeclaration, Type, AnnotatedElement {
	private static final long serialVersionUID = 3206093459760846163L;
	private static ProtectionDomain AllPermissionsPD;

	private static Map classNameMap = null;

	private static final int SYNTHETIC = 0x1000;
	private static final int ANNOTATION = 0x2000;
	private static final int ENUM = 0x4000;

	private static final int j9Version = 0x02020231;

	private static final long j9Config = 0x7363617200000000L;	// 'scar\0'

	private static Class[] EmptyParameters = new Class[0];

/**
 * Prevents this class from being instantiated. Instances
 * created by the virtual machine only.
 */
private Class() {}

private void checkMemberAccess(int type) {
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
		ClassLoader callerClassLoader = ClassLoader.getStackClassLoader(2);
		if (callerClassLoader != ClassLoader.systemClassLoader) {
			security.checkMemberAccess(this, type);
			String packageName = this.getPackageName();
			ClassLoader loader = getClassLoaderImpl();
			if (packageName != "" && callerClassLoader != loader &&
				!callerClassLoader.isAncestorOf(loader))
			{
				security.checkPackageAccess(packageName);
			}
		}
	}
}

/**
 * Answers a Class object which represents the class
 * named by the argument. The name should be the name
 * of a class as described in the class definition of
 * java.lang.Class, however Classes representing base
 * types can not be found using this method.
 *
 * @param		className	The name of the non-base type class to find
 * @return		the named Class
 * @throws		ClassNotFoundException If the class could not be found
 *
 * @see			java.lang.Class
 */
public static Class<?> forName(String className) throws ClassNotFoundException {
	ClassLoader defaultClassLoader = ClassLoader.callerClassLoader();
	return forNameImpl(className, true, defaultClassLoader);
}

/**
 * Answers a Class object which represents the class
 * named by the argument. The name should be the name
 * of a class as described in the class definition of
 * java.lang.Class, however Classes representing base
 * types can not be found using this method.
 * Security rules will be obeyed.
 *
 * @param		className			The name of the non-base type class to find
 * @param		initializeBoolean	A boolean indicating whether the class should be
 *									initialized
 * @param		classLoader			The classloader to use to load the class
 * @return		the named class.
 * @throws		ClassNotFoundException If the class could not be found
 *
 * @see			java.lang.Class
 */
public static Class<?> forName(String className, boolean initializeBoolean, ClassLoader classLoader)
	throws ClassNotFoundException
{
	/* perform security checks */
	if (null == classLoader) {
		ClassLoader callerClassLoader = ClassLoader.callerClassLoader();
		if (callerClassLoader != null) {
			/* only allowed if caller has RuntimePermission("getClassLoader") permission */
			SecurityManager security = System.getSecurityManager();
			if (null != security)
				security.checkPermission(RuntimePermission.permissionToGetClassLoader);
		}
	}
	return forNameImpl(className, initializeBoolean, classLoader);
}

/**
 * Answers a Class object which represents the class
 * named by the argument. The name should be the name
 * of a class as described in the class definition of
 * java.lang.Class, however Classes representing base
 * types can not be found using this method.
 *
 * @param		className			The name of the non-base type class to find
 * @param		initializeBoolean	A boolean indicating whether the class should be
 *									initialized
 * @param		classLoader			The classloader to use to load the class
 * @return		the named class.
 * @throws		ClassNotFoundException If the class could not be found
 *
 * @see			java.lang.Class
 */
private static native Class forNameImpl(String className,
                            boolean initializeBoolean,
                            ClassLoader classLoader)
	throws ClassNotFoundException;

/**
 * Answers an array containing all public class members
 * of the class which the receiver represents and its
 * superclasses and interfaces
 *
 * @return		the class' public class members
 * @throws		SecurityException If member access is not allowed
 *
 * @see			java.lang.Class
 */
public Class[] getClasses() {
	checkMemberAccess(Member.PUBLIC);

	java.util.Vector publicClasses = new java.util.Vector();
	Class current = this;
	Class[] classes;
	while(current != null) {
		classes = current.getDeclaredClassesImpl();
		for (int i = 0; i < classes.length; i++)
			if (Modifier.isPublic(classes[i].getModifiers()))
				publicClasses.addElement(classes[i]);
		current = current.getSuperclass();
	}
	classes = new Class[publicClasses.size()];
	publicClasses.copyInto(classes);
	return classes;
}

/**
 * Answers the classloader which was used to load the
 * class represented by the receiver. Answer null if the
 * class was loaded by the system class loader
 *
 * @return		the receiver's class loader or nil
 *
 * @see			java.lang.ClassLoader
 */
public ClassLoader getClassLoader() {
	ClassLoader loader = getClassLoaderImpl();
	SecurityManager security = System.getSecurityManager();
	if (security != null) {
		// No check for systemClassLoader, contrary to
		// spec but testing reveals this is correct behavior.
		if (loader == ClassLoader.systemClassLoader) return null;
		ClassLoader callersClassLoader = ClassLoader.callerClassLoader();
		if (callersClassLoader != null && callersClassLoader != loader
			&& !callersClassLoader.isAncestorOf(loader))
				security.checkPermission(
					RuntimePermission.permissionToGetClassLoader);
	}
	if (loader == ClassLoader.systemClassLoader) return null;
	return loader;
}

/**
 * Return the ClassLoader for this Class without doing any security
 * checks. The bootstrap ClassLoader is returned, unlike getClassLoader()
 * which returns null in place of the bootstrap ClassLoader.
 *
 * @return the ClassLoader
 *
 * @see ClassLoader#isSystemClassLoader()
 */
native ClassLoader getClassLoaderImpl();

/**
 * Answers a Class object which represents the receiver's
 * component type if the receiver represents an array type.
 * Otherwise answers nil. The component type of an array
 * type is the type of the elements of the array.
 *
 * @return		the component type of the receiver.
 *
 * @see			java.lang.Class
 */
public native Class<?> getComponentType();

private void throwNoSuchMethodException(String name, Class[] types) throws NoSuchMethodException {
	StringBuffer error = new StringBuffer();
	error.append(getName()).append('.').append(name).append('(');
	if (types.length > 0) {
		error.append(types[0] == null ? null : types[0].getName());
		for (int i=1; i<types.length; i++) {
			error.append(", ").append(types[i] == null ? null : types[i].getName());
		}
	}
	error.append(')');
	throw new NoSuchMethodException(error.toString());
}

/**
 * Answers a public Constructor object which represents the
 * constructor described by the arguments.
 *
 * @param		parameterTypes	the types of the arguments.
 * @return		the constructor described by the arguments.
 * @throws		NoSuchMethodException if the constructor could not be found.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getConstructors
 */
public Constructor<T> getConstructor(Class... parameterTypes) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.PUBLIC);

	// Handle the default constructor case upfront
	if(parameterTypes == null || parameterTypes.length == 0) {
		Constructor rc = getConstructorImpl(EmptyParameters, "()V");
		if (rc == null) throwNoSuchMethodException("<init>", EmptyParameters);
		return rc;
	}

	// Build a signature for the requested method.
	int total = 3;
	String[] sigs = new String[parameterTypes.length];
	for(int i = 0; i < parameterTypes.length; i++) {
		if (parameterTypes[i] != null) {
			sigs[i] = parameterTypes[i].getSignature();
			total += sigs[i].length();
		} else throwNoSuchMethodException("<init>", parameterTypes);
	}
	StringBuffer signature = new StringBuffer(total);
	signature.append('(');
	for(int i = 0; i < parameterTypes.length; i++)
		signature.append(sigs[i]);
	signature.append(")V");

	Constructor rc = getConstructorImpl((Class[])parameterTypes.clone(), signature.toString());
	if (rc == null) throwNoSuchMethodException("<init>", parameterTypes);
	return rc;
}

/**
 * Answers a public Constructor object which represents the
 * constructor described by the arguments.
 *
 * @param		parameterTypes	the types of the arguments.
 * @param		signature		the signature of the method.
 * @return		the constructor described by the arguments.
 *
 * @see			#getConstructors
 */
private native Constructor getConstructorImpl(Class parameterTypes[], String signature);

/**
 * Answers an array containing Constructor objects describing
 * all constructors which are visible from the current execution
 * context.
 *
 * @return		all visible constructors starting from the receiver.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getMethods
 */
public Constructor[] getConstructors() throws SecurityException {
	checkMemberAccess(Member.PUBLIC);
	return getConstructorsImpl();
}

/**
 * Answers an array containing Constructor objects describing
 * all constructors which are visible from the current execution
 * context.
 *
 * @return		all visible constructors starting from the receiver.
 *
 * @see			#getMethods
 */
private native Constructor[] getConstructorsImpl();

/**
 * Answers an array containing all class members of the class
 * which the receiver represents. Note that some of the fields
 * which are returned may not be visible in the current
 * execution context.
 *
 * @return		the class' class members
 * @throws		SecurityException if member access is not allowed
 *
 * @see			java.lang.Class
 */
public Class[] getDeclaredClasses() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredClassesImpl();
}

/**
 * Answers an array containing all class members of the class
 * which the receiver represents. Note that some of the fields
 * which are returned may not be visible in the current
 * execution context.
 *
 * @return		the class' class members
 *
 * @see			java.lang.Class
 */
private native Class[] getDeclaredClassesImpl();

/**
 * Answers a Constructor object which represents the
 * constructor described by the arguments.
 *
 * @param		parameterTypes	the types of the arguments.
 * @return		the constructor described by the arguments.
 * @throws		NoSuchMethodException if the constructor could not be found.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getConstructors
 */
public Constructor<T> getDeclaredConstructor(Class... parameterTypes) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.DECLARED);

	// Handle the default constructor case upfront
	if(parameterTypes == null || parameterTypes.length == 0) {
		Constructor rc = getDeclaredConstructorImpl(EmptyParameters, "()V");
		if (rc == null) throwNoSuchMethodException("<init>", EmptyParameters);
		return rc;
	}

	// Build a signature for the requested method.
	int total = 3;
	String[] sigs = new String[parameterTypes.length];
	for(int i = 0; i < parameterTypes.length; i++) {
		if (parameterTypes[i] != null) {
			sigs[i] = parameterTypes[i].getSignature();
			total += sigs[i].length();
		} else throwNoSuchMethodException("<init>", parameterTypes);
	}
	StringBuffer signature = new StringBuffer(total);
	signature.append('(');
	for(int i = 0; i < parameterTypes.length; i++)
		signature.append(sigs[i]);
	signature.append(")V");

	Constructor rc = getDeclaredConstructorImpl((Class[])parameterTypes.clone(), signature.toString());
	if (rc == null) throwNoSuchMethodException("<init>", parameterTypes);
	return rc;
}

/**
 * Answers a Constructor object which represents the
 * constructor described by the arguments.
 *
 * @param		parameterTypes	the types of the arguments.
 * @param		signature		the signature of the method.
 * @return		the constructor described by the arguments.
 *
 * @see			#getConstructors
 */
private native Constructor getDeclaredConstructorImpl(Class parameterTypes[], String signature);

/**
 * Answers an array containing Constructor objects describing
 * all constructor which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @return		the receiver's constructors.
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getMethods
 */
public Constructor[] getDeclaredConstructors() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredConstructorsImpl();
}

/**
 * Answers an array containing Constructor objects describing
 * all constructor which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @return		the receiver's constructors.
 *
 * @see			#getMethods
 */
private native Constructor[] getDeclaredConstructorsImpl();

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument. Note that the Constructor may not be
 * visible from the current execution context.
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException if the requested field could not be found
 * @throws		SecurityException if member access is not allowed
 *
 * @see			#getDeclaredFields
 */
public Field getDeclaredField(String name) throws NoSuchFieldException, SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredFieldImpl(name);
}

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument. Note that the Constructor may not be
 * visible from the current execution context.
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException If the given field does not exist
 *
 * @see			#getDeclaredFields
 */
private native Field getDeclaredFieldImpl(String name) throws NoSuchFieldException;

/**
 * Answers an array containing Field objects describing
 * all fields which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @return		the receiver's fields.
 * @throws		SecurityException If member access is not allowed
 *
 * @see			#getFields
 */
public Field[] getDeclaredFields() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredFieldsImpl();
}

/**
 * Answers an array containing Field objects describing
 * all fields which are defined by the receiver. Note that
 * some of the fields which are returned may not be visible
 * in the current execution context.
 *
 * @return		the receiver's fields.
 *
 * @see			#getFields
 */
private native Field[] getDeclaredFieldsImpl();

/**
 * Answers a Method object which represents the method
 * described by the arguments. Note that the associated
 * method may not be visible from the current execution
 * context.
 *
 * @param		name			the name of the method
 * @param		parameterTypes	the types of the arguments.
 * @return		the method described by the arguments.
 * @throws		NoSuchMethodException if the method could not be found.
 * @throws		SecurityException If member access is not allowed
 *
 * @see			#getMethods
 */
public Method getDeclaredMethod(String name, Class... parameterTypes) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.DECLARED);
	Method result, bestCandidate;
	int maxDepth;
	String strSig;
	Class[] params;

	// Handle the no parameter case upfront
	if(name == null || parameterTypes == null || parameterTypes.length == 0) {
		strSig = "()";
		params = EmptyParameters;
	} else {
		// Build a signature for the requested method.
		int total = 2;
		String[] sigs = new String[parameterTypes.length];
		for(int i = 0; i < parameterTypes.length; i++) {
			if (parameterTypes[i] != null) {
				sigs[i] = parameterTypes[i].getSignature();
				total += sigs[i].length();
			} else throwNoSuchMethodException(name, parameterTypes);
		}
		StringBuffer signature = new StringBuffer(total);
		signature.append('(');
		for(int i = 0; i < parameterTypes.length; i++)
			signature.append(sigs[i]);
		signature.append(')');

		strSig = signature.toString();
		params = (Class[])parameterTypes.clone();
	}
	result = getDeclaredMethodImpl(name, params, strSig, null);

	if (result == null) throwNoSuchMethodException(name, params);

	/* [PR 113003] The native is called repeatedly until it returns null,
	 * as each call returns another match if one exists.
	 * If more than one match is found, the code below selects the
	 * candidate method whose return type has the largest depth. The latter
	 * case is expected to occur only in certain JCK tests, as most Java
	 * compilers will refuse to produce a class file with multiple methods
	 * of the same name differing only in return type.
	 *
	 * Selecting by largest depth is one possible algorithm that satisfies the
	 * spec.
	 */
	bestCandidate = result;
	maxDepth = result.getReturnType().getClassDepth();
	while( true ) {
		result = getDeclaredMethodImpl( name, params, strSig, result );
		if( result == null ) {
			break;
		}
		int resultDepth = result.getReturnType().getClassDepth();
		if( resultDepth > maxDepth ) {
			bestCandidate = result;
			maxDepth = resultDepth;
		}
	}
	return bestCandidate;
}

/**
 * This native iterates over methods matching the provided name and signature
 * in the receiver class. The startingPoint parameter is passed the last
 * method returned (or null on the first use), and the native returns the next
 * matching method or null if there are no more matches.
 * Note that the associated method may not be visible from the
 * current execution context.
 *
 * @param		name				the name of the method
 * @param		parameterTypes		the types of the arguments.
 * @param		partialSignature	the signature of the method, without return type.
 * @param		startingPoint		the method to start searching after, or null to start at the beginning
 * @return		the next Method described by the arguments
 *
 * @see			#getMethods
 */
private native Method getDeclaredMethodImpl(String name, Class parameterTypes[], String partialSignature, Method startingPoint);

/**
 * Answers an array containing Method objects describing
 * all methods which are defined by the receiver. Note that
 * some of the methods which are returned may not be visible
 * in the current execution context.
 *
 * @throws		SecurityException	if member access is not allowed
 * @return		the receiver's methods.
 *
 * @see			#getMethods
 */
public Method[] getDeclaredMethods() throws SecurityException {
	checkMemberAccess(Member.DECLARED);
	return getDeclaredMethodsImpl();
}

/**
 * Answers an array containing Method objects describing
 * all methods which are defined by the receiver. Note that
 * some of the methods which are returned may not be visible
 * in the current execution context.
 *
 * @return		the receiver's methods.
 *
 * @see			#getMethods
 */
private native Method[] getDeclaredMethodsImpl();

/**
 * Answers the class which declared the class represented
 * by the receiver. This will return null if the receiver
 * is a member of another class.
 *
 * @return		the declaring class of the receiver.
 */
public Class<?> getDeclaringClass() {
	return getDeclaringClassImpl();
}

/**
 * Answers the class which declared the class represented
 * by the receiver. This will return null if the receiver
 * is a member of another class.
 *
 * @return		the declaring class of the receiver.
 */
private native Class getDeclaringClassImpl();

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument which must be visible from the current
 * execution context.
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException If the given field does not exist
 * @throws		SecurityException If access is denied
 *
 * @see			#getDeclaredFields
 */
public Field getField(String name) throws NoSuchFieldException, SecurityException {
	checkMemberAccess(Member.PUBLIC);
	return getFieldImpl(name);
}

/**
 * Answers a Field object describing the field in the receiver
 * named by the argument which must be visible from the current
 * execution context.
 *
 * @param		name		The name of the field to look for.
 * @return		the field in the receiver named by the argument.
 * @throws		NoSuchFieldException If the given field does not exist
 *
 * @see			#getDeclaredFields
 */
private native Field getFieldImpl(String name) throws NoSuchFieldException;

/**
 * Answers an array containing Field objects describing
 * all fields which are visible from the current execution
 * context.
 *
 * @return		all visible fields starting from the receiver.
 * @throws		SecurityException If member access is not allowed
 *
 * @see			#getDeclaredFields
 */
public Field[] getFields() throws SecurityException {
	checkMemberAccess(Member.PUBLIC);
	return getFieldsImpl();
}

/**
 * Answers an array containing Field objects describing
 * all fields which are visible from the current execution
 * context.
 *
 * @return		all visible fields starting from the receiver.
 *
 * @see			#getDeclaredFields
 */
private native Field[] getFieldsImpl();

/**
 * Answers an array of Class objects which match the interfaces
 * specified in the receiver classes <code>implements</code>
 * declaration
 *
 * @return		Class[]
 *					the interfaces the receiver claims to implement.
 */
public native Class[] getInterfaces();

/**
 * Answers a Method object which represents the method
 * described by the arguments.
 *
 * @param		name String
 *					the name of the method
 * @param		parameterTypes Class[]
 *					the types of the arguments.
 * @return		Method
 *					the method described by the arguments.
 * @throws	NoSuchMethodException
 *					if the method could not be found.
 * @throws	SecurityException
 *					if member access is not allowed
 *
 * @see			#getMethods
 */
public Method getMethod(String name, Class... parameterTypes) throws NoSuchMethodException, SecurityException {
	checkMemberAccess(Member.PUBLIC);
	Method result, bestCandidate;
	int maxDepth;
	String strSig;
	Class[] params;

	// Handle the no parameter case upfront
	if(name == null || parameterTypes == null || parameterTypes.length == 0) {
		strSig = "()";
		params = EmptyParameters;
	} else {
		// Build a signature for the requested method.
		int total = 2;
		String[] sigs = new String[parameterTypes.length];
		for(int i = 0; i < parameterTypes.length; i++) {
			if (parameterTypes[i] != null) {
				sigs[i] = parameterTypes[i].getSignature();
				total += sigs[i].length();
			} else throwNoSuchMethodException(name, parameterTypes);
		}
		StringBuffer signature = new StringBuffer(total);
		signature.append('(');
		for(int i = 0; i < parameterTypes.length; i++)
			signature.append(sigs[i]);
		signature.append(')');

		strSig = signature.toString();
		params = (Class[])parameterTypes.clone();
	}
	result = getMethodImpl(name, params, strSig);

	if (result == null) throwNoSuchMethodException(name, params);

	/* [PR 113003] The native is called repeatedly until it returns null,
	 * as each call returns another match if one exists. The first call uses
	 * getMethodImpl which searches across superclasses and interfaces, but
	 * since the spec requires that we only weigh multiple matches against
	 * each other if they are in the same class, on subsequent calls we call
	 * getDeclaredMethodImpl on the declaring class of the first hit.
	 * If more than one match is found, the code below selects the
	 * candidate method whose return type has the largest depth. This case
	 * case is expected to occur only in certain JCK tests, as most Java
	 * compilers will refuse to produce a class file with multiple methods
	 * of the same name differing only in return type.
	 *
	 * Selecting by largest depth is one possible algorithm that satisfies the
	 * spec.
	 */
	bestCandidate = result;
	maxDepth = result.getReturnType().getClassDepth();
	Class declaringClass = result.getDeclaringClass();
	while( true ) {
		result = declaringClass.getDeclaredMethodImpl( name, params, strSig, result );
		if( result == null ) {
			break;
		}
		if( (result.getModifiers() & Modifier.PUBLIC) != 0 ) {
			int resultDepth = result.getReturnType().getClassDepth();
			if( resultDepth > maxDepth ) {
				bestCandidate = result;
				maxDepth = resultDepth;
			}
		}
	}

	return bestCandidate;
}

/**
 * Answers a Method object which represents the first method found matching
 * the arguments.
 *
 * @param		name String
 *					the name of the method
 * @param		parameterTypes Class[]
 *					the types of the arguments.
 * @param		partialSignature String
 *					the signature of the method, without return type.
 * @return		Object
 *					the first Method found matching the arguments
 *
 * @see			#getMethods
 */
private native Method getMethodImpl(String name, Class parameterTypes[], String partialSignature);

/**
 * Answers an array containing Method objects describing
 * all methods which are visible from the current execution
 * context.
 *
 * @return		Method[]
 *					all visible methods starting from the receiver.
 * @throws	SecurityException
 *					if member access is not allowed
 *
 * @see			#getDeclaredMethods
 */
public Method[] getMethods() throws SecurityException {
	checkMemberAccess(Member.PUBLIC);
	Method[] methods;

	if(isPrimitive()) return new Method[0];
	if(isInterface())
	{
		return getInterfaceMethodsImpl();
	}
	else
	{
		int vCount = getVirtualMethodCountImpl();

		methods = (Method[])Method.class.allocateAndFillArray(vCount + getStaticMethodCountImpl());
		getVirtualMethodsImpl(methods, 0);
		getStaticMethodsImpl(methods, vCount);
		return methods;
	}
}

private boolean methodsEqual(Method m1, Method m2) {
	Class[] m1Parms, m2Parms;

	if(!m1.getName().equals(m2.getName())) {
		return false;
	}
	if(!m1.getReturnType().equals(m2.getReturnType())) {
		return false;
	}
	m1Parms = m1.getParameterTypes();
	m2Parms = m2.getParameterTypes();
	if(m1Parms.length != m2Parms.length) {
		return false;
	}
	for(int i = 0; i < m1Parms.length; i++) {
		if(m1Parms[i] != m2Parms[i]) {
			return false;
		}
	}
	return true;
}

private int getInterfaceMethodCountImpl()
{
	Class[] parents;
	int count;

	count = getDeclaredMethods().length;
	parents = getInterfaces();
	for(int i = 0; i < parents.length; i++) {
		count += parents[i].getInterfaceMethodCountImpl();
	}
	return count;
}

private Method[] getInterfaceMethodsImpl()
{
	Method[] scratch, unique, local;
	Class[] parents;
	int index, count;

	/* Get a pessimal amount of scratch space */
	scratch = new Method[getInterfaceMethodCountImpl()];

	/* Obtain the local methods. These are guaranteed to be returned */
	local = getDeclaredMethods();
	index = 0;
	for(int i = 0; i < local.length; i++) {
		/* <clinit> is never abstract */
		if(Modifier.isAbstract(local[i].getModifiers())) {
			scratch[index++] = local[i];
		} else {
			local[i] = null;
		}
	}

	/* Walk each superinterface */
	parents = getInterfaces();
	for(int i = 0; i < parents.length; i++) {
		/* Get the superinterface's (swept) methods */
		Method[] parentMethods = parents[i].getInterfaceMethodsImpl();
		for(int j = 0; j < local.length; j++) {
			if(local[j] != null) {
				/* Sweep out any local overrides */
				for(int k = 0; k < parentMethods.length; k++) {
					if((parentMethods[k] != null)&& methodsEqual(local[j], parentMethods[k])) {
						parentMethods[k] = null;
					}
				}
			}
		}
		/* Store any remaining methods */
		for(int j = 0; j < parentMethods.length; j++) {
			if(parentMethods[j] != null) {
				scratch[index++] = parentMethods[j];
			}
		}
	}

	/* Sweep out any duplicates */
	for(int i = 0; i < scratch.length; i++) {
		if(scratch[i] != null) {
			for(int j = i+1; j < scratch.length; j++) {
				if((scratch[j] != null) && scratch[i].equals(scratch[j])) {
					scratch[j] = null;
				}
			}
		}
	}

	/* Copy into a correctly sized array and return */
	count = 0;
	for(int i = 0; i < scratch.length; i++) {
		if(scratch[i] != null) count++;
	}
	unique = new Method[count];
	index = 0;
	for(int i = 0; i < scratch.length; i++) {
		if(scratch[i] != null) {
			unique[index++] = scratch[i];
		}
	}

	return unique;
}

private native int getVirtualMethodCountImpl();
private native void getVirtualMethodsImpl(Method[] array, int start);
private native int getStaticMethodCountImpl();
private native void getStaticMethodsImpl(Method[] array, int start);
private native Object[] allocateAndFillArray(int size);

/**
 * Answers an integer which which is the receiver's modifiers.
 * Note that the constants which describe the bits which are
 * returned are implemented in class java.lang.reflect.Modifier
 * which may not be available on the target.
 *
 * @return		the receiver's modifiers
 */
public int getModifiers() {
	int rawModifiers = getModifiersImpl();
	if (isArray()) {
		return rawModifiers &
			(Modifier.PUBLIC | Modifier.PRIVATE | Modifier.PROTECTED |
				Modifier.ABSTRACT | Modifier.FINAL);
	} else {
		return rawModifiers &
			(Modifier.PUBLIC | Modifier.PRIVATE | Modifier.PROTECTED |
					Modifier.STATIC | Modifier.FINAL | Modifier.INTERFACE |
					Modifier.ABSTRACT | SYNTHETIC | ENUM | ANNOTATION);
	}
}

private native int getModifiersImpl();

/**
 * Answers the name of the class which the receiver represents.
 * For a description of the format which is used, see the class
 * definition of java.lang.Class.
 *
 * @return		the receiver's name.
 *
 * @see			java.lang.Class
 */
public String getName() {
	if (classNameMap == null) {
		// don't start the hashtable until class init is done.
		return getNameImpl().intern();
	}

	String resultName = (String)classNameMap.get(this);
	if (resultName == null) {
		resultName = getNameImpl().intern();
		classNameMap.put(this, resultName);
	}
	return resultName;
}

native String getNameImpl();

static void setClassNameMap() {
	classNameMap = Collections.synchronizedMap(new WeakHashMap(200));
}

/**
 * Answers the ProtectionDomain of the receiver.
 * <p>
 * Note: In order to conserve space in embedded targets, we allow this
 * method to answer null for classes in the system protection domain
 * (i.e. for system classes). System classes are always given full
 * permissions (i.e. AllPermission). This is not changeable via the
 * java.security.Policy.
 *
 * @return		ProtectionDomain
 *					the receiver's ProtectionDomain.
 *
 * @see			java.lang.Class
 */
public ProtectionDomain getProtectionDomain() {
	SecurityManager security = System.getSecurityManager();
	if (security != null)
		security.checkPermission(
			RuntimePermission.permissionToGetProtectionDomain);

	ProtectionDomain result = getPDImpl();
	if (result != null) return result;

	if (AllPermissionsPD == null) {
			Permissions collection = new Permissions();
			collection.add(new AllPermission());
			AllPermissionsPD = new ProtectionDomain(null, collection);
	}
	return AllPermissionsPD;
}

/**
 * Answers the ProtectionDomain of the receiver.
 * <p>
 * This method is for internal use only.
 *
 * @return		ProtectionDomain
 *					the receiver's ProtectionDomain.
 *
 * @see			java.lang.Class
 */
native ProtectionDomain getPDImpl();

/**
 * Answers the name of the package to which the receiver belongs.
 * For example, Object.class.getPackageName() returns "java.lang".
 *
 * @return		the receiver's package name.
 *
 * @see			#getPackage
 */
String getPackageName() {
	String name = getName();
	int index = name.lastIndexOf('.');
	if (index >= 0) return name.substring(0, index);
	return "";
}

/**
 * Answers a read-only stream on the contents of the
 * resource specified by resName. The mapping between
 * the resource name and the stream is managed by the
 * class' class loader.
 *
 * @param		resName 	the name of the resource.
 * @return		a stream on the resource.
 *
 * @see			java.lang.ClassLoader
 */
public URL getResource(String resName) {
	ClassLoader loader = this.getClassLoaderImpl();
	if (loader == ClassLoader.systemClassLoader)
		return ClassLoader.getSystemResource(this.toResourceName(resName));
	else
		return loader.getResource(this.toResourceName(resName));
}

/**
 * Answers a read-only stream on the contents of the
 * resource specified by resName. The mapping between
 * the resource name and the stream is managed by the
 * class' class loader.
 *
 * @param		resName		the name of the resource.
 * @return		a stream on the resource.
 *
 * @see			java.lang.ClassLoader
 */
public InputStream getResourceAsStream(String resName) {
	ClassLoader loader = this.getClassLoaderImpl();
	if (loader == ClassLoader.systemClassLoader)
		return ClassLoader.getSystemResourceAsStream(this.toResourceName(resName));
	else
		return loader.getResourceAsStream(this.toResourceName(resName));
}

/**
 * Answers a String object which represents the class's
 * signature, as described in the class definition of
 * java.lang.Class.
 *
 * @return		the signature of the class.
 *
 * @see			java.lang.Class
 */
private String getSignature() {
	if(isArray()) return getName(); // Array classes are named with their signature
	if(isPrimitive()) {
		// Special cases for each base type.
		if(this == void.class) return "V";
		if(this == boolean.class) return "Z";
		if(this == byte.class) return "B";
		if(this == char.class) return "C";
		if(this == short.class) return "S";
		if(this == int.class) return "I";
		if(this == long.class) return "J";
		if(this == float.class) return "F";
		if(this == double.class) return "D";
	}

	// General case.
	// Create a StringBuffer of the correct size
	String name = getName();
	return new StringBuffer(name.length() + 2).
		append('L').append(name).append(';').toString();
}

/**
 * Answers the signers for the class represented by the
 * receiver, or null if there are no signers.
 *
 * @return		the signers of the receiver.
 *
 * @see			#getMethods
 */
public Object[] getSigners() {
	 return getClassLoaderImpl().getSigners(this);
}

/**
 * Answers the Class which represents the receiver's
 * superclass. For Classes which represent base types,
 * interfaces, and for java.lang.Object the method
 * answers null.
 *
 * @return		the receiver's superclass.
 */
public native Class<? super T> getSuperclass();

/**
 * Answers true if the receiver represents an array class.
 *
 * @return		<code>true</code>
 *					if the receiver represents an array class
 *              <code>false</code>
 *                  if it does not represent an array class
 */
public native boolean isArray();

/**
 * Answers true if the type represented by the argument
 * can be converted via an identity conversion or a widening
 * reference conversion (i.e. if either the receiver or the
 * argument represent primitive types, only the identity
 * conversion applies).
 *
 * @return		<code>true</code>
 *					the argument can be assigned into the receiver
 *              <code>false</code>
 *					the argument cannot be assigned into the receiver
 * @param		cls	Class
 *					the class to test
 * @throws	NullPointerException
 *					if the parameter is null
 *
 */
public native boolean isAssignableFrom(Class<?> cls);

/**
 * Answers true if the argument is non-null and can be
 * cast to the type of the receiver. This is the runtime
 * version of the <code>instanceof</code> operator.
 *
 * @return		<code>true</code>
 *					the argument can be cast to the type of the receiver
 *              <code>false</code>
 *					the argument is null or cannot be cast to the
 *					type of the receiver
 *
 * @param		object Object
 *					the object to test
 */
public native boolean isInstance(Object object);

/**
 * Answers true if the receiver represents an interface.
 *
 * @return		<code>true</code>
 *					if the receiver represents an interface
 *              <code>false</code>
 *                  if it does not represent an interface
 */
public boolean isInterface() {
	return (getModifiers() & 512 /* AccInterface */) != 0;
}

/**
 * Answers true if the receiver represents a base type.
 *
 * @return		<code>true</code>
 *					if the receiver represents a base type
 *              <code>false</code>
 *                  if it does not represent a base type
 */
public native boolean isPrimitive();

/**
 * Answers a new instance of the class represented by the
 * receiver, created by invoking the default (i.e. zero-argument)
 * constructor. If there is no such constructor, or if the
 * creation fails (either because of a lack of available memory or
 * because an exception is thrown by the constructor), an
 * InstantiationException is thrown. If the default constructor
 * exists, but is not accessible from the context where this
 * message is sent, an IllegalAccessException is thrown.
 *
 * @return		a new instance of the class represented by the receiver.
 * @throws		IllegalAccessException if the constructor is not visible to the sender.
 * @throws		InstantiationException if the instance could not be created.
 */
public T newInstance() throws IllegalAccessException, InstantiationException {
	checkMemberAccess(Member.PUBLIC);
	return (T)newInstanceImpl();
}

/**
 * Used as a prototype for the jit.
 *
 * @param 		cl
 * @return		the object
 * @throws 		InstantiationException
 */
private Object newInstancePrototype(Class cl) throws InstantiationException {
	throw new InstantiationException(cl);
}

/**
 * Answers a new instance of the class represented by the
 * receiver, created by invoking the default (i.e. zero-argument)
 * constructor. If there is no such constructor, or if the
 * creation fails (either because of a lack of available memory or
 * because an exception is thrown by the constructor), an
 * InstantiationException is thrown. If the default constructor
 * exists, but is not accessible from the context where this
 * message is sent, an IllegalAccessException is thrown.
 *
 * @return		a new instance of the class represented by the receiver.
 * @throws		IllegalAccessException if the constructor is not visible to the sender.
 * @throws		InstantiationException if the instance could not be created.
 */
private native Object newInstanceImpl()
	throws IllegalAccessException, InstantiationException;

/**
 * Answers a string describing a path to the receiver's appropriate
 * package specific subdirectory, with the argument appended if the
 * argument did not begin with a slash. If it did, answer just the
 * argument with the leading slash removed.
 *
 * @return		String
 *					the path to the resource.
 * @param		resName	String
 *					the name of the resource.
 *
 * @see			#getResource
 * @see			#getResourceAsStream
 */
private String toResourceName(String resName) {
	// Turn package name into a directory path
	if (resName.length() > 0 && resName.charAt(0) == '/')
		return resName.substring(1);

	String qualifiedClassName = getName();
	int classIndex = qualifiedClassName.lastIndexOf('.');
	if (classIndex == -1) return resName; // from a default package
	return qualifiedClassName.substring(0, classIndex + 1).replace('.', '/') + resName;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	// Note change from 1.1.7 to 1.2: For primitive types,
	// return just the type name.
	if (isPrimitive()) return getName();
	return (isInterface() ? "interface " : "class ") + getName();
}

/**
 * Returns the Package of which this class is a member.
 * A class has a Package iff it was loaded from a SecureClassLoader
 *
 * @return		Package the Package of which this class is a member or null
 *
 */
public Package getPackage() {
	return getClassLoaderImpl().getPackage(getPackageName());
}

static Class getPrimitiveClass(String name) {
	if (name.equals("float"))
		return new float[0].getClass().getComponentType();
	if (name.equals("double"))
		return new double[0].getClass().getComponentType();
	if (name.equals("int"))
		return new int[0].getClass().getComponentType();
	if (name.equals("long"))
		return new long[0].getClass().getComponentType();
	if (name.equals("char"))
		return new char[0].getClass().getComponentType();
	if (name.equals("byte"))
		return new byte[0].getClass().getComponentType();
	if (name.equals("boolean"))
		return new boolean[0].getClass().getComponentType();
	if (name.equals("short"))
		return new short[0].getClass().getComponentType();
	if (name.equals("void")) {
		try {
			java.lang.reflect.Method method = Runnable.class.getMethod("run", new Class[0]);
			return method.getReturnType();
		} catch (Exception e) {
			com.ibm.oti.vm.VM.dumpString("Cannot initialize Void.TYPE\n");
		}
	}
	throw new Error("Unknown primitive type: " + name);
}

/**
 * Returns the assertion status for this class.
 * Assertion is enabled/disabled based on
 * classloader default, package or class default at runtime
 *
 * @since 1.4
 *
 * @return		the assertion status for this class
 */
public boolean desiredAssertionStatus() {
	ClassLoader cldr = getClassLoaderImpl();
	if (cldr != null) {
		return cldr.getClassAssertionStatus(getName());
	}
	return false;
}

/**
 * Answer the class at depth.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 *	 2) The item at index zero describes the caller of this method.
 *
 * @param 		depth
 * @return		the class at the given depth
 */
static final native Class getStackClass(int depth);

/**
 * Walk the stack and answer an array containing the maxDepth
 * most recent classes on the stack of the calling thread.
 *
 * Starting with the caller of the caller of getStackClasses(), return an
 * array of not more than maxDepth Classes representing the classes of
 * running methods on the stack (including native methods).  Frames
 * representing the VM implementation of java.lang.reflect are not included
 * in the list.  If stopAtPrivileged is true, the walk will terminate at any
 * frame running one of the following methods:
 *
 * <code><ul>
 * <li>java/security/AccessController.doPrivileged(Ljava/security/PrivilegedAction;)Ljava/lang/Object;</li>
 * <li>java/security/AccessController.doPrivileged(Ljava/security/PrivilegedExceptionAction;)Ljava/lang/Object;</li>
 * <li>java/security/AccessController.doPrivileged(Ljava/security/PrivilegedAction;Ljava/security/AccessControlContext;)Ljava/lang/Object;</li>
 * <li>java/security/AccessController.doPrivileged(Ljava/security/PrivilegedExceptionAction;Ljava/security/AccessControlContext;)Ljava/lang/Object;</li>
 * </ul></code>
 *
 * If one of the doPrivileged methods is found, the walk terminate and that frame is NOT included in the returned array.
 *
 * Notes: <ul>
 * 	 <li> This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers. </li>
 *
 *	 <li> The item at index zero in the result array describes the caller of
 *		the caller of this method. </li>
 *</ul>
 *
 * @param 		maxDepth			maximum depth to walk the stack, -1 for the entire stack
 * @param 		stopAtPrivileged	stop at priviledged classes
 * @return		the array of the most recent classes on the stack
 */
static final native Class[] getStackClasses(int maxDepth, boolean stopAtPrivileged);

/**
 * Called from JVM_ClassDepth.
 * Answers the index in the stack of the first method which
 * is contained in a class called <code>name</code>. If no
 * methods from this class are in the stack, return -1.
 *
 * @param		name String
 *					the name of the class to look for.
 * @return		int
 *					the depth in the stack of a the first
 *					method found.
 */
static int classDepth (String name) {
	Class[] classes = getStackClasses(-1, false);
	for (int i=1; i<classes.length; i++)
		if (classes[i].getName().equals(name))
			return i - 1;
	return -1;
}

/**
 * Called from JVM_ClassLoaderDepth.
 * Answers the index in the stack of thee first class
 * whose class loader is not a system class loader.
 *
 * @return		the frame index of the first method whose class was loaded by a non-system class loader.
 */
static int classLoaderDepth() {
	// Now, check if there are any non-system class loaders in
	// the stack up to the first privileged method (or the end
	// of the stack.
	Class[] classes = getStackClasses(-1, true);
	for (int i=1; i<classes.length; i++) {
		ClassLoader cl = classes[i].getClassLoaderImpl();
		if (!cl.isSystemClassLoader()) return i - 1;
	}
	return -1;
}

/**
 * Called from JVM_CurrentClassLoader.
 * Answers the class loader of the first class in the stack
 * whose class loader is not a system class loader.
 *
 * @return		the most recent non-system class loader.
 */
static ClassLoader currentClassLoader() {
	// Now, check if there are any non-system class loaders in
	// the stack up to the first privileged method (or the end
	// of the stack.
	Class[] classes = getStackClasses(-1, true);
	for (int i=1; i<classes.length; i++) {
		ClassLoader cl = classes[i].getClassLoaderImpl();
		if (!cl.isSystemClassLoader()) return cl;
	}
	return null;
}

/**
 * Called from JVM_CurrentLoadedClass.
 * Answers the first class in the stack which was loaded
 * by a class loader which is not a system class loader.
 *
 * @return		the most recent class loaded by a non-system class loader.
 */
static Class currentLoadedClass() {
	// Now, check if there are any non-system class loaders in
	// the stack up to the first privileged method (or the end
	// of the stack.
	Class[] classes = getStackClasses(-1, true);
	for (int i=1; i<classes.length; i++) {
		ClassLoader cl = classes[i].getClassLoaderImpl();
		if (!cl.isSystemClassLoader()) return classes[i];
	}
	return null;
}

/**
 * Return the specified Annotation for this Class. Inherited Annotations
 * are searched.
 *
 * @param annotation the Annotation type
 * @return the specified Annotation or null
 *
 * @since 1.5
 */
public <A extends Annotation> A getAnnotation(Class<A> annotation) {
	if (annotation == null) throw new NullPointerException();
	Annotation[] ans = getAnnotations();
	for (int i = 0; i < ans.length; i++) {
		if (ans[i].annotationType() == annotation) {
			return (A)ans[i];
		}
	}
	return null;
}

/**
 * Return the Annotations for this Class, including the Annotations
 * inherited from superclasses.
 *
 * @return an array of Annotation
 *
 * @since 1.5
 */
public Annotation[] getAnnotations() {
	HashMap<String,Annotation> annotations = new HashMap<String,Annotation>();
	//Store annotations from the current class
	Annotation[] anns = getDeclaredAnnotations();
	for (int i = 0; i < anns.length; i++) {
		annotations.put(anns[i].annotationType().getName(), anns[i]);
	}
	Class c = getSuperclass();
	while (c != null) {
		Annotation[] ann = c.getDeclaredAnnotations();
		if (ann != null) {
			for (int i = 0; i < ann.length; i++) {
				// if we have an annotation of this type stored skip it
				if (!annotations.containsKey(ann[i].annotationType().getName())) {
					// if the annotation is Inherited store the annotation
					if (ann[i].annotationType().getAnnotation(Inherited.class) != null) {
						annotations.put(ann[i].annotationType().getName(), ann[i]);
					}
				}
			}
		}
		c = c.getSuperclass();
	}

	Annotation[] annArray = new Annotation[annotations.size()];
	annotations.values().toArray(annArray);

	return annArray;
}

/**
 * Return the Annotations only for this Class, not including Annotations
 * inherited from superclasses.
 *
 * @return an array of Annotation
 *
 * @since 1.5
 */
public Annotation[] getDeclaredAnnotations() {
	java.util.Hashtable collection = getClassLoaderImpl().getAnnotationCache();
	Annotation[] annotations = (Annotation[])collection.get(this);
	if (annotations == null) {
		annotations = AnnotationHelper.getDeclaredAnnotations(this, AnnotationHelper.ANNOTATION_TYPE_CLASS, null);
		collection.put(this, annotations);
	}
	return (Annotation[])annotations.clone();
}

/**
 * Answer if this class is an Annotation.
 *
 * @return true if this class is an Annotation
 *
 * @since 1.5
 */
public boolean isAnnotation() {
	return (getModifiers() & ANNOTATION) != 0;
}

/**
 * Answer if the specified Annotation exists for this Class. Inherited
 * Annotations are searched.
 *
 * @param annotation the Annotation type
 * @return true if the specified Annotation exists
 *
 * @since 1.5
 */
public boolean isAnnotationPresent(Class<? extends Annotation> annotation) {
	if (annotation == null) throw new NullPointerException();
	return getAnnotation(annotation) != null;
}

/**
 * Cast this Class to a subclass of the specified Class.
 *
 * @return this Class, cast to a subclass of the specified Class
 *
 * @throws ClassCastException if this Class is not the same or a subclass
 *		of the specified Class
 *
 * @since 1.5
 */
public <U> Class<? extends U> asSubclass(Class<U> cls) {
	if (!cls.isAssignableFrom(this))
		throw new ClassCastException(this.toString());
	return (Class<? extends U>)this;
}

/**
 * Cast the specified object to this Class.
 *
 * @param object the object to cast
 *
 * @return the specified object, cast to this Class
 *
 * @throws ClassCastException if the specified object cannot be cast
 *		to this Class
 *
 * @since 1.5
 */
public T cast(Object object) {
	if (object != null && !this.isAssignableFrom(object.getClass()))
		throw new ClassCastException(com.ibm.oti.util.Msg.getString("K0336", object.getClass(), this));
	return (T)object;
}

/**
 * Answer if this Class is an enum.
 *
 * @return true if this Class is an enum
 *
 * @since 1.5
 */
public boolean isEnum() {
	return (getModifiers() & ENUM) != 0 &&
		getSuperclass() == Enum.class;
}

Map<String, T> enumConstantDirectory() {
	T[] enums = getEnumConstants();
	if (enums == null) throw new IllegalArgumentException(getName() + " is not an Enum");
	Map<String, T> map = new HashMap<String, T>(enums.length * 2);
	for (int i=0; i<enums.length; i++)
		map.put(((Enum)enums[i]).name(), enums[i]);
	return map;
}

/**
 * Answer the array of enum constants for this Class. Returns null if
 * this class is not an enum.
 *
 * @return the array of enum constants, or null
 *
 * @since 1.5
 */
public T[] getEnumConstants() {
	if (!isEnum()) return null;
	try {
		Method values = AccessController.doPrivileged(new PrivilegedExceptionAction<Method>() {
			public Method run() throws Exception {
				Method method = getMethod("values");
				// the enum class may not be visible
				method.setAccessible(true);
				return method;
			}
		});
		T[] enums = (T[])values.invoke(this);
		return (T[])enums.clone();
	} catch (Exception e) {
		return null;
	}
}

/**
 * Answer if this Class is synthetic. A synthetic Class is created by
 * the compiler.
 *
 * @return true if this Class is synthetic.
 *
 * @since 1.5
 */
public boolean isSynthetic() {
	return (getModifiers() & SYNTHETIC) != 0;
}

private native String getGenericSignature();

private CoreReflectionFactory getFactory() {
	return CoreReflectionFactory.make(this, ClassScope.make(this));
}

private ClassRepository getClassRepository(String signature) {
	java.util.Hashtable collection = getClassLoaderImpl().getGenericRepository();
	ClassRepository classRepository = (ClassRepository)collection.get(this);
	if (classRepository == null) {
		classRepository = ClassRepository.make(signature, getFactory());
		collection.put(this, classRepository);
	}
	return classRepository;
}

/**
 * Answers an array of TypeVariable for the generic parameters declared
 * on this Class.
 *
 * @return		the TypeVariable[] for the generic parameters
 *
 * @since 1.5
 */
public TypeVariable<Class<T>>[] getTypeParameters() {
	String signature = getGenericSignature();
	if (signature == null) return new TypeVariable[0];
	ClassRepository repository = getClassRepository(signature);
	return repository.getTypeParameters();
}

/**
 * Answers an array of Type for the Class objects which match the
 * interfaces specified in the receiver classes <code>implements</code>
 * declaration.
 *
 * @return		Type[]
 *					the interfaces the receiver claims to implement.
 *
 * @since 1.5
 */
public Type[] getGenericInterfaces() {
	String signature = getGenericSignature();
	if (signature == null) return getInterfaces();
	ClassRepository repository = getClassRepository(signature);
	return repository.getSuperInterfaces();
}

/**
 * Answers the Type for the Class which represents the receiver's
 * superclass. For classes which represent base types,
 * interfaces, and for java.lang.Object the method
 * answers null.
 *
 * @return		the Type for the receiver's superclass.
 *
 * @since 1.5
 */
public Type getGenericSuperclass() {
	String signature = getGenericSignature();
	if (signature == null) return getSuperclass();
	if (isInterface()) return null;
	ClassRepository repository = getClassRepository(signature);
	return repository.getSuperclass();
}

private native Object getEnclosingObject();

/**
 * If this Class is defined inside a constructor, return the Constructor.
 *
 * @return the enclosing Constructor or null
 *
 * @since 1.5
 *
 * @see #isAnonymousClass()
 * @see #isLocalClass()
 */
public Constructor<?> getEnclosingConstructor() {
	Object enclosing = getEnclosingObject();
	if (enclosing instanceof Constructor) {
		return (Constructor)enclosing;
	}
	return null;
}

/**
 * If this Class is defined inside a method, return the Method.
 *
 * @return the enclosing Method or null
 *
 * @since 1.5
 *
 * @see #isAnonymousClass()
 * @see #isLocalClass()
 */
public Method getEnclosingMethod() {
	Object enclosing = getEnclosingObject();
	if (enclosing instanceof Method) {
		return (Method)enclosing;
	}
	return null;
}

private native Class getEnclosingObjectClass();

/**
 * Return the enclosing Class of this Class. Unlike getDeclaringClass(),
 * this method works on any nested Class, not just classes nested directly
 * in other classes.
 *
 * @return the enclosing Class or null
 *
 * @since 1.5
 *
 * @see #getDeclaringClass()
 * @see #isAnonymousClass()
 * @see #isLocalClass()
 * @see #isMemberClass()
 */
public Class<?> getEnclosingClass() {
	Class declaringClass = getDeclaringClass();
	if (declaringClass != null) {
		return declaringClass;
	}
	Class enclosingClass = getEnclosingObjectClass();
	return enclosingClass;
}

private native String getSimpleNameImpl();

/**
 * Return the simple name of this Class. The simple name does not include
 * the package or the name of the enclosing class. The simple name of an
 * anonymous class is "".
 *
 * @return the simple name
 *
 * @since 1.5
 *
 * @see #isAnonymousClass()
 */
public String getSimpleName() {
	int arrayCount = 0;
	Class baseType = this;
	if (isArray()) {
		arrayCount = 1;
		while ((baseType = baseType.getComponentType()).isArray()) {
			arrayCount++;
		}
	}
	String simpleName = baseType.getSimpleNameImpl();
	if (simpleName == null) {
		// either a base class, or anonymous class
		if (baseType.getEnclosingObjectClass() != null) {
			simpleName = "";
		} else {
			// remove the package name
			simpleName = baseType.getName();
			int index = simpleName.lastIndexOf('.');
			if (index != -1) {
				simpleName = simpleName.substring(index+1);
			}
		}
	}
	if (arrayCount > 0) {
		StringBuffer result = new StringBuffer(simpleName);
		for (int i=0; i<arrayCount; i++) {
			result.append("[]");
		}
		return result.toString();
	}
	return simpleName;
}

/**
 * Return the canonical name of this Class. The canonical name is null
 * for a local or anonymous class. The canonical name includes the package
 * and the name of the enclosing class.
 *
 * @return the canonical name or null
 *
 * @since 1.5
 *
 * @see #isAnonymousClass()
 * @see #isLocalClass()
 */
public String getCanonicalName() {
	int arrayCount = 0;
	Class baseType = this;
	if (isArray()) {
		arrayCount = 1;
		while ((baseType = baseType.getComponentType()).isArray()) {
			arrayCount++;
		}
	}
	if (baseType.getEnclosingObjectClass() != null) {
		// local or anonymous class
		return null;
	}
	String canonicalName;
	Class declaringClass = baseType.getDeclaringClass();
	if (declaringClass == null) {
		canonicalName = baseType.getName();
	} else {
		String declaringClassCanonicalName = declaringClass.getCanonicalName();
		if (declaringClassCanonicalName == null) return null;
		// remove the enclosingClass from the name, including the $
		String simpleName = baseType.getName().substring(declaringClass.getName().length() + 1);
		canonicalName = declaringClassCanonicalName + '.' + simpleName;
	}

	if (arrayCount > 0) {
		StringBuffer result = new StringBuffer(canonicalName);
		for (int i=0; i<arrayCount; i++) {
			result.append("[]");
		}
		return result.toString();
	}
	return canonicalName;
}

/**
 * Answer if this Class is anonymous. An unnamed Class defined
 * inside a method.
 *
 * @return true if this Class is anonymous.
 *
 * @since 1.5
 *
 * @see #isLocalClass()
 */
public boolean isAnonymousClass() {
	return getSimpleNameImpl() == null && getEnclosingObjectClass() != null;
}

/**
 * Answer if this Class is local. A named Class defined inside
 * a method.
 *
 * @return true if this Class is local.
 *
 * @since 1.5
 *
 * @see #isAnonymousClass()
 */
public boolean isLocalClass() {
	return getEnclosingObjectClass() != null && getSimpleNameImpl() != null;
}

/**
 * Answer if this Class is a member Class. A Class defined inside another
 * Class.
 *
 * @return true if this Class is local.
 *
 * @since 1.5
 *
 * @see #isLocalClass()
 */
public boolean isMemberClass() {
	return getEnclosingObjectClass() == null && getDeclaringClass() != null;
}

/**
 * Return the depth in the class hierarchy of the receiver.
 * Base type classes and Object return 0.
 *
 * @return receiver's class depth
 *
 * @see #getDeclaredMethod
 * @see #getMethod
 */
private native int getClassDepth();

}
