package com.ibm.swat.password;

import java.io.*;
import java.util.*;
import java.net.*;

import javax.naming.*;
import javax.naming.directory.*;

/* HOW GROUP SEARCHING WORKS

  Searching a 'flat' groups (depth 0)
    1) Bind to Bluepages
    2) Search Bluepages for DN of person
    3) Search Bluegroups to see if person is in group
    4) Unbind Bluepages
  
  Searching a group (depth n)
    1) Bind to Bluepages
    2) Search Bluepages for DN of person
    3) Search Bluegroups to see if person is in group
    4) If person in group, go to step 7
    5) if (depth-- >= 0) Search Bluegroups for all groupDN's in current group else goto step 7
    6) Goto 3
    7) Unbind Bluepages

*/

/**
 * cwa2 is a class containing methods for authenticating and group querying 
 *
 * @author      Brian Olore
 * @version     3
 */

public class cwa2
{
  private String SEARCH_BASE = "ou=bluepages,o=ibm.com";
  private String MEMBERLIST_BASE = "ou=memberlist,ou=ibmgroups,o=ibm.com";
  private String METADATA_BASE = "ou=metadata,ou=ibmgroups,o=ibm.com";
  private static final String DEFAULT_BLUEPAGES_LDAP_SERVER = "ldap://bluepages.ibm.com:389";
  private static final String DEFAULT_BLUEGROUPS_LDAP_SERVER = "ldap://bluegroups.ibm.com:389";
  private static final int DEFAULT_LDAP_VERSION = 3;
  private static final int DEFAULT_DEPTH = 0;
  private String HTTPS_BASE = "https://bluepages.ibm.com/tools/groups/protect/groups.wss";
  private int CONNTIMEOUT = 0;
  private int SEARCHTIMELIMIT = 0;

  private String bluepages_ldap_server;
  private String bluegroups_ldap_server;
  private String ldap_version = "3";
  private boolean useThreaded = false;
  private ThreadGroup thread_group = null;  

  public static final String EVERYONE ="Everyone";
  public static final String OWN_ADMIN = "Owner/Admins";
  public static final String MEMBERLIST = "members";
  public static final String FILTER = "filter";

  static {
      try {
          /*** SUN JSSE ***/
          //java.security.Security.addProvider(new com.sun.net.ssl.internal.ssl.Provider());
          //System.setProperty("java.protocol.handler.pkgs","com.sun.net.ssl.internal.www.protocol");
          /***          ***/

          /*** IBM JSSE ***/
          java.security.Security.addProvider(new com.ibm.jsse.JSSEProvider());
          System.setProperty("java.protocol.handler.pkgs", "com.ibm.net.ssl.internal.www.protocol");          
      } catch (Exception e) {
          System.err.println(e);
          e.printStackTrace();
      }
  }

  /**
    * Default Constuctor
    *
    */
  public cwa2() {
      initialize(DEFAULT_BLUEPAGES_LDAP_SERVER, DEFAULT_BLUEGROUPS_LDAP_SERVER, DEFAULT_LDAP_VERSION, SEARCH_BASE, MEMBERLIST_BASE, METADATA_BASE, HTTPS_BASE);
  }

  /**
    * Constructor which allows you to specify the Bluepages server 
    * and Bluegroups server to use
    *
    * @param bluepages The hostname of the Bluepages server to use
    * @param bluegroups The hostname of the Bluegroups server to use
    */
  public cwa2(String bluepages, String bluegroups) {      
      initialize(bluepages, bluegroups, DEFAULT_LDAP_VERSION, SEARCH_BASE, MEMBERLIST_BASE, METADATA_BASE, HTTPS_BASE);
  }

  /**
    * Constructor which allows you to specify the Bluepages server 
    * and Bluegroups server and ldap version to use
    *
    * @param bluepages The hostname of the Bluepages server to use
    * @param bluegroups The hostname of the Bluegroups server to use
    * @param v The version of LDAP Server you are connecting to
    */
  public cwa2(String bluepages, String bluegroups, int v) {
      initialize(bluepages, bluegroups, v, SEARCH_BASE, MEMBERLIST_BASE, METADATA_BASE, HTTPS_BASE);
  }

  /**
    * Constructor which allows you to specify the Bluepages server 
    * and Bluegroups server and ldap version to use
    *
    * @param bluepages The hostname of the Bluepages server to use
    * @param bluegroups The hostname of the Bluegroups server to use
    * @param v The version of LDAP Server you are connecting to
    * @param bp_base search base for people searches
    * @param memberlist_base search base for group member list
    * @param metadata_base search base for group metadata
    * @param https_base The HTTP URL to connect to GUI, default it http://w3.ibm.com/tools/groups/protect/groups.wss
    */
  public cwa2(String bluepages, String bluegroups, int v, String bp_base, String memberlist_base, String metadata_base, String https_base) {
      initialize(bluepages, bluegroups, v, bp_base, memberlist_base, metadata_base, https_base);
  }

  private void initialize(String bluepages, String bluegroups, int v, String bp_base, String memberlist_base, String metadata_base, String https_base) {
      if (bluepages.startsWith("ldaps://")==true) {
          bluepages_ldap_server = "ldap://" + bluepages.substring(8, bluepages.length());

          if (bluepages.endsWith(":636")==false) {
              bluepages_ldap_server = bluepages_ldap_server + ":636";
          }
      } else if (bluepages.startsWith("ldap://")==true) {
          if (bluepages.endsWith(":389")==false && bluepages.endsWith(":636")==false) {
              bluepages_ldap_server = bluepages + ":389";
          } else {
              bluepages_ldap_server = bluepages;
          }
      } else {
          if (bluepages.endsWith(":389")==false && bluepages.endsWith(":636")==false) {
              bluepages_ldap_server = "ldap://" + bluepages + ":389";
          } else {
              bluepages_ldap_server = "ldap://" + bluepages;
          }
      }

      if (bluegroups.startsWith("ldaps://")==true) {
          bluegroups_ldap_server = "ldap://" + bluegroups.substring(8, bluegroups.length());

          if (bluegroups.endsWith(":636")==false) {
              bluegroups_ldap_server = bluegroups_ldap_server + ":636";
          }
      } else if (bluegroups.startsWith("ldap://")==true) {
          if (bluegroups.endsWith(":389")==false && bluegroups.endsWith(":636")==false) {
              bluegroups_ldap_server = bluegroups + ":389";
          } else {
              bluegroups_ldap_server = bluegroups;
          }
      } else {
          if (bluegroups.endsWith(":389")==false && bluegroups.endsWith(":636")==false) {
              bluegroups_ldap_server = "ldap://" + bluegroups + ":389";
          } else {
              bluegroups_ldap_server = "ldap://" + bluegroups;
          }
      }

      ldap_version = new Integer(v).toString();

      if (bp_base != null) {
          SEARCH_BASE = bp_base;
      }

      if (memberlist_base != null) {
          MEMBERLIST_BASE = memberlist_base;
      }

      if (metadata_base != null) {
          METADATA_BASE = metadata_base;
      }

      if (https_base != null) {
          HTTPS_BASE = https_base;
      }
  }

  /**
    * @deprecated
    * No longer needed, use set methods instead
    *
    * @return ReturnCode based on result
    */
  public ReturnCode init(){return init(false, 0, 0);}

    /**
    * @deprecated
    * No longer needed, use set methods instead
    * 
    * @param _useThreaded Specify true to use multiple threads when searching nested groups
    *
    * @return ReturnCode based on result
    */
  public ReturnCode init(boolean _useThreaded){return init(_useThreaded, 0, 0);}

  /**
    * @deprecated      
    * No longer needed, use set methods instead
    * 
    * @param conntimeout Connection to LDAP timeout in ms
    * @param searchtimelimit Search to LDAP timelimit in ms
    *
    * @return ReturnCode based on result
    */
  public ReturnCode init(int conntimeout, int searchtimelimit){return init(false, conntimeout, searchtimelimit);}

  /**
    * @deprecated 
    * No longer needed, use set methods instead
    * 
    * @param _useThreaded Specify true to use multiple threads when searching nested groups    
    * @param conntimeout Connection to LDAP timeout in ms
    * @param searchtimelimit Search to LDAP timelimit in ms
    *
    * @return ReturnCode based on result
    */
  public ReturnCode init(boolean _useThreaded, int conntimeout, int searchtimelimit)
  {
    CONNTIMEOUT = conntimeout;
    SEARCHTIMELIMIT = searchtimelimit;

    useThreaded = _useThreaded;
    if (useThreaded)
      thread_group = new ThreadGroup("MyThreadGroup");
    return cwaapi.SUCCESS;
  }

  /**
    * Uses threaded searches to determine if a member is in a Group
    *
    * @param _useThreaded
    */
  public void setUseThreaded(boolean _useThreaded) {
      useThreaded = _useThreaded;      
      if (useThreaded)
        thread_group = new ThreadGroup("MyThreadGroup");
  }

  /**
    * Set the LDAP connection timeout
    *
    * @param conntimeout Connection timeout in milliseconds
    */
  public void setConnTimeOut(int conntimeout) {
      CONNTIMEOUT = conntimeout;      
  }

  /**
    * Set the LDAP search timeout
    *
    * @param searchtimelimit Search timeout in milliseconds
    */
  public void setSearchTimeOut(int searchtimelimit) {
      SEARCHTIMELIMIT = searchtimelimit;      
  }

  /**
   * @deprecated
   * No longer needed since cleanup is done in every method
   */
  public void cleanup()
  {
  }

  private void cleanupCtx(DirContext ctx) {
      if ( ctx != null )
      {
        try
        {
          ctx.close();
        }
        catch ( Throwable t )
        {
        }
        ctx = null;
      }
  }

  private void cleanupNamingEnum(NamingEnumeration ne) {
      if ( ne != null )
      {
        try
        {
          // Iterate through enumeration due to JNDI bug in JRE 1.3.1
          while (ne.hasMore()) {
              SearchResult sr = (SearchResult)ne.next();
          }
        }
        catch ( Throwable t )
        {
        }

        try
        {
          ne.close();
        }
        catch ( Throwable t )
        {
        }

        ne = null;
      }
  }

  /**
    * Autenticates a user, using the default ldap server
    *
    * @param email The user's internet email address as listed in Bluepages
    * @param pw The user's password
    * @return ReturnCode based on result
    */
  public ReturnCode authenticate(String email, String pw)
  {return authenticate(email, pw, bluepages_ldap_server);}


  /**
    * Autenticates a user, using a user given ldap server
    *
    * @param email The user's internet email address as listed in Bluepages
    * @param pw The user's password    
    * @param server The LDAP Server to use for authentication    
    * @return ReturnCode based on result
    */
  public ReturnCode authenticate(String email, String pw, String server)
  {
    return authenticate("mail", email, pw, server);
  }

  /**
    * Autenticates a user, using a user given ldap server
    *
    * @param attrval The user's attribute to search on
    * @param attrname The user's value to search for
    * @param pw The user's password    
    * @param server The LDAP Server to use for authentication, it should be FQDN ldap://servername:port
    * @return ReturnCode based on result
    */
  public ReturnCode authenticate(String attrname, String attrval, String pw, String server)
  {
    //new as of 07.25.2001  ... check for null or blanks!
    if (attrname==null || attrname.equals("") || attrval==null || attrval.equals(""))
      return cwaapi.NO_RECORD_ERROR;

    if (pw==null || pw.equals(""))
      return cwaapi.INVALID_CRED_ERROR;


    String DN = new String();

    if (server.startsWith("ldap://") == false) {
        server = "ldap://" + server;
    }

    if (server.endsWith(":389") == false || server.endsWith(":636") == false) {
        server = server + ":636";
    }

    String sserver = server.substring(0, server.indexOf(":", 10)) + ":636";
    server = server.substring(0, server.indexOf(":", 10)) + ":389";

    Properties env1 = new Properties();
    env1.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env1.put("java.naming.ldap.derefAliases", "never");    
    env1.put("java.naming.ldap.version", ldap_version); 
    env1.put(Context.REFERRAL, "follow");
    env1.put("java.naming.ldap.referral.bind", "true");

    if (bluepages_ldap_server.endsWith("636")) {
        env1.put(Context.SECURITY_PROTOCOL, "ssl");
        env1.put(Context.PROVIDER_URL,sserver);    
    } else {
        env1.put(Context.PROVIDER_URL,server);    
    }

    Vector v = new Vector();
    Hashtable h = new Hashtable();

    DirContext ctx = null;
    NamingEnumeration results = null;

    try
    {
      String attrlist[] = {"uid"};
      String filter = "(" + attrname + "=" + attrval + ")";
      SearchControls constraints = new SearchControls();
      constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
      constraints.setTimeLimit(SEARCHTIMELIMIT);
      constraints.setReturningAttributes(attrlist);
      ctx = new InitialDirContext(env1); 
      results = ctx.search(SEARCH_BASE, filter ,constraints);

      SearchResult sr;
      Attributes attrs;
      Attribute attr;
      Enumeration vals;
      Object q;
      while (results.hasMore())
      {
        sr = (SearchResult) results.next();
        attrs = sr.getAttributes();
        //snag the DN
        DN = sr.getName() + ","+ SEARCH_BASE;

        h = new Hashtable();            
        for (NamingEnumeration ae = attrs.getAll(); ae.hasMore();)
        {
          attr = (Attribute) ae.next();                            
          vals = attr.getAll();                           
          q = vals.nextElement();
          try
          {
            h.put(attr.getID(),(String)q);
          } //put the String in the hash
          catch (ClassCastException cce)      //it's not a String, so "decode" it (int'l chars)
          {
            try
            {
              byte b[] = (byte[])q;
              String s = new String(b,"ISO-8859-1"); //"UTF8");  //UTF8 doesn't work for "palmero"
              h.put(attr.getID(),s); //put the string in the hash
            }
            catch (UnsupportedEncodingException ue)
            {
            }
          }
        }  
        if (h.size()>0) v.addElement(h); //no reason to add the hash if it is empty        
      }

    }
    catch (NamingException ne)
    {
      System.err.println("[cwa2] authenticate: " + ne);
      ne.printStackTrace();
      return cwaapi.LDAP_ERROR;
    } 
    finally 
    {
      cleanupNamingEnum(results);
      cleanupCtx(ctx);
    }

    if (v.size()==0)
      return cwaapi.NO_RECORD_ERROR;
    else
      if (v.size()>1)
      return cwaapi.MULTI_RECORD_ERROR;

    Properties env = new Properties();
    env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env.put("java.naming.ldap.derefAliases", "never");
    env.put(Context.PROVIDER_URL,sserver);    
    env.put("java.naming.ldap.version", ldap_version); 
    env.put(Context.SECURITY_AUTHENTICATION, "simple");
    env.put(Context.SECURITY_PRINCIPAL, DN);
    env.put(Context.SECURITY_CREDENTIALS, pw);
    env.put(Context.REFERRAL, "follow");
    env.put("java.naming.ldap.referral.bind", "true");
    env.put(Context.SECURITY_PROTOCOL, "ssl");    

    try {
        ctx = new InitialDirContext(env); 
    } catch (AuthenticationException ae) {        
        return cwaapi.INVALID_CRED_ERROR;
    } catch (NamingException ne) {
        System.err.println("[cwa2] authenticate: " + ne);
        ne.printStackTrace();
        return cwaapi.LDAP_ERROR;
    } finally {
        cleanupCtx(ctx);
    }

    return cwaapi.SUCCESS;
  }


  /**
    * Autenticates a user, using the default ldap server, throws Exceptions on failure.
    *
    * @param email The user's internet email address as listed in Bluepages
    * @param pw The user's password    
    * @return true if the user is authenticated
    * @exception NamingException An error has occurred
    * @throws NamingException An error has occurred
    */
  public boolean authenticate_throw(String email, String pw) throws NamingException
  {
      return authenticate_throw(email, pw, bluepages_ldap_server);
  }


  /**
    * Autenticates a user, throws Exceptions on failure
    *
    * @param server The LDAP Server to use for authentication
    * @param email The user's internet email address as listed in Bluepages    
    * @param pw The user's password    
    * @return true If the user is authenticated
    * @exception NamingException An error has occurred
    * @throws NamingException An error has occurred
    */
  public boolean authenticate_throw(String email, String pw, String server) throws NamingException
  {
      return authenticate_throw("mail", email, pw, server);
  }

  /**
    * Autenticates a user, throws Exceptions on failure
    *
    * @param attrval The user's attribute to search on
    * @param attrname The user's value to search for
    * @param pw The user's password    
    * @param server The LDAP Server to use for authentication
    * @return true If the user is authenticated
    * @exception NamingException An error has occurred
    * @throws NamingException An error has occurred
    */
  public boolean authenticate_throw(String attrname, String attrval, String pw, String server) throws NamingException
  {
    //new as of 07.25.2001  ... check for null or blanks!
    if (attrname==null || attrname.equals("") || attrval==null || attrval.equals(""))
      throw new NamingException("mail is either NULL or blank");

    if (pw==null || pw.equals(""))
      throw new NamingException("password is either NULL or blank");

    String DN = new String();
    String sserver = server.substring(0, server.indexOf(":", 10)) + ":636";
    server = server.substring(0, server.indexOf(":", 10)) + ":389";

    Properties env1 = new Properties();
    env1.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env1.put("java.naming.ldap.derefAliases", "never");
    env1.put("java.naming.ldap.version", ldap_version); 
    env1.put(Context.REFERRAL, "follow");
    env1.put("java.naming.ldap.referral.bind", "true");

    if (bluepages_ldap_server.endsWith("636")) {
        env1.put(Context.SECURITY_PROTOCOL, "ssl");
        env1.put(Context.PROVIDER_URL,sserver);    
    } else {
        env1.put(Context.PROVIDER_URL,server);    
    }

    Vector v = new Vector();
    Hashtable h = new Hashtable();

    String attrlist[] = {"uid"};
    String filter = "(" + attrname + "=" + attrval + ")";
    SearchControls constraints = new SearchControls();
    constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
    constraints.setTimeLimit(SEARCHTIMELIMIT);
    constraints.setReturningAttributes(attrlist);

    DirContext ctx = null;
    NamingEnumeration results = null;

    try {
        ctx = new InitialDirContext(env1);
        results = ctx.search(SEARCH_BASE, filter ,constraints); 

        SearchResult sr;
        Attributes attrs;
        Attribute attr;
        Enumeration vals;
        Object q;
        while (results.hasMore())
        {
          sr = (SearchResult) results.next();
          attrs = sr.getAttributes();
          //snag the DN
          DN = sr.getName() + ","+ SEARCH_BASE;

          h = new Hashtable();            
          for (NamingEnumeration ae = attrs.getAll(); ae.hasMore();)
          {
            attr = (Attribute) ae.next();                            
            vals = attr.getAll();                           
            q = vals.nextElement();
            try
            {
              h.put(attr.getID(),(String)q);
            } //put the String in the hash
            catch (ClassCastException cce)      //it's not a String, so "decode" it (int'l chars)
            {
              try
              {
                byte b[] = (byte[])q;
                String s = new String(b,"ISO-8859-1"); //"UTF8");  //UTF8 doesn't work for "palmero"
                h.put(attr.getID(),s); //put the string in the hash
              }
              catch (UnsupportedEncodingException ue)
              {
              }
            }
          }  
          if (h.size()>0) v.addElement(h); //no reason to add the hash if it is empty        
        }
    } finally {
        cleanupNamingEnum(results);
        cleanupCtx(ctx);    
    }

    if (v.size()==0)
    {      
      return false;
    }
    else
      if (v.size()>1)
    {      
      return false;
    }

    Properties env = new Properties();
    
    
    env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env.put("java.naming.ldap.derefAliases", "never");
    env.put(Context.PROVIDER_URL,sserver);    
    env.put("java.naming.ldap.version", ldap_version); 
    env.put(Context.SECURITY_AUTHENTICATION, "simple");
    env.put(Context.SECURITY_PRINCIPAL, DN);
    env.put(Context.SECURITY_CREDENTIALS, pw);
    env.put(Context.REFERRAL, "follow");
    env.put("java.naming.ldap.referral.bind", "true");
    env.put(Context.SECURITY_PROTOCOL, "ssl");

    try {
        ctx = new InitialDirContext(env);
    } catch (AuthenticationException ae) {
        throw new NamingException(ae.toString());
    } finally {
        cleanupCtx(ctx);
    }

    return true;
  }

  /**
    * Gets a person's Distinguished Name (DN)
    *
    * @param ctx An already established DirContext
    * @param filter The LDAP search filter to use to build the DN ex: (mail=xxxxx@us.ibm.com)
    * @exception NamingException An error has occurred
    * @throws NamingException An error has occurred
    * @return The DN of the person
    */
  private String getDNByFilter(DirContext ctx, String filter) throws NamingException
  {
    String DN = null;
    NamingEnumeration results = null;

    try {
        String attrlist[] = {"uid"};
        SearchControls constraints = new SearchControls();
        constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
        constraints.setReturningAttributes(attrlist);  
        constraints.setTimeLimit(SEARCHTIMELIMIT);
        results = ctx.search(SEARCH_BASE, filter ,constraints);  
        Enumeration vals;

        if (results.hasMore()) {
            SearchResult sr = (SearchResult) results.next();
            DN = sr.getName() + ","+ SEARCH_BASE;
        }
    } finally {
        cleanupNamingEnum(results);
    }
    
    return DN;
  }

    /**
    * Gets an attribute from an entry using a filter
    *
    * @param ctx An already established DirContext
    * @param filter The LDAP search filter to use to get the attribute ex: (mail=xxxxx@us.ibm.com)
    * @param attr The attribute to return ex: uid
    * @exception NamingException An error has occurred
    * @throws NamingException An error has occurred
    * @return The DN of the person
    */
  private String getAttrByFilter(DirContext ctx, String filter, String attr) throws NamingException
  {
    String attrlist[] = {attr};
    SearchControls constraints = new SearchControls();
    constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
    constraints.setReturningAttributes(attrlist);  
    constraints.setTimeLimit(SEARCHTIMELIMIT);

    NamingEnumeration results = null;
    

    try {
        results = ctx.search(SEARCH_BASE, filter ,constraints);
        
        if (results.hasMore()) {
            Enumeration vals;
            SearchResult sr = (SearchResult) results.next();

            Attributes attrs = sr.getAttributes();

            Attribute attrib = (BasicAttribute)attrs.get(attr);                 
            NamingEnumeration ne = attrib.getAll();
            String str_results = (String)ne.nextElement();
            
            return str_results;
        } else {
            return "";
        }
    } finally {
        cleanupNamingEnum(results);
    }
  }

  private NamingEnumeration getNEAttrByFilter(DirContext ctx, String filter, String base, String attrlist[]) throws NamingException
  {
    SearchControls constraints = new SearchControls();
    constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
    constraints.setReturningAttributes(attrlist);  
    constraints.setTimeLimit(SEARCHTIMELIMIT);

    NamingEnumeration results = ctx.search(base, filter ,constraints);
    return results;
  }

  /**
    * Creates and returns an LDAP connection
    *
    * @param server The LDAP server - for example: ldap://swatv3.bluepages.ibm.com
    * @param version The LDAP version
    * @return An established DirContext
    * @exception NamingException An error has occurred
    * @throws NamingException An error has occurred
    */
  private DirContext getLdapConnection(String server, String version) throws NamingException {
    DirContext ctx = null;
      
    Properties env = new Properties();
    env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env.put("java.naming.ldap.derefAliases", "never");
    env.put(Context.SECURITY_AUTHENTICATION, "simple");    
    env.put(Context.PROVIDER_URL,            server);
    env.put("java.naming.ldap.version",      version);
    env.put(Context.REFERRAL,                "follow");
    env.put("java.naming.ldap.referral.bind","true");
    env.put("com.sun.jndi.ldap.connect.timeout", (new Integer(CONNTIMEOUT)).toString());

    if (server.startsWith("ldaps://") || server.endsWith("636")) {
        env.put(Context.SECURITY_PROTOCOL, "ssl");                            
    }

    ctx = new InitialDirContext(env);
    
    return ctx;     
  }


  /**
    * Checks group membersip using the default ldap server and the default depth
    *
    * @param email Internet email address to check
    * @param group The group to check  
    * @return ReturnCode based on result
    */
  public ReturnCode inAGroup(String email, String group)
  {
    return inAGroup(email, group, DEFAULT_DEPTH);
  }

  /**
    * Checks group membersip
    *
    * @param email Internet email address to check
    * @param group The group to check
    * @param depth how many levels below to check
    * @return ReturnCode based on result
    */
  public ReturnCode inAGroup(String email, String group, int depth)
  {
      return inAGroup("mail", email, group, depth);
  }

  /**
    * Checks group membersip
    *
    * @param attrname attribute to search on
    * @param attrval value to search for
    * @param group The group to check
    * @param depth how many levels below to check
    * @return ReturnCode based on result
    */
  public ReturnCode inAGroup(String attrname, String attrval, String group, int depth)
  {    
    DirContext bp_ctx = null;
    DirContext bg_ctx = null;

    String userDN;    
    boolean found[] = new boolean[1];
    found[0] = false;

    NamingEnumeration results = null;

    try
    {      
      bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);
      userDN = getDNByFilter(bp_ctx, "(" + attrname + "=" + attrval + ")");

      if (userDN == null || userDN.length() == 0) {
          return cwaapi.NO_RECORD_ERROR;
      }

      String attrlist[] = {"cn"};
      String filter = "(&(cn=" + group + ")(uniquemember=" + substitute(userDN, "\\", "\\5c") + "))";
      
      SearchControls constraints = new SearchControls();
      constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
      constraints.setReturningAttributes(attrlist);
      constraints.setTimeLimit(SEARCHTIMELIMIT);
      

      if (bluepages_ldap_server.equalsIgnoreCase(bluegroups_ldap_server)) {
          bg_ctx = bp_ctx;
      } else {
          bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
      }      
      
      results = bg_ctx.search(MEMBERLIST_BASE, filter ,constraints);  

      Enumeration vals;
      if (results.hasMore())
      {
        return cwaapi.SUCCESS;
      }
      else
      {
        //look in groups below if depth>0
        //search and get all groups in this group
        //System.out.println("checking depth: " + depth);
        if (depth>0)
        {
          String attrlist2[] = { "uniqueGroup"};
          filter = "(cn=" + group + ")";
          constraints = new SearchControls();
          constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
          constraints.setReturningAttributes(attrlist2);  
          constraints.setTimeLimit(SEARCHTIMELIMIT);
          
          results = bg_ctx.search(MEMBERLIST_BASE, filter ,constraints);

          if (results.hasMore()) {
              BasicAttribute attr;
              NamingEnumeration ne = null;

              try {
                  SearchResult sr = (SearchResult) results.next();

                  Attributes attrs = sr.getAttributes();

                  if ( attrs == null ) {
                      return cwaapi.NOT_IN_GROUP;
                  } else {
                      attr = (BasicAttribute)attrs.get("uniqueGroup");
                  }

                  if ( attr == null) {
                      return cwaapi.NOT_IN_GROUP;
                  } else {
                      ne = attr.getAll();
                  }            

                  //if it finds something, search that group!
                  while ( ne.hasMore() )
                  {
                    String grpDN = (String)ne.nextElement();

                    if (useThreaded)
                    {
                      if (found[0]) {
                          return cwaapi.SUCCESS;
                      }

                      Thread t1 = new Thread(thread_group, new GroupSearcher(thread_group, depth-1, grpDN, userDN, found, bluepages_ldap_server, bluegroups_ldap_server, ldap_version));
                      t1.start();
                    }
                    else
                    {
                      ReturnCode rc = inAGroup(depth-1, userDN, grpDN);
                      if (rc == cwaapi.SUCCESS || rc == cwaapi.LDAP_ERROR) {
                          return rc;
                      }

                    }
                  }
              } finally {
                  cleanupNamingEnum(ne);
              }

              if (useThreaded)
              {
                while (thread_group.activeCount()>0)
                {                  
                  try
                  {
                    Thread.sleep(500);
                  }
                  catch (Exception e)
                  {
                  }
                }

                if (found[0]) {
                    return cwaapi.SUCCESS;
                }

              }
          } else {              
              return cwaapi.GROUP_DOES_NOT_EXIST;

          }
        }

      }
    }
    catch (NamingException ne)
    {        
      System.err.println("[cwa2] inAGroup: " + ne);
      System.out.println("[cwa2] inAGroup: " + ne);
      ne.printStackTrace();
      return cwaapi.LDAP_ERROR;
    }
    finally {        
        cleanupNamingEnum(results);

        cleanupCtx(bp_ctx);
        cleanupCtx(bg_ctx);
    }
    
    return cwaapi.NOT_IN_GROUP;
  }

  /**
    * Checks group membersip - This is the recursive call need GroupDN and not CN! Notice the searchbase is the groupDN
    *
    * @param depth how many levels below to check
    * @param userDN Distinguised name of user, example: uid=123456789,c=us,ou=bluepages,o=ibm.com
    * @param groupDN Distinguished name of group, example: cn=test,ou=memberlist,ou=ibmgroups,o=ibm.com
    * @return ReturnCode based on result
    */
  private ReturnCode inAGroup(int depth, String userDN, String groupDN)
  {
    DirContext bg_ctx = null;
    NamingEnumeration results = null;
    
    try
    {
      String attrlist[] = {"cn"};
      String filter = "(uniquemember=" + substitute(userDN, "\\", "\\5c") + ")";
      SearchControls constraints = new SearchControls();
      constraints.setSearchScope(SearchControls.OBJECT_SCOPE);
      constraints.setReturningAttributes(attrlist);
      constraints.setTimeLimit(SEARCHTIMELIMIT);
      
      bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
      results = bg_ctx.search(groupDN, filter ,constraints);  
      
      Enumeration vals;
      if (results.hasMore())
      {
        //userDN is in group .. party!
        return cwaapi.SUCCESS;
      }
      else
      {
        //look in groups below if depth>0
        //search and get all groups in this group
        //System.out.println("checking depth: " + depth);
        if (depth>0)
        {
          String attrlist2[] = { "uniqueGroup"};
          filter = "(objectclass=groupOfUniqueNames)";
          constraints = new SearchControls();
          constraints.setSearchScope(SearchControls.OBJECT_SCOPE);
          constraints.setReturningAttributes(attrlist2);  
          constraints.setTimeLimit(SEARCHTIMELIMIT);
          
          results = bg_ctx.search(groupDN, filter ,constraints);

          if (results.hasMore()) {
              BasicAttribute attr;
              NamingEnumeration ne;
              SearchResult sr = (SearchResult) results.next();
              Attributes attrs = sr.getAttributes();

              if ( attrs == null ) {
                  return cwaapi.NOT_IN_GROUP;
              } else {
                  attr = (BasicAttribute)attrs.get("uniqueGroup");
              }

              if ( attr == null) {
                  return cwaapi.NOT_IN_GROUP;
              } else {
                  ne = attr.getAll();
              }            

              //if it finds something, search that group!
              while ( ne.hasMore() )
              {
                String grpDN = (String)ne.nextElement();                

                ReturnCode rc = inAGroup(depth-1, userDN, grpDN);
                if (rc == cwaapi.SUCCESS || rc == cwaapi.LDAP_ERROR) {
                    return rc;
                }

              }
          } else {
              return cwaapi.GROUP_DOES_NOT_EXIST;
          }
        }
      }
    }
    catch (NamingException ne)
    {      
      System.err.println("[cwa2] inAGroup: " + ne);
      ne.printStackTrace();      
    }
    finally 
    {
      cleanupNamingEnum(results);
      cleanupCtx(bg_ctx);
    }

    return cwaapi.NOT_IN_GROUP;
  }


  /**
    * Checks group membersip in any of the provided groups using the default ldap server
    * and the default depth
    *
    * @param email Internet email address to check
    * @param group The group to check   
    * @return ReturnCode based on result
    */
  public ReturnCode inAnyGroup(String email, Vector group)
  {
    return inAnyGroup("mail",email,group,DEFAULT_DEPTH);
  }

  /**
    * Checks group membersip in any of the provided groups using the default ldap server
    * and the default depth
    *
    * @param email Internet email address to check
    * @param group The group to check   
    * @return ReturnCode based on result
    */
  public ReturnCode inAnyGroup(String email, Vector group, int depth)
  {
    return inAnyGroup("mail",email,group,depth);
  }

  /**
    * Checks group membersip in any of the provided groups
    *      
    * @param attrname attribute to search on
    * @param attrval value to search for
    * @param group A Vector of group names
    * @param depth The number of groups deep to search
    * @return ReturnCode based on result
    */
  public ReturnCode inAnyGroup(String attrname, String attrval, Vector group, int depth)
  {
    DirContext bp_ctx = null;
    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    String filter = "(&(|";

    try {
        Vector subgroups = new Vector();

        for (int i=1; i<=group.size(); i++) {
            listGroups((String)group.elementAt(i-1), depth-1, subgroups);
        }

        for (int i=1; i<=subgroups.size(); i++) {
            group.addElement((String)subgroups.elementAt(i-1));
        }

        bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

        if (bluepages_ldap_server.equalsIgnoreCase(bluegroups_ldap_server)) {
            bg_ctx = bp_ctx;
        } else {
            bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
        }        

        String memberdn = getDNByFilter(bp_ctx, "(" + attrname + "=" + attrval + ")");
        String g;

        for (int i=1; i<=group.size();i++)
        {
          g = (String)group.elementAt(i-1);
          filter = filter + "(cn=" + g + ")";

          if (i % 100 == 0) {
              filter = filter + ")";

              filter = filter + "(uniquemember=" + substitute(memberdn, "\\", "\\5c") + "))";

              results = getNEAttrByFilter(bg_ctx, filter, MEMBERLIST_BASE, new String[]{"1.1"});

              if (results.hasMore()) {
                  return cwaapi.SUCCESS;
              } else {
                  filter = "(&(|";                  
              }
          }          
        }

        if (filter.length() > 10) {
            filter = filter + ")";

            filter = filter + "(uniquemember=" + substitute(memberdn, "\\", "\\5c") + "))";

            results = getNEAttrByFilter(bg_ctx, filter, MEMBERLIST_BASE, new String[]{"1.1"});

            if (results.hasMore()) {
                return cwaapi.SUCCESS;
            }
        }
    } catch (NamingException ne) {
        System.err.println("[cwa2] inAnyGroup: " + ne);
        ne.printStackTrace();
        return cwaapi.LDAP_ERROR;        
    }
    finally {        
        cleanupNamingEnum(results);

        cleanupCtx(bp_ctx);
        cleanupCtx(bg_ctx);
    }

    return cwaapi.NOT_IN_GROUP;
  }

  /**
    * Checks group membersip in all of the provided groups using the default ldap server
    * and the default depth
    *
    * @param email Internet email address to check
    * @param group A Vector of group names
    * @return ReturnCode based on result
    */
  public ReturnCode inAllGroups(String email, Vector group)
  {
    return inAllGroups("mail", email, group, DEFAULT_DEPTH);
  }

  /**
    * Checks group membersip in all of the provided groups using the default ldap server
    *
    * @param email Internet email address to check
    * @param group A Vector of group names
    * @param depth The number of groups deep to search
    * @return ReturnCode based on result
    */
  public ReturnCode inAllGroups(String email, Vector group, int depth)
  {
    return inAllGroups("mail", email, group, depth);
  }

  /**
    * Checks group membersip in all of the provided groups
    *
    * @param attrname attribute to search on
    * @param attrval value to search for
    * @param group A Vector of group names
    * @param depth The number of groups deep to search
    * @return ReturnCode based on result
    */
  public ReturnCode inAllGroups(String attrname, String attrval, Vector group, int depth)
  {
    String g;

    for (int i=0; i<group.size();i++)
    {
      g = (String)group.elementAt(i);
      ReturnCode rc = inAGroup(attrname, attrval, g, depth);
      if (rc == cwaapi.NOT_IN_GROUP || rc == cwaapi.GROUP_DOES_NOT_EXIST || rc == cwaapi.LDAP_ERROR)
        return rc;
    }
    return cwaapi.SUCCESS;
  }

  /*
    These last methods give cwa2.jar functionality to list the members of a group 
    Thanks to clq@us.ibm.com !
  
  */

  /**
    * List the members of the given group (depth of 0), uses member's mail attribute
    *
    * @param group The group name
    * @param list Vector to return the members in
    * @return ReturnCode based on result
    */
  public ReturnCode listMembers(String group, Vector list)
  { return listMembers(group, DEFAULT_DEPTH, list, "mail");}

  /**
    * List the members of the given group (depth of 0), uses specified attribute
    *
    * @param group The group name
    * @param list Vector to return the members in
    * @param returnAttrib The Bluepages attribute to return
    * @return ReturnCode based on result
    */
  public ReturnCode listMembers(String group, Vector list, String attrib)
  { return listMembers(group, DEFAULT_DEPTH, list, attrib);}

  /**
    * List the members of the given group, to the depth specified, uses member's mail attribute
    *
    * @param group The group name
    * @param depth Depth to search 
    * @param list Vector to return the members in
    * @return ReturnCode based on result
    */
  public ReturnCode listMembers(String group, int depth, Vector list)
  { return listMembers(group, depth, list, "mail");}

  /**
    * List the members of the given group, to the depth specified, uses specified attribute
    *
    * @param group The group name
    * @param depth Depth to search 
    * @param list Vector to return the members in
    * @param returnAttrib The Bluepages attribute to return
    * @return ReturnCode based on result
    */
  public ReturnCode listMembers(String group, int depth, Vector list, String returnAttrib)
  {
    ReturnCode rc = null;

    if ( depth >= 0 )
    {
      rc = getMembers(getGroupMLDN(group), depth, list, returnAttrib);
    }    

    return rc;
  }

  /**
    * List the members of the given group, to the depth specified, uses specified attribute
    *
    * @param group The group name
    * @param depth Depth to search 
    * @param list Vector to return the members in where each element is an array. arr[0] = dn, arr[n+1] are returned attributes
    * @param returnAttrib The Bluepages attribute to return
    * @return ReturnCode based on result
    */
  public ReturnCode listMembers(String group, int depth, Vector list, String[] returnAttrib)
  {
    ReturnCode rc = null;
    
    if ( depth >= 0 )
    {
      rc = getMembers(getGroupMLDN(group), depth, list, returnAttrib);
    }
    
    return rc;
  }


  /**
    * List the admins of the given group, to the depth specified, uses specified attribute
    *
    * @param group The group name
    * @param depth Depth to search 
    * @param list Vector to return the members in
    * @param returnAttrib The Bluepages attribute to return
    * @return ReturnCode based on result
    */
  public ReturnCode listAdmins(String group, int depth, Vector list, String returnAttrib)
  {
    ReturnCode rc = null;
    
    if ( depth >= 0 )
    {
      rc = getAdmins(getGroupMetaDN(group), depth, list, returnAttrib);
    }

    return rc;
  }

  private ReturnCode getMembers(String groupDN, int depth, Vector list, String returnAttrib)
  {
    DirContext bp_ctx = null;
    DirContext bg_ctx = null;
    NamingEnumeration results = null;
    
    String attriblist[] = {returnAttrib};

    if ( depth >= 0 )
    {
      try {
          bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

          if (bluepages_ldap_server.equalsIgnoreCase(bluegroups_ldap_server)) {
              bg_ctx = bp_ctx;
          } else {
              bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
          }        

          if (returnAttrib.equalsIgnoreCase("dn")) {
              BasicAttributes attrs = null;
              BasicAttribute attr = null;
             
              attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN, new String[]{"uniquemember"});

              if ( attrs != null ) {
                  attr = (BasicAttribute)attrs.get("uniquemember");

                  if ( attr != null ) {
                      results = attr.getAll();

                      while (results.hasMore()) {
                          list.addElement(results.nextElement());
                      }
                  }
              }

              attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN,new String[]{"uniquegroup"});

              if ( attrs == null ) {
                  return cwaapi.SUCCESS;
              }

              attr = (BasicAttribute)attrs.get("uniquegroup");

              if ( attr == null ) {
                  return cwaapi.SUCCESS;
              }

              results = attr.getAll();

              while ( results.hasMore() ) {                 
                  String memDN = (String) results.nextElement();                  
                  getMembers(memDN, depth-1, list, returnAttrib);
              }
          } else {
              Vector templist = new Vector();
              getMembers(groupDN, depth, templist, attriblist);

              for (int i=0; i<templist.size(); i++) {
                  String[] attrarray = (String[])templist.elementAt(i);

                  // add 2nd element in array since 1st element is DN
                  if (attrarray[1] != null) {
                      list.addElement(attrarray[1]);
                  }                 
              }
          }
      } catch (NamingException ne) {
          System.err.println("[cwa2] getMembers: " + ne);
          ne.printStackTrace();
          return cwaapi.LDAP_ERROR;
      } finally {
          cleanupNamingEnum(results);
          
          cleanupCtx(bp_ctx);
          cleanupCtx(bg_ctx);
      }
    }

    return cwaapi.SUCCESS;
  }

  private ReturnCode getMembers(String groupDN, int depth, Vector list, String[] returnAttrib)
  {
    DirContext bp_ctx = null;
    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    if ( depth >= 0 )
    {
      try {
          bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

          if (bluepages_ldap_server.equalsIgnoreCase(bluegroups_ldap_server)) {
              bg_ctx = bp_ctx;
          } else {
              bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
          }        

          BasicAttributes attrs = null;
          BasicAttribute attr = null;

          attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN, new String[]{"uniquemember"});

          if ( attrs != null ) {
              attr = (BasicAttribute)attrs.get("uniquemember");

              if ( attr != null ) {
                  results = attr.getAll();

                  while ( results.hasMore() ) {
                      String filter = "(|";

                      for (int i = 1; i <= 200; i++) {
                          String memDN = (String) results.nextElement();

                          if (memDN.indexOf("\"") > -1) {
                              memDN = memDN.substring(5,memDN.indexOf("\",c"));
                          } else {
                              if (SEARCH_BASE.toLowerCase().indexOf("bluepages") > -1) {
                                  memDN = memDN.substring(4,memDN.indexOf(",c"));
                              } else {
                                  memDN = memDN.substring(4,memDN.indexOf(",ou"));
                              }
                          }

                          filter = filter + "(uid=" + memDN + ")";                

                          if (results.hasMore() == false) {
                              break;
                          }
                      }

                      if (filter.length() > 2) {
                          filter = filter + ")";

                          NamingEnumeration tempNE = null;

                          try {
                              tempNE = getNEAttrByFilter(bp_ctx, filter, SEARCH_BASE, returnAttrib);

                              if (tempNE != null) {
                                  while (tempNE.hasMore()) {
                                      String[] attrarray = new String[returnAttrib.length + 1];
                                      SearchResult sr = (SearchResult) tempNE.next();
                                      Attributes tempattrs = sr.getAttributes();                    
                                      NamingEnumeration attridNE = tempattrs.getIDs();

                                      attrarray[0] = sr.getName() + "," + SEARCH_BASE;
                                      int i = 1;

                                      while (attridNE.hasMore()) {                        
                                          Attribute tempattrib = tempattrs.get((String)attridNE.next());
                                          
                                          if (tempattrib != null) {                                
                                              attrarray[i] = (String)tempattrib.get();
                                          }

                                          i++;
                                      }

                                      list.addElement(attrarray);
                                  }
                              }
                          } finally {
                              cleanupNamingEnum(tempNE);
                          }

                      }            
                  }
                  
              }            
          }

          attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN,new String[]{"uniquegroup"});
          if ( attrs == null ) {
              return cwaapi.SUCCESS;
          }
          attr = (BasicAttribute)attrs.get("uniquegroup");
          if ( attr == null ) {
              return cwaapi.SUCCESS;
          }

          results = attr.getAll();

          while ( results.hasMore() )
          {
            String memDN = (String) results.nextElement();
            getMembers(memDN, depth-1, list, returnAttrib);
          }
      } catch (NamingException ne) {
          System.err.println("[cwa2] getMembers: " + ne);
          ne.printStackTrace();
          return cwaapi.LDAP_ERROR;
      } finally {
          cleanupNamingEnum(results);
          cleanupCtx(bp_ctx);
          cleanupCtx(bg_ctx);
      }

    }

    return cwaapi.SUCCESS;
  }

  private ReturnCode getAdmins(String groupDN, int depth, Vector list, String returnAttrib)
  {
    DirContext bp_ctx = null;
    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    String attriblist[] = {returnAttrib};

    if ( depth >= 0 )
    {
      try {
          bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

          if (bluepages_ldap_server.equalsIgnoreCase(bluegroups_ldap_server)) {
              bg_ctx = bp_ctx;
          } else {
              bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
          }        

          BasicAttributes attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN, new String[]{"admin"});
          if ( attrs == null )
            return cwaapi.SUCCESS;
          BasicAttribute attr = (BasicAttribute)attrs.get("admin");
          if ( attr == null )
            return cwaapi.SUCCESS;

          NamingEnumeration ne = null;

          try {
              ne = attr.getAll();

              if (returnAttrib.equalsIgnoreCase("dn")) {
                  while (ne.hasMore()) {
                      list.addElement(ne.nextElement());
                  }          
              } else {
                  while ( ne.hasMore() )
                  {
                    String filter = "(|";

                    for (int i = 1; i <= 200; i++) {
                        String memDN = (String) ne.nextElement();

                        if (memDN.indexOf("\"") > -1) {
                            memDN = memDN.substring(5,memDN.indexOf("\",c"));
                        } else {
                            if (SEARCH_BASE.toLowerCase().indexOf("bluepages") > -1) {
                                memDN = memDN.substring(4,memDN.indexOf(",c"));
                            } else {
                                memDN = memDN.substring(4,memDN.indexOf(",ou"));
                            }
                        }

                        filter = filter + "(uid=" + memDN + ")";

                        if (ne.hasMore() == false) {
                            break;
                        }
                    }        

                    if (filter.length() > 2) {
                        filter = filter + ")";

                        results = getNEAttrByFilter(bp_ctx, filter, SEARCH_BASE, attriblist);

                        if (results != null) {
                            while (results.hasMore()) {
                                SearchResult sr = (SearchResult) results.next();
                                Attributes tempattrs = sr.getAttributes();
                                Attribute tempattrib = tempattrs.get(returnAttrib);

                                if (tempattrib != null) {                                
                                    list.addElement((String)tempattrib.get());
                                }                        
                            }
                        }                    
                    }

                  }
              }
          } finally {
              cleanupNamingEnum(ne);
          }
      } catch (NamingException ne) {
          System.err.println("[cwa2] getAdmins: " + ne);
          ne.printStackTrace();
          return cwaapi.LDAP_ERROR;
      } finally {
          cleanupNamingEnum(results);

          cleanupCtx(bp_ctx);
          cleanupCtx(bg_ctx);
      }
    }

    return cwaapi.SUCCESS;
  }


  /**
    * List all of the groups a person is a member of. 
    *
    * @param userDN DN of person, for example, uid=123456896,c=us,ou=bluepages,o=ibm.com
    * @param attrlist return attributes
    * @return Vector results
    */
  public Vector listAllGroupsMemberOf(String userDN, String attrlist[])
  {    
    DirContext bg_ctx = null;
    NamingEnumeration results = null;
    
    Vector list = new Vector();

    try
    {      
      String filter = "(uniquemember=" + substitute(userDN, "\\", "\\5c") + ")";
      bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);

      results = getNEAttrByFilter(bg_ctx, filter, MEMBERLIST_BASE, attrlist);

      Enumeration vals;
      while (results.hasMore()) {
         Vector groupchar = new Vector();
         SearchResult sr  = (SearchResult) results.next();
         Attributes attrs = sr.getAttributes();
         
         for (NamingEnumeration ae = attrs.getAll(); ae.hasMore();)
         {
           Attribute attr = (Attribute) ae.next();
           groupchar.addElement(attr.get());
         }
         
         list.addElement(groupchar);
      }
    } catch (NamingException ne) {
      System.err.println("[cwa2] listAllGroupsMemberOf: " + ne);
      ne.printStackTrace();
    } finally {
        cleanupNamingEnum(results);
        
        cleanupCtx(bg_ctx);
    }

    return list;
  }

  /**
    * List all of the groups of which it is a subgroup of
    *
    * @param gName name of group
    * @param attrlist return attributes
    * @return Vector results
    */
  public Vector listAllGroupsSubGroupOf(String gName, String attrlist[])
  {
    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    Vector list = new Vector();

    try
    {      
      String filter = "(uniquegroup=cn=" + substitute(gName, "\\", "\\5c") + "," + MEMBERLIST_BASE + ")";
      bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);

      results = getNEAttrByFilter(bg_ctx, filter, MEMBERLIST_BASE, attrlist);

      Enumeration vals;
      while (results.hasMore()) {
         Vector groupchar = new Vector();
         SearchResult sr  = (SearchResult) results.next();
         Attributes attrs = sr.getAttributes();
         
         for (NamingEnumeration ae = attrs.getAll(); ae.hasMore();)
         {
           Attribute attr = (Attribute) ae.next();
           groupchar.addElement(attr.get());
         }
         
         list.addElement(groupchar);
      }
    } catch (NamingException ne) {
      System.err.println("[cwa2] listAllGroupsSubGroupOf: " + ne);
      ne.printStackTrace();
    } finally {
        cleanupNamingEnum(results);
        
        cleanupCtx(bg_ctx);
    }

    return list;
  }

  /**
    * List the groups in the given group (depth of 0)
    *
    * @param group The group name
    * @param list Vector to return the groups in
    * @return ReturnCode based on result
    */
  public ReturnCode listGroups(String group, Vector list)
  { return listGroups(group, DEFAULT_DEPTH, list, "cn");}

  /**
    * List the groups in the given group, using the given depth
    *
    * @param group The group name
    * @param depth The depth to search (depth=1, search 1 level deep .... 0 is default)
    * @param list Vector to return the groups in
    * @return ReturnCode based on result
    */
  public ReturnCode listGroups(String group, int depth, Vector list)
  { return listGroups(group, depth, list, "cn");}

  /**
    * List the groups in the given group, using the default depth, returning the give attribute
    *
    * @param group The group name
    * @param list Vector to return the groups in
    * @return ReturnCode based on result
    */
  public ReturnCode listGroups(String group, Vector list, String returnAttrib )
  { return listGroups(group, DEFAULT_DEPTH, list, returnAttrib);}

  /**
    * List the groups in the given group, with the given depth
    *
    * @param group The group name
    * @param depth The depth to search (depth=1, search 1 level deep .... 0 is default)
    * @param list Vector to return the groups in
    * @param returnAttrib The attribute to return (default is "cn")
    * @return ReturnCode based on result
    */
  public ReturnCode listGroups(String group, int depth, Vector list, String returnAttrib)
  { 
    ReturnCode rc = null;

    if ( depth >= 0 )
    {
      rc = getGroups(getGroupMLDN(group), depth, list, returnAttrib);
    }
   
    return rc;
  }

 private ReturnCode getGroups(String groupDN, int depth, Vector list, String returnAttrib)
  {
    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    if ( depth >= 0 )
    {
      try {
          bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
          BasicAttributes attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN, new String[]{"uniquegroup"});

          if ( attrs == null ) {
              return cwaapi.SUCCESS;
          }
            
          BasicAttribute attr = (BasicAttribute)attrs.get("uniquegroup");

          if ( attr == null ) {
              return cwaapi.SUCCESS;
          }
            
          
          results = attr.getAll();

          while ( results.hasMore() )
          {
            String grpDN = (String) results.nextElement();
            list.addElement(getAttrib(bg_ctx, grpDN, returnAttrib));
          }

          attrs = (BasicAttributes)bg_ctx.getAttributes(groupDN,new String[]{"uniquegroup"});
          if ( attrs == null )
            return cwaapi.SUCCESS;
          attr = (BasicAttribute)attrs.get("uniquegroup");
          if ( attr == null )
            return cwaapi.SUCCESS;
          results = attr.getAll();

          while ( results.hasMore() )
          {
            String grpDN = (String) results.nextElement();        
            getGroups(grpDN, depth-1, list, returnAttrib);
          }
      } catch ( javax.naming.NamingException ne ) {
          System.err.println("[cwa2] getGroups: " + ne);
          ne.printStackTrace();
          return cwaapi.LDAP_ERROR;
      } finally {
          cleanupNamingEnum(results);
          
          cleanupCtx(bg_ctx);
      }
    }

    return cwaapi.SUCCESS;
  }

  private String getAttrib(DirContext ctx, String DN, String attrib) throws javax.naming.NamingException
  {
    if ( DN == null )
    {
      return "";
    }

    try
    {
      BasicAttributes attrs = (BasicAttributes)ctx.getAttributes(DN,new String[]{attrib});
      if ( attrs == null )
        return DN;
      BasicAttribute attr = (BasicAttribute)attrs.get(attrib);
      if ( attr == null )
        return DN;
      return(String) attr.get();
    }
    catch ( Throwable t )
    {
      return DN;
    }

  }

  /**
  * Takes group name and builds the DN that represents the group.
  */
  //this is not the base way to do this !!
  private String getGroupMLDN(String groupName) {
    if ( groupName.startsWith("cn=") )
    {
      return groupName;
    }
    else
    {
      return("cn="+escape(groupName)+"," + MEMBERLIST_BASE);
    }
  }

  /**
  * Takes group name and builds the DN that represents the group.
  */
  //this is not the base way to do this !!
  private String getGroupMetaDN(String groupName) {
    if ( groupName.startsWith("cn=") )
    {
      return groupName;
    }
    else
    {
      return("cn="+escape(groupName)+"," + METADATA_BASE);
    }
  }

  /**
   * Escape special characters
   * 
   * @param str
   * @return escaped string
   **/
  private String escape(String str) {
      String temp = str;
      temp = substitute(temp, "\\", "\\\\");
      temp = substitute(temp, "=", "\\=");
      temp = substitute(temp, "+", "\\+");
      temp = substitute(temp, "<", "\\<");
      temp = substitute(temp, ">", "\\>");
      temp = substitute(temp, "#", "\\#");
      temp = substitute(temp, ";", "\\;");
      temp = substitute(temp, ",", "\\,");

      return temp;
  }


  /**
  * Substitute a String for a String in a message.
  *
  * @param message  The message where the values are being substituted.
  * @param oldvalue The value being substituted.
  * @param newvalue The value to substitute with.
  * @return The message after substitution.
  */
  private String substitute(String message,
                                                    String oldvalue,
                                                    String newvalue)
  {
          StringBuffer newMsg = new StringBuffer();
          int index = 0;
          int fromIndex = 0;

          while ( (index = message.indexOf(oldvalue, fromIndex)) >= 0 ) {
                  if ( (index - fromIndex) > 0 ) {
                          newMsg.append(message.substring(fromIndex, index));
                  }
                  newMsg.append(newvalue);
                  fromIndex = index + oldvalue.length();
          }

          newMsg.append(message.substring(fromIndex));

          return newMsg.toString();
  }

  /**
    * Runs the URL.
    * 
    * @param authInfo Name and password in the format name:password. 
    *                 For example, kyhsiao@us.ibm.com:password
    * @param query Task that is to be executed. 
    * 
    * @return int based on return code output from executing the https. The return codes are defined as this:
    *    0=success, 
    *    1=not authorized, 
    *    2=Group is not a memberlist, 
    *    3=No members were specified for add/delete
    *    4=Error occured while performing function, 
    *    5=An Unexpected Error Occurred
    */
  private ReturnCode executeURL(String authInfo, String query) 
  {
    URL textURL = null;
    int rc = -1;
    String rcMessage = "";
    int retries = 3;

    byte returnVal[] = new byte[1];

    try
    {
      textURL = new URL(HTTPS_BASE); 
    }
    catch (MalformedURLException MUE)
    {      
      System.err.println("[cwa2] executeURL: " + MUE);
      MUE.printStackTrace();
      return cwaapi.URL_ERROR;
    }

    try
    {
      String sReturnCode = "";
      URLConnection connection = textURL.openConnection();

      InputStream in = null;

      // get redirected URL
      HttpURLConnection http = (HttpURLConnection)connection;
      //http.setInstanceFollowRedirects(false);      
      String loc = http.getHeaderField("Location");      

      try
      {
        if (loc != null && loc.length() > 0) {
            textURL = new URL(loc); 
        }        
      }
      catch (MalformedURLException MUE)
      {        
        System.err.println("[cwa2] executeURL: " + MUE);
        MUE.printStackTrace();
        return cwaapi.URL_ERROR;
      }

      while (retries > 0) {
          connection = textURL.openConnection();
          connection.setUseCaches(false);
          connection.setDefaultUseCaches(false);
          connection.setDoOutput(true);

          String encoding = new sun.misc.BASE64Encoder().encode (authInfo.getBytes());
          connection.setRequestProperty ("Authorization", "Basic " + encoding);          

          PrintWriter outGmRequest = new PrintWriter( connection.getOutputStream(), true );

          outGmRequest.print(query);
          outGmRequest.close();

          in = connection.getInputStream();

          if (in.read(returnVal) != -1)
          {  
             try {                 
                 rc = Integer.parseInt((new String(returnVal)).trim());                 

                 if (rc == 0 || rc == 1) {
                     retries = 0;
                 } else {                     
                     System.err.println("Retrying exceuteURL " + retries + ": " + query);
                 }
             } catch (NumberFormatException nfe) {
                 System.err.println("executeURL: error executing URL - " + HTTPS_BASE + "?" + query);

                 System.err.print((new String(returnVal)));

                 while (in.read(returnVal) != -1) {
                     System.err.print((new String(returnVal)));
                 }

                 retries = 0;
             }

          }

          retries--;          
          in.close();
      }

      if (rc == 4 || rc == 5) {
          while (in.read(returnVal) != -1) {
              rcMessage = rcMessage + (new String(returnVal));
          }

          rcMessage = rcMessage.trim();
      }
    }
    catch (IOException IOE)
    {      
      System.err.println("[cwa2] executeURL: " + IOE);
      IOE.printStackTrace();
      return cwaapi.URL_ERROR;
    }

    //rc contains the XML return code
    //change that into a cwaapi.ReturnCode
    switch(rc)
    {      
      case 0: return cwaapi.SUCCESS;
      case 1: return new ReturnCode(1, "Not Authorized");
      case 2: return new ReturnCode(2, "Group is not a memberlist or group is in a buildlock");      
      case 3: return new ReturnCode(3, "No members / filters specified");
      case 4: return new ReturnCode(4, rcMessage);
      default: return new ReturnCode(5, rcMessage);
    }
  }    

  /**
    * Creates a new group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added. 
    * @param desc Description of group. 
    * @param Acc View Access level. If 0, everyone can see, else only Owner/Admins can see. 
    * @param year Expiration year. 
    * @param mth Expiration month. 
    * @param day Expiration day. 
    * 
    * @return int based on return code output from executeURL.
    */  
  public ReturnCode createGroup(String userName, String pw, String groupName, String desc, int Acc, int year, int mth, int day) 
  {
      String acType;

      if (Acc == 0)
        acType = EVERYONE;
      else
        acType = OWN_ADMIN;  
      
      return createGroup(userName, pw, groupName, desc, acType, year, mth, day, MEMBERLIST);
  }

  /**
    * Creates a new group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added. 
    * @param desc Description of group. 
    * @param Acc View Access level. If 0, everyone can see, else only Owner/Admins can see. 
    * @param year Expiration year. 
    * @param mth Expiration month. 
    * @param day Expiration day. 
    * 
    * @return int based on return code output from executeURL.
    */  
  public ReturnCode createGroup(String userName, String pw, String groupName, String desc, String Acc, int year, int mth, int day, String mode) 
  {
    ReturnCode rc;

    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);

    rc = executeURL(authInfo, "task=GoNew&selectOn=" + gName + "&gDesc=" + desc + "&mode=" + mode + "&vAcc=" + Acc + "&Y=" + year + "&M=" + mth + "&D=" + day + "&API=1");   

    return rc;
  }

  /**
    * Deletes a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted.
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode deleteGroup(String userName, String pw, String groupName) 
  {
    ReturnCode rc;

    if (!groupExist(groupName))
      return cwaapi.GROUP_DOES_NOT_EXIST;

    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);

    rc = executeURL(authInfo, "task=GoDel&gName=" + gName + "&API=1");   

    return rc;
  }

  /**
    * Adds a member to a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param mail person to add such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addMember(String userName, String pw, String groupName, String mail) {
      return addMember(userName, pw, groupName, "mail", mail);

  }

  /**
    * Adds a member to a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval value to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addMember(String userName, String pw, String groupName, String attrname, String attrval) 
  {
    return addMember(userName, pw, groupName, attrname, new String[]{attrval});
  }

  /**
    * Adds a member to a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval array of values to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addMember(String userName, String pw, String groupName, String attrname, String[] attrval) 
  {
    DirContext bp_ctx = null;    

    ReturnCode rc;
    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);

    String uri = "gName=" + gName + "&task=Members&";    

    try {
        bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

        for (int i = 0; i < attrval.length; i++) {
            if (attrval[i] != null && attrval[i].length() > 0) {
                String UID = new String();    
        
                if (attrname.equalsIgnoreCase("uid")) {
                    UID = attrval[i];
                } else if (attrname.equalsIgnoreCase("notesemail")) {
                    String temp = "";
    
                    if (attrval[i].indexOf("OU=") < 1) {
                        temp = "CN=" + attrval[i].substring(0, attrval[i].indexOf("/") + 1);
    
                        if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) > -1) {
                            temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1);
    
                            if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) != attrval[i].lastIndexOf("/")) {
                                temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1, attrval[i].lastIndexOf("/") + 1);
                            }
                            temp = temp + "O=" + attrval[i].substring(attrval[i].lastIndexOf("/") + 1, attrval[i].length());
                        } else {
                            temp = temp + "O=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].length());
                        }
    
                        UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + temp + "*)", "uid");
                    }
                } else {
                    UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + attrval[i] + ")", "uid");
                }
    
                if (UID.length() < 1) {
                    return cwaapi.NO_RECORD_ERROR;
                }
    
                UID = URLEncoder.encode(UID);    
                
                uri = uri + "mebox=" + UID + "&";
            }
        }    
    } catch(NamingException ne) {
        System.err.println("[cwa2] addMember: " + ne);
        ne.printStackTrace();
        return cwaapi.LDAP_ERROR;
    } finally {
        cleanupCtx(bp_ctx);
    }

    uri = uri + "Select=Add+Members&API=1";
    rc = executeURL(authInfo, uri);   

    return rc;
  }

  /**
    * Adds a member to a group by a file. This goes through HTTPS so its extremely. Not recommended for use. Better to use Bulkload
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param mail mail address of member to be added. 
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addMemberbyFile(String userName, String pw, String groupName, String filtertype, String filelist) throws NamingException 
  {
    DirContext bp_ctx = null;
    NamingEnumeration results = null;

    String attriblist[] = {"uid"};

    ReturnCode rc = new ReturnCode(0, "Success");
    FileReader fr;
    File list = new File(filelist);

    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);
    
    try {
        fr = new FileReader(list);
    } catch (FileNotFoundException fnfe) {
        System.err.println("[cwa2] addMemberbyFile: " + fnfe);
        fnfe.printStackTrace();
        rc = new ReturnCode(8, "File not found!");
        return rc;
    }

    BufferedReader br = new BufferedReader(fr);    

    try {
        bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

        String line = br.readLine();

        if (line != null) {
            if (filtertype.equalsIgnoreCase("uid")) {
                while (line != null) {
                    line = line.trim();
                    rc = executeURL(authInfo, "gName=" + gName + "&task=Members&mebox=" + line + "&Select=Add+Members&API=1");

                    line = br.readLine();
                }
            } else {
                while (line != null) {
                    String filter = "(|";

                    for (int i = 1; i <= 50; i++) {
                        line = line.trim();

                        if (filtertype.equalsIgnoreCase("notesemail")) {
                            
                            if (line.indexOf("OU=") < 1) {
                                String temp = "CN=" + line.substring(0, line.indexOf("/") + 1) + "OU=" + line.substring(line.indexOf("/") + 1, line.indexOf("/", line.indexOf("/") + 1) + 1);

                                if (line.indexOf("/", line.indexOf("/") + 1) != line.lastIndexOf("/")) {
                                    temp = temp + "OU=" + line.substring(line.indexOf("/", line.indexOf("/") + 1) + 1, line.lastIndexOf("/") + 1);
                                }
                                temp = temp + "O=" + line.substring(line.lastIndexOf("/") + 1, line.length());
                                filter = filter + "(" + filtertype + "=" + temp + "*)";
                            }
                        } else {
                            filter = filter + "(" + filtertype + "=" + line + ")";
                        }                                                

                        line = br.readLine();

                        if (line == null) {
                            break;
                        }
                    }                    

                    if (filter.length() > 2) {
                        filter = filter + ")";
                        System.out.println(filter);
                        results = getNEAttrByFilter(bp_ctx, filter, SEARCH_BASE, attriblist);                        

                        if (results != null) {
                            while (results.hasMore()) {
                                SearchResult sr = (SearchResult) results.next();
                                Attributes tempattrs = sr.getAttributes();
                                Attribute tempattrib1 = (BasicAttribute)tempattrs.get("uid");

                                if (tempattrib1 != null) {                                    
                                    rc = executeURL(authInfo, "gName=" + gName + "&task=Members&mebox=" + URLEncoder.encode((String)tempattrib1.get()) + "&Select=Add+Members&API=1");
                                }

                                if (rc.getCode() != 0) {                                    
                                    System.out.println("Failed to add this person:" + (String)tempattrib1.get());
                                    System.out.println(rc.getMessage());
                                }
                            }                                
                        }

                    }

                }

            }
        }
    } catch (IOException IOE) {
        System.err.println("[cwa2] addMemberbyFile: " + IOE);
        IOE.printStackTrace();
        rc = new ReturnCode(9, "IO Error in file!");
        return rc;
    } finally {
        cleanupNamingEnum(results);
        cleanupCtx(bp_ctx);
    }

    return rc;
  }

  /**
    * Adds a subgroup to a group. A subgroup can be any group that already exists
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param subGroupName subGroup to be added to groupName.
    *
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addSubGroup(String userName, String pw, String groupName, String subGroupName)
  {
    ReturnCode rc;
    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);
    String sgName = URLEncoder.encode(subGroupName);

    rc = executeURL(authInfo, "gName=" + gName + "&task=Members&mgbox=" + sgName + "&Select=Add+Members&API=1");

    return rc;
  }
  
  /**
    * Removes a subgroup to a group. A subgroup can be any group that already exists
    *
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be removed from.
    * @param subGroupName subGroup to be removed.
    *
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeSubGroup(String userName, String pw, String groupName, String subGroupName)
  {
    ReturnCode rc;
    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);
    String sgName = URLEncoder.encode(subGroupName);    

    rc = executeURL(authInfo, "Delete=Delete+Checked&gName=" + gName + "&task=DelMem&mgbox=" + sgName + "&API=1");

    return rc;
  }

  /**
    * Removes a member from a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be removed from.
    * @param mail person to delete such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeMember(String userName, String pw, String groupName, String mail) {
      return removeMember(userName, pw, groupName, "mail", mail);
  }

  /**
    * Removes a member from a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be removed from.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval value to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeMember(String userName, String pw, String groupName, String attrname, String attrval) 
  {
    return removeMember(userName, pw, groupName, attrname, new String[]{attrval});
  }

  /**
    * Removes a member from a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be removed from.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval value to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeMember(String userName, String pw, String groupName, String attrname, String[] attrval) 
  {
    DirContext bp_ctx = null;
    
    ReturnCode rc;
    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);

    String uri = "Delete=Delete+Checked&gName=" + gName + "&task=DelMem&";

    try {                
        bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

        for (int i = 0; i < attrval.length; i++) {
            if (attrval[i] != null && attrval[i].length() > 0) {
                String UID = new String();    

                if (attrname.equalsIgnoreCase("uid")) {
                    UID = attrval[i];
                } else if (attrname.equalsIgnoreCase("notesemail")) {
                    String temp = "";

                    if (attrval[i].indexOf("OU=") < 1) {
                        temp = "CN=" + attrval[i].substring(0, attrval[i].indexOf("/") + 1);

                        if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) > -1) {
                            temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1);

                            if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) != attrval[i].lastIndexOf("/")) {
                                temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1, attrval[i].lastIndexOf("/") + 1);
                            }
                            temp = temp + "O=" + attrval[i].substring(attrval[i].lastIndexOf("/") + 1, attrval[i].length());
                        } else {
                            temp = temp + "O=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].length());
                        }

                        UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + temp + "*)", "uid");                
                    }
                } else {
                    UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + attrval[i] + ")", "uid");
                }

                if (UID.length() < 1) {
                    return cwaapi.NO_RECORD_ERROR;
                }

                UID = URLEncoder.encode(UID);

                uri = uri + "mebox=" + UID + "&";
            }
        }
    } catch(NamingException ne) {
        System.err.println("[cwa2] removeMember: " + ne);
        ne.printStackTrace();
        return cwaapi.LDAP_ERROR;
    } finally {
        cleanupCtx(bp_ctx);
    }
    
    uri = uri + "API=1";

    rc = executeURL(authInfo, uri);   

    return rc;
  }

  /**
    * Adds an Administrator to a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param mail person to add such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addAdmin(String userName, String pw, String groupName, String mail) {
      return addAdmin(userName, pw, groupName, "mail", mail);
  }

  /**
    * Adds an Administrator to a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval value to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addAdmin(String userName, String pw, String groupName, String attrname, String attrval)
  {
    return addAdmin(userName, pw, groupName, attrname, new String[]{attrval});
  }

  /**
    * Adds an Administrator to a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be added to.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval array of values to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode addAdmin(String userName, String pw, String groupName, String attrname, String[] attrval)
  {
    DirContext bp_ctx = null;

    ReturnCode rc;
    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);

    String uri = "gName=" + gName + "&task=Administrators&";

    try {                
        bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

        for (int i = 0; i < attrval.length; i++) {
            if (attrval[i] != null && attrval[i].length() > 0) {
                String UID = new String();    

                if (attrname.equalsIgnoreCase("uid")) {
                    UID = attrval[i];
                } else if (attrname.equalsIgnoreCase("notesemail")) {
                    String temp = "";

                    if (attrval[i].indexOf("OU=") < 1) {
                        temp = "CN=" + attrval[i].substring(0, attrval[i].indexOf("/") + 1);

                        if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) > -1) {
                            temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1);

                            if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) != attrval[i].lastIndexOf("/")) {
                                temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1, attrval[i].lastIndexOf("/") + 1);
                            }
                            temp = temp + "O=" + attrval[i].substring(attrval[i].lastIndexOf("/") + 1, attrval[i].length());
                        } else {
                            temp = temp + "O=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].length());
                        }

                        UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + temp + "*)", "uid");
                    }
                } else {
                    UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + attrval[i] + ")", "uid");
                }

                if (UID.length() < 1) {
                    return cwaapi.NO_RECORD_ERROR;
                }

                UID = URLEncoder.encode(UID);    

                uri = uri + "mebox=" + UID + "&";
            }
        }
    } catch(NamingException ne) {
        System.err.println("[cwa2] addAdmin: " + ne);
        ne.printStackTrace();
        return cwaapi.LDAP_ERROR;
    } finally {
        cleanupCtx(bp_ctx);
    }

    uri = uri + "&Submit=Add+Administrators&API=1";
    rc = executeURL(authInfo, uri);   

    return rc;
  }

  /**
    * Deletes an Administrator from a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param mail person to remove such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeAdmin(String userName, String pw, String groupName, String mail) {
      return removeAdmin(userName, pw, groupName, "mail", mail);
  }

  /**
    * Deletes an Administrator from a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval value to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeAdmin(String userName, String pw, String groupName, String attrname, String attrval) 
  {
    return removeAdmin(userName, pw, groupName, attrname, new String[]{attrval});
  }

  /**
    * Deletes an Administrator from a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval array of values to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode removeAdmin(String userName, String pw, String groupName, String attrname, String[] attrval) 
  {
    DirContext bp_ctx = null;
    
    ReturnCode rc;
    String authInfo = userName + ":" + pw;
    String gName = URLEncoder.encode(groupName);

    String uri = "gName=" + gName + "&task=DelAdm&";

    try {
        bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

        for (int i = 0; i < attrval.length; i++) {
            if (attrval[i] != null && attrval[i].length() > 0) {
                String UID = new String();    

                if (attrname.equalsIgnoreCase("uid")) {
                    UID = attrval[i];
                } else if (attrname.equalsIgnoreCase("notesemail")) {
                    String temp = "";

                    if (attrval[i].indexOf("OU=") < 1) {
                        temp = "CN=" + attrval[i].substring(0, attrval[i].indexOf("/") + 1);

                        if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) > -1) {
                            temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1);

                            if (attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) != attrval[i].lastIndexOf("/")) {
                                temp = temp + "OU=" + attrval[i].substring(attrval[i].indexOf("/", attrval[i].indexOf("/") + 1) + 1, attrval[i].lastIndexOf("/") + 1);
                            }
                            temp = temp + "O=" + attrval[i].substring(attrval[i].lastIndexOf("/") + 1, attrval[i].length());
                        } else {
                            temp = temp + "O=" + attrval[i].substring(attrval[i].indexOf("/") + 1, attrval[i].length());
                        }

                        UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + temp + "*)", "uid");
                    }
                } else {
                    UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + attrval[i] + ")", "uid");
                }

                if (UID.length() < 1) {
                    return cwaapi.NO_RECORD_ERROR;
                }

                UID = URLEncoder.encode(UID);    

                uri = uri + "mebox=" + UID + "&";
            }
        }
    } catch(NamingException ne) {
        System.err.println("[cwa2] removeAdmin: " + ne);
        ne.printStackTrace();
        return cwaapi.LDAP_ERROR;
    } finally {
        cleanupCtx(bp_ctx);
    }

    uri = uri + "&API=1";
    rc = executeURL(authInfo, uri);   

    return rc;
  }

  /**
    * Changes an owner of a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param attrname attribute to create the filter with such as mail
    * @param attrval value to search for such as jondoe@us.ibm.com
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode changeOwner(String userName, String pw, String groupName, String attrname, String attrval) {
      DirContext bp_ctx = null;

      ReturnCode rc;

      String authInfo = userName + ":" + pw;
      String gName = URLEncoder.encode(groupName);

      //there is a better way to handle this ...
      //we should use better Rc's or something
      String UID = new String();    

      try {
          bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

          if (attrname.equalsIgnoreCase("uid")) {
              UID = attrval;
          } else if (attrname.equalsIgnoreCase("notesemail")) {
              String temp = "";

              if (attrval.indexOf("OU=") < 1) {
                  temp = "CN=" + attrval.substring(0, attrval.indexOf("/") + 1);

                  if (attrval.indexOf("/", attrval.indexOf("/") + 1) > -1) {
                      temp = temp + "OU=" + attrval.substring(attrval.indexOf("/") + 1, attrval.indexOf("/", attrval.indexOf("/") + 1) + 1);

                      if (attrval.indexOf("/", attrval.indexOf("/") + 1) != attrval.lastIndexOf("/")) {
                          temp = temp + "OU=" + attrval.substring(attrval.indexOf("/", attrval.indexOf("/") + 1) + 1, attrval.lastIndexOf("/") + 1);
                      }
                      temp = temp + "O=" + attrval.substring(attrval.lastIndexOf("/") + 1, attrval.length());
                  } else {
                      temp = temp + "O=" + attrval.substring(attrval.indexOf("/") + 1, attrval.length());
                  }

                  UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + temp + "*)", "uid");
              }
          } else {
              UID = getAttrByFilter(bp_ctx, "(" + attrname + "=" + attrval + ")", "uid");
          }

          if (UID.length() < 1) {
              return cwaapi.NO_RECORD_ERROR;
          }

          UID = URLEncoder.encode(UID);    
      } catch(NamingException ne) {
          System.err.println("[cwa2] changeOwner: " + ne);
          ne.printStackTrace();
          return cwaapi.LDAP_ERROR;
      } finally {
          cleanupCtx(bp_ctx);
      }
      rc = executeURL(authInfo, "gName=" + gName + "&task=GoCO&mebox=" + UID + "&API=1");   

      return rc;
  }

  /**
    * Changes Filter of a group. These attributes are ANDed together
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param newGroupName New Group name, leave null otherwise
    * @param description New description, leave null otherwise
    * @param viewAccess Use either cwa2.EVERYONE or cwa2.OWN_ADMIN
    * @param y Year to expire
    * @param m Month to expire
    * @param d Day to expire
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode changeFilter(String userName, String pw, String groupName, 
                                 String callupname, 
                                 String employeetype, 
                                 String notesemail, 
                                 String mail, 
                                 String ismanager, 
                                 String employeeserialnumber,
                                 String workloc,
                                 String employeecountrycode,
                                 String directoryalias,
                                 String dept,
                                 String div) {
      ReturnCode rc;      
      
      String authInfo = userName + ":" + pw;
      String gName = URLEncoder.encode(groupName);

      String query = "gName=" + gName + "&task=GoCf&Save=Build";

      if (callupname != null && callupname.length() > 0) {
          query = query + "&callupname=" + URLEncoder.encode(callupname);
      }

      if (employeetype != null && employeetype.length() > 0) {
          query = query + "&employeetype=" + URLEncoder.encode(employeetype);
      }

      if (notesemail != null && notesemail.length() > 0) {
          query = query + "&notesemail=" + URLEncoder.encode(notesemail);
      }

      if (mail != null && mail.length() > 0) {
          query = query + "&mail=" + URLEncoder.encode(mail);
      }

      if (ismanager != null && ismanager.length() > 0) {
          query = query + "&ismanager=" + URLEncoder.encode(ismanager);
      }
      
      if (employeeserialnumber != null && employeeserialnumber.length() > 0) {
          query = query + "&employeeserialnumber=" + URLEncoder.encode(employeeserialnumber);
      }

      if (workloc != null && workloc.length() > 0) {
          query = query + "&workloc=" + URLEncoder.encode(workloc);
      }

      if (employeecountrycode != null && employeecountrycode.length() > 0) {
          query = query + "&employeecountrycode=" + URLEncoder.encode(employeecountrycode);
      }

      if (directoryalias != null && directoryalias.length() > 0) {
          query = query + "&directoryalias=" + URLEncoder.encode(directoryalias);
      }

      if (dept != null && dept.length() > 0) {
          query = query + "&dept=" + URLEncoder.encode(dept);
      }

      if (div != null && div.length() > 0) {
          query = query + "&div=" + URLEncoder.encode(div);
      }

      rc = executeURL(authInfo, query + "&API=1");

      return rc;      
  }

  /**
    * Changes Filter of a group. These attributes are ANDed together
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param newGroupName New Group name, leave null otherwise
    * @param description New description, leave null otherwise
    * @param viewAccess Use either cwa2.EVERYONE or cwa2.OWN_ADMIN
    * @param y Year to expire
    * @param m Month to expire
    * @param d Day to expire
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode changeFilter(String userName, String pw, String groupName, String ldapfilter) {
      ReturnCode rc;      
      
      String authInfo = userName + ":" + pw;
      String gName = URLEncoder.encode(groupName);

      String query = "gName=" + gName + "&task=GoCf&Save=ldapfilter";

      if (ldapfilter != null) {
          query = query + "&ldapfilter=" + URLEncoder.encode(ldapfilter);
      }

      rc = executeURL(authInfo, query + "&API=1");

      return rc;
  }

  /**
    * Changes Characteristics of a group.
    * 
    * @param userName User name
    * @param pw Password
    * @param groupName Group to be deleted from.
    * @param newGroupName New Group name, leave null otherwise
    * @param description New description, leave null otherwise
    * @param viewAccess Use either cwa2.EVERYONE or cwa2.OWN_ADMIN
    * @param y Year to expire
    * @param m Month to expire
    * @param d Day to expire
    * 
    * @return int based on return code output from executeURL.
    */
  public ReturnCode changeCharacteristics(String userName, String pw, String groupName, String newGroupName, String description, String viewAccess, int y, int m, int d) {
      ReturnCode rc;      
      
      String authInfo = userName + ":" + pw;
      String gName = URLEncoder.encode(groupName);
      String newgName = newGroupName;
      String desc = description;

      String query = "gName=" + gName + "&task=GoCc";      

      if (newgName != null) {
          newgName = URLEncoder.encode(newgName);
          query = query + "&selectOn=" + newgName;
      }

      if (desc != null) {
          desc = URLEncoder.encode(desc);
          query = query + "&gDesc=" + desc;
      }

      if (viewAccess != null) {
          if (viewAccess.equalsIgnoreCase(cwa2.EVERYONE)) {
              query = query + "&vAcc=" + EVERYONE;
          } else if (viewAccess.equalsIgnoreCase(cwa2.OWN_ADMIN)) {
              query = query + "&vAcc=" + URLEncoder.encode(OWN_ADMIN);
          }
      }

      if (y > 2000 && m > 0 && d > 0) {
          query = query + "&Y=" + y + "&M=" + m + "&D=" + d;
      }

      rc = executeURL(authInfo, query + "&API=1");

      return rc;
  }

  /**
    * Check if the group exists
    * 
    * @param group Name of the Group to check
    * 
    * @return boolean
    */
  public boolean groupExist(String group)
  {
    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    try
    {
      String attrlist[] = {"cn"};
      String filter = "(cn=" + group + ")";
      SearchControls constraints = new SearchControls();
      constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
      constraints.setReturningAttributes(attrlist);
      constraints.setTimeLimit(SEARCHTIMELIMIT);

      bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
      results = bg_ctx.search(METADATA_BASE, filter ,constraints);  

      Enumeration vals;
      if (results.hasMore())
      {
        return true;
      }
      else
      {
        return false;
      }            
    }
    catch (NamingException ne)
    { 
      System.err.println("[cwa2] groupExist: " + ne);
      ne.printStackTrace();
      if (ne.getMessage().equals("java.util.NoSuchElementException"))
      {        
        return false;
      }
      else
        return false;

    } finally {
      cleanupNamingEnum(results);
      cleanupCtx(bg_ctx);
    }
  }

  /**
    * Checks if member is an admin of group 
    *
    * @param email Internet email address to check
    * @param group The group to check
    * @return true or false
    */
  public boolean isGroupAdmin(String uid, String group)
  {    
    DirContext bp_ctx = null;    
    
    String userDN = null;    

    try
    {      
      bp_ctx = getLdapConnection(bluepages_ldap_server, ldap_version);

      if (uid.indexOf("@") > 0) {
          userDN = getDNByFilter(bp_ctx, "(mail=" + uid + ")"); 
      } else {
          userDN = getDNByFilter(bp_ctx, "(uid=" + uid + ")"); 
      }

      //userDN should be like this "uid=\"780507897\",c=us,ou=bluepages,o=ibm.com" 

      if (userDN == null || userDN.length() == 0) {
          return false;
      }
    } catch (NamingException nne){        
        if (!nne.getMessage().equalsIgnoreCase("java.util.NoSuchElementException")) {
            System.err.println("[cwa2] isGroupAdmin: " + nne);
            nne.printStackTrace();
        }        
                
        return false;        
    } finally {
      cleanupCtx(bp_ctx);
    }

    DirContext bg_ctx = null;
    NamingEnumeration results = null;

    try
    {
      String attrlist[] = {"cn"};
      String filter = "(&(cn=" + group + ")(admin=" + substitute(userDN, "\\", "\\5c") + "))";
     
      SearchControls constraints = new SearchControls();
      constraints.setSearchScope(SearchControls.SUBTREE_SCOPE);
      constraints.setReturningAttributes(attrlist);
      constraints.setTimeLimit(SEARCHTIMELIMIT);

      bg_ctx = getLdapConnection(bluegroups_ldap_server, ldap_version);
      results = bg_ctx.search(METADATA_BASE, filter ,constraints);  
        
      Enumeration vals;
      if (results.hasMore())
      {
        return true;
      }
      else
      {
        return false;
      }      
    }
    catch (NamingException ne)
    {      
      if (ne.getMessage().equals("java.util.NoSuchElementException"))
      { 
        System.err.println("[cwa2] isGroupAdmin: " + ne);
        ne.printStackTrace();
        return false;
      }
      else
        return false;

    } finally {
      cleanupNamingEnum(results);
      cleanupCtx(bg_ctx);
    }
  }


}
