/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2005, 2009. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
dojo.provide("com.ibm.team.apt.internal.ui.quickquery.QuickQueryParser"); //$NON-NLS-1$

dojo.require("com.ibm.team.apt.client.PlanItem"); //$NON-NLS-1$
dojo.require("com.ibm.team.apt.client.PlanningAttributeType"); //$NON-NLS-1$

dojo.require("com.ibm.team.apt.internal.ui.quickquery.Term"); //$NON-NLS-1$
dojo.require("com.ibm.team.apt.internal.ui.quickquery.ProposalMatcher"); //$NON-NLS-1$

dojo.require("dojo.string"); //$NON-NLS-1$
dojo.require("dojo.i18n"); //$NON-NLS-1$
dojo.requireLocalization("com.ibm.team.apt.internal.ui", "QuickQueryMessages"); //$NON-NLS-1$ //$NON-NLS-2$

(function() {

var PlanItem						= com.ibm.team.apt.client.PlanItem;
var PlanningAttributeType			= com.ibm.team.apt.client.PlanningAttributeType;
	
var Term							= com.ibm.team.apt.internal.ui.quickquery.Term;
var ProposalMatcher					= com.ibm.team.apt.internal.ui.quickquery.ProposalMatcher;

// QuickQueryProposal is not yet available in the Web UI
var QuickQueryProposal 				= dojo.getObject("com.ibm.team.apt.internal.ui.quickquery.QuickQueryProposal"); //$NON-NLS-1$

var bind= dojo.string.substitute;
var Messages= dojo.i18n.getLocalization("com.ibm.team.apt.internal.ui", "QuickQueryMessages"); //$NON-NLS-1$ //$NON-NLS-2$

function ParsedCommon(type, input, start, end) {
	this.type= type;
	this.start= start;
	this.end= end;
	this.text= input.substring(start, end);
}

function ParsedError(errorMessage, input, start, end) {
	ParsedCommon.call(this, ParsedError, input, start, end);
	this.errorMessage= errorMessage;
}

function ParsedAttribute(attribute, operator, parameter, negate, input, start, end) {
	ParsedCommon.call(this, ParsedAttribute, input, start, end);
	this.attribute= attribute;
	this.operator= operator;
	this.parameter= parameter;
	this.negate= negate;
}

function ParsedKeyword(keyword, negate, input, start, end) {
	ParsedCommon.call(this, ParsedKeyword, input, start, end);
	this.keyword= keyword;
	this.negate= negate;
}

function __buildCharMap(characters, baseMap) {
	var result= {};
	if (baseMap) {
		dojo.mixin(result, baseMap);
	}
	for (var i = 0; i < characters.length; i++) {
		result[characters.charAt(i)]= true;
	}
	return result;
}
// JS does not offer locale dependent whitespace and character information
var whitespaceMap= __buildCharMap(" \f\n\r\t\u00A0\u2028\u2029"); //$NON-NLS-1$
var stopCharacterMap= __buildCharMap(':~=<>"', whitespaceMap); //$NON-NLS-1$

var IDENTIFIER= /[a-zA-Z_][\w]*/;
var OPERATORS= {
	DEFAULT: ":", //$NON-NLS-1$
	SUBTREE: "~", //$NON-NLS-1$
	EQ: "=", //$NON-NLS-1$
	LT: "<", //$NON-NLS-1$
	GT: ">" //$NON-NLS-1$
};

function skipWhitespace(chars) {
	var result= false;
	while (chars.hasNext()) {
		if (!whitespaceMap[chars.next()]) {
			chars.pushback();
			break;
		}
		result= true;
	}
	return result;
}

function parseWord(chars) {
	var start= chars.getIndex();
	while (chars.hasNext()) {
		if (stopCharacterMap[chars.next()]) {
			chars.pushback();
			break;
		}
	}

	return start != chars.getIndex() ? chars.substring(start) : null;
}

function parseParameter(chars) {
	var start= chars.getIndex();
	while (chars.hasNext()) {
		if (whitespaceMap[chars.next()]) {
			chars.pushback();
			break;
		}
	}
	
	return start != chars.getIndex() ? chars.substring(start) : null;
}

function parseString(chars) {
	if (!chars.hasNext())
		return null;

	var c= chars.next();
	if (c != '"') { //$NON-NLS-1$
		chars.pushback(); 
		return null;
	}

	var result= ""; //$NON-NLS-1$
	var foundEnd= false;
	while (chars.hasNext()) {
		c= chars.next();

		if (c == '"') { //$NON-NLS-1$
			foundEnd= true;
			break;
		} else if (c == '\\') { //$NON-NLS-1$
			if (!chars.hasNext())
				return null;
			switch (chars.next()) {
				case '\\': //$NON-NLS-1$
					result+= '\\'; //$NON-NLS-1$
					break;
				case '"': //$NON-NLS-1$
					result+= '"'; //$NON-NLS-1$
					break;
				case "'": //$NON-NLS-1$
					result+= "'"; //$NON-NLS-1$
					break;
				case 'r': //$NON-NLS-1$
					result+= '\r'; //$NON-NLS-1$
					break;
				case 'n': //$NON-NLS-1$
					result+= '\n'; //$NON-NLS-1$
					break;
				case 't': //$NON-NLS-1$
					result+= '\t'; //$NON-NLS-1$
					break;
				default:
					// error, illegal escape sequence
			}
		} else {
			result+= c;
		}
	}

	return foundEnd ? result : null;
}

function doParse(input) {
	result= [];
	chars= new com.ibm.team.apt.internal.ui.quickquery.QuickQueryParser._CharSequenceIterator(input);

	var negate= false;
	var posElementStart= -1;
	var maxLoops= input.length;

	while (chars.hasNext()) {
		if (maxLoops-- == 0) {
			// looks like we are stuck somewhere in the input -> give up and ignore the rest of the input
			break;
		}

		if (skipWhitespace(chars))
			continue;

		if (posElementStart == -1)
			posElementStart= chars.getIndex();

		var c= chars.next();
		if (c == '!' || c == '-' || c == '+') { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			negate= c == '-' || c == '!'; //$NON-NLS-1$ //$NON-NLS-2$
			continue;
		} else {
			chars.pushback();
		}

		var foundOperation= false;
		var posOperationNameStart= chars.getIndex();
		var keyword= parseWord(chars);
		if (keyword != null && IDENTIFIER.test(keyword) && chars.hasNext()) {
			var posOperationNameEnd= chars.getIndex();

			var operator= null;
			var operatorChar= chars.next();
			for (var operatorId in OPERATORS) {
				if (OPERATORS[operatorId] == operatorChar) {
					operator= OPERATORS[operatorId];
					break;
				}
			}
			
			if (operator != null) {
				var posAfterColon= chars.getIndex();
				var parameter= parseString(chars);
				if (parameter == null) {
					chars.reset(posAfterColon);
					parameter= parseParameter(chars);
					if (parameter == null) {
						// error, parameter expected
						result.push(new ParsedError(Messages['quickQuery_failureParameterExpected'], input, posAfterColon, chars.getIndex())); //$NON-NLS-1$
					}
				}

				var parseResult= new ParsedAttribute(keyword, operator, parameter, negate, input, posOperationNameStart, chars.getIndex());
				parseResult.attributeStart= posElementStart;
				parseResult.attributeEnd= posOperationNameEnd;
				parseResult.parameterStart= posAfterColon;
				parseResult.parameterEnd= chars.getIndex();
				result.push(parseResult);
				foundOperation= true;
			} else {
				chars.pushback();
			}
		} else if (keyword == null) {
			chars.reset(posOperationNameStart);
			keyword= parseString(chars);
		}

		if (!foundOperation) {
			if (keyword != null) {
				result.push(new ParsedKeyword(keyword, negate, input, posOperationNameStart, chars.getIndex()));
			} else if (chars.hasNext()) {
				// error, wasn't able to consume char -> swallow it to not be stuck 
				chars.next();
				result.push(new ParsedError(Messages['quickQuery_failureInvalidInput'], input, posOperationNameStart, chars.getIndex())); //$NON-NLS-1$
			}
		}
		
		negate= false;
		posElementStart= -1;

	}
	return result;
}

dojo.declare("com.ibm.team.apt.internal.ui.quickquery.QuickQueryParser", null, { //$NON-NLS-1$

	__quickQueryDefinition: null,
	
	constructor: function(quickQueryDefinition) {
		this.__quickQueryDefinition= quickQueryDefinition;
	},
	
	// ---- API ------------------------------------------------------------------------------------------------------------

	parse: function(input) {
		return doParse(input);
	},

	getExpression: function(input) {
		var result= new Term(Term.AND);
		var hasExpressions= false;

		var parsed= doParse(input);
		var expressionByAttribute= {};
		
		for (var i= 0; i < parsed.length; i++) {
			var element= parsed[i];
			if (element.type == ParsedKeyword) {
				var expression= this.__quickQueryDefinition.getWordExpression(element.keyword);
				result.add(element.negate ? new Term(Term.NOT, expression) : expression);
				hasExpressions= true;
			} else if (element.type == ParsedAttribute) {
				var expression= this.__quickQueryDefinition.getAttributeExpression(element.attribute, element.operator, element.parameter);
				if (expression != null) {
					if (element.negate)
						expression= new Term(Term.NOT, expression);
					
					var expressions= expressionByAttribute[element.attribute];
					if (!expressions) {
						expressions= [];
						expressionByAttribute[element.attribute]= expressions;
					}
					expressions.push(expression);
				} else {
					// ignore unknown attributes for now...
				}
			} else {
				// ignore parse errors for now..
			}
		}

		for (var attributeId in expressionByAttribute) {
			var expressionRoot= result;
			if (this.__quickQueryDefinition.getAttributeCombination(attributeId) == Term.OR) {
				expressionRoot= new Term(Term.OR);
				result.add(expressionRoot);
			}
			
			dojo.forEach(expressionByAttribute[attributeId], function(expression) {
				expressionRoot.add(expression);
			});
			hasExpressions= true;
		}

		return hasExpressions ? result : null;
	},
	
	getContentProposals: function(input, position) {
		var parsed= doParse(input);

		var currentPredicate= null;
		for (var i= 0; i < parsed.length; i++) {
			var element= parsed[i];
			if (element.start < position && element.end >= position) {
				currentPredicate= element;
				break;
			}
		}

		var result= [];
		
		if (currentPredicate != null) {
			var attributeStart= currentPredicate.start;
			var attributeEnd= currentPredicate.end;

			if (currentPredicate instanceof ParsedAttribute) {
				attributeStart= currentPredicate.attributeStart;
				attributeEnd= currentPredicate.attributeEnd;
			}

			if (position <= attributeEnd) {
				var attributeMatcher= new ProposalMatcher(input.substring(attributeStart, position));
				this.__withQueryableAttributes(function(attribute) {
					if (attributeMatcher.matches(attribute.getQueryName())) {
						result.push(QuickQueryProposal.createAttributeProposal(attribute, OPERATORS.DEFAULT, currentPredicate.start, currentPredicate.end));
					}

					// match attributes and auto complete attribute + parameter: "hi" -> "priority:High"
					var candidateParameters= this.__calculateParameterProposals(attribute);
					for (var i= 0; i < candidateParameters.length; i++) {
						var candidateParameter= candidateParameters[i];
						if (attributeMatcher.matches(candidateParameter.value)) {
							result.push(QuickQueryProposal.createCompleteProposal(attribute, OPERATORS.DEFAULT, candidateParameter.value, currentPredicate.start, currentPredicate.end));
						}
					}
				});
			} else if (currentPredicate instanceof ParsedAttribute) {
				this.__withQueryableAttributes(function(attribute) {
					if (attribute.getQueryName().equals(currentPredicate.attribute)) {
						var candidateParameters= this.__calculateParameterProposals(attribute);

						var prefix= null;
						if (position > currentPredicate.parameterStart)
							prefix= input.substring(currentPredicate.parameterStart, position);

						var parameterMatcher= new ProposalMatcher(prefix);
						for (var j= 0; j < candidateParameters.length; j++) {
							var candidateParameter= candidateParameters[j];
							if (parameterMatcher.matches(candidateParameter.value)) {
								result.push(QuickQueryProposal.createParameterProposal(attribute, candidateParameter.value, currentPredicate.parameterStart, currentPredicate.end));
							}
						}
					}
				});
			}
		} else {
			this.__withQueryableAttributes(function(attribute) {
				result.push(QuickQueryProposal.createAttributeProposal(attribute, OPERATORS.DEFAULT, position, position));
			});
		}

		
		return result;
	},
	
	// ---- implementation -------------------------------------------------------------------------------------------------
	
	__calculateParameterProposals: function(attribute) {
		var result= [];

		var plan= this.__quickQueryDefinition.getPlan();
		switch (attribute.attributeType) {
			case PlanningAttributeType.TAGS:
				dojo.forEach(attribute.getValueSet().allValues, function(value) {
					result.push({value : value});
				});
				break;

			case PlanningAttributeType.ITEM:
			case PlanningAttributeType.WORKITEM_TYPE:
			case PlanningAttributeType.ENUMERATION:
				dojo.forEach(attribute.getValueSet().allValues, function(value) {
					result.push({value : value.label});
				});
				break;

			case PlanningAttributeType.WORKFLOW_STATE: {
				var seenWorkflows= {};
				var stateNames= {};
				plan.accept(function(planElement) {
					if (planElement instanceof PlanItem) {
						var workflowInfo=planElement.getWorkflowInfo();
						if (workflowInfo != null && !seenWorkflows[workflowInfo.getId()]) {
							dojo.forEach(workflowInfo.getStateIds(), function(stateId) {
								var stateName= workflowInfo.getStateLabel(stateId);
								if (!stateNames[stateName]) {
									result.push({value : stateName });
									stateNames[stateName]= true;
								}
							});
							seenWorkflows[workflowInfo.getId()]= true;
						}
					}
					return true;
				}, this);
				break;
			}

			case PlanningAttributeType.BOOLEAN:
				result.push({value : "true"}); //$NON-NLS-1$
				result.push({value : "false"}); //$NON-NLS-1$
				break;
				
			case PlanningAttributeType.DURATION:
				result.push({value : "1"}); //$NON-NLS-1$
				result.push({value : "set"}); //$NON-NLS-1$
				result.push({value : "unset"}); //$NON-NLS-1$
				break;

			case PlanningAttributeType.CHECKERREPORT: {
				var planChecks= plan.getPlanChecks();
				var shortNames= {};
				for (var i= 0; i < planChecks.length; i++) {
					var problemDefinitions= planChecks[i].problemDefinitions;
					for (var problemId in problemDefinitions) {
						var shortName= problemDefinitions[problemId].shortName;
						if (shortName && !shortNames[shortName]) {
							result.push({value : shortName});
							shortNames[shortName]= true;
						}
					}
				}
				break;
			}
		}

		return result;
	},
	
	__withQueryableAttributes: function(code) {
		var allAttributes= this.__quickQueryDefinition.getAllAttributes();
		for (var i= 0; i < allAttributes.length; i++) {
			if (allAttributes[i].getQueryName()) {
				code.call(this, allAttributes[i]);
			}
		}
	},
	
	__sentinel: null // terminates this class definition
});

dojo.declare("com.ibm.team.apt.internal.ui.quickquery.QuickQueryParser._CharSequenceIterator", null, { //$NON-NLS-1$

	constructor: function(input) {
		this.__input= input;
		this.__length= input.length;
		this.__index= 0;
	},
	
	// ---- API ------------------------------------------------------------------------------------------------------------

	substring: function(start, end) {
		if (arguments.length > 1) {
			return this.__input.substring(start, end);
		} else {
			return this.__input.substring(start, this.__index);
		}
	},

	getIndex: function() {
		return this.__index;
	},

	hasNext: function() {
		return this.__index < this.__length;
	},

	next: function() {
		if (this.__index >= this.__length)
			throw new Error("StringOutOfBoundsException"); //$NON-NLS-1$

		return this.__input.charAt(this.__index++);
	},

	pushback: function() {
		if (this.__index == 0)
			throw new Error("StringOutOfBoundsException"); //$NON-NLS-1$

		this.__index--;
	},

	reset: function(position) {
		if (position < 0 || position > this.__length)
			throw new Error("StringOutOfBoundsException"); //$NON-NLS-1$

		this.__index= position;
	},

	__sentinel: null // terminates this class definition
});

})();

