/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2005, 2009. All Rights Reserved.
 * 
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *******************************************************************************/
dojo.provide("com.ibm.team.apt.internal.ui.quickquery.AttributeCondition"); //$NON-NLS-1$

dojo.require("com.ibm.team.apt.client.PlanningAttributeType"); //$NON-NLS-1$

(function() {

var PlanningAttributeType			= com.ibm.team.apt.client.PlanningAttributeType;

var OP_DEFAULT= ":"; //$NON-NLS-1$
var OP_SUBTREE= "~"; //$NON-NLS-1$
var OP_EQ= "="; //$NON-NLS-1$
var OP_LT= "<"; //$NON-NLS-1$
var OP_GT= ">"; //$NON-NLS-1$

dojo.declare("com.ibm.team.apt.internal.ui.quickquery.AttributeCondition", null, { //$NON-NLS-1$

	constructor: function(attribute, operator, value, plan) {
		this.__attribute= attribute;
		this.__operator= operator;
		this.__value= this.__convertValue(value, plan);
		this.__evaluateFn= this.__getEvaluateFn();
	},
	
	// ---- API ------------------------------------------------------------------------------------------------------------

	evaluate: function(planItem) {
		if (this.__value == null)
			return false;
		
		var planItemValue= planItem.getAttributeValue(this.__attribute);
		return planItemValue != undefined ? this.__evaluateFn(planItemValue, planItem) : false;
	},

	// ---- implementation -------------------------------------------------------------------------------------------------
	
	__evaluateString: function(planItemValue) {
		return planItemValue.toLowerCase().indexOf(this.__value) != -1;
	},

	__evaluateTags: function(planItemValue) {
		return dojo.some(planItemValue, function(tag) {
			return tag.toLowerCase() == this.__value;
		}, this);
	},
	
	__evaluateNumber: function(planItemValue) {
		switch (this.__operator) {
			case OP_DEFAULT:
			case OP_EQ:
				return planItemValue == this.__value;
			case OP_LT:
				return planItemValue <= this.__value;
			case OP_GT:
				return planItemValue >= this.__value;
		}
		
		return false;
	},

	__evaluateBoolean: function(planItemValue) {
		return planItemValue == this.__value;
	},
	
	__evaluateInstant: function(planItemValue) {
		return false;
	},
	
	__evaluateDuration: function(planItemValue) {
		if (this.__value == -1)
			return !planItemValue.specified;
		
		if (this.__value == -2)
			return planItemValue.specified;

		if (!isNaN(this.__value) && planItemValue.specified) {
			var ms= planItemValue.getMilliseconds();
			switch (this.__operator) {
				case OP_DEFAULT:
				case OP_EQ:
					return ms == this.__value;
				case OP_LT:
					return ms < this.__value;
				case OP_GT:
					return ms > this.__value;
			}
		}

		return false;
	},

	__evaluateItem: function(planItemValue) {
		return planItemValue.label.toLowerCase() == this.__value;
	},

	__evaluateState: function(planItemValue, planItem) {
		var workflowInfo= planItem.getWorkflowInfo();
		if (workflowInfo != null) {
			var stateName= workflowInfo.getStateLabel(planItemValue.getId());
			if (stateName != null) {
				return stateName.toLowerCase() == this.__value;
			}
		}
		return false;
	},

	__evaluateEnumeration: function(planItemValue) {
		var compared= this.__value.compareTo(planItemValue);
		if (this.__operator == OP_DEFAULT || this.__operator == OP_EQ) {
			return compared == 0;
		} else if (this.__operator == OP_LT) {
			return compared > 0;
		} else if (this.__operator == OP_GT) {
			return compared < 0;
		}
		return false;
	},
	
	__evaluateProblemReport: function(planItemValue) {
		var length= this.__value.length;
		for (var i= 0; i < length; i++) {
			if (planItemValue.getProblem(this.__value[i]) != null)
				return true;
		}
		
		return false;
	},

	__evaluateFalse: function(planItemValue) {
		return false;
	},

	__getEvaluateFn: function() {
		switch (this.__attribute.attributeType) {
			case PlanningAttributeType.STRING:
			case PlanningAttributeType.HTML:
				return this.__evaluateString;

			case PlanningAttributeType.TAGS:
				return this.__evaluateTags;

			case PlanningAttributeType.INTEGER:
			case PlanningAttributeType.FLOAT:
				return this.__evaluateNumber;
				
			case PlanningAttributeType.BOOLEAN:
				return this.__evaluateBoolean;

			case PlanningAttributeType.INSTANT:
				return this.__evaluateInstant;
				
			case PlanningAttributeType.DURATION:
				return this.__evaluateDuration;
				
			case PlanningAttributeType.ITEM:
			case PlanningAttributeType.WORKITEM_TYPE:
				return this.__evaluateItem;
			case PlanningAttributeType.WORKFLOW_STATE:
				return this.__evaluateState;

			case PlanningAttributeType.ENUMERATION:
				return this.__evaluateEnumeration;

			case PlanningAttributeType.CHECKERREPORT:
				return this.__evaluateProblemReport;

			default:
				return this.__evaluateFalse;
		}
	},

	__convertValue: function(value, plan) {
		if (value == null)
			return null;
		
		switch (this.__attribute.attributeType) {
			case PlanningAttributeType.STRING:
			case PlanningAttributeType.HTML:
			case PlanningAttributeType.TAGS:
				return value.toLowerCase();
			
			case PlanningAttributeType.INTEGER:
			case PlanningAttributeType.FLOAT:
				return Number(value);
				
			case PlanningAttributeType.BOOLEAN:
				return value.toLowerCase() == "true"; //$NON-NLS-1$
				
			case PlanningAttributeType.INSTANT:
				// TODO
				return value;
			case PlanningAttributeType.DURATION:
				switch (value.toLowerCase()) {
					case "set": //$NON-NLS-1$
						return -2;
					case "unset": //$NON-NLS-1$
						return -1;
					default:
						return Number(value) * 60 * 60 * 1000;
				}
				
			case PlanningAttributeType.ITEM:
			case PlanningAttributeType.WORKITEM_TYPE:
				return value.toLowerCase();
			case PlanningAttributeType.WORKFLOW_STATE:
				return value.toLowerCase();
			case PlanningAttributeType.ENUMERATION: {
				var valueSet= this.__attribute.getValueSet().values;
				for (var i= 0; i < valueSet.length; i++) {
					if (valueSet[i].label.toLowerCase() == value.toLowerCase()) {
						return valueSet[i];
					}
				}
				return null;
			}
			
			case PlanningAttributeType.CHECKERREPORT: {
				var planChecks= plan.getPlanChecks();
				var problemIds= [];
				for (var i= 0; i < planChecks.length; i++) {
					var problemDefinitions= planChecks[i].problemDefinitions;
					for (var problemId in problemDefinitions) {
						var problemDefinition= problemDefinitions[problemId];
						if (problemDefinition.shortName && problemDefinition.shortName.toLowerCase() == value.toLowerCase()) {
							problemIds.push(problemId);
						}
					}
				}
				
				return problemIds;
			}
		}

		return value;
	},

	equals: function(other) {
		if (this === other)
			return true;

		if (this.declaredClass !== other.declaredClass)
			return false;

		if (this.__attribute.equals(other.__attribute))
			return false;
		
		if (this.__operator != other.__operator)
			return false;

		if (this.__evaluateFn != other.__evaluateFn)
			return false;

		if (this.__attribute.attributeType == PlanningAttributeType.ENUMERATION) {
			return this.__value.label == other.__value.label;
		} else {
			return this.__value == other.__value;
		}
	},

	__sentinel: null // terminates this class definition
});

})();
